<?php
if (!defined('ABSPATH')) {
    exit;
}

class Crsh_Public {

    public $last_error = '';

    // No nonce needed for frontend search — removed entirely

    public function __construct() {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_shortcode('crashguru_search', [$this, 'render_form']);
        add_action('template_redirect', [$this, 'handle_search_redirect']);
        add_action('init', [$this, 'add_sku_search_filters']);
        add_filter('template_include', [$this, 'override_search_template'], 99);
    }

    public function enqueue_scripts() {
        $width  = intval(get_option('crsh_input_width', 300));
        $height = intval(get_option('crsh_input_height', 40));
        $width  = max(100, min(1200, $width));
        $height = max(20, min(200, $height));
        $icon_size = intval($height * 0.75);

        wp_enqueue_style(
            'crsh-public-style',
            plugin_dir_url(__FILE__) . '../assets/css/search.css',
            [],
            defined('CRSH_VERSION') ? CRSH_VERSION : '1.0.0'
        );

        // Dynamically inject CSS (now properly enqueued via inline style)
        wp_add_inline_style('crsh-public-style', $this->generate_custom_css($width, $height, $icon_size));

        if (is_search() && class_exists('WooCommerce')) {
            wp_enqueue_style('woocommerce-general');
            wp_enqueue_style('woocommerce-layout');
            wp_enqueue_style('woocommerce-smallscreen');
        }

        wp_enqueue_script(
            'crsh-public-script',
            plugin_dir_url(__FILE__) . '../assets/js/search.js',
            [],
            defined('CRSH_VERSION') ? CRSH_VERSION : '1.0.0',
            true
        );
    }

    private function generate_custom_css($width, $height, $icon_size) {
        // Escape all values for safe CSS output
        $safe_width     = absint($width);
        $safe_height    = absint($height);
        $safe_icon_size = absint($icon_size);

        return "
        .ai-search-container {
            margin-bottom:20px;
            text-align:center;
        }
        .ai-search-container input[type=\"text\"] {
            width: {$safe_width}px;
            height: {$safe_height}px;
            padding:10px;
            border:2px solid #ddd;
            border-radius:4px;
            font-size:16px;
            transition:all .3s ease;
            box-sizing:border-box;
        }
        .ai-search-container button {
            background: transparent !important;
            border: none !important;
            padding: 0 !important;
            margin-left: 8px !important;
            width: auto !important;
            height: auto !important;
            line-height: 1 !important;
            font-size: {$safe_icon_size}px !important;
            color: #333 !important;
            box-shadow: none !important;
            vertical-align: middle !important;
            cursor: pointer;
        }
        .ai-search-container button:hover,
        .ai-search-container button:focus {
            color: #0073aa !important;
            outline: none !important;
        }
        .ai-search-container button img,
        .ai-search-container button:before,
        .ai-search-container button:after {
            border: none !important;
            box-shadow: none !important;
            background: none !important;
        }
        .ai-search-spinner {
            display:inline-block;
            width: {$safe_icon_size}px;
            height: {$safe_icon_size}px;
            border: 2px solid #ccc;
            border-top: 2px solid #0073aa;
            border-radius: 50%;
            animation: ai-spin 0.6s linear infinite;
            vertical-align: middle;
        }
        @keyframes ai-spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        .ai-search-mixed-results {
            margin-left:40px;
            margin-right:20px;
            display:flex;flex-wrap:wrap;gap:20px;
        }
        .page-header { margin-left:40px; }
        .ai-search-mixed-results .search-result-product {
            flex:0 0 calc(33.333% - 14px);
            list-style:none;border:1px solid #ddd;border-radius:4px;
        }
        .ai-search-mixed-results .search-result-post {
            flex:0 0 100%;
            border-bottom:1px solid #eee;padding-bottom:20px;margin-bottom:20px;
        }
        @media(max-width:768px){.ai-search-mixed-results .search-result-product{flex:0 0 calc(50% - 10px);}}
        @media(max-width:480px){.ai-search-mixed-results .search-result-product{flex:0 0 100%;}}
        ";
    }

    public function render_form($atts) {
        if (!class_exists('WooCommerce')) {
            return '<div class="woocommerce-error">' . esc_html__('AI Search requires WooCommerce', 'smarter-search-products-posts') . '</div>';
        }

        $atts = shortcode_atts(['placeholder' => ''], $atts, 'crashguru_search');

        $search_for = get_option('crsh_products_only', 'yes') === 'yes' ? 'products_only' : 'mixed';
        $default_placeholder = ($search_for === 'products_only')
            ? __('Search products…', 'smarter-search-products-posts')
            : __('Search…', 'smarter-search-products-posts');

        $placeholder = !empty($atts['placeholder']) ? sanitize_text_field($atts['placeholder']) : $default_placeholder;

        ob_start(); ?>
        <div class="ai-search-container">
            <form id="ai-search-form" method="get" action="<?php echo esc_url(home_url('/')); ?>">
                <input type="text" name="s" placeholder="<?php echo esc_attr($placeholder); ?>" required maxlength="200">
                <?php if ($search_for === 'products_only'): ?>
                    <input type="hidden" name="post_type" value="product">
                <?php endif; ?>
                <input type="hidden" name="crsh_search" value="1">
                <button type="submit" aria-label="<?php esc_attr_e('Search', 'smarter-search-products-posts'); ?>">
                    <span class="ai-search-icon">🔍</span>
                    <span class="ai-search-spinner" style="display:none"></span>
                </button>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }

    public function override_search_template($template) {
        if (is_search() && !is_admin()) {
            global $wp_query;
            $post_types = array_unique(array_map(function($post) {
                return $post->post_type;
            }, $wp_query->posts));

            if (count($post_types) === 1 && $post_types[0] === 'product') {
                $product_template = dirname(__FILE__) . '/../templates/search-results-product.php';
                if (file_exists($product_template)) {
                    return $product_template;
                }
            } elseif (count($post_types) > 1) {
                $mixed_template = dirname(__FILE__) . '/../templates/search-results-mixed.php';
                if (file_exists($mixed_template)) {
                    return $mixed_template;
                }
            }
        }
        return $template;
    }

    // === SKU Search (unchanged logic, safe) ===
    public function add_sku_search_filters() {
        add_filter('posts_join', [$this, 'join_postmeta_for_sku_search'], 10, 2);
        add_filter('posts_where', [$this, 'where_sku_search'], 10, 2);
        add_filter('posts_distinct', [$this, 'distinct_sku_search'], 10, 2);
    }

    private function is_product_search($query) {
        if (!$query->is_search()) return false;
        $post_type = $query->get('post_type');
        return $post_type === 'product' || (is_array($post_type) && in_array('product', $post_type, true));
    }

    public function join_postmeta_for_sku_search($join, $query) {
        if (is_admin() || !$this->is_product_search($query)) return $join;
        global $wpdb;
        if (strpos($join, 'crsh_sku_pm') === false) {
            $join .= " LEFT JOIN {$wpdb->postmeta} AS crsh_sku_pm ON ({$wpdb->posts}.ID = crsh_sku_pm.post_id) ";
        }
        return $join;
    }

    public function where_sku_search($where, $query) {
        if (is_admin() || !$this->is_product_search($query)) return $where;
        global $wpdb;
        $s = sanitize_text_field(wp_unslash($query->get('s')));
        if (empty($s)) return $where;
        $like = '%' . $wpdb->esc_like($s) . '%';
        $where .= $wpdb->prepare(
            " OR ( crsh_sku_pm.meta_key = %s AND crsh_sku_pm.meta_value LIKE %s ) ",
            '_sku',
            $like
        );
        return $where;
    }

    public function distinct_sku_search($distinct, $query) {
        if (is_admin() || !$this->is_product_search($query)) return $distinct;
        return "DISTINCT";
    }

    // === MAIN SEARCH HANDLER (no nonce) ===
    public function handle_search_redirect() {
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (!isset($_GET['crsh_search']) || $_GET['crsh_search'] !== '1') {
        return;
    }
    if (!is_search()) {
        return;
    }

        $search_query = get_search_query(false);
        if (empty($search_query) || strlen($search_query) > 200) {
            return;
        }

        $search_for = get_option('crsh_products_only', 'yes') === 'yes' ? 'products_only' : 'mixed';

        if ($this->is_exact_sku($search_query)) {
            $this->redirect_to_sku_search($search_query);
            return;
        }

        $this->process_ai_search_and_redirect($search_query);
    }

    private function is_exact_sku($query) {
        if (!class_exists('WooCommerce')) return false;
        $product_id = wc_get_product_id_by_sku(sanitize_text_field($query));
        return $product_id > 0;
    }

    private function redirect_to_sku_search($sku) {
        $product_id = wc_get_product_id_by_sku(sanitize_text_field($sku));
        if ($product_id > 0) {
            wp_safe_redirect(get_permalink($product_id));
            exit;
        }
        wp_safe_redirect(add_query_arg(['s' => sanitize_text_field($sku), 'post_type' => 'product'], home_url('/')));
        exit;
    }

    private function redirect_to_normal_search($query, $post_types = null) {
        $args = ['s' => sanitize_text_field($query)];
        if ($post_types) {
            $args['post_type'] = $post_types;
        }
        wp_safe_redirect(add_query_arg($args, home_url('/')));
        exit;
    }

    private function process_ai_search_and_redirect($query) {
        $api_key     = sanitize_text_field(get_option('crsh_api_key'));
        $model       = sanitize_text_field(get_option('crsh_llm_model', 'llama-3.1-8b-instant'));
        $temperature = max(0, min(2, floatval(get_option('crsh_temperature', 0.7))));
        $products_only = get_option('crsh_products_only', 'yes') === 'yes';

        // Load prompt (fallback to default if missing)
        $base_prompt = get_option('crsh_prompt', '');
        if (empty($base_prompt)) {
            // Reuse logic or define default
            $base_prompt = 'Extract search parameters from query. Return JSON with: name, category, min_price, max_price';
        }

        $critical_rules = "\n\nCRITICAL RULES (ALWAYS FOLLOW):\n1. Response MUST be ONLY raw JSON.\n2. For products: taxonomy=\"product_cat\"\n3. For posts: taxonomy=\"category\"\n4. May return multiple post_types.\n5. Irrelevant → return {}.\n6. NO HTML.";

        $final_prompt = $base_prompt . $critical_rules;

        if (empty($api_key)) {
            $fallback = $products_only ? 'product' : ['product', 'post', 'page'];
            $this->redirect_to_normal_search($query, $fallback);
            return;
        }

        $response = wp_remote_post('https://api.groq.com/openai/v1/chat/completions', [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode([
                'model'       => $model,
                'messages'    => [
                    ['role' => 'system', 'content' => $final_prompt],
                    ['role' => 'user',   'content' => sanitize_text_field(wp_strip_all_tags($query))],
                ],
                'temperature' => $temperature,
                'max_tokens'  => 500,
            ]),
            'timeout'   => 20,
            'sslverify' => true,
        ]);

        if (is_wp_error($response)) {
            $fallback = $products_only ? 'product' : ['product', 'post', 'page'];
            $this->redirect_to_normal_search($query, $fallback);
            return;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE || empty($data['choices'][0]['message']['content'])) {
            $fallback = $products_only ? 'product' : ['product', 'post', 'page'];
            $this->redirect_to_normal_search($query, $fallback);
            return;
        }

        $ai_content = trim(wp_strip_all_tags($data['choices'][0]['message']['content']));
        if (!preg_match('/\{.*\}/s', $ai_content, $matches)) {
            $fallback = $products_only ? 'product' : ['product', 'post', 'page'];
            $this->redirect_to_normal_search($query, $fallback);
            return;
        }

        $json_string = wp_strip_all_tags($matches[0]);
        $params = json_decode($json_string, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $fallback = $products_only ? 'product' : ['product', 'post', 'page'];
            $this->redirect_to_normal_search($query, $fallback);
            return;
        }

        // Handle SKU redirect
        if (!empty($params['sku'])) {
            $this->redirect_to_sku_search(sanitize_text_field(wp_strip_all_tags($params['sku'])));
            return;
        }

        // Build redirect args
        $redirect_args = ['s' => sanitize_text_field(wp_strip_all_tags($query))];

        if ($products_only) {
            $redirect_args['post_type'] = 'product';
        } elseif (!empty($params['post_types']) && is_array($params['post_types'])) {
            $redirect_args['post_type'] = array_map('sanitize_text_field', $params['post_types']);
        } elseif (!empty($params['taxonomy']) && $params['taxonomy'] === 'product_cat') {
            $redirect_args['post_type'] = 'product';
        } elseif (!empty($params['taxonomy']) && $params['taxonomy'] === 'category') {
            $redirect_args['post_type'] = ['post', 'page'];
        } else {
            $redirect_args['post_type'] = ['product', 'post', 'page'];
        }

        if (!empty($params['name'])) {
            $redirect_args['s'] = sanitize_text_field(wp_strip_all_tags($params['name']));
        }

        if (!empty($params['taxonomy']) && $params['taxonomy'] === 'product_cat'
            && !empty($params['category']) && term_exists($params['category'], 'product_cat')) {
            $redirect_args['product_cat'] = sanitize_title($params['category']);
        }

        if (!empty($params['taxonomy']) && $params['taxonomy'] === 'category'
            && !empty($params['category']) && term_exists($params['category'], 'category')) {
            $redirect_args['category_name'] = sanitize_title($params['category']);
        }

        if (!empty($params['min_price']) && is_numeric($params['min_price'])) {
            $redirect_args['min_price'] = max(0, floatval($params['min_price']));
            if (empty($params['post_types'])) $redirect_args['post_type'] = 'product';
        }

        if (!empty($params['max_price']) && is_numeric($params['max_price'])) {
            $redirect_args['max_price'] = max(0, floatval($params['max_price']));
            if (empty($params['post_types'])) $redirect_args['post_type'] = 'product';
        }

        wp_safe_redirect(add_query_arg($redirect_args, home_url('/')));
        exit;
    }
}