<?php
if (!defined('ABSPATH')) {
    exit;
}

class Crsh_Admin {

    public function __construct() {
        add_action('admin_menu', [$this, 'add_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('wp_ajax_crsh_generate_prompt', [$this, 'handle_generate_prompt']);
    }

    public function enqueue_assets($hook) {
        if ('settings_page_crsh' !== $hook) {
            return;
        }

        $plugin_url = plugin_dir_url(dirname(__FILE__));
        $version    = defined('CRSH_VERSION') ? CRSH_VERSION : '1.0.0';

        wp_enqueue_style(
            'crsh-admin-style',
            $plugin_url . 'assets/css/admin.css',
            [],
            $version
        );

        wp_enqueue_script(
            'crsh-admin-script',
            $plugin_url . 'assets/js/admin.js',
            ['jquery'],
            $version,
            true
        );

        wp_localize_script(
            'crsh-admin-script',
            'crshAjax',
            [
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce'   => wp_create_nonce('crsh_admin_nonce'),
            ]
        );
    }

    public function add_menu() {
        add_options_page(
            esc_html__('Smarter Search for WooCommerce Settings', 'smarter-search-products-posts'),
            esc_html__('Smarter Search', 'smarter-search-products-posts'),
            'manage_options',
            'crsh',
            [$this, 'render_page']
        );
    }

    public function register_settings() {
        register_setting('crsh_settings', 'crsh_api_key', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('crsh_settings', 'crsh_llm_model', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('crsh_settings', 'crsh_prompt', ['sanitize_callback' => 'sanitize_textarea_field']);
        register_setting('crsh_settings', 'crsh_temperature', ['sanitize_callback' => [$this, 'sanitize_temperature']]);
        register_setting('crsh_settings', 'crsh_input_width', ['sanitize_callback' => [$this, 'sanitize_width']]);
        register_setting('crsh_settings', 'crsh_input_height', ['sanitize_callback' => [$this, 'sanitize_height']]);
        register_setting('crsh_settings', 'crsh_products_only', ['sanitize_callback' => [$this, 'sanitize_products_only']]);
        register_setting('crsh_settings', 'crsh_relevant_categories', ['sanitize_callback' => 'sanitize_text_field']);

        // Auto-fill relevant categories with PRIMARY categories only
        if (get_option('crsh_relevant_categories', '') === '') {
            $product_cats = get_terms([
                'taxonomy'   => 'product_cat',
                'hide_empty' => true,
                'parent'     => 0, // ← Only top-level categories
                'orderby'    => 'count',
                'order'      => 'DESC',
                'number'     => 20,
            ]);
            if (!is_wp_error($product_cats) && !empty($product_cats)) {
                $slugs = wp_list_pluck($product_cats, 'slug');
                update_option('crsh_relevant_categories', implode(', ', $slugs));
            }
        }

        // Set default prompt if empty
        if (get_option('crsh_prompt', '') === '') {
            update_option('crsh_prompt', $this->generate_shop_specific_prompt());
        }

        add_settings_section('main', esc_html__('API Configuration', 'smarter-search-products-posts'), null, 'crsh');
        add_settings_section('search_scope', esc_html__('Search Scope', 'smarter-search-products-posts'), null, 'crsh');
        add_settings_section('search_input', esc_html__('Search Input Settings', 'smarter-search-products-posts'), null, 'crsh');

        add_settings_field('api_key', esc_html__('Groq API Key', 'smarter-search-products-posts'), [$this, 'field_api_key'], 'crsh', 'main');
        add_settings_field('llm_model', esc_html__('LLM Model', 'smarter-search-products-posts'), [$this, 'field_llm_model'], 'crsh', 'main');
        add_settings_field('prompt', esc_html__('Search Prompt', 'smarter-search-products-posts'), [$this, 'field_prompt'], 'crsh', 'main');
        add_settings_field('temperature', esc_html__('Temperature', 'smarter-search-products-posts'), [$this, 'field_temperature'], 'crsh', 'main');

        add_settings_field('relevant_categories', esc_html__('Relevant Categories (slugs)', 'smarter-search-products-posts'), [$this, 'field_relevant_categories'], 'crsh', 'search_scope');
        add_settings_field('products_only', esc_html__('Search in', 'smarter-search-products-posts'), [$this, 'field_products_only'], 'crsh', 'search_scope');
        add_settings_field('input_width', esc_html__('Input Width (px)', 'smarter-search-products-posts'), [$this, 'field_input_width'], 'crsh', 'search_input');
        add_settings_field('input_height', esc_html__('Input Height (px)', 'smarter-search-products-posts'), [$this, 'field_input_height'], 'crsh', 'search_input');
    }

    public function field_api_key() {
        $api_key = get_option('crsh_api_key');
        echo '<input type="password" name="crsh_api_key" value="' . esc_attr($api_key) . '" class="regular-text">';
        echo '<p class="description">' . wp_kses(
            __('Enter your Groq API key. Get one free at <a href="https://console.groq.com/keys" target="_blank">console.groq.com/keys</a>', 'smarter-search-products-posts'),
            ['a' => ['href' => [], 'target' => []]]
        ) . '</p>';
    }

    public function field_llm_model() {
        $current = get_option('crsh_llm_model', 'llama-3.1-8b-instant');
        $models = [
            'llama-3.1-8b-instant'                            => 'Llama 3.1 8B Instant',
            'meta-llama/llama-4-scout-17b-16e-instruct'       => 'Llama 4 Scout 17B 16E Instruct',
            'openai/gpt-oss-20b'                              => 'GPT-OSS 20B',
            'meta-llama/llama-guard-4-12b'                    => 'Llama Guard 4 12B',
        ];
        echo '<select name="crsh_llm_model">';
        foreach ($models as $value => $label) {
            printf(
                '<option value="%s"%s>%s</option>',
                esc_attr($value),
                selected($value, $current, false),
                esc_html($label)
            );
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__(
            'Choose the AI model for processing search queries. Llama 3.1 8B Instant is recommended for speed and accuracy.',
            'smarter-search-products-posts'
        ) . '</p>';
    }

    public function field_relevant_categories() {
        $value = get_option('crsh_relevant_categories', '');
        echo '<input type="text" name="crsh_relevant_categories" value="' . esc_attr($value) . '" class="regular-text">';
        echo '<p class="description">' .
             esc_html__('Enter a comma-separated list of primary category slugs to use in AI prompts. Example: tagestour, selbstfahrer, walvis-bay. If left empty, the plugin will auto-detect the top 20 primary (top-level) product categories.', 'smarter-search-products-posts') .
             '</p>';
    }

    public function field_prompt() {
        $saved_prompt   = get_option('crsh_prompt', '');
        $example_prompt = $this->generate_shop_specific_prompt();

        echo '<p><strong>' . esc_html__('Instructions:', 'smarter-search-products-posts') . '</strong><br>' .
             esc_html__('This prompt defines how the AI interprets customer search queries. You can paste your own custom version below, or use the "Generate AI-Optimized Prompt" button to auto-fill.', 'smarter-search-products-posts') .
             '</p>';

        echo '<p><em>' . esc_html__('Suggested Prompt for Your Shop (example):', 'smarter-search-products-posts') . '</em></p>';
        echo '<pre style="background:#f5f5f5;border:1px solid #ddd;padding:10px;white-space:pre-wrap;max-height:250px;overflow:auto;">' .
             esc_html($example_prompt) .
             '</pre>';

        echo '<textarea name="crsh_prompt" rows="8" class="large-text" placeholder="' .
             esc_attr(__('Enter a custom prompt here, or click Generate below to use a shop-specific version.', 'smarter-search-products-posts')) . '">';
        echo esc_textarea($saved_prompt);
        echo '</textarea>';

        echo '<br><button type="button" id="generate-ai-prompt" class="button button-secondary" style="margin-top:10px;">' .
             esc_html__('Generate AI-Optimized Prompt', 'smarter-search-products-posts') .
             '</button>';

        echo '<p class="description">' .
             esc_html__('If left blank, the plugin will fall back on its built-in shop-specific default.', 'smarter-search-products-posts') .
             '</p>';
    }

    public function field_temperature() {
        $temperature = get_option('crsh_temperature', 0.7);
        echo '<input type="number" name="crsh_temperature" value="' . esc_attr($temperature) . '" min="0" max="2" step="0.1" class="regular-text">';
        echo '<p class="description">' . esc_html__(
            'Controls AI creativity. Lower values (0.1-0.3) for precise searches, higher values (0.7-1.0) for creative interpretation.',
            'smarter-search-products-posts'
        ) . '</p>';
    }

    public function field_products_only() {
        $current = get_option('crsh_products_only', 'yes');
        echo '<select name="crsh_products_only">';
        printf(
            '<option value="yes"%s>%s</option>',
            selected('yes', $current, false),
            esc_html__('Products only', 'smarter-search-products-posts')
        );
        printf(
            '<option value="no"%s>%s</option>',
            selected('no', $current, false),
            esc_html__('Products & Posts', 'smarter-search-products-posts')
        );
        echo '</select>';
    }

    public function sanitize_products_only($value) {
        return in_array($value, ['yes', 'no'], true) ? $value : 'yes';
    }

    public function field_input_width() {
        $width = get_option('crsh_input_width', '300');
        echo '<input type="number" name="crsh_input_width" value="' . esc_attr($width) . '" min="50" max="1000" class="regular-text">';
    }

    public function field_input_height() {
        $height = get_option('crsh_input_height', '40');
        echo '<input type="number" name="crsh_input_height" value="' . esc_attr($height) . '" min="20" max="200" class="regular-text">';
    }

    public function handle_generate_prompt() {
        check_ajax_referer('crsh_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions', 'smarter-search-products-posts'));
        }

        $generated_prompt = $this->generate_shop_specific_prompt();
        wp_send_json_success(['prompt' => $generated_prompt]);
    }

    public function generate_shop_specific_prompt() {
        $shop_name = sanitize_text_field(get_bloginfo('name'));
        if (empty($shop_name)) {
            $shop_name = __('this website', 'smarter-search-products-posts');
        }

        $products_only = get_option('crsh_products_only', 'yes') === 'yes';

        // Fetch ONLY PRIMARY (top-level) product categories
        $product_cats = get_terms([
            'taxonomy'   => 'product_cat',
            'hide_empty' => true,
            'parent'     => 0, // ← Critical: top-level only
            'orderby'    => 'count',
            'order'      => 'DESC',
            'number'     => 20,
        ]);
        $product_slugs = (!is_wp_error($product_cats) && !empty($product_cats))
            ? wp_list_pluck($product_cats, 'slug')
            : [];

        $post_cats = get_terms([
            'taxonomy'   => 'category',
            'hide_empty' => true,
            'parent'     => 0, // also top-level for posts (optional, but consistent)
            'orderby'    => 'count',
            'order'      => 'DESC',
            'number'     => 20,
        ]);
        $post_slugs = (!is_wp_error($post_cats) && !empty($post_cats))
            ? wp_list_pluck($post_cats, 'slug')
            : [];

        // Sanitize all category slugs for safe inclusion in the prompt
        $product_slugs = array_map('sanitize_text_field', $product_slugs);
        $post_slugs    = array_map('sanitize_text_field', $post_slugs);

        $product_string = !empty($product_slugs) ? implode(', ', $product_slugs) : __('(no product categories defined)', 'smarter-search-products-posts');
        $post_string    = !empty($post_slugs) ? implode(', ', $post_slugs) : __('(no post categories defined)', 'smarter-search-products-posts');

        if ($products_only) {
            return "You are a world-class search parameter assistant for {$shop_name}.\nYour task is to convert a user's search query into a strict JSON object for PRODUCT SEARCHES ONLY.\n\nCRITICAL RULES:\n1. Your response MUST be ONLY the raw JSON object. Do not include any text, greetings, or markdown.\n2. ONLY search for WooCommerce products - use product_cat slugs from this list: {$product_string}\n3. Always use taxonomy=\"product_cat\" and post_types=[\"product\"]\n4. If the query is not product-related, return an empty JSON object: {}\n5. NEVER return HTML (e.g. <span>) or formatting.\n\nThe JSON can contain these keys:\n- \"name\": product name or keywords\n- \"category\": product category slug\n- \"min_price\": minimum price (number only, if mentioned)\n- \"max_price\": maximum price (number only, if mentioned)\n- \"sku\": if the query looks like a SKU code\n";
        }

        return "You are a world-class search parameter assistant for {$shop_name}.\nYour task is to convert a user's search query into a strict JSON object describing either product searches or content (posts/pages).\n\nCRITICAL RULES:\n1. Your response MUST be ONLY the raw JSON object. Do not include any text, greetings, or markdown.\n2. For WooCommerce products, ONLY use product_cat slugs from this list: {$product_string}\n3. For blog/articles or other site content, ONLY use post category slugs from this list: {$post_string}\n4. Always decide which taxonomy you are using (\"product_cat\" or \"category\") and include it in the JSON.\n5. If both products AND posts are relevant, you may include multiple post_types: [\"product\",\"post\"]\n6. If the query is irrelevant, return an empty JSON object: {}\n7. NEVER return HTML (e.g. <span>) or formatting.\n\nThe JSON can contain these keys:\n- \"name\": product name or keywords\n- \"category\": product category slug\n- \"min_price\": minimum price (number only, if mentioned)\n- \"max_price\": maximum price (number only, if mentioned)\n- \"sku\": if the query looks like a SKU code\n";
    }

    public function sanitize_temperature($value) {
        $value = floatval($value);
        return max(0, min(2, $value));
    }

    public function sanitize_width($value) {
        $value = intval($value);
        return max(50, min(1000, $value));
    }

    public function sanitize_height($value) {
        $value = intval($value);
        return max(20, min(200, $value));
    }

    public function render_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Smarter Search for WooCommerce Settings', 'smarter-search-products-posts'); ?></h1>

            <?php if (!class_exists('WooCommerce')) : ?>
                <div class="notice notice-warning">
                    <p><?php echo esc_html__('WooCommerce is not installed or activated. This plugin requires WooCommerce to function properly.', 'smarter-search-products-posts'); ?></p>
                </div>
            <?php endif; ?>

            <form method="post" action="options.php">
                <?php
                settings_fields('crsh_settings');
                do_settings_sections('crsh');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }
}