# RAG Collection API Schema

## Authentication

All API requests require authentication using the API key in the request header:

```
X-API-Key: your-secret-api-key-here
```

The API key can be obtained from the admin panel under "API Settings" section.

---

## Base URL

All endpoints are prefixed with `/api/rag-collections` and are served from:

```
https://smartsupport.pro
```

---

## 1. Create RAG Collection

### Endpoint
```
POST /api/rag-collections/create
```

### Request Headers
```
X-API-Key: {your-api-key}
Content-Type: application/json
```

### Request Body Schema

```json
{
  "collection_name": "string (required)",
  "description": "string (required)",
  "tokens": ["string"] (optional, array of agent public_ids),
  "fields": [
    {
      "field_name": "string (required)",
      "type": "string (required)",
      "embed": "boolean (required)"
    }
  ]
}
```

### Field Descriptions

- **collection_name** (string, required): The name of the RAG collection
- **description** (string, required): Description of the RAG collection
- **tokens** (array of strings, optional): Array of agent public_ids (UUIDs) to associate with this collection. Only agents belonging to the authenticated user will be linked.
- **fields** (array, required): Array of field definitions for the collection
  - **field_name** (string, required): Name of the field
  - **type** (string, required): Type of the field
  - **embed** (boolean, required): Whether this field should be embedded in the content for vector search

### Example Request

```json
{
  "collection_name": "Product Knowledge Base",
  "description": "Knowledge base containing product information and specifications",
  "tokens": [
    "550e8400-e29b-41d4-a716-446655440000",
    "6ba7b810-9dad-11d1-80b4-00c04fd430c8"
  ],
  "fields": [
    {
      "field_name": "product_name",
      "type": "string",
      "embed": true
    },
    {
      "field_name": "product_id",
      "type": "string",
      "embed": false
    },
    {
      "field_name": "description",
      "type": "text",
      "embed": true
    },
    {
      "field_name": "price",
      "type": "number",
      "embed": false
    }
  ]
}
```

### Response

```json
{
  "ragCollection": {
    "id": 1,
    "collection_name": "Product Knowledge Base",
    "collection_unique": "1234567890_Product_Knowledge_Base",
    "description": "Knowledge base containing product information and specifications",
    "user_id": 1,
    "records_count": 0,
    "created_at": "2025-11-07T18:00:00.000000Z",
    "updated_at": "2025-11-07T18:00:00.000000Z",
    "fields": [
      {
        "id": 101,
        "field_name": "product_name",
        "field_unique": "1709999999product_name",
        "type": "string",
        "embed": true,
        "collection_id": 1,
        "created_at": "2025-11-07T18:00:00.000000Z",
        "updated_at": "2025-11-07T18:00:00.000000Z"
      },
      {
        "id": 102,
        "field_name": "description",
        "field_unique": "1709999999description",
        "type": "text",
        "embed": true,
        "collection_id": 1,
        "created_at": "2025-11-07T18:00:00.000000Z",
        "updated_at": "2025-11-07T18:00:00.000000Z"
      },
      {
        "id": 103,
        "field_name": "price",
        "field_unique": "1709999999price",
        "type": "number",
        "embed": false,
        "collection_id": 1,
        "created_at": "2025-11-07T18:00:00.000000Z",
        "updated_at": "2025-11-07T18:00:00.000000Z"
      }
    ]
  }
}
```

---

## 2. Update RAG Collection

### Endpoint
```
POST /api/rag-collections/update/{ragCollection}
```

Where `{ragCollection}` is the ID of the RAG collection to update.

### Request Headers
```
X-API-Key: {your-api-key}
Content-Type: application/json
```

### Request Body Schema

```json
{
  "collection_name": "string (required)",
  "description": "string (required)",
  "tokens": ["string"] (optional, array of agent public_ids)
}
```

### Field Descriptions

- **collection_name** (string, required): The updated name of the RAG collection
- **description** (string, required): Updated description of the RAG collection
- **tokens** (array of strings, optional): Array of agent public_ids (UUIDs) to associate with this collection. Only agents belonging to the authenticated user will be linked. Existing associations are preserved (syncWithoutDetaching).

### Example Request

```json
{
  "collection_name": "Updated Product Knowledge Base",
  "description": "Updated knowledge base with latest product information",
  "tokens": [
    "550e8400-e29b-41d4-a716-446655440000",
    "6ba7b810-9dad-11d1-80b4-00c04fd430c8",
    "7ca8c920-0ebe-22e2-91c5-11d5fd541d9d"
  ]
}
```

### Response

```json
{
  "ragCollection": {
    "id": 1,
    "collection_name": "Updated Product Knowledge Base",
    "collection_unique": "1234567890_Product_Knowledge_Base",
    "description": "Updated knowledge base with latest product information",
    "user_id": 1,
    "records_count": 0,
    "created_at": "2025-11-07T18:00:00.000000Z",
    "updated_at": "2025-11-07T18:30:00.000000Z",
    "fields": [
      {
        "id": 101,
        "field_name": "product_name",
        "field_unique": "1709999999product_name",
        "type": "string",
        "embed": true,
        "collection_id": 1,
        "created_at": "2025-11-07T18:00:00.000000Z",
        "updated_at": "2025-11-07T18:00:00.000000Z"
      },
      {
        "id": 102,
        "field_name": "description",
        "field_unique": "1709999999description",
        "type": "text",
        "embed": true,
        "collection_id": 1,
        "created_at": "2025-11-07T18:00:00.000000Z",
        "updated_at": "2025-11-07T18:00:00.000000Z"
      },
      {
        "id": 103,
        "field_name": "price",
        "field_unique": "1709999999price",
        "type": "number",
        "embed": false,
        "collection_id": 1,
        "created_at": "2025-11-07T18:00:00.000000Z",
        "updated_at": "2025-11-07T18:00:00.000000Z"
      }
    ]
  }
}
```

---

## 3. Delete RAG Collection

### Endpoint
```
DELETE /api/rag-collections/{ragCollection}
```

Where `{ragCollection}` is the numeric ID of the collection you want to delete.

### Request Headers
```
X-API-Key: {your-api-key}
```

### Request Body

No request body is required.

### Response

```
HTTP 200 OK
```

Empty JSON body.

> **Warning:** Deleting a collection permanently removes the collection, all associated fields, and any stored records/chroma data.

---

## 4. Create RAG Record

### Endpoint
```
POST /api/rag-collections/{ragCollection}/records
```

Where `{ragCollection}` is the numeric ID of the collection that will own the record.

### Request Headers
```
X-API-Key: {your-api-key}
Content-Type: application/json
```

### Request Body Schema

```json
{
  "fields": [
    {
      "field_name": "string (required)",
      "value": "string | number | boolean (required)"
    }
  ]
}
```

### Field Descriptions

- **fields** (array, required): Each item maps data to a collection field
  - **field_name** (string, required): The `field_unique` value of an existing collection field. Use `GET /api/rag-collections/{id}` to retrieve the available fields and their `field_unique` values.
  - **value** (mixed, required): The content to store for the field. Stored as metadata and, for fields marked with `embed = true`, contributes to the vector content body.

### Example Request

```json
{
  "fields": [
    {
      "field_name": "1709999999product_name",
      "value": "SuperWidget 3000"
    },
    {
      "field_name": "1709999999description",
      "value": "Next generation widget with advanced AI support"
    },
    {
      "field_name": "1709999999price",
      "value": "199.99"
    }
  ]
}
```

### Response

Returns the metadata that was stored for the record along with the generated record ids:

```json
{
  "metadata": {
    "product_name": "SuperWidget 3000",
    "description": "Next generation widget with advanced AI support",
    "price": "199.99"
  },
  "ids": [
    "Product_Knowledge_Base_9f1a2b3c4d"
  ]
}
```

---

## 5. Update RAG Record

### Endpoint
```
PATCH /api/rag-collections/{ragCollection}/records
```

### Request Headers
```
X-API-Key: {your-api-key}
Content-Type: application/json
```

### Request Body Schema

```json
{
  "ids": ["string"],
  "fields": [
    {
      "field_name": "string (required)",
      "value": "string | number | boolean (required)"
    }
  ]
}
```

### Field Descriptions

- **ids** (array of strings, required): Collection record IDs to update. These IDs are returned by the `/get-records` endpoint (`collection.ids`). Provide one or more IDs to update the stored documents in place.
- **fields** (array, required): Same structure as in record creation. All supplied IDs receive the same updated data.

### Example Request

```json
{
  "ids": ["collection_21f42f8d0b"],
  "fields": [
    {
      "field_name": "1709999999description",
      "value": "Updated description with the latest specifications"
    },
    {
      "field_name": "1709999999price",
      "value": "179.99"
    }
  ]
}
```

### Response

Returns the metadata that was stored after the update:

```json
{
  "description": "Updated description with the latest specifications",
  "price": "179.99"
}
```

---

## 6. Delete RAG Record

### Endpoint
```
DELETE /api/rag-collections/{ragCollection}/records/{recordId}
```

- `{ragCollection}`: Numeric ID of the collection
- `{recordId}`: The record ID retrieved from `/get-records`

### Request Headers
```
X-API-Key: {your-api-key}
```

### Request Body

No request body required.

### Response

```
HTTP 200 OK
```

Empty JSON body.

---

## Agent Management API

These endpoints let you manage agents programmatically. All requests require the `X-API-Key` header and operate on the authenticated user's agents. Agent identifiers in the API are the agent `public_id` values (also referred to as *tokens*).

### 1. List Agents

**Endpoint**
```
GET /api/agents
```

**Response**
```json
{
  "agents": [
    {
      "id": 1,
      "public_id": "550e8400-e29b-41d4-a716-446655440000",
      "name": "Support Bot",
      "description": "Handles first-line customer support",
      "personality": "friendly",
      "user_id": 42,
      "created_at": "2025-11-07T18:00:00.000000Z",
      "updated_at": "2025-11-07T18:30:00.000000Z",
      "tickets_count": 12,
      "tools": [
        {
          "id": 5,
          "agent_id": 1,
          "type": "telegram",
          "tool_settings": null,
          "is_active": true,
          "created_at": "2025-11-07T18:10:00.000000Z",
          "updated_at": "2025-11-07T18:10:00.000000Z"
        }
      ]
    }
  ]
}
```

### 2. Create Agent

**Endpoint**
```
POST /api/agents
```

**Request Headers**
```
X-API-Key: {your-api-key}
Content-Type: application/json
```

**Body Schema**
```json
{
  "name": "string (required)",
  "description": "string (optional)",
  "personality": "string (required: professional | friendly | technical | casual)",
  "tools": ["string"] (optional, list of tool types to activate)
}
```

**Example Request**
```json
{
  "name": "Proactive Helper",
  "description": "AI assistant for onboarding questions",
  "personality": "professional",
  "tools": ["telegram", "website"]
}
```

**Response**
```json
{
  "agent": {
    "id": 7,
    "public_id": "f0a3c090-9b44-4efc-8d73-83d91f1d5f9e",
    "name": "Proactive Helper",
    "description": "AI assistant for onboarding questions",
    "personality": "professional",
    "user_id": 42,
    "created_at": "2025-11-07T19:00:00.000000Z",
    "updated_at": "2025-11-07T19:00:00.000000Z",
    "tickets_count": 0,
    "tools": [
      {
        "id": 21,
        "agent_id": 7,
        "type": "telegram",
        "tool_settings": null,
        "is_active": true,
        "created_at": "2025-11-07T19:00:00.000000Z",
        "updated_at": "2025-11-07T19:00:00.000000Z"
      },
      {
        "id": 22,
        "agent_id": 7,
        "type": "website",
        "tool_settings": null,
        "is_active": true,
        "created_at": "2025-11-07T19:00:00.000000Z",
        "updated_at": "2025-11-07T19:00:00.000000Z"
      }
    ]
  }
}
```

### 3. Show Agent

**Endpoint**
```
GET /api/agents/{agent_token}
```

`{agent_token}` is the agent `public_id` string returned by the list/create endpoints.

Returns the requested agent (must belong to the authenticated user) with the same structure as the create response.

### 4. Update Agent

**Endpoint**
```
PUT /api/agents/{agent_token}
```

`PATCH` is also supported.

**Request Headers**
```
X-API-Key: {your-api-key}
Content-Type: application/json
```

**Body Schema** (same as create)
```json
{
  "name": "string (required)",
  "description": "string (optional)",
  "personality": "string (required: professional | friendly | technical | casual)",
  "tools": ["string"] (optional)
}
```

Passing the `tools` array replaces the list of active tools. Supply an empty array to deactivate all tools.

**Response**
```json
{
  "agent": {
    "id": 7,
    "public_id": "f0a3c090-9b44-4efc-8d73-83d91f1d5f9e",
    "name": "Proactive Helper",
    "description": "Updated AI assistant",
    "personality": "friendly",
    "user_id": 42,
    "created_at": "2025-11-07T19:00:00.000000Z",
    "updated_at": "2025-11-07T19:30:00.000000Z",
    "tickets_count": 0,
    "tools": [
      {
        "id": 21,
        "agent_id": 7,
        "type": "telegram",
        "tool_settings": null,
        "is_active": true,
        "created_at": "2025-11-07T19:00:00.000000Z",
        "updated_at": "2025-11-07T19:30:00.000000Z"
      }
    ]
  }
}
```

### 5. Delete Agent

**Endpoint**
```
DELETE /api/agents/{agent_token}
```

**Response**
```
HTTP 204 No Content
```

---

## Important Notes

1. **Authentication**: All requests must include the `X-API-Key` header with a valid API key.

2. **Agent Tokens**: The `tokens` field accepts an array of agent `public_id` values (UUIDs). The system will:
   - Validate that all tokens belong to agents owned by the authenticated user
   - Convert tokens to internal agent IDs
   - Only link agents that pass validation

3. **Fields in Create**: When creating a collection, you must provide at least one field definition. Fields define the structure of data that will be stored in the collection. The API response now returns the full list of fields (with IDs, field_unique, and metadata) that were attached to the collection during creation.

4. **Fields in Update**: The update endpoint does NOT modify fields. Fields are only set during collection creation, but the response includes the current set of fields for reference.

5. **Agent Association**: 
   - In **Create**: Agents are attached to the collection
   - In **Update**: Agents are added to existing associations (existing associations are preserved)

6. **Record Creation Response**: The record creation endpoint returns both the stored metadata and the generated collection record IDs. Save these IDs if you plan to update or delete the records later.

7. **Agent Management**: Agent endpoints use `public_id` values (tokens) in path parameters, return `tickets_count`, and include the full list of tools. Only tools supplied in the `tools` array remain active; omitted tools are deactivated.

8. **Error Responses**: 
   - `401 Unauthorized`: Invalid or missing API key
   - `403 Forbidden`: Attempted access to an agent or collection owned by another user
   - `500 Internal Server Error`: Server error with error message in response body

---

## Example cURL Commands

### Create Collection
```bash
curl -X POST "https://smartsupport.pro/api/rag-collections/create" \
  -H "X-API-Key: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "collection_name": "Product Knowledge Base",
    "description": "Knowledge base containing product information",
    "tokens": ["550e8400-e29b-41d4-a716-446655440000"],
    "fields": [
      {
        "field_name": "product_name",
        "type": "string",
        "embed": true
      },
      {
        "field_name": "product_id",
        "type": "string",
        "embed": false
      }
    ]
  }'
```

### Update Collection
```bash
curl -X POST "https://smartsupport.pro/api/rag-collections/update/1" \
  -H "X-API-Key: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "collection_name": "Updated Product Knowledge Base",
    "description": "Updated description",
    "tokens": ["550e8400-e29b-41d4-a716-446655440000"]
  }'
```

### Create Agent
```bash
curl -X POST "https://smartsupport.pro/api/agents" \
  -H "X-API-Key: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Proactive Helper",
    "description": "AI assistant for onboarding questions",
    "personality": "professional",
    "tools": ["telegram", "website"]
  }'
```

### Update Agent
```bash
curl -X PATCH "https://smartsupport.pro/api/agents/f0a3c090-9b44-4efc-8d73-83d91f1d5f9e" \
  -H "X-API-Key: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Proactive Helper",
    "description": "Updated AI assistant",
    "personality": "friendly",
    "tools": ["telegram"]
  }'
```