<?php
/**
 * Plugin Name: Smart Support Chat Widget
 * Plugin URI: https://smartsupport.pro/chat-widget
 * Description: Integrate Smart Support AI chat widget into your WordPress site. Simply add your agent token and customize the chat widget appearance.
 * Short Description: Embed the Smart Support AI chat widget with customizable visuals and configuration.
 * Version: 1.0.4
 * Author: Smart Support
 * Author URI: https://smartsupport.pro/about
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: smart-support-chat-widget
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('SMARTSUPPORT_WP_VERSION', '1.0.0');
define('SMARTSUPPORT_WP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SMARTSUPPORT_WP_PLUGIN_URL', plugin_dir_url(__FILE__));

// Load configuration
require_once SMARTSUPPORT_WP_PLUGIN_DIR . 'includes/config.php';

/**
 * Main plugin class
 */
class SmartSupport_WP {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Data for rendering collection metadata in frontend
     */
    private $pending_collection_metadata = null;

    /**
     * Get instance of this class
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init();
    }

    /**
     * Initialize plugin
     */
    private function init() {
        // Load admin functionality (includes shared functionality used by webhooks/indexing)
        require_once SMARTSUPPORT_WP_PLUGIN_DIR . 'includes/class-smartsupport-admin.php';
        new SmartSupport_Admin();

        // Load frontend functionality
        add_action('wp_enqueue_scripts', array($this, 'enqueue_chat_widget'));
    }

    /**
     * Enqueue chat widget script on frontend
     */
    public function enqueue_chat_widget() {
        $settings = get_option('smartsupport_wp_settings', array());

        // Only enqueue if agent token is set
        if (empty($settings['agent_id']) || empty(trim($settings['agent_id']))) {
            return;
        }

        // Get settings with defaults
        $agent_id = sanitize_text_field($settings['agent_id']);
        $socket_url = smartsupport_get_base_url();
        $api_base_url = smartsupport_get_base_url();
        $title = !empty($settings['title']) ? sanitize_text_field($settings['title']) : 'Chat Support';
        $primary_color = !empty($settings['primary_color']) ? sanitize_hex_color($settings['primary_color']) : '#6366f1';
        $position = !empty($settings['position']) ? sanitize_text_field($settings['position']) : 'bottom-right';
        $greeting_message = !empty($settings['greeting_message']) ? sanitize_text_field($settings['greeting_message']) : 'Welcome! How may I assist you today?';
        $show_header = isset($settings['show_header']) ? (bool) $settings['show_header'] : true;
        $show_toggle = isset($settings['show_toggle']) ? (bool) $settings['show_toggle'] : true;
        $default_open = isset($settings['default_open']) ? (bool) $settings['default_open'] : false;

        // Create inline script with configuration
        $config_script = sprintf(
            "window.chatWidgetConfig = {\n" .
            "    agentId: %s,\n" .
            "    socketUrl: %s,\n" .
            "    apiBaseUrl: %s,\n" .
            "    title: %s,\n" .
            "    primaryColor: %s,\n" .
            "    position: %s,\n" .
            "    greetingMessage: %s,\n" .
            "    showHeader: %s,\n" .
            "    showToggle: %s,\n" .
            "    defaultOpen: %s\n" .
            "};",
            wp_json_encode($agent_id),
            wp_json_encode($socket_url),
            wp_json_encode($api_base_url),
            wp_json_encode($title),
            wp_json_encode($primary_color),
            wp_json_encode($position),
            wp_json_encode($greeting_message),
            $show_header ? 'true' : 'false',
            $show_toggle ? 'true' : 'false',
            $default_open ? 'true' : 'false'
        );

        // Register and enqueue config script in head (before chat widget)
        wp_register_script('smartsupport-chat-config', '', array(), SMARTSUPPORT_WP_VERSION, false);
        wp_enqueue_script('smartsupport-chat-config');
        wp_add_inline_script('smartsupport-chat-config', $config_script);

        // Enqueue the chat widget script with defer attribute
        wp_enqueue_script(
            'smartsupport-chat-widget',
            esc_url_raw($api_base_url . '/js/webhooks/chat.js'),
            array('smartsupport-chat-config'),
            SMARTSUPPORT_WP_VERSION,
            true
        );

        if (is_singular()) {
            global $post;
            if ($post instanceof WP_Post) {
                $metadata = $this->resolve_collection_metadata_for_post($post);
                if (!empty($metadata)) {
                    $this->pending_collection_metadata = $metadata;
                    add_action('wp_print_footer_scripts', array($this, 'render_collection_metadata_inputs'), 5);
                }
            }
        }

        // Add defer attribute to chat widget script
        add_filter('script_loader_tag', function($tag, $handle) {
            if ('smartsupport-chat-widget' === $handle) {
                return str_replace(' src', ' defer src', $tag);
            }
            return $tag;
        }, 10, 2);
    }

    /**
     * Resolve SmartSupport collection metadata for the current post
     */
    private function resolve_collection_metadata_for_post(WP_Post $post) {
        $collections = get_option('smartsupport_wp_rag_collections', array());
        if (empty($collections) || !is_array($collections)) {
            return null;
        }

        foreach ($collections as $collection_key => $collection) {
            if (!is_array($collection)) {
                continue;
            }

            $post_type = isset($collection['post_type']) ? $collection['post_type'] : '';
            if ($post_type !== $post->post_type) {
                continue;
            }

            $collection_key_sanitized = sanitize_key($collection_key);
            $meta_key = '_smartsupport_rag_record_' . $collection_key_sanitized;
            $record_id = get_post_meta($post->ID, $meta_key, true);

            if (empty($record_id)) {
                continue;
            }

            $collection_unique = isset($collection['collection_unique']) ? $collection['collection_unique'] : '';
            if (empty($collection_unique)) {
                $collection_unique = $collection_key;
            }

            return array(
                'collection_name' => sanitize_text_field($collection_unique),
                'record_id' => sanitize_text_field($record_id)
            );
        }

        return null;
    }

    /**
     * Render hidden inputs containing collection metadata
     */
    public function render_collection_metadata_inputs() {
        if (empty($this->pending_collection_metadata) || !is_array($this->pending_collection_metadata)) {
            return;
        }

        $collection_name = isset($this->pending_collection_metadata['collection_name']) ? $this->pending_collection_metadata['collection_name'] : '';
        $record_id = isset($this->pending_collection_metadata['record_id']) ? $this->pending_collection_metadata['record_id'] : '';

        if (empty($collection_name) || empty($record_id)) {
            return;
        }

        printf(
            '<input type="hidden" id="wordpressSmartSupportCollectionName" value="%s" />' . "\n" .
            '<input type="hidden" id="wordpressSmartSupportRecordId" value="%s" />' . "\n",
            esc_attr($collection_name),
            esc_attr($record_id)
        );

        // Prevent duplicate rendering
        $this->pending_collection_metadata = null;
    }
}

/**
 * Initialize the plugin
 */
function smartsupport_wp_init() {
    return SmartSupport_WP::get_instance();
}

// Start the plugin
add_action('plugins_loaded', 'smartsupport_wp_init');

/**
 * Redirect to plugin settings page after activation
 */
function smartsupport_wp_activation_redirect() {
    // Set transient to indicate plugin was just activated
    set_transient('smartsupport_wp_activation_redirect', true, 30);
}
register_activation_hook(__FILE__, 'smartsupport_wp_activation_redirect');

/**
 * Handle activation redirect
 */
function smartsupport_wp_handle_activation_redirect() {
    // Check if we should redirect
    if (!get_transient('smartsupport_wp_activation_redirect')) {
        return;
    }

    // Delete the transient so we only redirect once
    delete_transient('smartsupport_wp_activation_redirect');

    // Don't redirect if user doesn't have permission
    if (!current_user_can('manage_options')) {
        return;
    }

    // Don't redirect if doing AJAX, cron, or if user is activating multiple plugins
    $activate_multi = isset($_GET['activate-multi']) ? sanitize_text_field(wp_unslash($_GET['activate-multi'])) : '';
    if (wp_doing_ajax() || wp_doing_cron() || $activate_multi === 'true') {
        return;
    }

    // Redirect to plugin settings page
    wp_safe_redirect(admin_url('admin.php?page=smart-support-chat-widget'));
    exit;
}
add_action('admin_init', 'smartsupport_wp_handle_activation_redirect');

