<?php
/**
 * Smart Support API Integration
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Load config if not already loaded
if (!function_exists('smartsupport_get_base_url')) {
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/config.php';
}

class SmartSupport_API {

    /**
     * API Base URL
     */
    private $api_base_url;

    /**
     * Constructor
     */
    public function __construct() {
        $this->api_base_url = smartsupport_get_base_url();
    }

    /**
     * Get API key from settings
     */
    private function get_api_key() {
        $settings = get_option('smartsupport_wp_settings', array());
        return isset($settings['api_key']) ? $settings['api_key'] : '';
    }

    /**
     * Make API request
     */
    private function make_request($endpoint, $method = 'POST', $data = array(), $success_message = null, $error_message = null) {
        $api_key = $this->get_api_key();

        if (empty($api_key)) {
            return array(
                'success' => false,
                'message' => __('API key is not configured.', 'smart-support-chat-widget')
            );
        }

        $method = strtoupper($method);

        $url = $this->api_base_url . '/api/rag-collections' . $endpoint;

        $args = array(
            'method' => $method,
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'timeout' => 30
        );

        $payload = $data;
        if (in_array($method, array('POST', 'PUT', 'PATCH'), true)) {
            $payload['api_key'] = $api_key;
        }

        if (!empty($payload) && in_array($method, array('POST', 'PUT', 'PATCH'), true)) {
            $args['body'] = wp_json_encode($payload);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();

            // Add detailed error information if WP_DEBUG is enabled
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $error_codes = $response->get_error_codes();
                $error_data = $response->get_error_data();

                $debug_info = array();
                if (!empty($error_codes)) {
                    $debug_info[] = __('Error codes:', 'smart-support-chat-widget') . ' ' . implode(', ', $error_codes);
                }
                if (!empty($error_data)) {
                    $debug_info[] = __('Error data:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($error_data, JSON_PRETTY_PRINT));
                }
                $debug_info[] = __('Request URL:', 'smart-support-chat-widget') . ' ' . $url;
                $debug_info[] = __('Request method:', 'smart-support-chat-widget') . ' ' . $method;

                if (!empty($debug_info)) {
                    $error_message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n" . implode("\n", $debug_info);
                }
            }

            return array(
                'success' => false,
                'message' => $error_message
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $body_data = json_decode($body, true);
        $success_message = $success_message ? $success_message : __('Request completed successfully.', 'smart-support-chat-widget');
        $default_error_message = __('Request failed.', 'smart-support-chat-widget');

        if ($status_code >= 200 && $status_code < 300) {
            return array(
                'success' => true,
                'message' => $success_message,
                'data' => $body_data,
                'status_code' => $status_code
            );
        } else {
            $error_message = $error_message ? $error_message : $default_error_message;
            if (isset($body_data['message'])) {
                $error_message = $body_data['message'];
            } elseif (isset($body_data['error'])) {
                $error_message = $body_data['error'];
            }

            // Add detailed error information if WP_DEBUG is enabled
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $debug_info = array();
                $debug_info[] = __('HTTP Status Code:', 'smart-support-chat-widget') . ' ' . $status_code;
                $debug_info[] = __('Request URL:', 'smart-support-chat-widget') . ' ' . $url;
                $debug_info[] = __('Request method:', 'smart-support-chat-widget') . ' ' . $method;

                if (!empty($body)) {
                    $debug_info[] = __('Response body:', 'smart-support-chat-widget') . "\n" . $body;
                }

                if (!empty($data)) {
                    $debug_info[] = __('Request data:', 'smart-support-chat-widget') . "\n" . esc_html(wp_json_encode($data, JSON_PRETTY_PRINT));
                }

                if (!empty($debug_info)) {
                    $error_message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n" . implode("\n\n", $debug_info);
                }
            }

            return array(
                'success' => false,
                'message' => $error_message,
                'status_code' => $status_code
            );
        }
    }

    /**
     * Make GET request
     */
    private function make_get_request($endpoint, $query = array(), $success_message = null, $error_message = null) {
        $api_key = $this->get_api_key();

        if (empty($api_key)) {
            return array(
                'success' => false,
                'message' => __('API key is not configured.', 'smart-support-chat-widget')
            );
        }

        $url = $this->api_base_url . '/api/rag-collections' . $endpoint;

        if (!empty($query) && is_array($query)) {
            $url = add_query_arg(array_map('sanitize_text_field', $query), $url);
        }

        $args = array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'timeout' => 30
        );

        $response = wp_remote_get($url, $args);

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();

            if (defined('WP_DEBUG') && WP_DEBUG) {
                $error_codes = $response->get_error_codes();
                $error_data = $response->get_error_data();

                $debug_info = array();
                if (!empty($error_codes)) {
                    $debug_info[] = __('Error codes:', 'smart-support-chat-widget') . ' ' . implode(', ', $error_codes);
                }
                if (!empty($error_data)) {
                    $debug_info[] = __('Error data:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($error_data, JSON_PRETTY_PRINT));
                }
                $debug_info[] = __('Request URL:', 'smart-support-chat-widget') . ' ' . $url;

                if (!empty($debug_info)) {
                    $error_message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n" . implode("\n", $debug_info);
                }
            }

            return array(
                'success' => false,
                'message' => $error_message
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $body_data = json_decode($body, true);

        $success_message = $success_message ? $success_message : __('Request completed successfully.', 'smart-support-chat-widget');
        $error_message = $error_message ? $error_message : __('Request failed.', 'smart-support-chat-widget');

        if ($status_code >= 200 && $status_code < 300) {
            return array(
                'success' => true,
                'message' => $success_message,
                'data' => $body_data,
                'status_code' => $status_code
            );
        }

        if (isset($body_data['message'])) {
            $error_message = $body_data['message'];
        } elseif (isset($body_data['error'])) {
            $error_message = $body_data['error'];
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            $debug_info = array();
            $debug_info[] = __('HTTP Status Code:', 'smart-support-chat-widget') . ' ' . $status_code;
            $debug_info[] = __('Request URL:', 'smart-support-chat-widget') . ' ' . $url;

            if (!empty($body)) {
                $debug_info[] = __('Response body:', 'smart-support-chat-widget') . "\n" . $body;
            }

            if (!empty($debug_info)) {
                $error_message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n" . implode("\n\n", $debug_info);
            }
        }

        return array(
            'success' => false,
            'message' => $error_message,
            'status_code' => $status_code
        );
    }

    /**
     * Create RAG collection
     *
     * @param string $collection_name Collection name (required)
     * @param string $description Collection description
     * @param array $tokens Array of agent public_ids (optional, alternative to agent_ids)
     * @param array $fields Array of field definitions with field_name, type, embed, metadata (required)
     * @param string $embedding_type Embedding type (optional, defaults to 'multilingual-e5-base')
     * @param array $chunking_settings Chunking settings (optional, defaults to enabled: true, chunk_size: 250, chunk_overlap: 50, chunk_by: 'semantic')
     * @return array API response
     */
    public function create_rag_collection($collection_name, $description, $tokens = array(), $fields = array(), $embedding_type = null, $chunking_settings = null) {
        $data = array(
            'collection_name' => $collection_name,
            'description' => $description,
            'fields' => $fields
        );

        if (!empty($tokens)) {
            $data['tokens'] = $tokens;
        }

        if (!empty($embedding_type)) {
            $data['embedding_type'] = $embedding_type;
        }

        if (!empty($chunking_settings) && is_array($chunking_settings)) {
            $data['chunking_settings'] = $chunking_settings;
        }

        return $this->make_request(
            '/create',
            'POST',
            $data,
            __('RAG collection created successfully!', 'smart-support-chat-widget'),
            __('Failed to create RAG collection.', 'smart-support-chat-widget')
        );
    }

    /**
     * Update RAG collection
     *
     * @param int $collection_id Collection ID (required)
     * @param string $collection_name Collection name (required)
     * @param string $description Collection description
     * @param array $tokens Array of agent public_ids (optional, alternative to agent_ids)
     * @param array $fields Array of field definitions with field_name, type, embed, metadata (optional)
     * @param array $chunking_settings Chunking settings (optional)
     * @return array API response
     */
    public function update_rag_collection($collection_id, $collection_name, $description, $tokens = array(), $fields = null, $chunking_settings = null) {
        $data = array(
            'collection_name' => $collection_name,
            'description' => $description
        );

        if (!empty($tokens)) {
            $data['tokens'] = $tokens;
        }

        if (!empty($fields) && is_array($fields)) {
            $data['fields'] = $fields;
        }

        if (!empty($chunking_settings) && is_array($chunking_settings)) {
            $data['chunking_settings'] = $chunking_settings;
        }

        return $this->make_request(
            '/update/' . intval($collection_id),
            'POST',
            $data,
            __('RAG collection updated successfully!', 'smart-support-chat-widget'),
            __('Failed to update RAG collection.', 'smart-support-chat-widget')
        );
    }

    /**
     * Create RAG record (V2 API - uses records-v2 endpoint)
     */
    public function create_rag_record($collection_id, $fields) {
        return $this->create_rag_record_v2($collection_id, $fields);
    }

    /**
     * Create RAG record (V2 API)
     * Fields should be array of {'field_name': string, 'value': mixed}
     */
    public function create_rag_record_v2($collection_id, $fields) {
        $collection_id = intval($collection_id);
        if ($collection_id <= 0) {
            return array(
                'success' => false,
                'message' => __('Invalid collection ID provided.', 'smart-support-chat-widget')
            );
        }

        if (empty($fields) || !is_array($fields)) {
            return array(
                'success' => false,
                'message' => __('No fields provided for record creation.', 'smart-support-chat-widget')
            );
        }

        $data = array(
            'fields' => array_values($fields)
        );

        $response = $this->make_request(
            '/' . $collection_id . '/records-v2',
            'POST',
            $data,
            __('RAG record created successfully!', 'smart-support-chat-widget'),
            __('Failed to create RAG record.', 'smart-support-chat-widget')
        );

        // Extract document_id from V2 response
        if (!empty($response['success']) && isset($response['data']['record']['id'])) {
            $response['document_id'] = $response['data']['record']['id'];
            $response['record_id'] = isset($response['data']['record']['record_id']) ? $response['data']['record']['record_id'] : '';
        }

        return $response;
    }

    /**
     * Update RAG record (V2 API - uses document_id)
     * @param int $collection_id Collection ID
     * @param string $document_id Document ID (from V2 API)
     * @param array $fields Fields array with field_name and value
     */
    public function update_rag_record($collection_id, $document_id, $fields) {
        return $this->update_rag_record_v2($collection_id, $document_id, $fields);
    }

    /**
     * Update RAG record (V2 API - uses records-v2/{documentId} endpoint)
     * Fields should be array of {'field_name': string, 'value': mixed}
     */
    public function update_rag_record_v2($collection_id, $document_id, $fields) {
        $collection_id = intval($collection_id);
        if ($collection_id <= 0) {
            return array(
                'success' => false,
                'message' => __('Invalid collection ID provided.', 'smart-support-chat-widget')
            );
        }

        $document_id = sanitize_text_field($document_id);
        if (empty($document_id)) {
            return array(
                'success' => false,
                'message' => __('No document ID provided for update.', 'smart-support-chat-widget')
            );
        }

        if (empty($fields) || !is_array($fields)) {
            return array(
                'success' => false,
                'message' => __('No fields provided for record update.', 'smart-support-chat-widget')
            );
        }

        $data = array(
            'fields' => array_values($fields)
        );

        $response = $this->make_request(
            '/' . $collection_id . '/records-v2/' . rawurlencode($document_id),
            'PATCH',
            $data,
            __('RAG record updated successfully!', 'smart-support-chat-widget'),
            __('Failed to update RAG record.', 'smart-support-chat-widget')
        );

        // Extract document_id from V2 response
        if (!empty($response['success']) && isset($response['data']['record']['id'])) {
            $response['document_id'] = $response['data']['record']['id'];
        }

        return $response;
    }

    /**
     * Retrieve RAG collection details (includes fields, metadata, etc.)
     */
    public function get_rag_collection($collection_id) {
        $collection_id = intval($collection_id);
        if ($collection_id <= 0) {
            return array(
                'success' => false,
                'message' => __('Invalid collection ID provided.', 'smart-support-chat-widget')
            );
        }

        $response = $this->make_get_request(
            '/' . $collection_id,
            array(),
            __('RAG collection retrieved successfully.', 'smart-support-chat-widget'),
            __('Failed to retrieve RAG collection.', 'smart-support-chat-widget')
        );

        // Extract collection data from response
        if (!empty($response['success']) && isset($response['data']['collection'])) {
            $response['collection'] = $response['data']['collection'];
        }

        return $response;
    }

    /**
     * Delete RAG collection
     */
    public function delete_rag_collection($collection_id) {
        $collection_id = intval($collection_id);
        if ($collection_id <= 0) {
            return array(
                'success' => false,
                'message' => __('Invalid collection ID provided.', 'smart-support-chat-widget')
            );
        }

        $response = $this->make_request(
            '/' . $collection_id,
            'DELETE',
            array(),
            __('RAG collection deleted successfully!', 'smart-support-chat-widget'),
            __('Failed to delete RAG collection.', 'smart-support-chat-widget')
        );

        if (!$response['success'] && isset($response['status_code']) && 404 === intval($response['status_code'])) {
            return array(
                'success' => true,
                'message' => __('Collection not found remotely; treated as deleted.', 'smart-support-chat-widget'),
                'status_code' => 404
            );
        }

        return $response;
    }

    /**
     * Retrieve records for a RAG collection (V2 - uses get-records-v2 endpoint)
     */
    public function get_rag_records($collection_id, $query = array()) {
        return $this->get_rag_records_v2($collection_id, $query);
    }

    /**
     * Retrieve records for a RAG collection (V2 API)
     */
    public function get_rag_records_v2($collection_id, $query = array()) {
        $collection_id = intval($collection_id);
        if ($collection_id <= 0) {
            return array(
                'success' => false,
                'message' => __('Invalid collection ID provided.', 'smart-support-chat-widget')
            );
        }

        if (!is_array($query)) {
            $query = array();
        }

        return $this->make_get_request(
            '/' . $collection_id . '/get-records-v2',
            $query,
            __('RAG records retrieved successfully.', 'smart-support-chat-widget'),
            __('Failed to retrieve RAG records.', 'smart-support-chat-widget')
        );
    }

    /**
     * Delete RAG record
     */
    public function delete_rag_record($collection_id, $record_id) {
        $collection_id = intval($collection_id);
        if ($collection_id <= 0) {
            return array(
                'success' => false,
                'message' => __('Invalid collection ID provided.', 'smart-support-chat-widget')
            );
        }

        $record_id = sanitize_text_field($record_id);
        if (empty($record_id)) {
            return array(
                'success' => false,
                'message' => __('Invalid record ID provided.', 'smart-support-chat-widget')
            );
        }

        return $this->make_request(
            '/' . $collection_id . '/records/' . rawurlencode($record_id),
            'DELETE',
            array(),
            __('RAG record deleted successfully!', 'smart-support-chat-widget'),
            __('Failed to delete RAG record.', 'smart-support-chat-widget')
        );
    }
}

