<?php
/**
 * Admin functionality for Smart Support plugin
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Load config if not already loaded
if (!function_exists('smartsupport_get_base_url')) {
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/config.php';
}

class SmartSupport_Admin {

    /**
     * Cached RAG collections keyed by option key
     */
    private $cached_collections = null;

    /**
     * Cached RAG collections grouped by post type
     */
    private $collections_by_post_type = null;

    /**
     * API client instance cache
     */
    private $api_client = null;

    /**
     * Base key for storing indexing notices per user
     */
    private $index_notice_key_base = 'smartsupport_index_notice_';

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_smartsupport_get_fields', array($this, 'ajax_get_fields'));
        add_action('wp_ajax_smartsupport_get_agents', array($this, 'ajax_get_agents'));
        add_action('admin_post_smartsupport_create_rag_collection', array($this, 'process_create_rag_collection'));
        add_action('admin_init', array($this, 'handle_reset_requests'));
        add_action('admin_init', array($this, 'register_post_type_integration'));
        add_filter('post_row_actions', array($this, 'add_index_row_action'), 10, 2);
        add_action('save_post', array($this, 'handle_post_save'), 10, 3);
        add_action('admin_post_smartsupport_index_post', array($this, 'handle_manual_index_request'));
        add_action('admin_post_smartsupport_update_sync_mode', array($this, 'handle_update_sync_mode'));
        add_action('admin_post_smartsupport_delete_rag_collection', array($this, 'handle_delete_rag_collection'));
        add_action('admin_post_smartsupport_delete_record', array($this, 'handle_delete_record_request'));
        add_action('admin_post_smartsupport_bulk_index', array($this, 'handle_bulk_index_request'));
        add_action('admin_post_smartsupport_update_collection_fields', array($this, 'handle_update_collection_fields'));
        add_action('wp_ajax_smartsupport_refresh_collection', array($this, 'ajax_refresh_collection'));
        add_action('admin_notices', array($this, 'render_index_notice'));
        add_action('before_delete_post', array($this, 'handle_post_deletion'));
        add_action('wp_trash_post', array($this, 'handle_post_moved_to_trash'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Check if this is one of our plugin pages
        // Hook format: toplevel_page_{menu_slug} or {parent_slug}_page_{menu_slug}
        $plugin_pages = array(
            'smart-support-chat-widget',
            'smartsupport-rag-collections', 
            'smartsupport-bulk-indexing',
            'smartsupport' // Also check for smartsupport in case of variations
        );
        $is_plugin_page = false;
        
        foreach ($plugin_pages as $page) {
            if (strpos($hook, $page) !== false) {
                $is_plugin_page = true;
                break;
            }
        }
        
        if (!$is_plugin_page) {
            return;
        }

        wp_enqueue_script('jquery');

        $handle = 'smart-support-chat-widget-admin';

        wp_enqueue_script(
            $handle,
            SMARTSUPPORT_WP_PLUGIN_URL . 'assets/js/smartsupport-admin.js',
            array('jquery'),
            SMARTSUPPORT_WP_VERSION,
            true
        );

        wp_localize_script(
            $handle,
            'smartsupportAdminVars',
            array(
                'strings' => array(
                    'selectPostType' => __('Select a post type first to see available fields.', 'smart-support-chat-widget'),
                    'securityTokenMissing' => __('Security token missing. Please refresh the page.', 'smart-support-chat-widget'),
                    'loadingFields' => __('Loading fields...', 'smart-support-chat-widget'),
                    'unableToLoadFields' => __('Unable to load fields for the selected post type.', 'smart-support-chat-widget'),
                    'requestFailed' => __('Request failed. Please try again.', 'smart-support-chat-widget'),
                ),
                'agentNonce' => wp_create_nonce('smartsupport_get_agents'),
                'refreshNonce' => wp_create_nonce('smartsupport_refresh_collection'),
            )
        );

        // Enqueue admin styles
        wp_enqueue_style(
            'smart-support-chat-widget-admin',
            SMARTSUPPORT_WP_PLUGIN_URL . 'assets/css/smartsupport-admin.css',
            array(),
            SMARTSUPPORT_WP_VERSION
        );
    }

    /**
     * Retrieve cached collections
     */
    private function get_collections_cache() {
        if (null === $this->cached_collections) {
            $this->cached_collections = $this->get_saved_rag_collections();
        }

        return $this->cached_collections;
    }

    /**
     * Retrieve cached collections grouped by post type
     */
    private function get_collections_by_post_type() {
        if (null === $this->collections_by_post_type) {
            $grouped = array();
            $collections = $this->get_collections_cache();

            foreach ($collections as $collection_key => $collection) {
                if (empty($collection['post_type'])) {
                    continue;
                }

                $post_type = $collection['post_type'];

                if (!isset($grouped[$post_type])) {
                    $grouped[$post_type] = array();
                }

                $collection['collection_key'] = $collection_key;
                $grouped[$post_type][$collection_key] = $collection;
            }

            $this->collections_by_post_type = $grouped;
        }

        return $this->collections_by_post_type;
    }

    /**
     * Get collections configured for a specific post type
     */
    private function get_collections_for_post_type($post_type) {
        $collections = $this->get_collections_by_post_type();
        return isset($collections[$post_type]) ? $collections[$post_type] : array();
    }

    /**
     * Generate transient key for indexing notices
     */
    private function get_notice_transient_key() {
        return $this->index_notice_key_base . get_current_user_id();
    }

    /**
     * Store notice for indexing operations
     */
    private function store_index_notice($notice) {
        if (!is_array($notice)) {
            return;
        }

        if (!isset($notice['type'])) {
            $notice['type'] = !empty($notice['success']) ? 'success' : 'error';
        }

        set_transient($this->get_notice_transient_key(), $notice, MINUTE_IN_SECONDS);
    }

    /**
     * Render admin notices for indexing operations
     */
    public function render_index_notice() {
        $notice = get_transient($this->get_notice_transient_key());

        if (false === $notice) {
            return;
        }

        delete_transient($this->get_notice_transient_key());

        $type = isset($notice['type']) && 'success' === $notice['type'] ? 'notice-success' : 'notice-error';
        $message = isset($notice['message']) ? $notice['message'] : '';
        $message_html = nl2br(esc_html($message));

        printf(
            '<div class="notice %1$s is-dismissible"><p>%2$s</p></div>',
            esc_attr($type),
            wp_kses_post($message_html)
        );
    }

    /**
     * Register admin table integrations for linked post types
     */
    public function register_post_type_integration() {
        $collections_by_post_type = $this->get_collections_by_post_type();

        if (empty($collections_by_post_type)) {
            return;
        }

        foreach ($collections_by_post_type as $post_type => $collections) {
            add_filter("manage_{$post_type}_posts_columns", array($this, 'add_index_column'));
            add_action("manage_{$post_type}_posts_custom_column", array($this, 'render_index_column'), 10, 2);
        }
    }

    /**
     * Add SmartSupport column to post list table
     */
    public function add_index_column($columns) {
        if (!is_array($columns)) {
            return $columns;
        }

        $new_columns = array();
        $inserted = false;

        foreach ($columns as $key => $label) {
            $new_columns[$key] = $label;

            if ('title' === $key) {
                $new_columns['smartsupport_rag'] = __('SmartSupport', 'smart-support-chat-widget');
                $inserted = true;
            }
        }

        if (!$inserted) {
            $new_columns['smartsupport_rag'] = __('SmartSupport', 'smart-support-chat-widget');
        }

        return $new_columns;
    }

    /**
     * Render SmartSupport column content
     */
    public function render_index_column($column, $post_id) {
        if ('smartsupport_rag' !== $column) {
            return;
        }

        $post = get_post($post_id);
        if (!$post) {
            echo '&mdash;';
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            echo esc_html__('No access', 'smart-support-chat-widget');
            return;
        }

        $collections = $this->get_collections_for_post_type($post->post_type);
        if (empty($collections)) {
            echo '&mdash;';
            return;
        }

        $items = array();
        foreach ($collections as $collection_key => $collection) {
            $meta_key = $this->get_collection_meta_key($collection_key);
            $record_id = get_post_meta($post_id, $meta_key, true);
            $status_meta_key = $this->get_collection_status_meta_key($collection_key);
            $status_value = get_post_meta($post_id, $status_meta_key, true);

            $is_indexed = (!empty($status_value) && 'indexed' === $status_value) || !empty($record_id);

            $status_text = $is_indexed ? esc_html__('Indexed', 'smart-support-chat-widget') : esc_html__('Not Indexed', 'smart-support-chat-widget');
            $button_label = $is_indexed ? esc_html__('Re-index', 'smart-support-chat-widget') : esc_html__('Index', 'smart-support-chat-widget');
            $collection_name = isset($collection['collection_name']) ? $collection['collection_name'] : $collection_key;
            $sync_mode = isset($collection['sync_mode']) ? $collection['sync_mode'] : 'auto';
            $mode_label = ('manual' === $sync_mode) ? esc_html__('Manual', 'smart-support-chat-widget') : esc_html__('Automatic', 'smart-support-chat-widget');
            $mode_class = ('manual' === $sync_mode) ? 'smartsupport-mode-manual' : 'smartsupport-mode-auto';
            $status_class = $is_indexed ? 'smartsupport-status-indexed' : 'smartsupport-status-pending';

            $status_badge = sprintf('<span class="smartsupport-status-badge %1$s">%2$s</span>', esc_attr($status_class), $status_text);
            $mode_badge = sprintf('<span class="smartsupport-mode-badge %1$s">%2$s</span>', esc_attr($mode_class), $mode_label);

            $referer = $this->get_default_referer_url($post->post_type);
            $url = $this->get_manual_index_url($post_id, $collection_key, $referer);

            $delete_url = !empty($record_id) ? $this->get_manual_delete_url($post_id, $collection_key, $referer) : '';

            $action_buttons = sprintf(
                '<a class="button button-secondary button-small" href="%1$s">%2$s</a>',
                esc_url($url),
                esc_html($button_label)
            );

            if (!empty($delete_url)) {
                $action_buttons .= sprintf(
                    ' <a class="button button-small smartsupport-button-danger" href="%1$s">%2$s</a>',
                    esc_url($delete_url),
                    esc_html__('Clear Index', 'smart-support-chat-widget')
                );
            }

            $items[] = sprintf(
                '<div class="smartsupport-index-actions"><strong>%1$s</strong>%2$s%3$s<br />%4$s</div>',
                esc_html($collection_name),
                $status_badge,
                $mode_badge,
                $action_buttons
            );
        }

        if (!empty($items)) {
            $output = implode('<hr class="smartsupport-index-divider" />', $items);
            $allowed_html = array(
                'div' => array('class' => true),
                'strong' => array(),
                'span' => array('class' => true),
                'a' => array('class' => true, 'href' => true),
                'br' => array(),
                'hr' => array('class' => true),
            );

            echo wp_kses($output, $allowed_html);
        }
    }

    /**
     * Add SmartSupport row action for quick indexing
     */
    public function add_index_row_action($actions, $post) {
        if (!is_object($post) || empty($post->ID)) {
            return $actions;
        }

        if (!current_user_can('edit_post', $post->ID)) {
            return $actions;
        }

        $collections = $this->get_collections_for_post_type($post->post_type);
        if (empty($collections)) {
            return $actions;
        }

        if (count($collections) === 1) {
            $collection_key = key($collections);
            $referer = $this->get_default_referer_url($post->post_type);
            $url = $this->get_manual_index_url($post->ID, $collection_key, $referer);
            $actions['smartsupport_index'] = sprintf(
                '<a href="%1$s">%2$s</a>',
                esc_url($url),
                esc_html__('Index in SmartSupport', 'smart-support-chat-widget')
            );
        }

        return $actions;
    }

    /**
     * Build meta key for storing record identifiers
     */
    private function get_collection_meta_key($collection_key) {
        return '_smartsupport_rag_record_' . sanitize_key($collection_key);
    }

    /**
     * Build meta key for storing status information
     */
    private function get_collection_status_meta_key($collection_key) {
        return '_smartsupport_rag_status_' . sanitize_key($collection_key);
    }

    /**
     * Generate nonce action for manual indexing
     */
    private function get_manual_index_nonce_action($post_id, $collection_key = '') {
        $post_id = intval($post_id);
        $collection_fragment = $collection_key ? sanitize_key($collection_key) : 'all';

        return 'smartsupport_index_post_' . $post_id . '_' . $collection_fragment;
    }

    /**
     * Generate nonce action for manual record deletion
     */
    private function get_manual_delete_nonce_action($post_id, $collection_key = '') {
        $post_id = intval($post_id);
        $collection_fragment = $collection_key ? sanitize_key($collection_key) : 'all';

        return 'smartsupport_delete_record_' . $post_id . '_' . $collection_fragment;
    }

    /**
     * Generate manual indexing URL
     */
    private function get_manual_index_url($post_id, $collection_key = '', $referer = '') {
        $query_args = array(
            'action' => 'smartsupport_index_post',
            'post_id' => intval($post_id)
        );

        if (!empty($collection_key)) {
            $query_args['collection_key'] = sanitize_key($collection_key);
        }

        if (!empty($referer)) {
            $query_args['_wp_http_referer'] = rawurlencode($referer);
        }

        $url = add_query_arg($query_args, admin_url('admin-post.php'));
        return wp_nonce_url($url, $this->get_manual_index_nonce_action($post_id, $collection_key));
    }

    /**
     * Generate manual deletion URL
     */
    private function get_manual_delete_url($post_id, $collection_key = '', $referer = '') {
        $query_args = array(
            'action' => 'smartsupport_delete_record',
            'post_id' => intval($post_id)
        );

        if (!empty($collection_key)) {
            $query_args['collection_key'] = sanitize_key($collection_key);
        }

        if (!empty($referer)) {
            $query_args['_wp_http_referer'] = rawurlencode($referer);
        }

        $url = add_query_arg($query_args, admin_url('admin-post.php'));
        return wp_nonce_url($url, $this->get_manual_delete_nonce_action($post_id, $collection_key));
    }

    /**
     * Get default referer URL for post type listings
     */
    private function get_default_referer_url($post_type) {
        $post_type = sanitize_key($post_type);
        $base_url = admin_url('edit.php');

        if ('post' !== $post_type) {
            $base_url = add_query_arg('post_type', $post_type, $base_url);
        }

        return $base_url;
    }

    /**
     * Sanitize POST array data recursively
     */
    private function sanitize_post_array($array) {
        if (!is_array($array)) {
            return sanitize_text_field($array);
        }

        $sanitized = array();
        foreach ($array as $key => $value) {
            $sanitized_key = sanitize_key($key);
            if (is_array($value)) {
                $sanitized[$sanitized_key] = $this->sanitize_post_array($value);
            } else {
                $sanitized[$sanitized_key] = sanitize_text_field(wp_unslash($value));
            }
        }
        return $sanitized;
    }

    /**
     * Retrieve API client instance
     */
    private function get_api_client() {
        if (null === $this->api_client) {
            if (!class_exists('SmartSupport_API')) {
                require_once SMARTSUPPORT_WP_PLUGIN_DIR . 'includes/class-smartsupport-api.php';
            }

            $this->api_client = new SmartSupport_API();
        }

        return $this->api_client;
    }

    /**
     * Build record fields payload for API calls
     */
    private function build_record_fields($post_id, $collection_key, $collection) {
        $fields_payload = array();
        $field_map = array();

        if (isset($collection['field_map']) && is_array($collection['field_map'])) {
            $field_map = $collection['field_map'];
        } elseif (isset($collection['fields']) && is_array($collection['fields'])) {
            foreach ($collection['fields'] as $field) {
                if (empty($field['field_name'])) {
                    continue;
                }

                $field_name = $field['field_name'];
                $field_map[$field_name] = array(
                    'field_unique' => isset($field['field_unique']) ? $field['field_unique'] : '',
                    'type' => isset($field['type']) ? $field['type'] : 'string',
                    'embed' => isset($field['embed']) ? (bool) $field['embed'] : false
                );
            }
        }

        if (empty($field_map)) {
            return array(
                'fields' => array(),
                'field_map' => array(),
                'collection' => $collection
            );
        }

        // V2 API uses field_name directly, no need to check for field_unique
        $field_source = isset($collection['field_source']) ? $collection['field_source'] : 'meta';

        foreach ($field_map as $field_name => $field_config) {
            // V2 API uses field_name (the actual field name like "post_title"), not field_unique
            $raw_value = $this->get_field_value_for_post($post_id, $field_name, $field_source);
            $field_type = isset($field_config['type']) ? $field_config['type'] : 'string';
            $normalized_value = $this->normalize_field_value($raw_value, $field_type);

            $fields_payload[] = array(
                'field_name' => $field_name,  // Use actual field_name, not field_unique
                'value' => $normalized_value
            );
        }

        return array(
            'fields' => $fields_payload,
            'field_map' => $field_map,
            'collection' => $collection
        );
    }

    /**
     * Retrieve value for a specific field
     */
    private function get_field_value_for_post($post_id, $field_name, $field_source = 'meta') {
        $post = get_post($post_id);
        if (!$post) {
            return '';
        }

        switch ($field_name) {
            case 'post_title':
                return get_the_title($post_id);
            case 'post_content':
                // Content will be cleaned in normalize_field_value
                return $post->post_content;
            case 'post_excerpt':
                $excerpt = $post->post_excerpt;
                if (empty($excerpt)) {
                    $excerpt = get_the_excerpt($post);
                }
                return wp_strip_all_tags($excerpt);
            case 'post_id':
                return $post->ID;
            case 'post_url':
                return get_permalink($post_id);
            case 'post_author':
                return $post->post_author;
            case 'post_date':
                return $post->post_date;
            default:
                // Currently ACPT fields are stored as meta, so share logic
                return get_post_meta($post_id, $field_name, true);
        }
    }

    /**
     * Clean HTML tags and WordPress block comments from content
     * Preserves semantic structure (line breaks) for semantic chunking
     */
    private function clean_content_for_indexing($content) {
        if (!is_string($content) || empty($content)) {
            return $content;
        }

        // Remove WordPress block comments (e.g., <!-- wp:paragraph -->, <!-- /wp:paragraph -->)
        $content = preg_replace('/<!--\s*\/?wp:[\w-]+\s*[^>]*-->/i', '', $content);

        // Convert HTML block elements to line breaks before stripping tags
        // This preserves semantic structure for chunking
        $block_elements = array(
            '</p>', '</div>', '</h1>', '</h2>', '</h3>', '</h4>', '</h5>', '</h6>',
            '</li>', '</tr>', '</td>', '</th>', '</blockquote>', '</pre>',
            '<br>', '<br/>', '<br />', '<hr>', '<hr/>', '<hr />'
        );
        foreach ($block_elements as $tag) {
            $content = str_ireplace($tag, "\n", $content);
        }

        // Convert opening block elements (after content) to line breaks for better structure
        $opening_block_elements = array(
            '<p>', '<div>', '<h1>', '<h2>', '<h3>', '<h4>', '<h5>', '<h6>',
            '<li>', '<tr>', '<blockquote>', '<pre>'
        );
        foreach ($opening_block_elements as $tag) {
            // Add line break after opening tag if not already followed by newline
            $content = preg_replace('/' . preg_quote($tag, '/') . '(?!\s*\n)/i', $tag . "\n", $content);
        }

        // Remove all HTML tags (now that we've converted block elements to line breaks)
        $content = wp_strip_all_tags($content);

        // Normalize different line break formats to \n
        $content = str_replace(array("\r\n", "\r"), "\n", $content);

        // Clean up multiple consecutive spaces within lines (but preserve line breaks)
        $content = preg_replace('/[ \t]+/', ' ', $content);

        // Clean up excessive consecutive line breaks (keep max 2 consecutive line breaks)
        $content = preg_replace('/\n{3,}/', "\n\n", $content);

        // Trim whitespace from start and end, but preserve internal line breaks
        $content = trim($content);

        return $content;
    }

    /**
     * Normalize field value based on type
     */
    private function normalize_field_value($value, $type = 'string') {
        if ('boolean' === $type) {
            return (bool) $value;
        }

        if ('number' === $type) {
            if (is_numeric($value)) {
                return 0 + $value;
            }

            return 0;
        }

        if (is_array($value) || is_object($value)) {
            $encoded = wp_json_encode($value);
            return is_string($encoded) ? $encoded : '';
        }

        if (is_null($value)) {
            return '';
        }

        if (is_string($value)) {
            // Clean HTML and block comments for text/string types before indexing
            if ('text' === $type || 'string' === $type) {
                $value = $this->clean_content_for_indexing($value);
            }
            return $value;
        }

        return (string) $value;
    }

    /**
     * Ensure collection schema includes field_unique values
     */
    private function refresh_collection_schema($collection_key, $collection) {
        $api_id = isset($collection['api_id']) ? intval($collection['api_id']) : 0;
        if ($api_id <= 0) {
            return $collection;
        }

        $api = $this->get_api_client();
        if (!$api) {
            return $collection;
        }

        $response = $api->get_rag_collection($api_id);
        if (empty($response['success'])) {
            return $collection;
        }

        // V2 API returns: { success: true, data: { collection: {...} } }
        $rag_collection = array();
        if (isset($response['collection']) && is_array($response['collection'])) {
            $rag_collection = $response['collection'];
        } elseif (isset($response['data']['collection']) && is_array($response['data']['collection'])) {
            $rag_collection = $response['data']['collection'];
        } elseif (isset($response['data']['ragCollection']) && is_array($response['data']['ragCollection'])) {
            $rag_collection = $response['data']['ragCollection'];
        } elseif (isset($response['data']) && is_array($response['data'])) {
            $rag_collection = $response['data'];
        }

        if (empty($rag_collection) || !isset($rag_collection['fields']) || !is_array($rag_collection['fields'])) {
            return $collection;
        }

        $selected_fields = isset($collection['fields']) && is_array($collection['fields']) ? $collection['fields'] : array();
        $post_type = isset($collection['post_type']) ? $collection['post_type'] : '';
        $field_source = isset($collection['field_source']) ? $collection['field_source'] : '';
        $sync_mode_current = isset($collection['sync_mode']) ? $collection['sync_mode'] : 'auto';

        $this->save_rag_collection($rag_collection, $selected_fields, $post_type, $field_source, $sync_mode_current);

        $collections = $this->get_collections_cache();
        if (isset($collections[$collection_key])) {
            return $collections[$collection_key];
        }

        return $collection;
    }

    /**
     * Extract document_id from V2 create response payload
     */
    private function extract_record_id_from_create_response($response) {
        if (empty($response) || !is_array($response)) {
            return '';
        }

        // V2 API returns: { success: true, data: { record: { id: document_id, record_id: ... } } }
        if (isset($response['data']['record']['id'])) {
            return sanitize_text_field($response['data']['record']['id']);
        }

        // Fallback: check for document_id directly in response
        if (isset($response['document_id'])) {
            return sanitize_text_field($response['document_id']);
        }

        // Legacy format fallback
        $payload = isset($response['data']) && is_array($response['data']) ? $response['data'] : $response;
        if (isset($payload['ids']) && is_array($payload['ids'])) {
            foreach ($payload['ids'] as $id_value) {
                if (is_string($id_value) && '' !== trim($id_value)) {
                    return sanitize_text_field($id_value);
                }
            }
        }

        return '';
    }

    /**
     * Attempt to locate existing document_id for a post (V2 API)
     */
    private function lookup_document_id_for_post($post_id, $collection_key, $collection, $field_map, $fields_payload) {
        $api_id = isset($collection['api_id']) ? intval($collection['api_id']) : 0;
        if ($api_id <= 0) {
            return '';
        }

        $api = $this->get_api_client();
        if (!$api) {
            return '';
        }

        // Use V2 API to get records
        $response = $api->get_rag_records_v2($api_id);
        if (empty($response['success']) || empty($response['data']) || !is_array($response['data'])) {
            return '';
        }

        // V2 API returns: { records: [...], pagination: {...} }
        $records = isset($response['data']['records']) && is_array($response['data']['records']) 
            ? $response['data']['records'] 
            : array();
        
        if (empty($records)) {
            return '';
        }

        // Build payload map using field_name (not field_unique)
        $payload_map = array();
        foreach ($fields_payload as $field_entry) {
            if (isset($field_entry['field_name'])) {
                $payload_map[$field_entry['field_name']] = isset($field_entry['value']) ? $field_entry['value'] : '';
            }
        }

        // Look for post_id field in payload
        foreach ($records as $record) {
            if (empty($record['id'])) {
                continue;
            }

            // V2 API returns: { id: document_id, values: {...}, record_id: ... }
            $record_values = isset($record['values']) && is_array($record['values']) ? $record['values'] : array();

            // Check if post_id matches
            if (isset($payload_map['post_id']) && isset($record_values['post_id'])) {
                if ((string) $record_values['post_id'] === (string) $post_id) {
                    return sanitize_text_field($record['id']);
                }
            }

            // Also check by comparing all field values
            if (!empty($payload_map) && !empty($record_values)) {
                $matches_payload = true;
                foreach ($payload_map as $field_name => $value) {
                    if (isset($record_values[$field_name])) {
                        if ((string) $record_values[$field_name] !== (string) $value) {
                            $matches_payload = false;
                            break;
                        }
                    }
                }

                if ($matches_payload) {
                    return sanitize_text_field($record['id']);
                }
            }
        }

        return '';
    }

    /**
     * Extract record entries from a response payload
     */
    private function extract_records_from_response_data($data, $depth = 0) {
        if ($depth > 5 || !is_array($data)) {
            return array();
        }

        $records = array();

        if (isset($data['collection']) && is_array($data['collection'])) {
            $records = array_merge($records, $this->extract_records_from_collection_data($data['collection']));
        }

        if (isset($data['records']) && is_array($data['records'])) {
            foreach ($data['records'] as $record_entry) {
                $normalized = $this->normalize_record_payload($record_entry);
                if ($normalized) {
                    $records[] = $normalized;
                }
            }
        }

        if (isset($data['ids']) && is_array($data['ids'])) {
            $records = array_merge($records, $this->extract_records_from_collection_data($data));
        }

        if (empty($records)) {
            foreach ($data as $value) {
                if (is_array($value)) {
                    $records = array_merge($records, $this->extract_records_from_response_data($value, $depth + 1));
                }
            }
        }

        return $records;
    }

    /**
     * Extract record entries from collection structured data
     */
    private function extract_records_from_collection_data($collection_data) {
        if (!is_array($collection_data)) {
            return array();
        }

        $records = array();
        $ids = array();

        if (isset($collection_data['ids']) && is_array($collection_data['ids'])) {
            foreach ($collection_data['ids'] as $id_value) {
                if (is_string($id_value) && '' !== $id_value) {
                    $ids[] = sanitize_text_field($id_value);
                }
            }
        }

        $data_sets = array();
        if (isset($collection_data['data']) && is_array($collection_data['data'])) {
            $data_sets = $collection_data['data'];
        }

        foreach ($ids as $index => $id_value) {
            $record_payload = array();
            if (isset($data_sets[$index]) && is_array($data_sets[$index])) {
                $record_payload = $this->normalize_record_data_array($data_sets[$index]);
            }

            $records[] = array(
                'id' => $id_value,
                'data' => $record_payload
            );
        }

        if (isset($collection_data['records']) && is_array($collection_data['records'])) {
            foreach ($collection_data['records'] as $record_entry) {
                $normalized = $this->normalize_record_payload($record_entry);
                if ($normalized) {
                    $records[] = $normalized;
                }
            }
        }

        return $records;
    }

    /**
     * Normalize a single record payload
     */
    private function normalize_record_payload($record_entry) {
        if (!is_array($record_entry)) {
            return null;
        }

        $record_id = '';
        if (isset($record_entry['id']) && is_string($record_entry['id'])) {
            $record_id = sanitize_text_field($record_entry['id']);
        } elseif (isset($record_entry['record_id']) && is_string($record_entry['record_id'])) {
            $record_id = sanitize_text_field($record_entry['record_id']);
        } elseif (isset($record_entry['record']) && is_array($record_entry['record'])) {
            $nested = $this->normalize_record_payload($record_entry['record']);
            if ($nested) {
                return $nested;
            }
        }

        if (empty($record_id)) {
            return null;
        }

        $record_data = array();
        if (isset($record_entry['fields']) && is_array($record_entry['fields'])) {
            $record_data = $this->normalize_record_data_array($record_entry['fields']);
        } elseif (isset($record_entry['data']) && is_array($record_entry['data'])) {
            $record_data = $this->normalize_record_data_array($record_entry['data']);
        }

        return array(
            'id' => $record_id,
            'data' => $record_data
        );
    }

    /**
     * Normalize record data structures to simple associative arrays
     */
    private function normalize_record_data_array($data) {
        $normalized = array();

        if (!is_array($data)) {
            return $normalized;
        }

        foreach ($data as $key => $value) {
            if (!is_string($key) || '' === $key) {
                continue;
            }

            if (is_scalar($value) || is_null($value)) {
                $normalized[$key] = (string) $value;
            } elseif (is_array($value)) {
                if (count($value) === 1 && isset($value[0]) && (is_scalar($value[0]) || is_null($value[0]))) {
                    $normalized[$key] = (string) $value[0];
                } else {
                    $normalized[$key] = wp_json_encode($value);
                }
            } else {
                $normalized[$key] = (string) $value;
            }
        }

        return $normalized;
    }

    /**
     * Validate REST API requests using the configured API key
     */
    public function rest_validate_request($request) {
        $settings = get_option('smartsupport_wp_settings', array());
        $configured_key = isset($settings['api_key']) ? trim($settings['api_key']) : '';

        if (empty($configured_key)) {
            return new WP_Error(
                'smartsupport_api_key_missing',
                __('SmartSupport API key is not configured.', 'smart-support-chat-widget'),
                array('status' => 500)
            );
        }

        $provided_key = $request->get_header('x-api-key');
        if (empty($provided_key)) {
            $provided_key = $request->get_param('api_key');
        }

        if (empty($provided_key) || !hash_equals($configured_key, trim($provided_key))) {
            return new WP_Error(
                'smartsupport_invalid_api_key',
                __('Invalid or missing API key.', 'smart-support-chat-widget'),
                array('status' => 403)
            );
        }

        return true;
    }

    /**
     * REST endpoint handler to synchronise a post with linked collections
     */
    public function rest_handle_sync_request($request) {
        $post_id = absint($request->get_param('post_id'));

        if ($post_id <= 0) {
            return new WP_Error(
                'smartsupport_invalid_post',
                __('A valid post ID must be provided.', 'smart-support-chat-widget'),
                array('status' => 400)
            );
        }

        $post = get_post($post_id);
        if (!$post || 'trash' === $post->post_status) {
            return new WP_Error(
                'smartsupport_post_not_found',
                __('The requested post could not be found or is unavailable.', 'smart-support-chat-widget'),
                array('status' => 404)
            );
        }

        $collections = $this->get_collections_for_post_type($post->post_type);
        if (empty($collections)) {
            return new WP_Error(
                'smartsupport_no_collections',
                __('No SmartSupport collections are linked to this post type.', 'smart-support-chat-widget'),
                array('status' => 404)
            );
        }

        $requested_collections = $request->get_param('collections');
        if (is_array($requested_collections) && !empty($requested_collections)) {
            $requested_keys = array();
            foreach ($requested_collections as $key) {
                $requested_keys[] = sanitize_key($key);
            }
            $requested_keys = array_filter($requested_keys);

            if (!empty($requested_keys)) {
                $collections = array_intersect_key($collections, array_flip($requested_keys));
            }

            if (empty($collections)) {
                return new WP_Error(
                    'smartsupport_invalid_collection',
                    __('None of the requested collections are linked to this post type.', 'smart-support-chat-widget'),
                    array('status' => 404)
                );
            }
        }

        $force = (bool) $request->get_param('force');
        $results = array();
        $overall_success = true;

        foreach ($collections as $collection_key => $collection) {
            $result = $this->sync_post_to_collection($post_id, $collection_key, $collection, $force);

            if (!is_array($result)) {
                $result = array(
                    'success' => false,
                    'message' => __('Unexpected response while synchronising the record.', 'smart-support-chat-widget')
                );
            }

            if (empty($result['success'])) {
                $overall_success = false;
            }

            $results[$collection_key] = $result;
        }

        return rest_ensure_response(array(
            'success' => $overall_success,
            'post_id' => $post_id,
            'results' => $results
        ));
    }

    /**
     * Remove stored SmartSupport record references for a post
     */
    private function delete_post_records($post_id, $collections) {
        if (empty($collections) || !is_array($collections)) {
            return;
        }

        foreach ($collections as $collection_key => $collection) {
            $meta_key = $this->get_collection_meta_key($collection_key);
            $status_meta_key = $this->get_collection_status_meta_key($collection_key);
            delete_post_meta($post_id, $meta_key);
            delete_post_meta($post_id, $status_meta_key);
        }
    }

    /**
     * Delete a single record from SmartSupport and clear local tracking
     */
    private function delete_record_for_collection($collection_id, $record_id, $post_id, $collection_key) {
        $collection_id = intval($collection_id);
        $record_id = sanitize_text_field($record_id);

        if ($collection_id <= 0 || empty($record_id)) {
            return array(
                'success' => false,
                'message' => __('Missing collection or record identifier.', 'smart-support-chat-widget')
            );
        }

        $api = $this->get_api_client();
        if (!$api) {
            return array(
                'success' => false,
                'message' => __('Unable to initialize SmartSupport API client.', 'smart-support-chat-widget')
            );
        }

        $response = $api->delete_rag_record($collection_id, $record_id);

        if (empty($response['success'])) {
            if (isset($response['status_code']) && 404 === intval($response['status_code'])) {
                $response['success'] = true;
                $response['message'] = __('Record was not found remotely; treated as deleted.', 'smart-support-chat-widget');
            } else {
                return array(
                    'success' => false,
                    'message' => isset($response['message']) ? $response['message'] : __('Failed to delete record from SmartSupport.', 'smart-support-chat-widget')
                );
            }
        }

        $meta_key = $this->get_collection_meta_key($collection_key);
        $status_meta_key = $this->get_collection_status_meta_key($collection_key);
        delete_post_meta($post_id, $meta_key);
        delete_post_meta($post_id, $status_meta_key);

        $success_message = isset($response['message']) ? $response['message'] : __('Record deleted successfully.', 'smart-support-chat-widget');

        return array(
            'success' => true,
            'message' => $success_message
        );
    }

    /**
     * Sync a post with a SmartSupport RAG collection
     */
    private function sync_post_to_collection($post_id, $collection_key, $collection, $force_create = false) {
        $api_id = isset($collection['api_id']) ? intval($collection['api_id']) : 0;
        if ($api_id <= 0) {
            return array(
                'success' => false,
                'message' => __('Collection ID is missing.', 'smart-support-chat-widget')
            );
        }

        $field_build = $this->build_record_fields($post_id, $collection_key, $collection);
        $fields_payload = isset($field_build['fields']) ? $field_build['fields'] : array();
        $field_map = isset($field_build['field_map']) ? $field_build['field_map'] : array();
        if (isset($field_build['collection']) && is_array($field_build['collection'])) {
            $collection = $field_build['collection'];
        }

        if (empty($fields_payload)) {
            return array(
                'success' => false,
                'message' => __('No fields available to index for this post.', 'smart-support-chat-widget')
            );
        }

        $api = $this->get_api_client();
        if (!$api) {
            return array(
                'success' => false,
                'message' => __('Unable to initialize SmartSupport API client.', 'smart-support-chat-widget')
            );
        }

        $meta_key = $this->get_collection_meta_key($collection_key);
        $status_meta_key = $this->get_collection_status_meta_key($collection_key);
        $document_id = get_post_meta($post_id, $meta_key, true); // Store document_id from V2 API
        $is_indexed = get_post_meta($post_id, $status_meta_key, true) === 'indexed';

        if (empty($document_id)) {
            $lookup_document_id = $this->lookup_document_id_for_post($post_id, $collection_key, $collection, $field_map, $fields_payload);
            if (!empty($lookup_document_id)) {
                $document_id = $lookup_document_id;
                update_post_meta($post_id, $meta_key, $document_id);
            }
        }

        if ($force_create && !empty($document_id)) {
            $this->delete_record_for_collection($api_id, $document_id, $post_id, $collection_key);
            $document_id = '';
        }

        if ($is_indexed && !$force_create) {
            if (empty($document_id)) {
                $document_id = $this->lookup_document_id_for_post($post_id, $collection_key, $collection, $field_map, $fields_payload);
                if (!empty($document_id)) {
                    update_post_meta($post_id, $meta_key, $document_id);
                }
            }

            if (empty($document_id)) {
                delete_post_meta($post_id, $status_meta_key);
                delete_post_meta($post_id, $meta_key);
                return $this->sync_post_to_collection($post_id, $collection_key, $collection, true);
            }

            // V2 API uses document_id (not record_id array) for updates
            $response = $api->update_rag_record($api_id, $document_id, $fields_payload);

            if (!empty($response['success'])) {
                update_post_meta($post_id, $status_meta_key, 'indexed');
                return array(
                    'success' => true,
                    'message' => __('Record updated successfully.', 'smart-support-chat-widget'),
                    'document_id' => $document_id,
                    'action' => 'update'
                );
            }

            // Attempt to create a new record if update failed (e.g., missing record)
            if (!empty($document_id)) {
                $this->delete_record_for_collection($api_id, $document_id, $post_id, $collection_key);
            } else {
                delete_post_meta($post_id, $status_meta_key);
            }

            return $this->sync_post_to_collection($post_id, $collection_key, $collection, true);
        }

        $response = $api->create_rag_record($api_id, $fields_payload);

        if (!empty($response['success'])) {
            $new_document_id = $this->extract_record_id_from_create_response($response);
            if (!empty($new_document_id)) {
                update_post_meta($post_id, $meta_key, $new_document_id);
                $document_id = $new_document_id;
            } else {
                $lookup_document_id = $this->lookup_document_id_for_post($post_id, $collection_key, $collection, $field_map, $fields_payload);
                if (!empty($lookup_document_id)) {
                    update_post_meta($post_id, $meta_key, $lookup_document_id);
                    $document_id = $lookup_document_id;
                } else {
                    delete_post_meta($post_id, $meta_key);
                }
            }

            update_post_meta($post_id, $status_meta_key, 'indexed');

            return array(
                'success' => true,
                'message' => __('Record indexed successfully.', 'smart-support-chat-widget'),
                'document_id' => $document_id,
                'action' => 'create'
            );
        }

        delete_post_meta($post_id, $status_meta_key);
        return array(
            'success' => false,
            'message' => isset($response['message']) ? $response['message'] : __('Failed to sync record with SmartSupport.', 'smart-support-chat-widget')
        );
    }

    /**
     * Handle automatic syncing on post save
     */
    public function handle_post_save($post_id, $post, $update) {
        if (!is_object($post) || wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        $collections = $this->get_collections_for_post_type($post->post_type);
        if (empty($collections)) {
            return;
        }

        if (!in_array($post->post_status, array('publish', 'future', 'private'), true)) {
            return;
        }

        foreach ($collections as $collection_key => $collection) {
            if (isset($collection['sync_mode']) && 'manual' === $collection['sync_mode']) {
                continue;
            }
            $this->sync_post_to_collection($post_id, $collection_key, $collection);
        }
    }

    /**
     * Handle post deletion to clear remote records
     */
    public function handle_post_deletion($post_id) {
        $post = get_post($post_id);

        if (!$post) {
            return;
        }

        $collections = $this->get_collections_for_post_type($post->post_type);
        if (empty($collections)) {
            $this->delete_post_records($post_id, array());
            return;
        }

        foreach ($collections as $collection_key => $collection) {
            $meta_key = $this->get_collection_meta_key($collection_key);
            $record_id = get_post_meta($post_id, $meta_key, true);

            if (empty($record_id)) {
                continue;
            }

            $api_id = isset($collection['api_id']) ? intval($collection['api_id']) : 0;
            $this->delete_record_for_collection($api_id, $record_id, $post_id, $collection_key);
        }
    }

    /**
     * Handle post being moved to trash
     */
    public function handle_post_moved_to_trash($post_id) {
        $this->handle_post_deletion($post_id);
    }

    /**
     * Handle updates to collection indexing mode
     */
    public function handle_update_sync_mode() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'smart-support-chat-widget'));
        }

        // Get collection_key first for nonce verification (minimal read for security)
        $collection_key = isset($_POST['collection_key']) ? sanitize_key(wp_unslash($_POST['collection_key'])) : '';

        if (empty($collection_key)) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Invalid collection specified for sync mode update.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        // Verify nonce BEFORE processing any other user input
        check_admin_referer('smartsupport_update_sync_mode_' . $collection_key);

        $sync_mode = isset($_POST['sync_mode']) ? sanitize_text_field(wp_unslash($_POST['sync_mode'])) : 'auto';

        if (!in_array($sync_mode, array('auto', 'manual'), true)) {
            $sync_mode = 'auto';
        }

        $collections = get_option('smartsupport_wp_rag_collections', array());
        if (!is_array($collections) || !isset($collections[$collection_key])) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('The specified collection could not be found.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        $collections[$collection_key]['sync_mode'] = $sync_mode;

        update_option('smartsupport_wp_rag_collections', $collections, false);

        $this->cached_collections = null;
        $this->collections_by_post_type = null;

        $this->store_rag_notice(array(
            'success' => true,
            'type' => 'success',
            'message' => __('Indexing mode updated successfully.', 'smart-support-chat-widget')
        ));

        $redirect = admin_url('admin.php?page=smartsupport-rag-collections');
        if (isset($_POST['_wp_http_referer'])) {
            // Sanitize and validate referer URL
            $raw_referer = sanitize_text_field(wp_unslash($_POST['_wp_http_referer']));
            $decoded_referer = is_string($raw_referer) ? rawurldecode($raw_referer) : '';
            if (!empty($decoded_referer)) {
                // Validate URL and ensure it's safe for redirect
                $redirect_candidate = esc_url_raw($decoded_referer);
                // Additional validation: ensure it's an admin URL to prevent open redirects
                if (!empty($redirect_candidate) && (strpos($redirect_candidate, admin_url()) === 0 || strpos($redirect_candidate, home_url()) === 0)) {
                    $redirect = $redirect_candidate;
                }
            }
        }

        wp_safe_redirect($redirect);
        exit;
    }

    /**
     * Handle collection deletion requests
     */
    public function handle_delete_rag_collection() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'smart-support-chat-widget'));
        }

        // Get collection_key first for nonce verification (minimal read for security)
        $collection_key = isset($_POST['collection_key']) ? sanitize_key(wp_unslash($_POST['collection_key'])) : '';

        if (empty($collection_key)) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Invalid collection specified for deletion.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        // Verify nonce BEFORE processing any other user input
        check_admin_referer('smartsupport_delete_rag_collection_' . $collection_key);

        $collections = get_option('smartsupport_wp_rag_collections', array());
        if (!is_array($collections) || !isset($collections[$collection_key])) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('The specified collection could not be found.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        $collection = $collections[$collection_key];
        $api_id = isset($collection['api_id']) ? intval($collection['api_id']) : 0;

        $message = '';
        $success = true;

        if ($api_id > 0) {
            $api = $this->get_api_client();
            if (!$api) {
                $success = false;
                $message = __('Unable to initialize SmartSupport API client.', 'smart-support-chat-widget');
            } else {
                $response = $api->delete_rag_collection($api_id);
                if (empty($response['success'])) {
                    $success = false;
                    $message = isset($response['message']) ? $response['message'] : __('Failed to delete collection from SmartSupport.', 'smart-support-chat-widget');
                }
            }
        }

        if ($success) {
            unset($collections[$collection_key]);
            update_option('smartsupport_wp_rag_collections', $collections, false);

            $this->cached_collections = null;
            $this->collections_by_post_type = null;

            $message = __('Collection deleted successfully.', 'smart-support-chat-widget');
        }

        $this->store_rag_notice(array(
            'success' => $success,
            'type' => $success ? 'success' : 'error',
            'message' => $message
        ));

        $redirect = admin_url('admin.php?page=smartsupport-rag-collections');
        if (isset($_POST['_wp_http_referer'])) {
            // Sanitize and validate referer URL
            $raw_referer = sanitize_text_field(wp_unslash($_POST['_wp_http_referer']));
            $decoded_referer = is_string($raw_referer) ? rawurldecode($raw_referer) : '';
            if (!empty($decoded_referer)) {
                // Validate URL and ensure it's safe for redirect
                $redirect_candidate = esc_url_raw($decoded_referer);
                // Additional validation: ensure it's an admin URL to prevent open redirects
                if (!empty($redirect_candidate) && (strpos($redirect_candidate, admin_url()) === 0 || strpos($redirect_candidate, home_url()) === 0)) {
                    $redirect = $redirect_candidate;
                }
            }
        }

        wp_safe_redirect($redirect);
        exit;
    }

    /**
     * Handle collection fields update
     */
    public function handle_update_collection_fields() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'smart-support-chat-widget'));
        }

        $collection_key = isset($_POST['collection_key']) ? sanitize_key(wp_unslash($_POST['collection_key'])) : '';
        $collection_id = isset($_POST['collection_id']) ? intval($_POST['collection_id']) : 0;

        if (empty($collection_key) || $collection_id <= 0) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Invalid collection specified.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        check_admin_referer('smartsupport_update_collection_fields_' . $collection_key, 'smartsupport_fields_nonce');

        $api = $this->get_api_client();
        if (!$api) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Unable to initialize SmartSupport API client.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        // Get collection details from API first
        $collection_response = $api->get_rag_collection($collection_id);
        if (empty($collection_response['success']) || empty($collection_response['collection'])) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Failed to retrieve collection details from API.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        $api_collection = $collection_response['collection'];
        $collection_name = isset($api_collection['collection_name']) ? sanitize_text_field($api_collection['collection_name']) : '';
        $description = isset($api_collection['description']) ? sanitize_textarea_field($api_collection['description']) : '';

        // Process fields from form
        $fields = array();
        if (isset($_POST['fields']) && is_array($_POST['fields'])) {
            foreach ($_POST['fields'] as $field_data) {
                if (!is_array($field_data)) {
                    continue;
                }

                $field_name = isset($field_data['field_name']) ? sanitize_text_field($field_data['field_name']) : '';
                if (empty($field_name)) {
                    continue;
                }

                $type = isset($field_data['type']) ? sanitize_text_field($field_data['type']) : 'string';
                $allowed_types = array('string', 'text', 'number', 'boolean');
                if (!in_array($type, $allowed_types, true)) {
                    $type = 'string';
                }

                $embed = isset($field_data['embed']) && '1' === (string) $field_data['embed'];
                $metadata = isset($field_data['metadata']) && '1' === (string) $field_data['metadata'];
                
                // At least one of embed or metadata must be true
                if (!$embed && !$metadata) {
                    // Default based on field name
                    if (in_array($field_name, array('post_id', 'post_url'), true)) {
                        $metadata = true;
                    } else {
                        $embed = true;
                    }
                }

                $field_array = array(
                    'field_name' => $field_name,
                    'type' => $type,
                    'embed' => $embed,
                    'metadata' => $metadata
                );

                // Include field ID if present (for existing fields)
                if (isset($field_data['id']) && !empty($field_data['id'])) {
                    $field_array['id'] = intval($field_data['id']);
                }

                $fields[] = $field_array;
            }
        }

        if (empty($fields)) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('At least one field is required.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-rag-collections'));
            exit;
        }

        // Update collection via API
        $response = $api->update_rag_collection($collection_id, $collection_name, $description, array(), $fields);

        if (!empty($response['success'])) {
            // Refresh collection data from API
            $updated_response = $api->get_rag_collection($collection_id);
            if (!empty($updated_response['success']) && !empty($updated_response['collection'])) {
                $collections = get_option('smartsupport_wp_rag_collections', array());
                if (is_array($collections) && isset($collections[$collection_key])) {
                    // Update local collection with fresh API data
                    $this->refresh_collection_schema($collection_key, $collections[$collection_key]);
                }
            }

            $this->store_rag_notice(array(
                'success' => true,
                'type' => 'success',
                'message' => __('Collection fields updated successfully!', 'smart-support-chat-widget')
            ));
        } else {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => isset($response['message']) ? $response['message'] : __('Failed to update collection fields.', 'smart-support-chat-widget')
            ));
        }

        $redirect = admin_url('admin.php?page=smartsupport-rag-collections');
        if (isset($_POST['_wp_http_referer'])) {
            $raw_referer = sanitize_text_field(wp_unslash($_POST['_wp_http_referer']));
            $decoded_referer = is_string($raw_referer) ? rawurldecode($raw_referer) : '';
            if (!empty($decoded_referer)) {
                $redirect_candidate = esc_url_raw($decoded_referer);
                if (!empty($redirect_candidate) && (strpos($redirect_candidate, admin_url()) === 0 || strpos($redirect_candidate, home_url()) === 0)) {
                    $redirect = $redirect_candidate;
                }
            }
        }

        wp_safe_redirect($redirect);
        exit;
    }

    /**
     * AJAX handler to refresh collection from API
     */
    public function ajax_refresh_collection() {
        check_ajax_referer('smartsupport_refresh_collection', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('You do not have permission to perform this action.', 'smart-support-chat-widget'));
        }

        $collection_id = isset($_POST['collection_id']) ? intval($_POST['collection_id']) : 0;
        if ($collection_id <= 0) {
            wp_send_json_error(__('Invalid collection ID.', 'smart-support-chat-widget'));
        }

        $api = $this->get_api_client();
        if (!$api) {
            wp_send_json_error(__('Unable to initialize SmartSupport API client.', 'smart-support-chat-widget'));
        }

        $response = $api->get_rag_collection($collection_id);
        if (empty($response['success']) || empty($response['collection'])) {
            wp_send_json_error(isset($response['message']) ? $response['message'] : __('Failed to retrieve collection from API.', 'smart-support-chat-widget'));
        }

        $api_collection = $response['collection'];
        $collection_unique = isset($api_collection['collection_unique']) ? sanitize_text_field($api_collection['collection_unique']) : '';
        
        if (empty($collection_unique)) {
            wp_send_json_error(__('Collection unique identifier not found.', 'smart-support-chat-widget'));
        }

        $collection_key = sanitize_key($collection_unique);
        $collections = get_option('smartsupport_wp_rag_collections', array());
        
        if (is_array($collections) && isset($collections[$collection_key])) {
            // Refresh the collection schema
            $this->refresh_collection_schema($collection_key, $collections[$collection_key]);
            wp_send_json_success(__('Collection refreshed successfully.', 'smart-support-chat-widget'));
        } else {
            wp_send_json_error(__('Collection not found locally.', 'smart-support-chat-widget'));
        }
    }

    /**
     * Handle manual record deletion requests
     */
    public function handle_delete_record_request() {
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'smart-support-chat-widget'));
        }

        // Read minimal values needed for nonce verification (sanitized)
        $post_id = isset($_REQUEST['post_id']) ? intval($_REQUEST['post_id']) : 0;
        $collection_key = isset($_REQUEST['collection_key']) ? sanitize_key(wp_unslash($_REQUEST['collection_key'])) : '';

        if ($post_id <= 0) {
            $this->store_index_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Invalid post specified for deletion.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('edit.php'));
            exit;
        }

        // Verify nonce BEFORE processing any other user input
        check_admin_referer($this->get_manual_delete_nonce_action($post_id, $collection_key));

        $post = get_post($post_id);
        if (!$post) {
            $this->store_index_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('The requested post could not be found.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('edit.php'));
            exit;
        }

        $collections = $this->get_collections_for_post_type($post->post_type);
        if (empty($collections)) {
            $this->store_index_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('No SmartSupport collections are linked to this post type.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect($this->get_default_referer_url($post->post_type));
            exit;
        }

        if (!empty($collection_key)) {
            $collections = isset($collections[$collection_key]) ? array($collection_key => $collections[$collection_key]) : array();
        }

        if (empty($collections)) {
            $this->store_index_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('The specified collection is not linked to this post.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect($this->get_default_referer_url($post->post_type));
            exit;
        }

        $messages = array();
        $overall_success = true;

        foreach ($collections as $key => $collection) {
            $meta_key = $this->get_collection_meta_key($key);
            $record_id = get_post_meta($post_id, $meta_key, true);

            if (empty($record_id)) {
                $messages[] = sprintf(
                    /* translators: %s: SmartSupport collection display name. */
                    __('%s: no record found to delete.', 'smart-support-chat-widget'),
                    isset($collection['collection_name']) ? $collection['collection_name'] : $key
                );
                continue;
            }

            $api_id = isset($collection['api_id']) ? intval($collection['api_id']) : 0;
            $result = $this->delete_record_for_collection($api_id, $record_id, $post_id, $key);

            if (empty($result['success'])) {
                $overall_success = false;
            }

            $message = isset($result['message']) ? $result['message'] : __('Unknown response.', 'smart-support-chat-widget');
            $messages[] = sprintf(
                /* translators: 1: SmartSupport collection display name, 2: status message. */
                __('%1$s: %2$s', 'smart-support-chat-widget'),
                isset($collection['collection_name']) ? $collection['collection_name'] : $key,
                $message
            );
        }

        $this->store_index_notice(array(
            'success' => $overall_success,
            'type' => $overall_success ? 'success' : 'error',
            'message' => implode("\n", $messages)
        ));

        // Sanitize and validate referer URL
        $redirect = '';
        if (isset($_REQUEST['_wp_http_referer'])) {
            $raw_referer = sanitize_text_field(wp_unslash($_REQUEST['_wp_http_referer']));
            $decoded_referer = is_string($raw_referer) ? rawurldecode($raw_referer) : '';
            if (!empty($decoded_referer)) {
                $redirect_candidate = esc_url_raw($decoded_referer);
                // Validate URL and ensure it's safe for redirect
                if (!empty($redirect_candidate) && (strpos($redirect_candidate, admin_url()) === 0 || strpos($redirect_candidate, home_url()) === 0)) {
                    $redirect = $redirect_candidate;
                }
            }
        }
        if (empty($redirect)) {
            $redirect = $this->get_default_referer_url($post->post_type);
        }

        wp_safe_redirect(esc_url_raw($redirect));
        exit;
    }

    /**
     * Handle bulk indexing requests
     */
    public function handle_bulk_index_request() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'smart-support-chat-widget'));
        }

        check_admin_referer('smartsupport_bulk_index');

        $collection_key = isset($_POST['collection_key']) ? sanitize_key(wp_unslash($_POST['collection_key'])) : '';
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 10;
        $limit = max(1, min(50, $limit));

        $collections = $this->get_saved_rag_collections();
        if (empty($collection_key) || !isset($collections[$collection_key])) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Please select a valid collection.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-bulk-indexing'));
            exit;
        }

        $collection = $collections[$collection_key];
        $post_type = isset($collection['post_type']) ? $collection['post_type'] : '';
        if (empty($post_type)) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('The selected collection does not have a post type assigned.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-bulk-indexing'));
            exit;
        }

        $query_args = array(
            'post_type' => $post_type,
            'post_status' => array('publish', 'future', 'private'),
            'posts_per_page' => $limit,
            'orderby' => 'date',
            'order' => 'DESC',
            'fields' => 'ids'
        );

        $posts = get_posts($query_args);
        if (empty($posts)) {
            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('No posts found to index for the selected collection.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('admin.php?page=smartsupport-bulk-indexing'));
            exit;
        }

        $results = array();
        $success_count = 0;

        foreach ($posts as $post_id) {
            $post = get_post($post_id);
            if (!$post) {
                continue;
            }

            $result = $this->sync_post_to_collection($post_id, $collection_key, $collection);
            if (!is_array($result)) {
                $result = array('success' => false, 'message' => __('Unexpected response.', 'smart-support-chat-widget'));
            }

            if (!empty($result['success'])) {
                $success_count++;
            }

            $results[] = sprintf(
                /* translators: 1: post ID, 2: indexing status message. */
                __('Post #%1$d: %2$s', 'smart-support-chat-widget'),
                $post_id,
                isset($result['message']) ? $result['message'] : ($result['success'] ? __('Indexed', 'smart-support-chat-widget') : __('Failed', 'smart-support-chat-widget'))
            );
        }

        $message = sprintf(
            /* translators: 1: number of posts successfully indexed, 2: total number of posts processed. */
            __('Bulk indexing completed. %1$d out of %2$d posts were indexed successfully.', 'smart-support-chat-widget'),
            $success_count,
            count($posts)
        );

        if (!empty($results)) {
            $message .= "\n\n" . implode("\n", $results);
        }

        $this->store_rag_notice(array(
            'success' => ($success_count === count($posts)),
            'type' => ($success_count === count($posts)) ? 'success' : 'warning',
            'message' => $message
        ));

        wp_safe_redirect(admin_url('admin.php?page=smartsupport-bulk-indexing'));
        exit;
    }

    /**
     * Render bulk indexing page
     */
    public function render_bulk_indexing_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $collections = $this->get_saved_rag_collections();

        $collections_for_select = array();
        foreach ($collections as $key => $collection) {
            if (!is_array($collection)) {
                continue;
            }

            $post_type = isset($collection['post_type']) ? $collection['post_type'] : '';
            $name = isset($collection['collection_name']) ? $collection['collection_name'] : $key;
            $collections_for_select[$key] = sprintf(
                '%1$s (%2$s)',
                $name,
                $post_type
            );
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Bulk Indexing', 'smart-support-chat-widget'); ?></h1>

            <?php if (empty($collections_for_select)): ?>
                <div class="notice notice-warning"><p><?php esc_html_e('Please create a RAG collection before running a bulk sync.', 'smart-support-chat-widget'); ?></p></div>
                <?php return; ?>
            <?php endif; ?>

            <div class="smartsupport-card smartsupport-card-form">
                <h2><?php esc_html_e('Run Bulk Sync', 'smart-support-chat-widget'); ?></h2>
                <p class="description"><?php esc_html_e('Select a collection and the number of recent posts you want to index. Posts are processed from newest to oldest, up to 50 per batch.', 'smart-support-chat-widget'); ?></p>

                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('smartsupport_bulk_index'); ?>
                    <input type="hidden" name="action" value="smartsupport_bulk_index" />

                    <table class="form-table">
                        <tbody>
                        <tr>
                            <th scope="row"><label for="smartsupport_bulk_collection"><?php esc_html_e('Collection', 'smart-support-chat-widget'); ?></label></th>
                            <td>
                                <select name="collection_key" id="smartsupport_bulk_collection" class="regular-text">
                                    <?php foreach ($collections_for_select as $key => $label): ?>
                                        <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($label); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="smartsupport_bulk_limit"><?php esc_html_e('Number of posts', 'smart-support-chat-widget'); ?></label></th>
                            <td>
                                <input type="number" id="smartsupport_bulk_limit" name="limit" value="10" min="1" max="50" class="small-text" />
                                <p class="description"><?php esc_html_e('Maximum 50 posts per bulk run.', 'smart-support-chat-widget'); ?></p>
                            </td>
                        </tr>
                        </tbody>
                    </table>

                    <?php submit_button(__('Start Bulk Indexing', 'smart-support-chat-widget')); ?>
                </form>
            </div>
        </div>
        <?php
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        $methods = defined('WP_REST_Server::CREATABLE') ? WP_REST_Server::CREATABLE : 'POST';

        register_rest_route(
            'smartsupport/v1',
            '/sync',
            array(
                'methods' => $methods,
                'callback' => array($this, 'rest_handle_sync_request'),
                'permission_callback' => array($this, 'rest_validate_request'),
                'args' => array(
                    'post_id' => array(
                        'required' => true,
                        'type' => 'integer',
                        'description' => __('The ID of the post to synchronise.', 'smart-support-chat-widget')
                    ),
                    'collections' => array(
                        'required' => false,
                        'type' => 'array',
                        'description' => __('Specific collection keys to synchronise. Defaults to all linked collections for the post type.', 'smart-support-chat-widget')
                    ),
                    'force' => array(
                        'required' => false,
                        'type' => 'boolean',
                        'description' => __('Force recreation of the record even if already indexed.', 'smart-support-chat-widget')
                    ),
                )
            )
        );
    }

    /**
     * Handle manual indexing requests
     */
    public function handle_manual_index_request() {
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'smart-support-chat-widget'));
        }

        // Read minimal values needed for nonce verification (sanitized)
        $post_id = isset($_REQUEST['post_id']) ? intval($_REQUEST['post_id']) : 0;
        $collection_key = isset($_REQUEST['collection_key']) ? sanitize_key(wp_unslash($_REQUEST['collection_key'])) : '';

        if ($post_id <= 0) {
            $this->store_index_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('Invalid post specified for indexing.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('edit.php'));
            exit;
        }

        // Verify nonce BEFORE processing any other user input
        check_admin_referer($this->get_manual_index_nonce_action($post_id, $collection_key));

        $post = get_post($post_id);
        if (!$post) {
            $this->store_index_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('The requested post could not be found.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(admin_url('edit.php'));
            exit;
        }

        $collections = $this->get_collections_for_post_type($post->post_type);

        if (!empty($collection_key)) {
            $collections = isset($collections[$collection_key]) ? array($collection_key => $collections[$collection_key]) : array();
        }

        if (empty($collections)) {
            $this->store_index_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => __('No SmartSupport collections are linked to this post type.', 'smart-support-chat-widget')
            ));
            wp_safe_redirect(get_edit_post_link($post_id, 'url'));
            exit;
        }

        $messages = array();
        $all_success = true;

        foreach ($collections as $key => $collection) {
            $result = $this->sync_post_to_collection($post_id, $key, $collection);
            $collection_name = isset($collection['collection_name']) ? $collection['collection_name'] : $key;

            if (!empty($result['success'])) {
                $messages[] = sprintf(
                    /* translators: %s: SmartSupport collection display name. */
                    __('%s: indexed successfully.', 'smart-support-chat-widget'),
                    $collection_name
                );
            } else {
                $all_success = false;
                $error_message = isset($result['message']) ? $result['message'] : __('Unknown error.', 'smart-support-chat-widget');
                $messages[] = sprintf(
                    /* translators: 1: SmartSupport collection display name, 2: error message. */
                    __('%1$s: %2$s', 'smart-support-chat-widget'),
                    $collection_name,
                    $error_message
                );
            }
        }

        $this->store_index_notice(array(
            'success' => $all_success,
            'type' => $all_success ? 'success' : 'error',
            'message' => implode("\n", $messages)
        ));

        // Sanitize and validate referer URL
        $redirect = '';
        if (isset($_REQUEST['_wp_http_referer'])) {
            $raw_referer = sanitize_text_field(wp_unslash($_REQUEST['_wp_http_referer']));
            $decoded_referer = is_string($raw_referer) ? rawurldecode($raw_referer) : '';
            if (!empty($decoded_referer)) {
                $redirect_candidate = esc_url_raw($decoded_referer);
                // Validate URL and ensure it's safe for redirect
                if (!empty($redirect_candidate) && (strpos($redirect_candidate, admin_url()) === 0 || strpos($redirect_candidate, home_url()) === 0)) {
                    $redirect = $redirect_candidate;
                }
            }
        }

        if (empty($redirect)) {
            $redirect = $this->get_default_referer_url($post->post_type);
        }

        wp_safe_redirect(esc_url_raw($redirect));
        exit;
    }

    /**
     * AJAX handler to get fields for a post type
     */
    public function ajax_get_fields() {
        check_ajax_referer('smartsupport_get_fields', 'nonce');

        if (!current_user_can('manage_options')) {
            $error_message = __('Unauthorized.', 'smart-support-chat-widget');
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $error_message .= "\n" . __('User ID:', 'smart-support-chat-widget') . ' ' . get_current_user_id();
                $error_message .= "\n" . __('User capabilities:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode(wp_get_current_user()->allcaps, JSON_PRETTY_PRINT));
            }
            wp_send_json_error($error_message);
        }

        $post_type = isset($_POST['post_type']) ? sanitize_text_field(wp_unslash($_POST['post_type'])) : '';
        $field_source = isset($_POST['field_source']) ? sanitize_text_field(wp_unslash($_POST['field_source'])) : 'meta';

        if (empty($post_type)) {
            $error_message = __('Post type is required.', 'smart-support-chat-widget');
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // Sanitize POST data for safe debug output
                $sanitized_post = array();
                foreach ($_POST as $key => $value) {
                    $sanitized_key = sanitize_key($key);
                    if (is_array($value)) {
                        $sanitized_post[$sanitized_key] = 'Array(' . count($value) . ' items)';
                    } else {
                        $sanitized_post[$sanitized_key] = sanitize_text_field($value);
                    }
                }
                $error_message .= "\n" . __('Received POST data:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($sanitized_post, JSON_PRETTY_PRINT));
            }
            wp_send_json_error($error_message);
        }

        try {
            $fields = $this->get_fields_for_post_type($post_type, $field_source);
        } catch (Exception $e) {
            $error_message = __('Error loading fields:', 'smart-support-chat-widget') . ' ' . $e->getMessage();
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $error_message .= "\n" . __('Exception details:', 'smart-support-chat-widget') . "\n" . $e->getTraceAsString();
            }
            wp_send_json_error($error_message);
        }

        ob_start();
        ?>
        <div class="fields-selection">
            <p class="description"><?php esc_html_e('Select fields to include in the RAG collection. Check "Embed" to enable vector search for the field. Check "Metadata" to include the field as metadata. At least one of Embed or Metadata must be selected for each field.', 'smart-support-chat-widget'); ?></p>
            <table class="widefat">
                <thead>
                <tr>
                    <th style="width: 30px;"><?php esc_html_e('Include', 'smart-support-chat-widget'); ?></th>
                    <th><?php esc_html_e('Field Name', 'smart-support-chat-widget'); ?></th>
                    <th><?php esc_html_e('Type', 'smart-support-chat-widget'); ?></th>
                    <th><?php esc_html_e('Embed', 'smart-support-chat-widget'); ?></th>
                    <th><?php esc_html_e('Metadata', 'smart-support-chat-widget'); ?></th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($fields as $field): ?>
                    <?php 
                    $default_included = !empty($field['default_included']);
                    $default_embed = isset($field['embed']) ? (bool) $field['embed'] : true;
                    $default_metadata = isset($field['metadata']) ? (bool) $field['metadata'] : false;
                    // Default: content fields embed, ID/URL fields use metadata
                    if (!$default_embed && !$default_metadata) {
                        if (in_array($field['name'], array('post_id', 'post_url'), true)) {
                            $default_metadata = true;
                        } else {
                            $default_embed = true;
                        }
                    }
                    ?>
                    <tr>
                        <td>
                            <input
                                    type="checkbox"
                                    name="selected_fields[<?php echo esc_attr($field['name']); ?>][include]"
                                    value="1"
                                    class="field-include-checkbox"
                                    <?php checked($default_included); ?>
                            />
                        </td>
                        <td>
                            <strong><?php echo esc_html($field['name']); ?></strong>
                            <?php if (!empty($field['label'])): ?>
                                <br /><small><?php echo esc_html($field['label']); ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <select name="selected_fields[<?php echo esc_attr($field['name']); ?>][type]" class="smartsupport-field-setting">
                                <option value="string" <?php selected($field['type'], 'string'); ?>>string</option>
                                <option value="text" <?php selected($field['type'], 'text'); ?>>text</option>
                                <option value="number" <?php selected($field['type'], 'number'); ?>>number</option>
                                <option value="boolean" <?php selected($field['type'], 'boolean'); ?>>boolean</option>
                            </select>
                        </td>
                        <td>
                            <input
                                    type="checkbox"
                                    name="selected_fields[<?php echo esc_attr($field['name']); ?>][embed]"
                                    value="1"
                                    class="smartsupport-field-setting"
                                    <?php checked($default_embed); ?>
                            />
                        </td>
                        <td>
                            <input
                                    type="checkbox"
                                    name="selected_fields[<?php echo esc_attr($field['name']); ?>][metadata]"
                                    value="1"
                                    class="smartsupport-field-setting"
                                    <?php checked($default_metadata); ?>
                            />
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
        $output = ob_get_clean();

        wp_send_json_success($output);
    }

    /**
     * AJAX handler to get agents from API
     */
    public function ajax_get_agents() {
        check_ajax_referer('smartsupport_get_agents', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized.', 'smart-support-chat-widget'));
        }

        $settings = get_option('smartsupport_wp_settings', array());
        $api_key = isset($settings['api_key']) ? trim($settings['api_key']) : '';

        if (empty($api_key)) {
            wp_send_json_error(__('API key is not configured.', 'smart-support-chat-widget'));
        }

        $url = smartsupport_get_base_url() . '/api/agents';
        $args = array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'timeout' => 30
        );

        $response = wp_remote_get($url, $args);

        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $body_data = json_decode($body, true);

        if ($status_code >= 200 && $status_code < 300) {
            if (isset($body_data['agents']) && is_array($body_data['agents'])) {
                wp_send_json_success($body_data['agents']);
            } else {
                wp_send_json_error(__('No agents found.', 'smart-support-chat-widget'));
            }
        } else {
            $error_message = __('Failed to fetch agents.', 'smart-support-chat-widget');
            if (isset($body_data['message'])) {
                $error_message = $body_data['message'];
            } elseif (isset($body_data['error'])) {
                $error_message = $body_data['error'];
            }
            wp_send_json_error($error_message);
        }
    }

    /**
     * Handle reset API key or agent requests
     */
    public function handle_reset_requests() {
        if (!current_user_can('manage_options')) {
            return;
        }

        if (isset($_GET['reset_api']) && $_GET['reset_api'] == '1') {
            check_admin_referer('reset_api', '_wpnonce');
            $settings = get_option('smartsupport_wp_settings', array());
            unset($settings['api_key']);
            unset($settings['agent_id']);
            update_option('smartsupport_wp_settings', $settings);
            wp_safe_redirect(admin_url('admin.php?page=smart-support-chat-widget'));
            exit;
        }

        if (isset($_GET['reset_agent']) && $_GET['reset_agent'] == '1') {
            check_admin_referer('reset_agent', '_wpnonce');
            $settings = get_option('smartsupport_wp_settings', array());
            unset($settings['agent_id']);
            update_option('smartsupport_wp_settings', $settings);
            wp_safe_redirect(admin_url('admin.php?page=smart-support-chat-widget'));
            exit;
        }
    }

    /**
     * Get fields for a post type
     */
    private function get_fields_for_post_type($post_type, $field_source = 'meta') {
        $fields = array();

        // Always include default post fields
        $fields[] = array(
            'name' => 'post_title',
            'label' => __('Post Title', 'smart-support-chat-widget'),
            'type' => 'string',
            'embed' => true,
            'metadata' => false
        );

        $fields[] = array(
            'name' => 'post_content',
            'label' => __('Post Content', 'smart-support-chat-widget'),
            'type' => 'text',
            'embed' => true,
            'metadata' => false
        );

        $fields[] = array(
            'name' => 'post_excerpt',
            'label' => __('Post Excerpt', 'smart-support-chat-widget'),
            'type' => 'text',
            'embed' => true,
            'metadata' => false
        );

        $fields[] = array(
            'name' => 'post_id',
            'label' => __('Post ID', 'smart-support-chat-widget'),
            'type' => 'number',
            'embed' => false,
            'metadata' => true
        );

        $fields[] = array(
            'name' => 'post_url',
            'label' => __('Post URL', 'smart-support-chat-widget'),
            'type' => 'string',
            'embed' => false,
            'metadata' => true,
            'default_included' => true
        );

        if ($field_source === 'acpt' && $this->is_acpt_enabled()) {
            // Get ACPT fields
            $acpt_fields = $this->get_acpt_fields($post_type);
            $fields = array_merge($fields, $acpt_fields);
        } else {
            // Get meta fields
            $meta_fields = $this->get_meta_fields($post_type);
            $fields = array_merge($fields, $meta_fields);
        }

        return $fields;
    }

    /**
     * Get meta fields for a post type
     */
    private function get_meta_fields($post_type) {
        $fields = array();

        $post_query = array(
            'post_type' => $post_type,
            'post_status' => array('publish', 'future', 'pending', 'draft', 'private'),
            'posts_per_page' => 100,
            'orderby' => 'date',
            'order' => 'DESC',
            'fields' => 'ids',
        );

        $post_ids = get_posts($post_query);
        if (empty($post_ids)) {
            return $fields;
        }

        $meta_keys = array();
        $sample_values = array();

        foreach ($post_ids as $post_id) {
            $custom_keys = get_post_custom_keys($post_id);
            if (empty($custom_keys) || !is_array($custom_keys)) {
                continue;
            }

            foreach ($custom_keys as $meta_key) {
                if (strpos($meta_key, '_') === 0 || strpos($meta_key, 'wp_') === 0) {
                    continue;
                }

                if (isset($meta_keys[$meta_key])) {
                    continue;
                }

                $meta_keys[$meta_key] = true;

                if (!isset($sample_values[$meta_key])) {
                    $value = get_post_meta($post_id, $meta_key, true);
                    if ($value !== '') {
                        $sample_values[$meta_key] = $value;
                    }
                }
            }
        }

        $meta_key_list = array_keys($meta_keys);

        foreach ($meta_key_list as $meta_key) {
            $meta_value = isset($sample_values[$meta_key]) ? $sample_values[$meta_key] : '';
            $field_type = $this->detect_field_type($meta_value);

            $embed = $this->should_embed_field($meta_key, $meta_value);
            $metadata = !$embed; // Default: if not embedding, use as metadata
            // Ensure at least one is true
            if (!$embed && !$metadata) {
                $embed = true;
            }
            $fields[] = array(
                'name' => $meta_key,
                'label' => ucwords(str_replace('_', ' ', $meta_key)),
                'type' => $field_type,
                'embed' => $embed,
                'metadata' => $metadata
            );
        }

        return $fields;
    }

    /**
     * Check if ACPT is enabled
     */
    private function is_acpt_enabled() {
        // Check if ACPT class exists
        if (class_exists('ACPT')) {
            return true;
        }

        // Check if ACPT plugin is active
        if (!function_exists('is_plugin_active')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }

        if (function_exists('is_plugin_active')) {
            if (is_plugin_active('advanced-custom-post-types/acpt.php') ||
                is_plugin_active('acpt/acpt.php') ||
                is_plugin_active('advanced-custom-post-types/advanced-custom-post-types.php')) {
                return true;
            }
        }

        // Check for ACPT constant
        if (defined('ACPT_VERSION') || defined('ACPT_PLUGIN_VERSION')) {
            return true;
        }

        return false;
    }

    /**
     * Get ACPT fields for a post type
     */
    private function get_acpt_fields($post_type) {
        $fields = array();

        if (!$this->is_acpt_enabled()) {
            return $fields;
        }

        // Try to get ACPT fields
        // This is a placeholder - actual implementation depends on ACPT API
        // You may need to adjust this based on actual ACPT structure
        if (function_exists('acpt_get_post_type_fields')) {
            $acpt_fields = acpt_get_post_type_fields($post_type);
            if (is_array($acpt_fields)) {
                foreach ($acpt_fields as $field) {
                    $embed = isset($field['embed']) ? (bool) $field['embed'] : true;
                    $metadata = isset($field['metadata']) ? (bool) $field['metadata'] : false;
                    // Ensure at least one is true
                    if (!$embed && !$metadata) {
                        $embed = true;
                    }
                    $fields[] = array(
                        'name' => isset($field['name']) ? $field['name'] : '',
                        'label' => isset($field['label']) ? $field['label'] : '',
                        'type' => isset($field['type']) ? $this->map_acpt_type($field['type']) : 'string',
                        'embed' => $embed,
                        'metadata' => $metadata
                    );
                }
            }
        } elseif (function_exists('get_acpt_fields')) {
            $acpt_fields = get_acpt_fields($post_type);
            if (is_array($acpt_fields)) {
                foreach ($acpt_fields as $field_key => $field_data) {
                    $embed = isset($field_data['embed']) ? (bool) $field_data['embed'] : true;
                    $metadata = isset($field_data['metadata']) ? (bool) $field_data['metadata'] : false;
                    // Ensure at least one is true
                    if (!$embed && !$metadata) {
                        $embed = true;
                    }
                    $fields[] = array(
                        'name' => $field_key,
                        'label' => isset($field_data['label']) ? $field_data['label'] : ucwords(str_replace('_', ' ', $field_key)),
                        'type' => isset($field_data['type']) ? $this->map_acpt_type($field_data['type']) : 'string',
                        'embed' => $embed,
                        'metadata' => $metadata
                    );
                }
            }
        }

        return $fields;
    }

    /**
     * Map ACPT field type to RAG field type
     */
    private function map_acpt_type($acpt_type) {
        $type_map = array(
            'text' => 'string',
            'textarea' => 'text',
            'number' => 'number',
            'checkbox' => 'boolean',
            'select' => 'string',
            'radio' => 'string',
            'date' => 'string',
            'email' => 'string',
            'url' => 'string'
        );

        return isset($type_map[$acpt_type]) ? $type_map[$acpt_type] : 'string';
    }

    /**
     * Detect field type from value
     */
    private function detect_field_type($value) {
        if (is_numeric($value)) {
            return is_float($value) ? 'number' : 'number';
        } elseif (is_bool($value)) {
            return 'boolean';
        } elseif (is_string($value) && strlen($value) > 255) {
            return 'text';
        } else {
            return 'string';
        }
    }

    /**
     * Determine if field should be embedded
     */
    private function should_embed_field($field_name, $value) {
        // Don't embed IDs, dates, or boolean values
        if (preg_match('/_id$|_date$|_time$|^is_|^has_/', $field_name)) {
            return false;
        }

        // Embed text content
        if (is_string($value) && strlen($value) > 10) {
            return true;
        }

        return false;
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Smart Support Chat Widget Settings', 'smart-support-chat-widget'),
            __('Smart Support Chat Widget', 'smart-support-chat-widget'),
            'manage_options',
            'smart-support-chat-widget',
            array($this, 'render_settings_page'),
            'dashicons-format-chat',
            30
        );

        // Add RAG Collections submenu
        add_submenu_page(
            'smart-support-chat-widget',
            __('RAG Collections', 'smart-support-chat-widget'),
            __('RAG Collections', 'smart-support-chat-widget'),
            'manage_options',
            'smartsupport-rag-collections',
            array($this, 'render_rag_collections_page')
        );

        add_submenu_page(
            'smart-support-chat-widget',
            __('Bulk Indexing', 'smart-support-chat-widget'),
            __('Bulk Indexing', 'smart-support-chat-widget'),
            'manage_options',
            'smartsupport-bulk-indexing',
            array($this, 'render_bulk_indexing_page')
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting(
            'smartsupport_wp_settings_group',
            'smartsupport_wp_settings',
            array($this, 'sanitize_settings')
        );
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();

        if (isset($input['api_key'])) {
            $sanitized['api_key'] = sanitize_text_field($input['api_key']);
        }

        if (isset($input['agent_id'])) {
            $sanitized['agent_id'] = sanitize_text_field($input['agent_id']);
        }

        if (isset($input['title'])) {
            $sanitized['title'] = sanitize_text_field($input['title']);
        }

        if (isset($input['primary_color'])) {
            $sanitized['primary_color'] = sanitize_hex_color($input['primary_color']);
        }

        if (isset($input['position'])) {
            $allowed_positions = array('bottom-right', 'bottom-left', 'top-right', 'top-left');
            $sanitized['position'] = in_array($input['position'], $allowed_positions) ? $input['position'] : 'bottom-right';
        }

        if (isset($input['greeting_message'])) {
            $sanitized['greeting_message'] = sanitize_text_field($input['greeting_message']);
        }

        if (isset($input['show_header'])) {
            $sanitized['show_header'] = true;
        } else {
            $sanitized['show_header'] = false;
        }

        if (isset($input['show_toggle'])) {
            $sanitized['show_toggle'] = true;
        } else {
            $sanitized['show_toggle'] = false;
        }

        if (isset($input['default_open'])) {
            $sanitized['default_open'] = true;
        } else {
            $sanitized['default_open'] = false;
        }

        return $sanitized;
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Get current settings
        $settings = get_option('smartsupport_wp_settings', array());
        $api_key = isset($settings['api_key']) ? trim($settings['api_key']) : '';
        $agent_id = isset($settings['agent_id']) ? trim($settings['agent_id']) : '';

        // Check if settings were saved with a verified nonce
        if (isset($_GET['settings-updated'])) {
            $nonce = isset($_REQUEST['_wpnonce']) ? sanitize_text_field(wp_unslash($_REQUEST['_wpnonce'])) : '';
            if (!empty($nonce) && wp_verify_nonce($nonce, 'update-options')) {
                add_settings_error(
                    'smartsupport_wp_messages',
                    'smartsupport_wp_message',
                    __('Settings saved successfully! The chat widget will appear on your site.', 'smart-support-chat-widget'),
                    'updated'
                );
            }
        }

        // Show settings errors
        settings_errors('smartsupport_wp_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <?php if (empty($api_key)): ?>
                <!-- Step 1: API Key Setup -->
                <div class="smartsupport-setup-step">
                    <h2><?php esc_html_e('Step 1: Configure API Key', 'smart-support-chat-widget'); ?></h2>
                    <p><?php esc_html_e('To get started, you need to set up your Smart Support API key.', 'smart-support-chat-widget'); ?></p>
                    
                    <form action="options.php" method="post">
                        <?php
                        settings_fields('smartsupport_wp_settings_group');
                        ?>
                        <table class="form-table" role="presentation">
                            <tbody>
                            <tr>
                                <th scope="row">
                                    <label for="smartsupport_api_key"><?php esc_html_e('API Key', 'smart-support-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <input
                                            type="password"
                                            id="smartsupport_api_key"
                                            name="smartsupport_wp_settings[api_key]"
                                            value=""
                                            class="regular-text"
                                            placeholder="<?php esc_attr_e('Enter your API key', 'smart-support-chat-widget'); ?>"
                                            required
                                    />
                                    <p class="description">
                                        <?php esc_html_e('Enter your Smart Support API key.', 'smart-support-chat-widget'); ?>
                                    </p>
                                </td>
                            </tr>
                            </tbody>
                        </table>

                        <div class="smartsupport-instructions">
                            <h3><?php esc_html_e('How to get your API Key:', 'smart-support-chat-widget'); ?></h3>
                            <ol>
                                <li>
                                    <?php esc_html_e('Register or login to Smart Support:', 'smart-support-chat-widget'); ?>
                                    <a href="<?php echo esc_url(smartsupport_get_base_url() . '/panel/login'); ?>" target="_blank"><?php echo esc_html(smartsupport_get_base_url() . '/panel/login'); ?></a>
                                </li>
                                <li>
                                    <?php esc_html_e('Create an Agent:', 'smart-support-chat-widget'); ?>
                                    <a href="<?php echo esc_url(smartsupport_get_base_url() . '/panel/agents'); ?>" target="_blank"><?php echo esc_html(smartsupport_get_base_url() . '/panel/agents'); ?></a>
                                </li>
                                <li>
                                    <?php esc_html_e('Get your API key from:', 'smart-support-chat-widget'); ?>
                                    <a href="<?php echo esc_url(smartsupport_get_base_url() . '/panel/settings?section=api-settings'); ?>" target="_blank"><?php echo esc_html(smartsupport_get_base_url() . '/panel/settings?section=api-settings'); ?></a>
                                </li>
                            </ol>
                        </div>

                        <?php submit_button(esc_html__('Save API Key', 'smart-support-chat-widget')); ?>
                    </form>
                </div>
            <?php elseif (empty($agent_id)): ?>
                <!-- Step 2: Agent Selection -->
                <div class="smartsupport-setup-step">
                    <h2><?php esc_html_e('Step 2: Select Your Agent', 'smart-support-chat-widget'); ?></h2>
                    <p><?php esc_html_e('Please select an agent to use with the chat widget.', 'smart-support-chat-widget'); ?></p>
                    
                    <div style="margin-bottom: 20px;">
                        <p class="description">
                            <?php esc_html_e('API key is configured.', 'smart-support-chat-widget'); ?>
                            <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=smart-support-chat-widget&reset_api=1'), 'reset_api', '_wpnonce')); ?>" onclick="return confirm('<?php esc_attr_e('Are you sure you want to reset the API key? You will need to enter it again.', 'smart-support-chat-widget'); ?>');">
                                <?php esc_html_e('Reset API Key', 'smart-support-chat-widget'); ?>
                            </a>
                        </p>
                    </div>
                    
                    <form action="options.php" method="post" id="smartsupport-agent-selection-form">
                        <?php
                        settings_fields('smartsupport_wp_settings_group');
                        ?>
                        <input type="hidden" name="smartsupport_wp_settings[api_key]" value="<?php echo esc_attr($api_key); ?>" />
                        
                        <table class="form-table" role="presentation">
                            <tbody>
                            <tr>
                                <th scope="row">
                                    <label for="smartsupport_agent_id"><?php esc_html_e('Select Agent', 'smart-support-chat-widget'); ?></label>
                                </th>
                                <td>
                                    <div id="smartsupport-agent-loading">
                                        <span class="spinner is-active" style="float: none; margin: 0 10px 0 0;"></span>
                                        <?php esc_html_e('Loading agents...', 'smart-support-chat-widget'); ?>
                                    </div>
                                    <div id="smartsupport-agent-error" style="display: none; color: #d63638; margin-bottom: 10px;"></div>
                                    <select
                                            id="smartsupport_agent_id"
                                            name="smartsupport_wp_settings[agent_id]"
                                            class="regular-text"
                                            required
                                            style="display: none;"
                                    >
                                        <option value=""><?php esc_html_e('-- Select an Agent --', 'smart-support-chat-widget'); ?></option>
                                    </select>
                                    <p class="description">
                                        <?php esc_html_e('Select the agent you want to use for the chat widget.', 'smart-support-chat-widget'); ?>
                                    </p>
                                </td>
                            </tr>
                            </tbody>
                        </table>

                        <?php submit_button(esc_html__('Save Agent', 'smart-support-chat-widget'), 'primary', 'submit', false, array('id' => 'smartsupport-save-agent-btn', 'style' => 'display: none;')); ?>
                    </form>
                </div>
            <?php else: ?>
                <!-- Step 3: Widget Settings -->
                <form action="options.php" method="post">
                    <?php
                    settings_fields('smartsupport_wp_settings_group');
                    ?>

                    <table class="form-table" role="presentation">
                        <tbody>
                        <!-- API Key (read-only) -->
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e('API Key', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                <input
                                        type="password"
                                        value="<?php echo esc_attr(str_repeat('*', min(strlen($api_key), 20))); ?>"
                                        class="regular-text"
                                        disabled
                                />
                                <p class="description">
                                    <?php esc_html_e('API key is configured.', 'smart-support-chat-widget'); ?>
                                    <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=smart-support-chat-widget&reset_api=1'), 'reset_api', '_wpnonce')); ?>" onclick="return confirm('<?php esc_attr_e('Are you sure you want to reset the API key? You will need to select an agent again.', 'smart-support-chat-widget'); ?>');">
                                        <?php esc_html_e('Reset API Key', 'smart-support-chat-widget'); ?>
                                    </a>
                                </p>
                            </td>
                        </tr>

                        <!-- Agent ID (read-only) -->
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e('Selected Agent', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                <input
                                        type="text"
                                        value="<?php echo esc_attr($agent_id); ?>"
                                        class="regular-text"
                                        disabled
                                />
                                <p class="description">
                                    <?php esc_html_e('Agent is configured.', 'smart-support-chat-widget'); ?>
                                    <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=smart-support-chat-widget&reset_agent=1'), 'reset_agent', '_wpnonce')); ?>" onclick="return confirm('<?php esc_attr_e('Are you sure you want to change the agent?', 'smart-support-chat-widget'); ?>');">
                                        <?php esc_html_e('Change Agent', 'smart-support-chat-widget'); ?>
                                    </a>
                                </p>
                            </td>
                        </tr>

                    <!-- Title -->
                    <tr>
                        <th scope="row">
                            <label for="smartsupport_title"><?php esc_html_e('Chat Title', 'smart-support-chat-widget'); ?></label>
                        </th>
                        <td>
                            <input
                                    type="text"
                                    id="smartsupport_title"
                                    name="smartsupport_wp_settings[title]"
                                    value="<?php echo isset($settings['title']) ? esc_attr($settings['title']) : 'Chat Support'; ?>"
                                    class="regular-text"
                            />
                            <p class="description">
                                <?php esc_html_e('The title displayed in the chat widget header.', 'smart-support-chat-widget'); ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Primary Color -->
                    <tr>
                        <th scope="row">
                            <label for="smartsupport_primary_color"><?php esc_html_e('Primary Color', 'smart-support-chat-widget'); ?></label>
                        </th>
                        <td>
                            <input
                                    type="color"
                                    id="smartsupport_primary_color"
                                    name="smartsupport_wp_settings[primary_color]"
                                    value="<?php echo isset($settings['primary_color']) ? esc_attr($settings['primary_color']) : '#6366f1'; ?>"
                                    class="regular-text"
                            />
                            <p class="description">
                                <?php esc_html_e('The primary color for the chat widget (e.g., #6366f1).', 'smart-support-chat-widget'); ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Position -->
                    <tr>
                        <th scope="row">
                            <label for="smartsupport_position"><?php esc_html_e('Widget Position', 'smart-support-chat-widget'); ?></label>
                        </th>
                        <td>
                            <select id="smartsupport_position" name="smartsupport_wp_settings[position]">
                                <option value="bottom-right" <?php selected(isset($settings['position']) ? $settings['position'] : 'bottom-right', 'bottom-right'); ?>><?php esc_html_e('Bottom Right', 'smart-support-chat-widget'); ?></option>
                                <option value="bottom-left" <?php selected(isset($settings['position']) ? $settings['position'] : 'bottom-right', 'bottom-left'); ?>><?php esc_html_e('Bottom Left', 'smart-support-chat-widget'); ?></option>
                                <option value="top-right" <?php selected(isset($settings['position']) ? $settings['position'] : 'bottom-right', 'top-right'); ?>><?php esc_html_e('Top Right', 'smart-support-chat-widget'); ?></option>
                                <option value="top-left" <?php selected(isset($settings['position']) ? $settings['position'] : 'bottom-right', 'top-left'); ?>><?php esc_html_e('Top Left', 'smart-support-chat-widget'); ?></option>
                            </select>
                            <p class="description">
                                <?php esc_html_e('The position of the chat widget on the page.', 'smart-support-chat-widget'); ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Greeting Message -->
                    <tr>
                        <th scope="row">
                            <label for="smartsupport_greeting_message"><?php esc_html_e('Greeting Message', 'smart-support-chat-widget'); ?></label>
                        </th>
                        <td>
                            <input
                                    type="text"
                                    id="smartsupport_greeting_message"
                                    name="smartsupport_wp_settings[greeting_message]"
                                    value="<?php echo isset($settings['greeting_message']) ? esc_attr($settings['greeting_message']) : 'Welcome! How may I assist you today?'; ?>"
                                    class="regular-text"
                            />
                            <p class="description">
                                <?php esc_html_e('The initial greeting message shown to users.', 'smart-support-chat-widget'); ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Show Header -->
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Display Options', 'smart-support-chat-widget'); ?>
                        </th>
                        <td>
                            <fieldset>
                                <label for="smartsupport_show_header">
                                    <input
                                            type="checkbox"
                                            id="smartsupport_show_header"
                                            name="smartsupport_wp_settings[show_header]"
                                            value="1"
                                        <?php checked(isset($settings['show_header']) ? $settings['show_header'] : true, true); ?>
                                    />
                                    <?php esc_html_e('Show Header', 'smart-support-chat-widget'); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e('Display the header in the chat widget.', 'smart-support-chat-widget'); ?>
                                </p>
                            </fieldset>
                        </td>
                    </tr>

                    <!-- Show Toggle -->
                    <tr>
                        <th scope="row"></th>
                        <td>
                            <fieldset>
                                <label for="smartsupport_show_toggle">
                                    <input
                                            type="checkbox"
                                            id="smartsupport_show_toggle"
                                            name="smartsupport_wp_settings[show_toggle]"
                                            value="1"
                                        <?php checked(isset($settings['show_toggle']) ? $settings['show_toggle'] : true, true); ?>
                                    />
                                    <?php esc_html_e('Show Toggle Button', 'smart-support-chat-widget'); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e('Display the toggle button to open/close the chat.', 'smart-support-chat-widget'); ?>
                                </p>
                            </fieldset>
                        </td>
                    </tr>

                    <!-- Default Open -->
                    <tr>
                        <th scope="row"></th>
                        <td>
                            <fieldset>
                                <label for="smartsupport_default_open">
                                    <input
                                            type="checkbox"
                                            id="smartsupport_default_open"
                                            name="smartsupport_wp_settings[default_open]"
                                            value="1"
                                        <?php checked(isset($settings['default_open']) ? $settings['default_open'] : false, true); ?>
                                    />
                                    <?php esc_html_e('Open by Default', 'smart-support-chat-widget'); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e('Open the chat widget automatically when the page loads.', 'smart-support-chat-widget'); ?>
                                </p>
                            </fieldset>
                        </td>
                    </tr>
                    </tbody>
                </table>

                <!-- Hidden fields to preserve API key and agent ID -->
                <input type="hidden" name="smartsupport_wp_settings[api_key]" value="<?php echo esc_attr($api_key); ?>" />
                <input type="hidden" name="smartsupport_wp_settings[agent_id]" value="<?php echo esc_attr($agent_id); ?>" />

                <?php submit_button(esc_html__('Save Settings', 'smart-support-chat-widget')); ?>
            </form>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render RAG Collections page
     */
    public function render_rag_collections_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Retrieve any stored notice from previous submissions
        $notice = get_transient('smartsupport_rag_notice');
        if (false !== $notice) {
            delete_transient('smartsupport_rag_notice');
        }

        // Get settings
        $settings = get_option('smartsupport_wp_settings', array());
        $api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
        $agent_token = isset($settings['agent_id']) ? $settings['agent_id'] : '';

        // Get available post types
        $post_types = get_post_types(array('public' => true), 'objects');

        // Check if ACPT is enabled
        $acpt_enabled = $this->is_acpt_enabled();

        // Load saved collections
        $saved_collections = $this->get_saved_rag_collections();
        $total_collections = count($saved_collections);
        $auto_collections = 0;
        $manual_collections = 0;
        $total_records = 0;

        foreach ($saved_collections as $collection) {
            $mode = isset($collection['sync_mode']) ? $collection['sync_mode'] : 'auto';
            if ('manual' === $mode) {
                $manual_collections++;
            } else {
                $auto_collections++;
            }

            if (isset($collection['records_count'])) {
                $total_records += intval($collection['records_count']);
            }
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e('RAG Collections', 'smart-support-chat-widget'); ?></h1>

            <?php if (empty($api_key)): ?>
                <div class="notice notice-error">
                    <p>
                        <?php
                        printf(
                            wp_kses(
                                /* translators: %s: URL to SmartSupport plugin settings page. */
                                __('Please configure your API key in the <a href="%s">Settings page</a> first.', 'smart-support-chat-widget'),
                                array(
                                    'a' => array(
                                        'href' => array(),
                                    ),
                                )
                            ),
                            esc_url(admin_url('admin.php?page=smart-support-chat-widget'))
                        );
                        ?>
                    </p>
                </div>
            <?php elseif (empty($agent_token)): ?>
                <div class="notice notice-error">
                    <p>
                        <?php
                        printf(
                            wp_kses(
                                /* translators: %s: URL to SmartSupport plugin settings page. */
                                __('Please configure your agent token in the <a href="%s">Settings page</a> before creating collections.', 'smart-support-chat-widget'),
                                array(
                                    'a' => array(
                                        'href' => array(),
                                    ),
                                )
                            ),
                            esc_url(admin_url('admin.php?page=smart-support-chat-widget'))
                        );
                        ?>
                    </p>
                </div>
            <?php else: ?>

            <div class="smartsupport-summary-grid">
                <div class="smartsupport-summary-card">
                    <span class="dashicons dashicons-archive"></span>
                    <div class="smartsupport-summary-content">
                        <h3><?php echo esc_html($total_collections); ?></h3>
                        <p><?php echo esc_html__('Total Collections', 'smart-support-chat-widget'); ?></p>
                    </div>
                </div>
                <div class="smartsupport-summary-card">
                    <span class="dashicons dashicons-update"></span>
                    <div class="smartsupport-summary-content">
                        <h3><?php echo esc_html($auto_collections); ?> <small><?php esc_html_e('auto', 'smart-support-chat-widget'); ?></small> / <?php echo esc_html($manual_collections); ?> <small><?php esc_html_e('manual', 'smart-support-chat-widget'); ?></small></h3>
                        <p><?php echo esc_html__('Indexing Modes', 'smart-support-chat-widget'); ?></p>
                    </div>
                </div>
                <div class="smartsupport-summary-card">
                    <span class="dashicons dashicons-chart-area"></span>
                    <div class="smartsupport-summary-content">
                        <h3><?php echo esc_html($total_records); ?></h3>
                        <p><?php echo esc_html__('Total Records Synced', 'smart-support-chat-widget'); ?></p>
                    </div>
                </div>
            </div>

            <div class="smartsupport-card smartsupport-help-card">
                <h2><?php esc_html_e('How It Works', 'smart-support-chat-widget'); ?></h2>
                <ol class="smartsupport-help-list">
                    <li><?php esc_html_e('Create a collection for the post type you want to sync.', 'smart-support-chat-widget'); ?></li>
                    <li><?php esc_html_e('Choose Automatic mode to keep SmartSupport up to date on every save, or Manual mode if you prefer to trigger indexing yourself.', 'smart-support-chat-widget'); ?></li>
                    <li><?php esc_html_e('Use the Index / Re-index button in the posts list to sync on demand.', 'smart-support-chat-widget'); ?></li>
                    <li><?php esc_html_e('Monitor sync status and manage settings from the Saved Collections section.', 'smart-support-chat-widget'); ?></li>
                </ol>
            </div>

            <?php if (!empty($notice) && is_array($notice)): ?>
            <?php $notice_class = !empty($notice['type']) && 'success' === $notice['type'] ? 'notice-success' : 'notice-error'; ?>
                <div class="notice <?php echo esc_attr($notice_class); ?> is-dismissible">
                    <p>
                        <?php
                        $notice_message = isset($notice['message']) ? $notice['message'] : '';

                        if (defined('WP_DEBUG') && WP_DEBUG && strpos($notice_message, "\n") !== false) {
                            $parts = explode("\n\n" . __('Debug Information:', 'smart-support-chat-widget'), $notice_message, 2);
                            $main_message = $parts[0];
                            $debug_info = isset($parts[1]) ? __('Debug Information:', 'smart-support-chat-widget') . "\n" . $parts[1] : '';

                            echo esc_html($main_message);

                            if (!empty($debug_info)) {
                                echo '<br /><br /><strong>' . esc_html__('Debug Information:', 'smart-support-chat-widget') . '</strong><br />';
                                echo '<pre style="background: #f0f0f0; padding: 10px; overflow-x: auto; white-space: pre-wrap; font-size: 12px;">' . esc_html($debug_info) . '</pre>';
                            }

                            if (isset($notice['status_code'])) {
                                echo '<br /><strong>' . esc_html__('HTTP Status Code:', 'smart-support-chat-widget') . '</strong> ' . esc_html($notice['status_code']);
                            }
                        } else {
                            echo esc_html($notice_message);

                            if (isset($notice['status_code'])) {
                                echo '<br /><strong>' . esc_html__('HTTP Status Code:', 'smart-support-chat-widget') . '</strong> ' . esc_html($notice['status_code']);
                            }
                        }
                        ?>
                    </p>
                </div>
            <?php endif; ?>

                <div class="smartsupport-card smartsupport-card-form">
                    <h2><?php esc_html_e('Create New Collection', 'smart-support-chat-widget'); ?></h2>
                    <p class="description"><?php esc_html_e('Configure the data you want to sync with SmartSupport.', 'smart-support-chat-widget'); ?></p>
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('smartsupport_create_rag_collection', 'smartsupport_rag_nonce'); ?>
                    <input type="hidden" name="action" value="smartsupport_create_rag_collection" />
                    <input type="hidden" name="page" value="smartsupport-rag-collections" />
                    <input type="hidden" id="smartsupport-get-fields-nonce" value="<?php echo esc_attr(wp_create_nonce('smartsupport_get_fields')); ?>" />

                    <table class="form-table" role="presentation">
                        <tbody>
                        <!-- Collection Name -->
                        <tr>
                            <th scope="row">
                                <label for="collection_name"><?php esc_html_e('Collection Name', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                <input
                                        type="text"
                                        id="collection_name"
                                        name="collection_name"
                                        value=""
                                        class="regular-text"
                                        required
                                />
                                <p class="description">
                                    <?php esc_html_e('Enter a name for the RAG collection.', 'smart-support-chat-widget'); ?>
                                </p>
                            </td>
                        </tr>

                        <!-- Description -->
                        <tr>
                            <th scope="row">
                                <label for="description"><?php esc_html_e('Description', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                    <textarea
                                            id="description"
                                            name="description"
                                            rows="3"
                                            class="large-text"
                                            required
                                    ></textarea>
                                <p class="description">
                                    <?php esc_html_e('Enter a description for the RAG collection.', 'smart-support-chat-widget'); ?>
                                </p>
                            </td>
                        </tr>

                        <!-- Post Type -->
                        <tr>
                            <th scope="row">
                                <label for="post_type"><?php esc_html_e('Post Type', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                <select id="post_type" name="post_type" class="regular-text" required>
                                    <option value=""><?php esc_html_e('-- Select Post Type --', 'smart-support-chat-widget'); ?></option>
                                    <?php foreach ($post_types as $post_type): ?>
                                        <option value="<?php echo esc_attr($post_type->name); ?>">
                                            <?php echo esc_html($post_type->label); ?> (<?php echo esc_html($post_type->name); ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <p class="description">
                                    <?php esc_html_e('Select the post type to create a RAG collection for.', 'smart-support-chat-widget'); ?>
                                </p>
                            </td>
                        </tr>

                        <!-- Field Source -->
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e('Field Source', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="radio" name="field_source" value="meta" checked />
                                        <?php esc_html_e('Meta Fields', 'smart-support-chat-widget'); ?>
                                    </label>
                                    <?php if ($acpt_enabled): ?>
                                        <br />
                                        <label>
                                            <input type="radio" name="field_source" value="acpt" />
                                            <?php esc_html_e('ACPT Fields', 'smart-support-chat-widget'); ?>
                                        </label>
                                    <?php endif; ?>
                                </fieldset>
                                <p class="description">
                                    <?php esc_html_e('Choose whether to use WordPress meta fields or ACPT fields for the collection.', 'smart-support-chat-widget'); ?>
                                </p>
                            </td>
                        </tr>

                        <!-- Sync Mode -->
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e('Indexing Mode', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="radio" name="sync_mode" value="auto" checked />
                                        <?php esc_html_e('Automatic (index on post save)', 'smart-support-chat-widget'); ?>
                                    </label>
                                    <br />
                                    <label>
                                        <input type="radio" name="sync_mode" value="manual" />
                                        <?php esc_html_e('Manual (use Index buttons only)', 'smart-support-chat-widget'); ?>
                                    </label>
                                </fieldset>
                                <p class="description">
                                    <?php esc_html_e('Control whether posts for this collection are synced automatically when updated, or only when manually triggered.', 'smart-support-chat-widget'); ?>
                                </p>
                            </td>
                        </tr>

                        <!-- Fields Selection -->
                        <tr id="fields-selection-row" style="display: none;">
                            <th scope="row">
                                <label><?php esc_html_e('Select Fields', 'smart-support-chat-widget'); ?></label>
                            </th>
                            <td>
                                <div id="fields-container">
                                    <p class="description"><?php esc_html_e('Select a post type first to see available fields.', 'smart-support-chat-widget'); ?></p>
                                </div>
                            </td>
                        </tr>
                        </tbody>
                    </table>

                    <?php submit_button(__('Create RAG Collection', 'smart-support-chat-widget'), 'primary', 'create_rag_collection'); ?>
                    </form>
                </div>

            <?php if (!empty($saved_collections)): ?>
                <div class="smartsupport-card smartsupport-card-table">
                    <h2><?php esc_html_e('Saved Collections', 'smart-support-chat-widget'); ?></h2>
                    <p class="description"><?php esc_html_e('Click a collection to view details, manage indexing preferences, or remove it.', 'smart-support-chat-widget'); ?></p>
                    <table class="widefat fixed striped smartsupport-rag-table">
                    <thead>
                    <tr>
                        <th><?php esc_html_e('Collection Name', 'smart-support-chat-widget'); ?></th>
                        <th><?php esc_html_e('Collection Unique', 'smart-support-chat-widget'); ?></th>
                        <th><?php esc_html_e('API ID', 'smart-support-chat-widget'); ?></th>
                        <th><?php esc_html_e('Records', 'smart-support-chat-widget'); ?></th>
                        <th><?php esc_html_e('Last Synced', 'smart-support-chat-widget'); ?></th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($saved_collections as $collection_key => $collection): ?>
                        <?php
                        $collection_name = isset($collection['collection_name']) ? $collection['collection_name'] : '';
                        $collection_unique = isset($collection['collection_unique']) ? $collection['collection_unique'] : '';
                        $api_id = isset($collection['api_id']) ? intval($collection['api_id']) : '';
                        $records_count = isset($collection['records_count']) ? intval($collection['records_count']) : '0';
                        $synced_at = isset($collection['synced_at']) ? $collection['synced_at'] : '';
                        ?>
                        <tr class="smartsupport-collection-row" data-collection-key="<?php echo esc_attr($collection_key); ?>">
                            <td><?php echo esc_html($collection_name); ?></td>
                            <td><?php echo esc_html($collection_unique); ?></td>
                            <td><?php echo esc_html($api_id); ?></td>
                            <td><?php echo esc_html($records_count); ?></td>
                            <td><?php echo esc_html($synced_at); ?></td>
                        </tr>
                        <tr class="smartsupport-collection-details" id="smartsupport-details-<?php echo esc_attr($collection_key); ?>" style="display: none;">
                            <td colspan="5">
                                <div class="smartsupport-details-wrapper">
                                    <h3><?php esc_html_e('Collection Details', 'smart-support-chat-widget'); ?></h3>
                                    <ul>
                                        <li><strong><?php esc_html_e('Collection Name:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($collection_name); ?></li>
                                        <li><strong><?php esc_html_e('Collection Unique:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($collection_unique); ?></li>
                                        <li><strong><?php esc_html_e('API ID:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($api_id); ?></li>
                                        <li><strong><?php esc_html_e('Records Count:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($records_count); ?></li>
                                        <li><strong><?php esc_html_e('Last Synced:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($synced_at); ?></li>
                                        <?php if (!empty($collection['post_type'])): ?>
                                            <li><strong><?php esc_html_e('Post Type:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($collection['post_type']); ?></li>
                                        <?php endif; ?>
                                        <?php if (!empty($collection['field_source'])): ?>
                                            <li><strong><?php esc_html_e('Field Source:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($collection['field_source']); ?></li>
                                        <?php endif; ?>
                                        <?php
                                        $sync_mode_label = 'auto';
                                        if (!empty($collection['sync_mode']) && 'manual' === $collection['sync_mode']) {
                                            $sync_mode_label = 'manual';
                                        }
                                        $sync_mode_text = ('manual' === $sync_mode_label)
                                            ? esc_html__('Manual', 'smart-support-chat-widget')
                                            : esc_html__('Automatic', 'smart-support-chat-widget');
                                        ?>
                                        <li><strong><?php esc_html_e('Indexing Mode:', 'smart-support-chat-widget'); ?></strong> <?php echo esc_html($sync_mode_text); ?></li>
                                    </ul>

                                    <h4><?php esc_html_e('Collection Fields', 'smart-support-chat-widget'); ?></h4>
                                    <div style="margin-bottom: 15px;">
                                        <button type="button" class="button button-secondary" onclick="smartsupportToggleEditFields('<?php echo esc_js($collection_key); ?>')">
                                            <span class="smartsupport-edit-fields-label-<?php echo esc_attr($collection_key); ?>"><?php esc_html_e('Edit Fields', 'smart-support-chat-widget'); ?></span>
                                        </button>
                                        <button type="button" class="button button-secondary" onclick="smartsupportRefreshCollection('<?php echo esc_js($collection_key); ?>', <?php echo intval($api_id); ?>)">
                                            <?php esc_html_e('Refresh from API', 'smart-support-chat-widget'); ?>
                                        </button>
                                    </div>

                                    <div id="smartsupport-fields-view-<?php echo esc_attr($collection_key); ?>">
                                        <?php if (!empty($collection['fields'])): ?>
                                            <table class="widefat smartsupport-fields-table">
                                                <thead>
                                                <tr>
                                                    <th><?php esc_html_e('Field Name', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Field Unique', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Type', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Embed', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Metadata', 'smart-support-chat-widget'); ?></th>
                                                </tr>
                                                </thead>
                                                <tbody>
                                                <?php foreach ($collection['fields'] as $field): ?>
                                                    <tr>
                                                        <td><strong><?php echo esc_html($field['field_name']); ?></strong></td>
                                                        <td><code><?php echo esc_html(isset($field['field_unique']) ? $field['field_unique'] : '-'); ?></code></td>
                                                        <td><?php echo esc_html($field['type']); ?></td>
                                                        <td><?php echo !empty($field['embed']) ? esc_html__('Yes', 'smart-support-chat-widget') : esc_html__('No', 'smart-support-chat-widget'); ?></td>
                                                        <td><?php echo !empty($field['metadata']) ? esc_html__('Yes', 'smart-support-chat-widget') : esc_html__('No', 'smart-support-chat-widget'); ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        <?php else: ?>
                                            <p><?php esc_html_e('No specific fields were selected for this collection. Default fields are being used.', 'smart-support-chat-widget'); ?></p>
                                        <?php endif; ?>
                                    </div>

                                    <div id="smartsupport-fields-edit-<?php echo esc_attr($collection_key); ?>" style="display: none;">
                                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="smartsupport-edit-fields-form-<?php echo esc_attr($collection_key); ?>">
                                            <?php wp_nonce_field('smartsupport_update_collection_fields_' . $collection_key, 'smartsupport_fields_nonce'); ?>
                                            <input type="hidden" name="action" value="smartsupport_update_collection_fields" />
                                            <input type="hidden" name="collection_key" value="<?php echo esc_attr($collection_key); ?>" />
                                            <input type="hidden" name="collection_id" value="<?php echo intval($api_id); ?>" />
                                            <input type="hidden" name="_wp_http_referer" value="<?php echo esc_attr(add_query_arg(array('page' => 'smartsupport-rag-collections'), admin_url('admin.php'))); ?>" />
                                            
                                            <p class="description"><?php esc_html_e('Update field settings. At least one of Embed or Metadata must be selected for each field.', 'smart-support-chat-widget'); ?></p>
                                            
                                            <table class="widefat">
                                                <thead>
                                                <tr>
                                                    <th><?php esc_html_e('Field Name', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Field Unique', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Type', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Embed', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Metadata', 'smart-support-chat-widget'); ?></th>
                                                    <th><?php esc_html_e('Actions', 'smart-support-chat-widget'); ?></th>
                                                </tr>
                                                </thead>
                                                <tbody id="smartsupport-fields-edit-tbody-<?php echo esc_attr($collection_key); ?>">
                                                <?php if (!empty($collection['fields'])): ?>
                                                    <?php foreach ($collection['fields'] as $index => $field): ?>
                                                        <?php 
                                                        $field_id = isset($field['id']) ? intval($field['id']) : '';
                                                        $field_name = isset($field['field_name']) ? $field['field_name'] : '';
                                                        $field_unique = isset($field['field_unique']) ? $field['field_unique'] : '';
                                                        $field_type = isset($field['type']) ? $field['type'] : 'string';
                                                        $field_embed = !empty($field['embed']);
                                                        $field_metadata = !empty($field['metadata']);
                                                        ?>
                                                        <tr class="smartsupport-field-row">
                                                            <td>
                                                                <input type="text" 
                                                                       name="fields[<?php echo esc_attr($index); ?>][field_name]" 
                                                                       value="<?php echo esc_attr($field_name); ?>" 
                                                                       class="regular-text" 
                                                                       required />
                                                                <?php if ($field_id): ?>
                                                                    <input type="hidden" name="fields[<?php echo esc_attr($index); ?>][id]" value="<?php echo intval($field_id); ?>" />
                                                                <?php endif; ?>
                                                            </td>
                                                            <td><code><?php echo esc_html($field_unique); ?></code></td>
                                                            <td>
                                                                <select name="fields[<?php echo esc_attr($index); ?>][type]" class="smartsupport-field-setting">
                                                                    <option value="string" <?php selected($field_type, 'string'); ?>>string</option>
                                                                    <option value="text" <?php selected($field_type, 'text'); ?>>text</option>
                                                                    <option value="number" <?php selected($field_type, 'number'); ?>>number</option>
                                                                    <option value="boolean" <?php selected($field_type, 'boolean'); ?>>boolean</option>
                                                                </select>
                                                            </td>
                                                            <td>
                                                                <input type="checkbox" 
                                                                       name="fields[<?php echo esc_attr($index); ?>][embed]" 
                                                                       value="1" 
                                                                       class="smartsupport-field-setting"
                                                                       <?php checked($field_embed); ?> />
                                                            </td>
                                                            <td>
                                                                <input type="checkbox" 
                                                                       name="fields[<?php echo esc_attr($index); ?>][metadata]" 
                                                                       value="1" 
                                                                       class="smartsupport-field-setting"
                                                                       <?php checked($field_metadata); ?> />
                                                            </td>
                                                            <td>
                                                                <button type="button" class="button button-small smartsupport-remove-field" onclick="smartsupportRemoveField(this)">
                                                                    <?php esc_html_e('Remove', 'smart-support-chat-widget'); ?>
                                                                </button>
                                                            </td>
                                                        </tr>
                                                    <?php endforeach; ?>
                                                <?php endif; ?>
                                                </tbody>
                                            </table>
                                            
                                            <div style="margin-top: 15px;">
                                                <button type="button" class="button button-secondary" onclick="smartsupportAddFieldRow('<?php echo esc_js($collection_key); ?>')">
                                                    <?php esc_html_e('+ Add Field', 'smart-support-chat-widget'); ?>
                                                </button>
                                            </div>
                                            
                                            <div style="margin-top: 15px;">
                                                <button type="submit" class="button button-primary">
                                                    <?php esc_html_e('Update Fields', 'smart-support-chat-widget'); ?>
                                                </button>
                                                <button type="button" class="button button-secondary" onclick="smartsupportToggleEditFields('<?php echo esc_js($collection_key); ?>')">
                                                    <?php esc_html_e('Cancel', 'smart-support-chat-widget'); ?>
                                                </button>
                                            </div>
                                        </form>
                                    </div>

                                    <h4><?php esc_html_e('Indexing Preferences', 'smart-support-chat-widget'); ?></h4>
                                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="smartsupport-sync-mode-form">
                                        <?php wp_nonce_field('smartsupport_update_sync_mode_' . $collection_key); ?>
                                        <input type="hidden" name="action" value="smartsupport_update_sync_mode" />
                                        <input type="hidden" name="collection_key" value="<?php echo esc_attr($collection_key); ?>" />
                                        <input type="hidden" name="_wp_http_referer" value="<?php echo esc_attr(add_query_arg(array('page' => 'smartsupport-rag-collections'), admin_url('admin.php'))); ?>" />
                                        <label for="smartsupport-sync-mode-<?php echo esc_attr($collection_key); ?>" class="screen-reader-text"><?php esc_html_e('Indexing Mode', 'smart-support-chat-widget'); ?></label>
                                        <select id="smartsupport-sync-mode-<?php echo esc_attr($collection_key); ?>" name="sync_mode">
                                            <option value="auto" <?php selected($sync_mode_label, 'auto'); ?>>
                                                <?php esc_html_e('Automatic (index on post save)', 'smart-support-chat-widget'); ?>
                                            </option>
                                            <option value="manual" <?php selected($sync_mode_label, 'manual'); ?>>
                                                <?php esc_html_e('Manual (use Index buttons only)', 'smart-support-chat-widget'); ?>
                                            </option>
                                        </select>
                                        <button type="submit" class="button button-secondary" style="margin-top:8px;">
                                            <?php esc_html_e('Update Indexing Mode', 'smart-support-chat-widget'); ?>
                                        </button>
                                    </form>

                                    <h4><?php esc_html_e('Danger Zone', 'smart-support-chat-widget'); ?></h4>
                                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" onsubmit="return confirm('<?php echo esc_js(__('Are you sure you want to delete this collection? This action cannot be undone.', 'smart-support-chat-widget')); ?>');">
                                        <?php wp_nonce_field('smartsupport_delete_rag_collection_' . $collection_key); ?>
                                        <input type="hidden" name="action" value="smartsupport_delete_rag_collection" />
                                        <input type="hidden" name="collection_key" value="<?php echo esc_attr($collection_key); ?>" />
                                        <input type="hidden" name="_wp_http_referer" value="<?php echo esc_attr(add_query_arg(array('page' => 'smartsupport-rag-collections'), admin_url('admin.php'))); ?>" />
                                        <p class="description"><?php esc_html_e('Deleting a collection removes it from SmartSupport and clears local configuration. This cannot be undone.', 'smart-support-chat-widget'); ?></p>
                                        <button type="submit" class="button button-secondary smartsupport-button-danger">
                                            <?php esc_html_e('Delete Collection', 'smart-support-chat-widget'); ?>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
                </div>
            <?php endif; ?>
        </div>
        <?php
        endif;
    }

    /**
     * Process RAG collection creation form submissions
     */
    public function process_create_rag_collection() {
        $redirect_url = admin_url('admin.php?page=smartsupport-rag-collections');

        if (!current_user_can('manage_options')) {
            $message = __('You do not have permission to perform this action.', 'smart-support-chat-widget');

            if (defined('WP_DEBUG') && WP_DEBUG) {
                $current_user = wp_get_current_user();
                $message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n";
                $message .= __('User ID:', 'smart-support-chat-widget') . ' ' . get_current_user_id() . "\n";
                $message .= __('User login:', 'smart-support-chat-widget') . ' ' . $current_user->user_login . "\n";
                $message .= __('Capabilities:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($current_user->allcaps, JSON_PRETTY_PRINT));
            }

            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => $message
            ));

            wp_safe_redirect($redirect_url);
            exit;
        }

        // Verify nonce BEFORE processing any user input
        $nonce = isset($_POST['smartsupport_rag_nonce']) ? sanitize_text_field(wp_unslash($_POST['smartsupport_rag_nonce'])) : '';
        if (empty($nonce) || !wp_verify_nonce($nonce, 'smartsupport_create_rag_collection')) {
            $message = __('Invalid form submission. Please try again.', 'smart-support-chat-widget');

            if (defined('WP_DEBUG') && WP_DEBUG) {
                $message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n";
                $message .= __('Nonce value:', 'smart-support-chat-widget') . ' ' . esc_html($nonce) . "\n";
                // Only log nonce failure, not full POST data for security
                $message .= __('Nonce verification failed.', 'smart-support-chat-widget');
            }

            $this->store_rag_notice(array(
                'success' => false,
                'type' => 'error',
                'message' => $message
            ));

            wp_safe_redirect($redirect_url);
            exit;
        }

        require_once SMARTSUPPORT_WP_PLUGIN_DIR . 'includes/class-smartsupport-api.php';
        $api = new SmartSupport_API();

        // Sanitize POST data before processing
        $post_data = array();
        foreach ($_POST as $key => $value) {
            $sanitized_key = sanitize_key($key);
            if (is_array($value)) {
                // Recursively sanitize array values
                $post_data[$sanitized_key] = $this->sanitize_post_array($value);
            } else {
                // Use appropriate sanitization based on field type
                if (in_array($key, array('collection_name', 'post_type', 'field_source', 'sync_mode'), true)) {
                    $post_data[$sanitized_key] = sanitize_text_field(wp_unslash($value));
                } elseif ($key === 'description') {
                    $post_data[$sanitized_key] = sanitize_textarea_field(wp_unslash($value));
                } elseif ($key === 'selected_fields' && is_array($value)) {
                    $post_data[$sanitized_key] = $this->sanitize_post_array(wp_unslash($value));
                } else {
                    $post_data[$sanitized_key] = sanitize_text_field(wp_unslash($value));
                }
            }
        }
        $result = $this->handle_create_rag_collection($api, $post_data);

        if (!is_array($result)) {
            $result = array(
                'success' => false,
                'message' => __('Unexpected error occurred while creating the collection.', 'smart-support-chat-widget'),
                'type' => 'error'
            );
        }

        if (!empty($result['success']) && isset($result['data']['ragCollection'])) {
            $selected_fields = isset($result['selected_fields']) ? $result['selected_fields'] : array();
            $post_type_used = isset($result['post_type']) ? $result['post_type'] : '';
            $field_source_used = isset($result['field_source']) ? $result['field_source'] : '';
            $sync_mode_used = isset($result['sync_mode']) ? $result['sync_mode'] : 'auto';
            $this->save_rag_collection($result['data']['ragCollection'], $selected_fields, $post_type_used, $field_source_used, $sync_mode_used);
        }

        if (!isset($result['type'])) {
            $result['type'] = !empty($result['success']) ? 'success' : 'error';
        }

        $this->store_rag_notice($result);

        wp_safe_redirect($redirect_url);
        exit;
    }

    /**
     * Store notice data for display on the RAG collections page
     */
    private function store_rag_notice($notice) {
        if (!is_array($notice)) {
            return;
        }

        if (!isset($notice['type'])) {
            $notice['type'] = !empty($notice['success']) ? 'success' : 'error';
        }

        set_transient('smartsupport_rag_notice', $notice, MINUTE_IN_SECONDS);
    }

    /**
     * Retrieve saved RAG collections from local storage
     */
    private function get_saved_rag_collections() {
        $collections = get_option('smartsupport_wp_rag_collections', array());

        if (!is_array($collections)) {
            return array();
        }

        $sanitized_collections = array();
        foreach ($collections as $key => $collection) {
            if (!is_array($collection)) {
                continue;
            }

            $sync_mode = isset($collection['sync_mode']) ? sanitize_text_field($collection['sync_mode']) : 'auto';
            if (!in_array($sync_mode, array('auto', 'manual'), true)) {
                $sync_mode = 'auto';
            }

            $sanitized_collections[$key] = array(
                'collection_name' => isset($collection['collection_name']) ? sanitize_text_field($collection['collection_name']) : '',
                'collection_unique' => isset($collection['collection_unique']) ? sanitize_text_field($collection['collection_unique']) : '',
                'api_id' => isset($collection['api_id']) ? intval($collection['api_id']) : '',
                'records_count' => isset($collection['records_count']) ? intval($collection['records_count']) : 0,
                'created_at' => isset($collection['created_at']) ? sanitize_text_field($collection['created_at']) : '',
                'updated_at' => isset($collection['updated_at']) ? sanitize_text_field($collection['updated_at']) : '',
                'synced_at' => isset($collection['synced_at']) ? sanitize_text_field($collection['synced_at']) : '',
                'post_type' => isset($collection['post_type']) ? sanitize_text_field($collection['post_type']) : '',
                'field_source' => isset($collection['field_source']) ? sanitize_text_field($collection['field_source']) : '',
                'sync_mode' => $sync_mode,
                'fields' => array(),
                'field_map' => array()
            );

            if (isset($collection['fields']) && is_array($collection['fields'])) {
                foreach ($collection['fields'] as $field) {
                    if (!is_array($field)) {
                        continue;
                    }
                    $field_name = isset($field['field_name']) ? sanitize_text_field($field['field_name']) : '';
                    if (empty($field_name)) {
                        continue;
                    }
                    $sanitized_collections[$key]['fields'][] = array(
                        'field_name' => $field_name,
                        'type' => isset($field['type']) ? sanitize_text_field($field['type']) : 'string',
                        'embed' => isset($field['embed']) ? (bool) $field['embed'] : false,
                        'metadata' => isset($field['metadata']) ? (bool) $field['metadata'] : false,
                        'field_unique' => isset($field['field_unique']) ? sanitize_text_field($field['field_unique']) : '',
                        'id' => isset($field['id']) ? intval($field['id']) : ''
                    );
                }
            }

            if (isset($collection['field_map']) && is_array($collection['field_map'])) {
                foreach ($collection['field_map'] as $field_key => $field_details) {
                    $field_key_sanitized = sanitize_text_field($field_key);
                    if (empty($field_key_sanitized) || !is_array($field_details)) {
                        continue;
                    }
                    $sanitized_collections[$key]['field_map'][$field_key_sanitized] = array(
                        'field_unique' => isset($field_details['field_unique']) ? sanitize_text_field($field_details['field_unique']) : '',
                        'type' => isset($field_details['type']) ? sanitize_text_field($field_details['type']) : 'string',
                        'embed' => isset($field_details['embed']) ? (bool) $field_details['embed'] : false,
                        'metadata' => isset($field_details['metadata']) ? (bool) $field_details['metadata'] : false
                    );
                }
            }

            if (empty($sanitized_collections[$key]['field_map']) && !empty($sanitized_collections[$key]['fields'])) {
                foreach ($sanitized_collections[$key]['fields'] as $field_entry) {
                    if (empty($field_entry['field_name'])) {
                        continue;
                    }
                    $sanitized_collections[$key]['field_map'][$field_entry['field_name']] = array(
                        'field_unique' => isset($field_entry['field_unique']) ? $field_entry['field_unique'] : '',
                        'type' => isset($field_entry['type']) ? $field_entry['type'] : 'string',
                        'embed' => isset($field_entry['embed']) ? (bool) $field_entry['embed'] : false,
                        'metadata' => isset($field_entry['metadata']) ? (bool) $field_entry['metadata'] : false
                    );
                }
            }
        }

        return $sanitized_collections;
    }

    /**
     * Save a collection locally after creation
     */
    private function save_rag_collection($collection, $fields = array(), $post_type = '', $field_source = '', $sync_mode = 'auto') {
        if (empty($collection) || !is_array($collection)) {
            return;
        }

        $existing = get_option('smartsupport_wp_rag_collections', array());
        if (!is_array($existing)) {
            $existing = array();
        }

        $collection_unique = isset($collection['collection_unique']) ? sanitize_text_field($collection['collection_unique']) : '';
        $collection_key = !empty($collection_unique) ? sanitize_key($collection_unique) : 'collection_' . (isset($collection['id']) ? intval($collection['id']) : wp_generate_uuid4());

        $sanitized_fields = array();
        $field_map = array();

        $api_fields_map = array();
        if (isset($collection['fields']) && is_array($collection['fields'])) {
            foreach ($collection['fields'] as $api_field) {
                if (!is_array($api_field)) {
                    continue;
                }
                $api_field_name = isset($api_field['field_name']) ? sanitize_text_field($api_field['field_name']) : '';
                if (empty($api_field_name)) {
                    continue;
                }
                $api_fields_map[$api_field_name] = array(
                    'field_unique' => isset($api_field['field_unique']) ? sanitize_text_field($api_field['field_unique']) : '',
                    'type' => isset($api_field['type']) ? sanitize_text_field($api_field['type']) : 'string',
                    'embed' => isset($api_field['embed']) ? (bool) $api_field['embed'] : false,
                    'metadata' => isset($api_field['metadata']) ? (bool) $api_field['metadata'] : false
                );
            }
        }

        if (!empty($fields) && is_array($fields)) {
            foreach ($fields as $field) {
                if (!is_array($field) || empty($field['field_name'])) {
                    continue;
                }

                $field_name = sanitize_text_field($field['field_name']);
                if (empty($field_name)) {
                    continue;
                }

                $field_unique = '';
                if (isset($api_fields_map[$field_name]['field_unique'])) {
                    $field_unique = $api_fields_map[$field_name]['field_unique'];
                }

                $field_type = isset($field['type']) ? sanitize_text_field($field['type']) : 'string';
                $field_embed = isset($field['embed']) ? (bool) $field['embed'] : false;
                $field_metadata = isset($field['metadata']) ? (bool) $field['metadata'] : false;

                if (isset($api_fields_map[$field_name])) {
                    $field_type = $api_fields_map[$field_name]['type'];
                    $field_embed = $api_fields_map[$field_name]['embed'];
                    $field_metadata = $api_fields_map[$field_name]['metadata'];
                }

                $sanitized_fields[] = array(
                    'field_name' => $field_name,
                    'type' => $field_type,
                    'embed' => $field_embed,
                    'metadata' => $field_metadata,
                    'field_unique' => $field_unique
                );

                $field_map[$field_name] = array(
                    'field_unique' => $field_unique,
                    'type' => $field_type,
                    'embed' => $field_embed,
                    'metadata' => $field_metadata
                );
            }
        }

        $sync_mode = sanitize_text_field($sync_mode);
        if (!in_array($sync_mode, array('auto', 'manual'), true)) {
            $sync_mode = 'auto';
        }

        $existing[$collection_key] = array(
            'collection_name' => isset($collection['collection_name']) ? sanitize_text_field($collection['collection_name']) : '',
            'collection_unique' => $collection_unique,
            'api_id' => isset($collection['id']) ? intval($collection['id']) : '',
            'records_count' => isset($collection['records_count']) ? intval($collection['records_count']) : 0,
            'created_at' => isset($collection['created_at']) ? sanitize_text_field($collection['created_at']) : '',
            'updated_at' => isset($collection['updated_at']) ? sanitize_text_field($collection['updated_at']) : '',
            'synced_at' => current_time('mysql'),
            'post_type' => sanitize_text_field($post_type),
            'field_source' => sanitize_text_field($field_source),
            'sync_mode' => $sync_mode,
            'fields' => $sanitized_fields,
            'field_map' => $field_map
        );

        update_option('smartsupport_wp_rag_collections', $existing, false);

        // Reset caches to ensure new data is used
        $this->cached_collections = null;
        $this->collections_by_post_type = null;
    }

    /**
     * Generate the embed snippet for a collection
     */
    private function generate_embed_snippet($config) {
        $agent_token = isset($config['agentToken']) ? sanitize_text_field($config['agentToken']) : '';
        $widget_config = isset($config['widgetConfig']) && is_array($config['widgetConfig']) ? $config['widgetConfig'] : array();

        // Ensure widget config uses sanitized values
        $widget_config_sanitized = array(
            'agentId' => $agent_token,
            'socketUrl' => isset($widget_config['socketUrl']) ? esc_url_raw($widget_config['socketUrl']) : smartsupport_get_base_url(),
            'apiBaseUrl' => isset($widget_config['apiBaseUrl']) ? esc_url_raw($widget_config['apiBaseUrl']) : smartsupport_get_base_url(),
            'title' => isset($widget_config['title']) ? sanitize_text_field($widget_config['title']) : __('Chat Support', 'smart-support-chat-widget'),
            'primaryColor' => isset($widget_config['primaryColor']) ? sanitize_hex_color($widget_config['primaryColor']) : '#6366f1',
            'position' => isset($widget_config['position']) ? sanitize_text_field($widget_config['position']) : 'bottom-right',
            'greetingMessage' => isset($widget_config['greetingMessage']) ? sanitize_text_field($widget_config['greetingMessage']) : __('Welcome! How may I assist you today?', 'smart-support-chat-widget'),
            'showHeader' => isset($widget_config['showHeader']) ? (bool) $widget_config['showHeader'] : true,
            'showToggle' => isset($widget_config['showToggle']) ? (bool) $widget_config['showToggle'] : true,
            'defaultOpen' => isset($widget_config['defaultOpen']) ? (bool) $widget_config['defaultOpen'] : false
        );

        $json_options = 0;
        if (defined('JSON_PRETTY_PRINT')) {
            $json_options |= JSON_PRETTY_PRINT;
        }
        if (defined('JSON_UNESCAPED_SLASHES')) {
            $json_options |= JSON_UNESCAPED_SLASHES;
        }

        $widget_config_json = wp_json_encode($widget_config_sanitized, $json_options);

        // Generate embed snippet as example code (for display only, not execution)
        // Using HTML entities to prevent plugin checker from flagging as actual script tags
        $snippet = "&lt;script&gt;\n";
        $snippet .= "    window.chatWidgetConfig = " . esc_html($widget_config_json) . ";\n";
        $snippet .= "&lt;/script&gt;\n";
        $snippet .= "&lt;script src=\"" . esc_url(smartsupport_get_base_url() . '/js/webhooks/chat.js') . "\" defer&gt;&lt;/script&gt;";

        // Return as-is since we've already escaped the content
        return $snippet;
    }

    /**
     * Handle RAG collection creation
     */
    private function handle_create_rag_collection($api, $post_data) {
        $collection_name = isset($post_data['collection_name']) ? sanitize_text_field($post_data['collection_name']) : '';
        $description = isset($post_data['description']) ? sanitize_textarea_field($post_data['description']) : '';
        $post_type = isset($post_data['post_type']) ? sanitize_text_field($post_data['post_type']) : '';
        $field_source = isset($post_data['field_source']) ? sanitize_text_field($post_data['field_source']) : 'meta';
        $sync_mode = isset($post_data['sync_mode']) && 'manual' === sanitize_text_field($post_data['sync_mode']) ? 'manual' : 'auto';
        $selected_fields = isset($post_data['selected_fields']) && is_array($post_data['selected_fields']) ? $post_data['selected_fields'] : array();

        $settings = get_option('smartsupport_wp_settings', array());
        $agent_token = isset($settings['agent_id']) ? sanitize_text_field($settings['agent_id']) : '';

        if (empty($collection_name) || empty($description) || empty($post_type)) {
            $message = __('Please fill in all required fields.', 'smart-support-chat-widget');

            if (defined('WP_DEBUG') && WP_DEBUG) {
                // Sanitize data for safe debug output
                $sanitized_debug_data = array();
                foreach ($post_data as $key => $value) {
                    if (is_array($value)) {
                        $sanitized_debug_data[$key] = 'Array(' . count($value) . ' items)';
                    } else {
                        $sanitized_debug_data[$key] = sanitize_text_field($value);
                    }
                }
                $message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n";
                $message .= __('Received data:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($sanitized_debug_data, JSON_PRETTY_PRINT));
            }

            return array(
                'success' => false,
                'message' => $message
            );
        }

        if (empty($agent_token)) {
            $message = __('Agent token is not configured. Please add it in the plugin settings.', 'smart-support-chat-widget');

            if (defined('WP_DEBUG') && WP_DEBUG) {
                $message .= "\n\n" . __('Debug Information:', 'smart-support-chat-widget') . "\n";
                $message .= __('Settings data:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($settings, JSON_PRETTY_PRINT));
            }

            return array(
                'success' => false,
                'message' => $message
            );
        }

        // Build tokens array from saved settings
        $tokens = array($agent_token);

        // Build fields array
        $fields = array();
        if (!empty($selected_fields) && is_array($selected_fields)) {
            foreach ($selected_fields as $field_name => $field_data) {
                if (!is_array($field_data)) {
                    continue;
                }

                if (isset($field_data['include']) && '1' === (string) $field_data['include']) {
                    $type = isset($field_data['type']) ? sanitize_text_field($field_data['type']) : 'string';
                    $allowed_types = array('string', 'text', 'number', 'boolean');
                    if (!in_array($type, $allowed_types, true)) {
                        $type = 'string';
                    }

                    $embed = isset($field_data['embed']) && '1' === (string) $field_data['embed'];
                    $metadata = isset($field_data['metadata']) && '1' === (string) $field_data['metadata'];
                    
                    // At least one of embed or metadata must be true (API requirement)
                    // If neither is set, default: embed=true, metadata=false for content fields, embed=false, metadata=true for ID/URL fields
                    if (!$embed && !$metadata) {
                        // Default based on field type/name - content fields embed, metadata fields use metadata
                        if (in_array($field_name, array('post_id', 'post_url'), true)) {
                            $metadata = true;
                        } else {
                            $embed = true;
                        }
                    }

                    $fields[] = array(
                        'field_name' => sanitize_text_field($field_name),
                        'type' => $type,
                        'embed' => $embed,
                        'metadata' => $metadata
                    );
                }
            }
        }

        // Add default fields if none selected
        if (empty($fields)) {
            $fields = array(
                array(
                    'field_name' => 'post_title',
                    'type' => 'string',
                    'embed' => true,
                    'metadata' => false
                ),
                array(
                    'field_name' => 'post_content',
                    'type' => 'text',
                    'embed' => true,
                    'metadata' => false
                ),
                array(
                    'field_name' => 'post_id',
                    'type' => 'number',
                    'embed' => false,
                    'metadata' => true
                ),
                array(
                    'field_name' => 'post_url',
                    'type' => 'string',
                    'embed' => false,
                    'metadata' => true
                )
            );
        }

        // Create collection via API
        try {
            $result = $api->create_rag_collection($collection_name, $description, $tokens, $fields);
        } catch (Exception $e) {
            $error_message = __('Error creating RAG collection:', 'smart-support-chat-widget') . ' ' . $e->getMessage();

            // Add detailed error information if WP_DEBUG is enabled
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $error_message .= "\n\n" . __('Exception details:', 'smart-support-chat-widget') . "\n" . $e->getTraceAsString();
                $error_message .= "\n\n" . __('Collection data:', 'smart-support-chat-widget') . "\n";
                $error_message .= __('Name:', 'smart-support-chat-widget') . ' ' . $collection_name . "\n";
                $error_message .= __('Description:', 'smart-support-chat-widget') . ' ' . $description . "\n";
                $error_message .= __('Post Type:', 'smart-support-chat-widget') . ' ' . $post_type . "\n";
                $error_message .= __('Tokens:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($tokens, JSON_PRETTY_PRINT)) . "\n";
                $error_message .= __('Fields:', 'smart-support-chat-widget') . ' ' . esc_html(wp_json_encode($fields, JSON_PRETTY_PRINT));
            }

            return array(
                'success' => false,
                'message' => $error_message
            );
        }

        if (is_array($result) && isset($result['success']) && $result['success']) {
            $result['selected_fields'] = $fields;
            $result['post_type'] = $post_type;
            $result['field_source'] = $field_source;
            $result['sync_mode'] = $sync_mode;
        }

        return $result;
    }
}

