(function ($) {
    'use strict';

    var vars = window.smartsupportAdminVars || {};
    var strings = vars.strings || {};

    function getString(key, fallback) {
        if (strings && Object.prototype.hasOwnProperty.call(strings, key) && strings[key]) {
            return strings[key];
        }
        return fallback || '';
    }

    function escapeHtml(text) {
        return $('<div />').text(text || '').html();
    }

    function updateFieldSettings($checkbox) {
        var $row = $checkbox.closest('tr');
        var $settings = $row.find('.smartsupport-field-setting');
        var shouldEnable = $checkbox.is(':checked');

        $settings.prop('disabled', !shouldEnable);
    }

    function initFieldSettingHandlers($context) {
        var $scope = $context || $(document);

        $scope.find('.field-include-checkbox').each(function () {
            updateFieldSettings($(this));
        });
    }

    $(document).on('change', '.field-include-checkbox', function () {
        updateFieldSettings($(this));
    });

    function createErrorNotice(message) {
        return '<div class="notice notice-error inline" style="margin: 0;"><p>' + escapeHtml(message) + '</p></div>';
    }

    function initFieldSelectors() {
        var $postType = $('#post_type');

        if (!$postType.length) {
            return;
        }

        var $fieldSource = $('input[name="field_source"]');
        var $fieldsRow = $('#fields-selection-row');
        var $fieldsContainer = $('#fields-container');
        var $nonceField = $('#smartsupport-get-fields-nonce');

        function resetFieldsContainer() {
            $fieldsContainer.html('<p class="description">' + escapeHtml(
                getString('selectPostType', 'Select a post type first to see available fields.')
            ) + '</p>');
            $fieldsRow.hide();
        }

        function renderError(message) {
            $fieldsContainer.html(createErrorNotice(
                message || getString('unableToLoadFields', 'Unable to load fields for the selected post type.')
            ));
            $fieldsRow.show();
        }

        function showLoading() {
            $fieldsRow.show();
            $fieldsContainer.html('<p class="description">' + escapeHtml(
                getString('loadingFields', 'Loading fields...')
            ) + '</p>');
        }

        function loadFields() {
            var postType = $postType.val();
            var fieldSource = $fieldSource.filter(':checked').val();
            var nonce = $nonceField.val();

            if (!postType) {
                resetFieldsContainer();
                return;
            }

            if (!nonce) {
                renderError(getString('securityTokenMissing', 'Security token missing. Please refresh the page.'));
                return;
            }

            showLoading();

            $.ajax({
                url: window.ajaxurl,
                method: 'POST',
                dataType: 'json',
                data: {
                    action: 'smartsupport_get_fields',
                    post_type: postType,
                    field_source: fieldSource,
                    nonce: nonce
                }
            }).done(function (response) {
                if (response && response.success && response.data) {
                    $fieldsContainer.html(response.data);
                    $fieldsRow.show();
                    initFieldSettingHandlers($fieldsContainer);
                } else {
                    renderError(response && response.data ? response.data : null);
                }
            }).fail(function (jqXHR) {
                var message = getString('requestFailed', 'Request failed. Please try again.');
                if (jqXHR && jqXHR.responseJSON && jqXHR.responseJSON.data) {
                    message = jqXHR.responseJSON.data;
                }
                renderError(message);
            });
        }

        $postType.on('change', loadFields);
        $fieldSource.on('change', function () {
            if ($postType.val()) {
                loadFields();
            }
        });

        resetFieldsContainer();
    }

    function initCollectionToggle() {
        $(document).on('click', '.smartsupport-collection-row', function (event) {
            if ($(event.target).closest('a, button, input, select, textarea, label, .button').length) {
                return;
            }

            var key = $(this).data('collection-key');
            var $detailsRow = $('#smartsupport-details-' + key);

            if (!$detailsRow.length) {
                return;
            }

            var isVisible = $detailsRow.is(':visible');

            $('.smartsupport-collection-details').not($detailsRow).slideUp(200);

            if (isVisible) {
                $detailsRow.slideUp(200);
            } else {
                $detailsRow.slideDown(200);
            }
        });
    }

    function initAgentSelector() {
        var $agentForm = $('#smartsupport-agent-selection-form');
        var $agentSelect = $('#smartsupport_agent_id');
        var $loadingDiv = $('#smartsupport-agent-loading');
        var $errorDiv = $('#smartsupport-agent-error');
        var $saveBtn = $('#smartsupport-save-agent-btn');

        if (!$agentForm.length || !$agentSelect.length) {
            return;
        }

        function showLoading() {
            $loadingDiv.show();
            $errorDiv.hide();
            $agentSelect.hide();
            $saveBtn.hide();
        }

        function showError(message) {
            $loadingDiv.hide();
            $errorDiv.html('<p>' + escapeHtml(message) + '</p>').show();
            $agentSelect.hide();
            $saveBtn.hide();
        }

        function showAgents(agents) {
            $loadingDiv.hide();
            $errorDiv.hide();
            
            // Clear existing options except the first one
            $agentSelect.find('option:not(:first)').remove();
            
            if (agents && agents.length > 0) {
                $.each(agents, function (index, agent) {
                    var optionText = escapeHtml(agent.name || agent.public_id);
                    $agentSelect.append(
                        $('<option></option>')
                            .attr('value', agent.public_id)
                            .text(optionText)
                    );
                });
                $agentSelect.show();
                $saveBtn.show();
            } else {
                showError('No agents found. Please create an agent first.');
            }
        }

        function loadAgents() {
            showLoading();

            // Get nonce safely
            var nonce = '';
            if (window.smartsupportAdminVars && window.smartsupportAdminVars.agentNonce) {
                nonce = window.smartsupportAdminVars.agentNonce;
            }

            if (!nonce) {
                console.error('SmartSupport: Nonce missing', window.smartsupportAdminVars);
                showError('Security token missing. Please refresh the page.');
                return;
            }

            if (!window.ajaxurl) {
                console.error('SmartSupport: ajaxurl missing');
                showError('AJAX URL is not configured.');
                return;
            }

            console.log('SmartSupport: Loading agents...', {
                ajaxurl: window.ajaxurl,
                action: 'smartsupport_get_agents',
                hasNonce: !!nonce
            });

            $.ajax({
                url: window.ajaxurl,
                method: 'POST',
                dataType: 'json',
                data: {
                    action: 'smartsupport_get_agents',
                    nonce: nonce
                }
            }).done(function (response) {
                console.log('SmartSupport: AJAX response', response);
                if (response && response.success && response.data) {
                    showAgents(response.data);
                } else {
                    var errorMsg = 'Failed to load agents.';
                    if (response && response.data) {
                        errorMsg = response.data;
                    }
                    console.error('SmartSupport: Error loading agents', response);
                    showError(errorMsg);
                }
            }).fail(function (jqXHR, textStatus, errorThrown) {
                console.error('SmartSupport: AJAX failed', {
                    jqXHR: jqXHR,
                    textStatus: textStatus,
                    errorThrown: errorThrown,
                    responseText: jqXHR && jqXHR.responseText
                });
                var message = 'Request failed. Please try again.';
                if (jqXHR && jqXHR.responseJSON) {
                    if (jqXHR.responseJSON.data) {
                        message = jqXHR.responseJSON.data;
                    } else if (jqXHR.responseJSON.message) {
                        message = jqXHR.responseJSON.message;
                    }
                } else if (errorThrown) {
                    message = 'Error: ' + errorThrown;
                } else if (jqXHR && jqXHR.responseText) {
                    try {
                        var errorResponse = JSON.parse(jqXHR.responseText);
                        if (errorResponse.data) {
                            message = errorResponse.data;
                        }
                    } catch (e) {
                        // Not JSON, use default message
                    }
                }
                showError(message);
            });
        }

        // Load agents on page load
        loadAgents();

        // Handle agent selection change
        $agentSelect.on('change', function () {
            if ($(this).val()) {
                $saveBtn.prop('disabled', false);
            } else {
                $saveBtn.prop('disabled', true);
            }
        });
    }

    // Toggle edit fields view
    window.smartsupportToggleEditFields = function(collectionKey) {
        var $view = $('#smartsupport-fields-view-' + collectionKey);
        var $edit = $('#smartsupport-fields-edit-' + collectionKey);
        var $label = $('.smartsupport-edit-fields-label-' + collectionKey);
        
        if ($view.is(':visible')) {
            $view.hide();
            $edit.show();
            $label.text('Cancel Editing');
        } else {
            $view.show();
            $edit.hide();
            $label.text('Edit Fields');
        }
    };

    // Add new field row
    window.smartsupportAddFieldRow = function(collectionKey) {
        var $tbody = $('#smartsupport-fields-edit-tbody-' + collectionKey);
        var rowCount = $tbody.find('tr').length;
        var newIndex = rowCount;
        
        var newRow = '<tr class="smartsupport-field-row">' +
            '<td><input type="text" name="fields[' + newIndex + '][field_name]" value="" class="regular-text" required placeholder="e.g. post_title" /></td>' +
            '<td><code>-</code> <small style="color: #666;">(will be generated)</small></td>' +
            '<td><select name="fields[' + newIndex + '][type]" class="smartsupport-field-setting">' +
            '<option value="string">string</option>' +
            '<option value="text">text</option>' +
            '<option value="number">number</option>' +
            '<option value="boolean">boolean</option>' +
            '</select></td>' +
            '<td><input type="checkbox" name="fields[' + newIndex + '][embed]" value="1" class="smartsupport-field-setting" checked /></td>' +
            '<td><input type="checkbox" name="fields[' + newIndex + '][metadata]" value="1" class="smartsupport-field-setting" /></td>' +
            '<td><button type="button" class="button button-small smartsupport-remove-field" onclick="smartsupportRemoveField(this)">Remove</button></td>' +
            '</tr>';
        
        $tbody.append(newRow);
    };

    // Remove field row
    window.smartsupportRemoveField = function(button) {
        if (confirm('Are you sure you want to remove this field?')) {
            $(button).closest('tr').remove();
        }
    };

    // Refresh collection from API
    window.smartsupportRefreshCollection = function(collectionKey, collectionId) {
        if (!window.ajaxurl) {
            alert('AJAX URL is not configured.');
            return;
        }

        if (!confirm('This will refresh the collection data from the API. Continue?')) {
            return;
        }

        $.ajax({
            url: window.ajaxurl,
            method: 'POST',
            dataType: 'json',
            data: {
                action: 'smartsupport_refresh_collection',
                collection_id: collectionId,
                nonce: window.smartsupportAdminVars && window.smartsupportAdminVars.refreshNonce ? window.smartsupportAdminVars.refreshNonce : ''
            }
        }).done(function (response) {
            if (response && response.success) {
                alert('Collection refreshed successfully. Please refresh the page to see updated data.');
                location.reload();
            } else {
                alert('Failed to refresh collection: ' + (response && response.data ? response.data : 'Unknown error'));
            }
        }).fail(function (jqXHR) {
            var message = 'Request failed. Please try again.';
            if (jqXHR && jqXHR.responseJSON && jqXHR.responseJSON.data) {
                message = jqXHR.responseJSON.data;
            }
            alert(message);
        });
    };

    $(function () {
        initCollectionToggle();
        initFieldSelectors();
        initFieldSettingHandlers($(document));
        initAgentSelector();
    });
})(jQuery);

