<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
class SmartPrefetch_21_Public
{
    /**
     * Smart Prefetch CDN domain constant
     */
    const SMART_PREFETCH_CDN_DOMAIN = 'https://pub-cdn.smartprefetch.link';
    public static function addActions()
    {
        //is_user_logged_in is a pluggable function and you could get a fatal error if you call it too early.
        add_action('init', 'SmartPrefetch_21_Public::init', 10);
    }

    public static function init()
    {
        add_action('wp_enqueue_scripts', 'SmartPrefetch_21_Public::injectScript');
        add_action('template_redirect', 'SmartPrefetch_21_Public::modifyCSP', 1);
        add_filter('wp_headers', 'SmartPrefetch_21_Public::filterCSPHeaders');
        add_action('rest_api_init', 'SmartPrefetch_21_Public::registerRestRoutes');
    }

    public static function injectScript()
    {
        wp_enqueue_script('smart-prefetch-index', self::SMART_PREFETCH_CDN_DOMAIN . '/prefetch/index.min.js', [], SMART_PREFETCH_PLUG_VERSION, ['strategy' => 'async', 'in_footer' => true]);
    }

    /**
     * Modify Content Security Policy headers to allow Smart Prefetch script
     * This function adds the necessary script-src directive to allow the Smart Prefetch script to load
     */
    public static function modifyCSP()
    {
        // Get existing CSP header
        $headers = headers_list();
        $csp_header = null;
        $csp_header_name = null;

        // Check for existing CSP headers (both standard and webkit)
        foreach ($headers as $header) {
            if (stripos($header, 'Content-Security-Policy:') === 0) {
                $csp_header = trim(substr($header, 25)); // Remove "Content-Security-Policy: "
                $csp_header_name = 'Content-Security-Policy';
                break;
            } elseif (stripos($header, 'Content-Security-Policy-Report-Only:') === 0) {
                $csp_header = trim(substr($header, 37)); // Remove "Content-Security-Policy-Report-Only: "
                $csp_header_name = 'Content-Security-Policy-Report-Only';
                break;
            }
        }

        // If CSP header exists, modify it
        if ($csp_header && $csp_header_name) {
            $smart_prefetch_domain = self::SMART_PREFETCH_CDN_DOMAIN;
            $new_csp = self::addScriptSrcToCSP($csp_header, $smart_prefetch_domain);

            // Only update header if changes were made
            if ($new_csp !== $csp_header) {
                header_remove($csp_header_name);
                header($csp_header_name . ': ' . $new_csp);
            }
        }
    }

    /**
     * Filter WordPress headers to add CSP support for Smart Prefetch
     * This provides an alternative method to modify CSP headers through WordPress filters
     */
    public static function filterCSPHeaders($headers)
    {
        $smart_prefetch_domain = self::SMART_PREFETCH_CDN_DOMAIN;

        // Check if CSP header exists in the headers array
        if (isset($headers['Content-Security-Policy'])) {
            $headers['Content-Security-Policy'] = self::addScriptSrcToCSP($headers['Content-Security-Policy'], $smart_prefetch_domain);
        } elseif (isset($headers['Content-Security-Policy-Report-Only'])) {
            $headers['Content-Security-Policy-Report-Only'] = self::addScriptSrcToCSP($headers['Content-Security-Policy-Report-Only'], $smart_prefetch_domain);
        }

        return $headers;
    }

    /**
     * Helper function to add script-src directive to CSP
     */
    private static function addScriptSrcToCSP($csp_header, $domain)
    {
        // Check if script-src directive exists
        if (preg_match('/script-src\s+([^;]+)/', $csp_header, $matches)) {
            $script_src_value = trim($matches[1]);

            // Check if our domain is already included
            if (strpos($script_src_value, $domain) === false) {
                // Add our domain to existing script-src
                $new_script_src = 'script-src ' . $script_src_value . ' ' . $domain;
                return preg_replace('/script-src\s+[^;]+/', $new_script_src, $csp_header);
            }
        } else {
            // No script-src directive exists, add one
            return $csp_header . '; script-src \'self\' ' . $domain;
        }

        return $csp_header;
    }

    /**
     * Register REST API routes for Smart Prefetch
     */
    public static function registerRestRoutes()
    {
        register_rest_route('smartprefetch/v1', '/validate-email', [
            'methods' => 'POST',
            'callback' => 'SmartPrefetch_21_Public::validateEmailEndpoint',
            'permission_callback' => '__return_true', // Public endpoint but with nonce verification
            'args' => [
                'email' => [
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => 'SmartPrefetch_21_Public::validateEmailFormat',
                    'sanitize_callback' => 'sanitize_email',
                ],
                'nonce' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ]
            ]
        ]);
    }

    /**
     * Validate email format
     */
    public static function validateEmailFormat($value, $request, $param)
    {
        return is_email($value);
    }

    /**
     * Email validation endpoint callback
     */
    public static function validateEmailEndpoint($request)
    {
        // Rate limiting check
        if (!self::checkRateLimit()) {
            return new WP_Error(
                'rate_limit_exceeded',
                'Too many requests. Please try again later.',
                ['status' => 429]
            );
        }

        $email = $request->get_param('email');
        $nonce = $request->get_param('nonce');

        // Verify nonce for security - using custom verification for external API calls
        if (!SmartPrefetch_21_Core::verifyApiNonce($nonce)) {
            return new WP_Error(
                'invalid_nonce',
                'Invalid security token.',
                ['status' => 403]
            );
        }

        // Additional email validation
        if (!is_email($email)) {
            return new WP_Error(
                'invalid_email',
                'Invalid email format.',
                ['status' => 400]
            );
        }

        // Check if email exists in WordPress users table with active status
        $is_valid = self::isValidActiveUser($email);

        return rest_ensure_response([
            'valid' => $is_valid
        ]);
    }

    /**
     * Check if email belongs to an active user
     */
    private static function isValidActiveUser($email)
    {
        global $wpdb;

        $user = $wpdb->get_row($wpdb->prepare(
            "SELECT user_status FROM {$wpdb->users} WHERE user_email = %s",
            $email
        ));

        // User exists and has active status (user_status = 0 means active)
        return $user && (int)$user->user_status === 0;
    }

    /**
     * Basic rate limiting implementation
     */
    private static function checkRateLimit()
    {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'NA';
        $transient_key = 'smartprefetch_rate_limit_' . md5($ip);

        $requests = get_transient($transient_key);

        if ($requests === false) {
            // First request from this IP
            set_transient($transient_key, 1, MINUTE_IN_SECONDS);
            return true;
        }

        if ($requests >= 3) { // Max 3 requests per minute
            return false;
        }

        set_transient($transient_key, $requests + 1, MINUTE_IN_SECONDS);
        return true;
    }
}
