<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
class SmartPrefetch_21_Core
{

    public static function getRLBaseDomain()
    {
        return SmartPrefetch_21_Util_Core::isDev() ? 'https://smartprefetch.link/' : 'https://smartprefetch.link/';
    }

    public static function getAPIDomain()
    {
        return SmartPrefetch_21_Util_Core::isDev() ? 'https://api.smartprefetch.link/' : 'https://api.smartprefetch.link/';
    }

    private static function isTemporaryError($apiMessage)
    {
        $temp_errors = ['timed out', 'Could not resolve host', 'error setting certificate', 'Connection reset', 'OpenSSL', 'getaddrinfo() thread', 'SSL connection timeout', 'Unknown SSL', 'SSL_ERROR_SYSCALL', 'Failed connect to', 'cURL error 77'];
        $found = false;
        foreach ($temp_errors as $msg) {
            if (stripos($apiMessage, $msg) !== false) {
                $found = true;
                break;
            }
        }
        return $found;
    }

    public static function &callGETAPI($endpoint, $payload, &$apiError, &$apiMessage)
    {
        $httpRes = [];
        $args = [];
        $apiError = false;

        $url = SmartPrefetch_21_Core::getAPIDomain();
        if (!empty($payload)) {
            $endpoint .= '?get_params=' . base64_encode(json_encode($payload));
        }
        $args['method'] = 'GET';
        $args['sslverify'] = !SmartPrefetch_21_Util_Core::isDev();
        $args['headers'] = [
            'Accept' => 'application/json',
        ];

        try {
            $http = wp_remote_get($url . $endpoint, $args);
            $code = wp_remote_retrieve_response_code($http);
            if (is_wp_error($http)) {
                $apiError = true;
                $apiMessage = $http->get_error_message();
                if (empty($apiMessage)) {
                    $apiMessage = '';
                }
                if (self::isTemporaryError($apiMessage)) {
                    //chill, it happens
                } else {
                    SmartPrefetch_21_Core::on_exception($http);
                }
            } else if (in_array($code, [401, 403])) {
                $apiError = true;
                $apiMessage = "Unauthorized access. Please disconnect and Login again.";
            } else {
                $httpRes =  json_decode(wp_remote_retrieve_body($http), true);
            }
        } catch (Throwable $e) {
            SmartPrefetch_21_Core::on_exception($e);
            $apiError = true;
            $apiMessage = $e->getMessage();
        }
        return $httpRes;
    }

    public static function &callPOSTAPI($endpoint, $payload, &$apiError, &$apiMessage)
    {
        $httpRes = [];
        $args = [];
        $apiError = false;

        $url = SmartPrefetch_21_Core::getAPIDomain();
        $args['method'] = 'POST';
        $args['body'] = json_encode($payload);
        $args['sslverify'] = !SmartPrefetch_21_Util_Core::isDev();
        $args['headers'] = [
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ];

        try {
            $http = wp_remote_post($url . $endpoint, $args);
            $code = wp_remote_retrieve_response_code($http);
            if (is_wp_error($http)) {
                $apiError = true;
                $apiMessage = $http->get_error_message();
                if (empty($apiMessage)) {
                    $apiMessage = '';
                }
                if (self::isTemporaryError($apiMessage)) {
                    //chill, it happens
                } else {
                    SmartPrefetch_21_Core::on_exception($http);
                }
            } else if (in_array($code, [401, 403])) {
                $apiError = true;
                $apiMessage = "Unauthorized access. Please disconnect and Login again.";
            } else {
                $httpRes =  json_decode(wp_remote_retrieve_body($http), true);
            }
        } catch (Throwable $e) {
            SmartPrefetch_21_Core::on_exception($e);
            $apiError = true;
            $apiMessage = $e->getMessage();
        }
        return $httpRes;
    }

    public static function on_exception($exception, $limit = 8)
    {
        if (WP_DEBUG) {
            error_log($exception->getMessage() . ' on ' . $exception->getFile() . ' : ' . $exception->getLine());
            //var_dump($exception);
        }
    }

    /**
     * Generate API token for our own use
     */
    public static function generateApiToken()
    {
        $timestamp = time();
        $site_key = self::getSiteKey();
        $data = $timestamp . '|' . $site_key;
        $hash = hash_hmac('sha256', $data, $site_key);

        return base64_encode($timestamp . '|' . $hash);
    }

    /**
     * Verify API token for external use
     */
    public static function verifyApiNonce($token)
    {
        if (empty($token)) {
            return false;
        }

        // First try WordPress nonce verification for backward compatibility
        if (wp_verify_nonce($token, 'smartprefetch_validate_email')) {
            return true;
        }

        // Try custom token verification for external API calls
        $decoded = base64_decode($token);
        if (!$decoded) {
            return false;
        }

        $parts = explode('|', $decoded);
        if (count($parts) !== 2) {
            return false;
        }

        list($timestamp, $hash) = $parts;

        // Check if token is not too old (24 hours max)
        if ((time() - (int)$timestamp) > (24 * HOUR_IN_SECONDS)) {
            return false;
        }

        // Verify hash
        $site_key = self::getSiteKey();
        $expected_data = $timestamp . '|' . $site_key;
        $expected_hash = hash_hmac('sha256', $expected_data, $site_key);

        return hash_equals($expected_hash, $hash);
    }

    /**
     * Get unique site key for token generation
     */
    private static function getSiteKey()
    {
        // Use WordPress constants and options to create a unique site key
        $key_components = [
            ABSPATH,
            get_option('siteurl'),
            get_option('admin_email'),
            defined('AUTH_KEY') ? AUTH_KEY : '',
            defined('SECURE_AUTH_KEY') ? SECURE_AUTH_KEY : ''
        ];

        return hash('sha256', implode('|', $key_components));
    }
}
