<?php
/**
 * Plugin Name: Smart Order Notifier
 * Description: Sends WhatsApp alerts via Meta Cloud API for new and updated WooCommerce orders.
 * Version: 1.3.0
 * Author: Muhammad Arslan Farooq
 * License: GPL-2.0-or-later
 * Text Domain: smart-order-notifier
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

function smartorder_load_textdomain() {
    load_plugin_textdomain('smart-order-notifier', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('plugins_loaded', 'smartorder_load_textdomain');

foreach (['admin-settings.php', 'admin-notifier.php', 'customer-notifier.php'] as $file) {
    $path = plugin_dir_path(__FILE__) . $file;
    if (file_exists($path)) {
        require_once $path;
    } elseif (defined('WP_DEBUG') && WP_DEBUG) {
        error_log("Smart Order Notifier: Missing required file $file");
    }
}

class SmartOrderNotifier_Plugin {
    private $option_name = 'smarorno_settings';
    private $settings;
    private $admin_notifier;
    private $customer_notifier;

    public function __construct() {
        $this->settings = get_option($this->option_name, []);
        new SmartOrderNotifier_Admin_Settings();

        $this->admin_notifier = new SmartOrderNotifier_Admin_Notifier($this->settings);
        $this->customer_notifier = new SmartOrderNotifier_Customer_Notifier($this->settings);

        add_action('woocommerce_new_order', [$this, 'on_new_order'], 10, 1);
        add_action('woocommerce_order_status_changed', [$this, 'on_order_status_changed'], 10, 4);
        add_action('admin_init', [$this, 'reload_settings']);
    }

    public function reload_settings() {
        $new_settings = get_option($this->option_name, []);
        if ($new_settings !== $this->settings) {
            $this->settings = $new_settings;
            $this->admin_notifier = new SmartOrderNotifier_Admin_Notifier($this->settings);
            $this->customer_notifier = new SmartOrderNotifier_Customer_Notifier($this->settings);
        }
    }

    public function on_new_order($order_id) {
        if (empty($this->settings)) return;
        if (!empty($this->settings['paid_orders_only'])) {
            $order = wc_get_order($order_id);
            if (!$order || !$order->is_paid()) return;
        }
        $this->admin_notifier->send_alert($order_id, __('Order Placed', 'smart-order-notifier'));
        $this->customer_notifier->send_alert($order_id, __('Placed', 'smart-order-notifier'));
    }

    public function on_order_status_changed($order_id, $old_status, $new_status, $order) {
        if (!$order) return;

        // translators: %s is the new order status (e.g., completed, cancelled).
        $this->admin_notifier->send_alert($order_id, sprintf(__('Status changed to %s', 'smart-order-notifier'), ucfirst($new_status)));
        $this->customer_notifier->send_alert($order_id, ucfirst($new_status));
    }
}

new SmartOrderNotifier_Plugin();
