<?php
// admin-settings.php
if (!defined('ABSPATH')) exit;

class SmartOrderNotifier_Admin_Settings {
    private $option_name = 'smarorno_settings';

    public function __construct() {
        add_action('admin_menu', [$this, 'register_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
    }

    public function register_settings_page() {
        add_options_page(
            __('Smart Order Notifier', 'smart-order-notifier'),
            __('Smart Order Notifier', 'smart-order-notifier'),
            'manage_options',
            'smarorno-admin',
            [$this, 'settings_page_html']
        );
    }

    public function register_settings() {
        register_setting(
            'smarorno_settings_group',
            $this->option_name,
            ['sanitize_callback' => [$this, 'sanitize_settings']]
        );
    }

    public function sanitize_settings($input) {
        $output = [];
        $output['access_token'] = sanitize_text_field($input['access_token'] ?? '');
        $output['phone_number_id'] = sanitize_text_field($input['phone_number_id'] ?? '');
        $output['template_name'] = sanitize_text_field($input['template_name'] ?? '');
        $output['template_language'] = sanitize_text_field($input['template_language'] ?? 'en_US');
        $output['customer_template_name'] = sanitize_text_field($input['customer_template_name'] ?? '');
        $output['customer_template_language'] = sanitize_text_field($input['customer_template_language'] ?? 'en_US');
        $output['admin_numbers'] = sanitize_text_field($input['admin_numbers'] ?? '');
        $output['button_url'] = esc_url_raw($input['button_url'] ?? '');
        $output['paid_orders_only'] = isset($input['paid_orders_only']) ? 1 : 0;
        $output['enable_customer'] = isset($input['enable_customer']) ? 1 : 0;
        $output['preserve_data'] = isset($input['preserve_data']) ? 1 : 0; // ✅ Add this line
        return $output;
    }


    public function settings_page_html() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'smart-order-notifier'));
        }

        $options = get_option($this->option_name);
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Smart Order Notifier Settings', 'smart-order-notifier'); ?></h1>
            <form method="post" action="options.php" style="max-width:700px;">
                <?php settings_fields('smarorno_settings_group'); ?>
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th><label for="access_token"><?php esc_html_e('Access Token', 'smart-order-notifier'); ?></label></th>
                            <td><input type="text" id="access_token" name="smarorno_settings[access_token]" value="<?php echo esc_attr($options['access_token'] ?? ''); ?>" class="regular-text" placeholder="Meta Cloud API Access Token" /></td>
                        </tr>
                        <tr>
                            <th><label for="phone_number_id"><?php esc_html_e('Phone Number ID', 'smart-order-notifier'); ?></label></th>
                            <td><input type="text" id="phone_number_id" name="smarorno_settings[phone_number_id]" value="<?php echo esc_attr($options['phone_number_id'] ?? ''); ?>" class="regular-text" placeholder="WhatsApp Business Phone Number ID" /></td>
                        </tr>
                        <!-- Admin template -->
                        <tr>
                            <th><label for="template_name"><?php esc_html_e('Admin Template Name', 'smart-order-notifier'); ?></label></th>
                            <td><input type="text" id="template_name" name="smarorno_settings[template_name]" value="<?php echo esc_attr($options['template_name'] ?? ''); ?>" class="regular-text" placeholder="Admin WhatsApp Template Name" /></td>
                        </tr>
                        <tr>
                            <th><label for="template_language"><?php esc_html_e('Admin Template Language', 'smart-order-notifier'); ?></label></th>
                            <td>
                                <input type="text" id="template_language" name="smarorno_settings[template_language]" value="<?php echo esc_attr($options['template_language'] ?? 'en_US'); ?>" class="regular-text" placeholder="e.g. en_US" />
                                <p class="description"><?php esc_html_e('WhatsApp template language code for admin messages.', 'smart-order-notifier'); ?></p>
                            </td>
                        </tr>
                        <!-- Customer template -->
                        <tr>
                            <th><label for="customer_template_name"><?php esc_html_e('Customer Template Name', 'smart-order-notifier'); ?></label></th>
                            <td><input type="text" id="customer_template_name" name="smarorno_settings[customer_template_name]" value="<?php echo esc_attr($options['customer_template_name'] ?? ''); ?>" class="regular-text" placeholder="Customer WhatsApp Template Name" /></td>
                        </tr>
                        <tr>
                            <th><label for="customer_template_language"><?php esc_html_e('Customer Template Language', 'smart-order-notifier'); ?></label></th>
                            <td>
                                <input type="text" id="customer_template_language" name="smarorno_settings[customer_template_language]" value="<?php echo esc_attr($options['customer_template_language'] ?? 'en_US'); ?>" class="regular-text" placeholder="e.g. en_US" />
                                <p class="description"><?php esc_html_e('WhatsApp template language code for customer messages.', 'smart-order-notifier'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="admin_numbers"><?php esc_html_e('Admin Phone Numbers', 'smart-order-notifier'); ?></label></th>
                            <td>
                                <input type="text" id="admin_numbers" name="smarorno_settings[admin_numbers]" value="<?php echo esc_attr($options['admin_numbers'] ?? ''); ?>" class="regular-text" placeholder="+1234567890, +10987654321" />
                                <p class="description"><?php esc_html_e('Comma-separated phone numbers with country codes.', 'smart-order-notifier'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="button_url"><?php esc_html_e('Button URL', 'smart-order-notifier'); ?></label></th>
                            <td>
                                <input type="url" id="button_url" name="smarorno_settings[button_url]" value="<?php echo esc_attr($options['button_url'] ?? ''); ?>" class="regular-text" placeholder="https://yourstore.com/order/{order_id}" />
                                <p class="description"><?php esc_html_e('Use {order_id} as placeholder for order URL.', 'smart-order-notifier'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Options', 'smart-order-notifier'); ?></th>
                            <td>
                                <label><input type="checkbox" name="smarorno_settings[paid_orders_only]" value="1" <?php checked($options['paid_orders_only'] ?? '', '1'); ?> /> <?php esc_html_e('Notify only for paid orders', 'smart-order-notifier'); ?></label><br />
                                <label><input type="checkbox" name="smarorno_settings[enable_customer]" value="1" <?php checked($options['enable_customer'] ?? '', '1'); ?> /> <?php esc_html_e('Enable customer notifications', 'smart-order-notifier'); ?></label>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="preserve_data"><?php esc_html_e('Preserve Data on Uninstall', 'smart-order-notifier'); ?></label></th>
                            <td>
                                <label>
                                    <input type="checkbox" id="preserve_data" name="smarorno_settings[preserve_data]" value="1"
                                        <?php checked($options['preserve_data'] ?? '', '1'); ?> />
                                    <?php esc_html_e('Do not delete plugin data when uninstalled.', 'smart-order-notifier'); ?>
                                </label>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}
