<?php
/**
 * Handle Product Meta for Instalment Settings.
 *
 * @package Smart_Instalments_Manager_For_Woocommerce
 */

if ( ! defined('ABSPATH') ) {
    exit;
}

if ( ! class_exists( 'WCSIM_Product_Meta' ) ) {
    class WCSIM_Product_Meta {
        private $logger;

        /**
         * Constructor
         */
        public function __construct() {
            $this->logger = wc_get_logger();

            add_action( 'woocommerce_product_after_variable_attributes', array( $this, 'wcsim_variation_field' ), 10, 3 );
            add_action( 'woocommerce_save_product_variation', array( $this, 'wcsim_save_variation' ), 10, 2 );

            add_action( 'woocommerce_product_options_general_product_data',array( $this, 'wcsim_simple_field' ) );
            add_action( 'woocommerce_admin_process_product_object', array( $this, 'wcsim_save_simple' ) );
        }

        /**
         * Output variation fields for instalments.
         *
         * @param int   $loop           Variation loop index.
         * @param array $variation_data Variation data array.
         * @param object $variation     WP_Post variation object.
         */
        public function wcsim_variation_field( $loop, $variation_data, $variation ) {
            wp_nonce_field( 'wcsim_save_variation_meta', 'wcsim_variation_nonce' );

            $wcsim_instalments_total    = get_post_meta( $variation->ID, '_wcsim_instalments_total', true );
            $wcsim_product_full_price   = get_post_meta( $variation->ID, '_wcsim_product_full_price', true );
            ?>
            <div class="variable_custom_field">
                <p class="form-field form-row form-row-full">
        <label for="variable_sale_price0">Sale price (£) <a href="#" class="sale_schedule">Schedule</a><a href="#" class="cancel_sale_schedule hidden">Cancel schedule</a></label><input type="text" class="short wc_input_price" style="" name="variable_sale_price[0]" id="variable_sale_price0" value="" placeholder=""> </p>

                <p class="form-field form-row form-row-full">
                    <label for="_wcsim_instalments_total<?php echo esc_attr( $loop ); ?>">
                        <?php esc_html_e( 'Product Full Price', 'smart-instalments-manager-for-woocommerce' ); ?>
                    </label>
                    <input 
                        class="short" 
                        type="text" 
                        name="_wcsim_product_full_price[<?php echo esc_attr( $loop ); ?>]" 
                        id="_wcsim_product_full_price<?php echo esc_attr( $loop ); ?>" 
                        value="<?php echo esc_attr( $wcsim_product_full_price ); ?>"  />
                </p>

                <p class="form-row form-row-full">
                    <label for="_wcsim_instalments_total<?php echo esc_attr( $loop ); ?>">
                        <?php esc_html_e( 'Instalments Total (Number of Months)', 'smart-instalments-manager-for-woocommerce' ); ?>
                    </label>
                    <input 
                        type="number" 
                        class="short" 
                        name="_wcsim_instalments_total[<?php echo esc_attr( $loop ); ?>]" 
                        id="_wcsim_instalments_total<?php echo esc_attr( $loop ); ?>" 
                        value="<?php echo esc_attr( $wcsim_instalments_total ); ?>" 
                        min="1" 
                        step="1" />
                </p>
            </div>
            <?php
        }

        /**
         * Save variation instalments meta.
         *
         * @param int $variation_id Variation post ID.
         * @param int $index        Loop index for variation submission.
         */
        public function wcsim_save_variation( $variation_id, $i ) {
            if ( ! current_user_can( 'manage_woocommerce' ) ) {
                return;
            }

            if ( ! isset( $_POST['wcsim_variation_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wcsim_variation_nonce'] ) ), 'wcsim_save_variation_meta' ) ) {
                return;
            }

            if ( isset( $_POST['_wcsim_instalments_total'][ $i ] ) ) {
                update_post_meta( $variation_id, '_wcsim_instalments_total', intval( $_POST['_wcsim_instalments_total'][ $i ] ) );
            }

            if ( isset( $_POST['_wcsim_product_full_price'][ $i ] ) ) {
                update_post_meta( $variation_id, '_wcsim_product_full_price', intval( $_POST['_wcsim_product_full_price'][ $i ] ) );
            }
        }

        /**
         * Output simple product field.
         *
         * Adds nonce field so saves can be verified.
         */
        public function wcsim_simple_field() {
            wp_nonce_field( 'wcsim_save_product_meta', 'wcsim_meta_nonce' );

            woocommerce_wp_text_input([
                'id'                => '_wcsim_instalments_total',
                'label'             => esc_html__('Instalments Total', 'smart-instalments-manager-for-woocommerce'),
                'desc_tip'          => true,
                'description'       => esc_html__('Enter the number of instalments for this product.', 'smart-instalments-manager-for-woocommerce'),
                'type'              => 'number',
                'custom_attributes' => [ 'min' => '1' ],
            ]);

            woocommerce_wp_text_input([
                'id'                => '_wcsim_product_full_price',
                'label'             => esc_html__('Product Full Price', 'smart-instalments-manager-for-woocommerce'),
                'desc_tip'          => true,
                'description'       => esc_html__('Enter the full price of this product.', 'smart-instalments-manager-for-woocommerce'),
                'type'              => 'text',
            ]);
        }

        /**
         * Save simple product instalments meta.
         *
         * @param WC_Product $product Product object passed from WooCommerce.
         */
        public function wcsim_save_simple( $product ) {
            if ( ! current_user_can( 'manage_woocommerce' ) ) {
                return;
            }

            if ( isset( $_POST['wcsim_meta_nonce'] ) ) {
                $nonce_value = sanitize_text_field( wp_unslash( $_POST['wcsim_meta_nonce'] ) );

                if ( ! check_admin_referer( 'wcsim_save_product_meta', 'wcsim_meta_nonce' ) ) {
                    return;
                }
            } else {
                return;
            }

            if ( isset( $_POST['_wcsim_instalments_total'] ) ) {
                $raw_value = sanitize_text_field( wp_unslash( $_POST['_wcsim_instalments_total'] ) );

                if ( '' !== $raw_value ) {
                    $instalments_int = absint( $raw_value );
                    $product->update_meta_data( '_wcsim_instalments_total', $instalments_int );
                } else {
                    $product->delete_meta_data( '_wcsim_instalments_total' );
                }
            }

            if ( isset( $_POST['_wcsim_product_full_price'] ) ) {
                $raw_value = sanitize_text_field( wp_unslash( $_POST['_wcsim_product_full_price'] ) );

                if ( '' !== $raw_value ) {
                    $instalments_int = absint( $raw_value );
                    $product->update_meta_data( '_wcsim_product_full_price', $instalments_int );
                } else {
                    $product->delete_meta_data( '_wcsim_product_full_price' );
                }
            }

            $product->save();
        }
    }
}