<?php
/**
 * Handle instalment tracking for WooCommerce Subscriptions.
 *
 * @package Smart_Instalments_Manager_For_Woocommerce
 */

if ( ! defined('ABSPATH') ) {
    exit;
}

if ( ! class_exists( 'WCSIM_Instalments' ) ) {
    class WCSIM_Instalments {
        private $logger;

        /**
         * Constructor
         */
        public function __construct() {
            $this->logger = function_exists( 'wc_get_logger' ) ? wc_get_logger() : null;

            add_action( 'wcs_create_subscription', array( $this, 'wcsim_on_subscription_created' ), 10, 1 );
            add_action( 'woocommerce_subscription_renewal_payment_complete', array( $this, 'wcsim_on_renewal_payment_complete' ), 10, 2 );
        }

        /**
         * Get instalment total from subscription or product metadata.
         * 
         * @param WC_Subscription $subscription Subscription object.
         * @return int
         */
        public function get_wcsim_instalments_total_from_subscription( $subscription ) {
            if ( ! is_a( $subscription, 'WC_Subscription' ) ) {
                return 0;
            }

            $wcsim_instalments_total = intval( $subscription->get_meta( '_wcsim_instalments_total', true ) );

            if ( $wcsim_instalments_total > 0 ) {
                return $wcsim_instalments_total;
            }

            $subscription_items = $subscription->get_items();

            if ( empty( $subscription_items ) ) {
                $this->logger->warning( "No items found in subscription {$subscription->get_id()}", ['source' => 'smart-instalments-manager-for-woocommerce'] );
            }

            foreach ( $subscription_items as $item ) {
                $product = $item->get_product();

                if ( ! $product ) {
                    continue;
                }

                $variation_id = $item->get_variation_id();

                if ( $variation_id ) {
                    $val = get_post_meta( $variation_id, '_wcsim_instalments_total', true );

                    if ( $val !== '' && $val !== false ) {
                        return intval( $val );
                    }
                } 
                else {
                    $val = get_post_meta( $product->get_id(), '_wcsim_instalments_total', true );

                    if ( $val !== '' && $val !== false ) {
                        return intval( $val );
                    }
                }
            }

            $this->logger->warning( "No _wcsim_instalment_total meta found for subscription {$subscription->get_id()}", ['source' => 'smart-instalments-manager-for-woocommerce'] );
            return 0;
        }

        /**
         * Runs when a subscription is created
         * 
         * @param WC_Subscription $subscription Subscription object.
         */
        public function wcsim_on_subscription_created( $subscription ) {
            if ( ! is_a( $subscription, 'WC_Subscription' ) ) {
                return;
            }

            $wcsim_instalments_total    = $this->get_wcsim_instalments_total_from_subscription( $subscription );
            $installments_completed     = 0;
            $parent_order_id            = $subscription->get_parent_id();

            if ( $parent_order_id ) {
                $parent_order = wc_get_order( $parent_order_id );

                if ( $parent_order ) {
                    $parent_total           = floatval( $parent_order->get_total() );
                    $installments_completed = $parent_total > 0 ? 1 : 0;

                    $subscription->update_meta_data( '_wcsim_parent_order_id', $parent_order_id );
                    $subscription->update_meta_data( '_wcsim_instalments_total', $wcsim_instalments_total );
                    $subscription->update_meta_data( '_wcsim_instalments_completed', $installments_completed );
                    $subscription->save();


                    $parent_order->update_meta_data( '_wcsim_instalment_type', 'parent' );
                    $parent_order->update_meta_data( '_wcsim_instalment_number', $installments_completed ? 1 : 0 );
                    $parent_order->update_meta_data( '_wcsim_instalments_total', $wcsim_instalments_total );
                    $parent_order->update_meta_data( '_wcsim_parent_order_id', $parent_order_id );
                    $parent_order->save();

                    return;
                }
            }

            // Fallback: still set subscription meta
            $subscription->update_meta_data( '_wcsim_instalments_total', $wcsim_instalments_total );
            $subscription->update_meta_data( '_wcsim_instalments_completed', $installments_completed );
            $subscription->save();
        }

        /**
         * Runs when a renewal order is paid.
         * 
         * @param WC_Subscription $subscription Subscription object.
         * @param WC_Order        $renewal_order Renewal order.
         */
        public function wcsim_on_renewal_payment_complete( $subscription, $renewal_order ) {
            if ( ! is_a( $subscription, 'WC_Subscription' ) ) {
                return;
            }
            if ( ! is_a( $renewal_order, 'WC_Order' ) ) {
                return;
            }

            $wcsim_instalments_total    = $this->get_wcsim_instalments_total_from_subscription( $subscription );
            $installments_completed     = intval( $subscription->get_meta( '_wcsim_instalments_completed', true ) );
            $next_installment           = $installments_completed + 1;
            $installment_type           = 'installment';

            if ( $wcsim_instalments_total > 0 && $installments_completed >= $wcsim_instalments_total ) {
                $installments_completed = 0;
                $next_installment       = 1;
                $installment_type       = 'parent';
            } 
            elseif ( $installments_completed === 0 ) {
                $next_installment = 1;
                $installment_type = 'installment';
            } 
            else {
                $next_installment = $installments_completed + 1;
                $installment_type = 'installment';
            }

            $installments_completed = $next_installment;

            $subscription->update_meta_data( '_wcsim_instalments_completed', intval( $installments_completed ) );
            $subscription->update_meta_data( '_wcsim_instalments_total', intval( $wcsim_instalments_total ) );
            $subscription->save();

            $renewal_order->update_meta_data( '_wcsim_instalment_number', intval( $next_installment ) );
            $renewal_order->update_meta_data( '_wcsim_instalment_type', $installment_type );
            $renewal_order->update_meta_data( '_wcsim_instalments_total', intval( $wcsim_instalments_total ) );

            $parent_order_id = intval( $subscription->get_meta( '_wcsim_parent_order_id', true ) );
            if ( $parent_order_id ) {
                $renewal_order->update_meta_data( '_wcsim_parent_order_id', $parent_order_id );
            }

            $renewal_order->save();

            if ( $installment_type === 'parent' ) {
                if ( 'processing' !== $renewal_order->get_status() ) {
                    $renewal_order->update_status( 'processing', esc_html__( 'New cycle started: Parent (1st Installment) marked for shipping import.', 'smart-instalments-manager-for-woocommerce' ) );
                }
            } 
            else {
                $wcsim_order_status_slug        = 'wc-balance-payable';
                $wcsim_order_status_label       = esc_html( apply_filters( 'wcsim_custom_order_status_label', 'Balance Payable' ) );

                $wcsim_admin_note               = sprintf(
                    /* translators: 1: Order status label (e.g. "Balance Payable") */
                    esc_html__( '%s installment (no box dispatch).', 'smart-instalments-manager-for-woocommerce' ),
                    $wcsim_order_status_label
                );

                if ( $wcsim_order_status_slug !== $renewal_order->get_status() ) {
                    $renewal_order->update_status( $wcsim_order_status_slug, $wcsim_admin_note );
                }
            }
        }
    }
}