<?php
/**
 * Plugin Name: Smart Initial Caps Titles
 * Description: Converts post titles to Initial Caps while preserving ALL CAPS and numeric acronyms like FG, INEC, 5G, 2FA, 50BN. Includes category-based rules and performance optimizations.
 * Version:     1.0.0
 * Author:      Ese Amadasun
 * Author URI:  https://amadasunese.pythonanywhere.com
 * Text Domain: smart-initial-caps-titles
 * License:     GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'SICT_OPTION', 'sict_settings' );
define( 'SICT_CACHE_GROUP', 'sict_titles' );

/**
 * Default settings with explicit types
 */
function sict_default_settings() {
    return array(
        'home'       => 1,
        'single'     => 1,
        'archive'    => 0,
        'min_length' => 4,
        'categories' => array(),
    );
}

/**
 * Get sanitized settings with fallback to prevent null errors
 */
function sict_get_settings() {
    $options = get_option( SICT_OPTION );
    return wp_parse_args( is_array($options) ? $options : array(), sict_default_settings() );
}

/**
 * Smart formatter
 */
function sict_format_title( $title ) {
    if ( ! is_string( $title ) || $title === '' ) {
        return $title;
    }

    $settings = sict_get_settings();
    $min_length = isset($settings['min_length']) ? (int)$settings['min_length'] : 4;

    if ( mb_strlen( $title ) < $min_length ) {
        return $title;
    }

    $cache_key = md5( $title );
    $cached    = wp_cache_get( $cache_key, SICT_CACHE_GROUP );
    if ( false !== $cached && is_string($cached) ) {
        return $cached;
    }

    // Capture Acronyms
    preg_match_all(
        '/(?<![A-Za-z0-9])(?:[A-Z]{2,}|\d+[A-Z]+|(?:[A-Z]\.){2,})(?![A-Za-z0-9])/',
        $title,
        $matches
    );
    $acronyms = !empty($matches[0]) ? array_unique( $matches[0] ) : array();

    // Convert to Initial Caps
    $formatted = ucwords( strtolower( $title ) );

    // Restore original acronyms
    if ( !empty($acronyms) ) {
        foreach ( $acronyms as $acro ) {
            $formatted = preg_replace(
                '/(?<![A-Za-z0-9])' . preg_quote( $acro, '/' ) . '(?![A-Za-z0-9])/i',
                $acro,
                $formatted
            );
        }
    }

    wp_cache_set( $cache_key, $formatted, SICT_CACHE_GROUP );
    return $formatted;
}

/**
 * Filter the title with improved safety
 */
function sict_filter_the_title( $title, $post_id = null ) {
    if ( is_admin() || empty( $title ) || !is_string($title) ) {
        return $title;
    }

    // Skip Nav Menus
    if ( $post_id ) {
        $post = get_post( $post_id );
        if ( $post && 'nav_menu_item' === $post->post_type ) {
            return $title;
        }
    }

    $settings = sict_get_settings();

    // Category logic
    if ( ! empty( $settings['categories'] ) && $post_id ) {
        $post_categories = wp_get_post_categories( $post_id );
        if ( ! array_intersect( (array)$settings['categories'], (array)$post_categories ) ) {
            return $title;
        }
    }

    $is_home    = ( is_home() || is_front_page() );
    $is_single  = is_singular();
    $is_archive = is_archive();

    if (
        ( ($settings['home'] ?? 0) && $is_home ) ||
        ( ($settings['single'] ?? 0) && $is_single ) ||
        ( ($settings['archive'] ?? 0) && $is_archive ) ||
        ( ! $is_single && ! $is_archive && ! $is_home )
    ) {
        return sict_format_title( $title );
    }

    return $title;
}
add_filter( 'the_title', 'sict_filter_the_title', 10, 2 );

/**
 * Admin Setup
 */
add_action( 'admin_menu', function () {
    add_options_page(
        'Smart Initial Caps Titles',
        'Initial Caps Titles',
        'manage_options',
        'sict-settings',
        'sict_render_settings_page'
    );
} );

add_action( 'admin_init', function () {
    register_setting( 'sict_settings_group', SICT_OPTION, array(
        'sanitize_callback' => 'sict_sanitize_settings',
        'default'           => sict_default_settings()
    ));
} );

function sict_sanitize_settings( $input ) {
    if ( ! is_array( $input ) ) return sict_default_settings();
    
    return array(
        'home'       => isset( $input['home'] ) ? 1 : 0,
        'single'     => isset( $input['single'] ) ? 1 : 0,
        'archive'    => isset( $input['archive'] ) ? 1 : 0,
        'min_length' => isset( $input['min_length'] ) ? absint( $input['min_length'] ) : 4,
        'categories' => ( isset( $input['categories'] ) && is_array( $input['categories'] ) ) ? array_map( 'absint', $input['categories'] ) : array(),
    );
}

function sict_render_settings_page() {
    $settings   = sict_get_settings();
    $categories = get_categories( array( 'hide_empty' => false ) );
    ?>
    <div class="wrap">
        <h1>Smart Initial Caps Titles</h1>
        <form method="post" action="options.php">
            <?php settings_fields( 'sict_settings_group' ); ?>
            <table class="form-table">
                <tr>
                    <th>Apply On</th>
                    <td>
                        <label><input type="checkbox" name="<?php echo SICT_OPTION; ?>[home]" value="1" <?php checked( $settings['home'] ); ?>> Homepage</label><br>
                        <label><input type="checkbox" name="<?php echo SICT_OPTION; ?>[single]" value="1" <?php checked( $settings['single'] ); ?>> Single Posts</label><br>
                        <label><input type="checkbox" name="<?php echo SICT_OPTION; ?>[archive]" value="1" <?php checked( $settings['archive'] ); ?>> Archives</label>
                    </td>
                </tr>
                <tr>
                    <th>Minimum Title Length</th>
                    <td>
                        <input type="number" name="<?php echo SICT_OPTION; ?>[min_length]" value="<?php echo esc_attr( $settings['min_length'] ); ?>" min="1">
                    </td>
                </tr>
                <tr>
                    <th>Categories</th>
                    <td>
                        <?php foreach ( $categories as $cat ) : ?>
                            <label style="display:block;">
                                <input type="checkbox" name="<?php echo SICT_OPTION; ?>[categories][]" value="<?php echo $cat->term_id; ?>" <?php checked( in_array( $cat->term_id, (array)$settings['categories'] ) ); ?>>
                                <?php echo esc_html( $cat->name ); ?>
                            </label>
                        <?php endforeach; ?>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}