<?php
/**
 * Plugin Name: Smart Duplicate Post & Page
 * Plugin URI:  https://wordpress.org/plugins/smart-duplicate-post-page/
 * Description: Easily duplicate posts, pages, and custom post types with a single click.
 * Version:     1.1.0
 * Author:      Ayush Guleria
 * Author URI:  https://profiles.wordpress.org/guleriaharbour/
 * License:     GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: smart-duplicate-post-page
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Duplicate a post as draft.
 */
function sdp_duplicate_post_as_draft() {

	// Verify required params.
	if ( ! isset( $_GET['post'] ) ) {
		wp_die( esc_html__( 'No post to duplicate has been supplied.', 'smart-duplicate-post-page' ) );
	}

	// Verify nonce (dies automatically if invalid).
	check_admin_referer( 'sdp_duplicate_post_as_draft', 'sdp_nonce' );

	$post_id = absint( $_GET['post'] );
	if ( ! $post_id ) {
		wp_die( esc_html__( 'Invalid post ID.', 'smart-duplicate-post-page' ) );
	}

	$original = get_post( $post_id );
	if ( ! $original instanceof WP_Post ) {
		wp_die( esc_html__( 'Post duplication failed: original post not found.', 'smart-duplicate-post-page' ) );
	}

	// Capability check.
	if ( ! current_user_can( 'edit_post', $post_id ) ) {
		wp_die( esc_html__( 'You are not allowed to duplicate this item.', 'smart-duplicate-post-page' ) );
	}

	$new_post_author = get_current_user_id();

	// Prepare new post data.
	$args = array(
		'comment_status' => $original->comment_status,
		'ping_status'    => $original->ping_status,
		'post_author'    => $new_post_author,
		'post_content'   => $original->post_content,
		'post_excerpt'   => $original->post_excerpt,
		'post_name'      => sanitize_title( $original->post_name . '-copy' ),
		'post_parent'    => $original->post_parent,
		'post_password'  => $original->post_password,
		'post_status'    => 'draft',
		'post_title'     => $original->post_title . ' (' . esc_html__( 'Copy', 'smart-duplicate-post-page' ) . ')',
		'post_type'      => $original->post_type,
		'to_ping'        => $original->to_ping,
		'menu_order'     => $original->menu_order,
	);

	$new_post_id = wp_insert_post( $args );

	if ( ! $new_post_id || is_wp_error( $new_post_id ) ) {
		wp_die( esc_html__( 'Post duplication failed while creating the new draft.', 'smart-duplicate-post-page' ) );
	}

	// Copy post meta.
	$all_meta = get_post_meta( $post_id );
	if ( ! empty( $all_meta ) ) {
		foreach ( $all_meta as $meta_key => $meta_values ) {
			foreach ( (array) $meta_values as $meta_value ) {
				add_post_meta( $new_post_id, $meta_key, maybe_unserialize( $meta_value ) );
			}
		}
	}

	// Redirect to edit screen.
	wp_safe_redirect( admin_url( 'post.php?action=edit&post=' . $new_post_id ) );
	exit;
}
add_action( 'admin_action_sdp_duplicate_post_as_draft', 'sdp_duplicate_post_as_draft' );

/**
 * Add "Duplicate" link to post and page row actions.
 */
function sdp_duplicate_post_row_action( $actions, $post ) {
	if ( current_user_can( 'edit_post', $post->ID ) ) {
		$url = wp_nonce_url(
			add_query_arg(
				array(
					'action' => 'sdp_duplicate_post_as_draft',
					'post'   => $post->ID,
				),
				admin_url( 'admin.php' )
			),
			'sdp_duplicate_post_as_draft',
			'sdp_nonce'
		);

		$actions['sdp_duplicate'] = '<a href="' . esc_url( $url ) . '" title="' . esc_attr__( 'Duplicate this item', 'smart-duplicate-post-page' ) . '">' . esc_html__( 'Duplicate', 'smart-duplicate-post-page' ) . '</a>';
	}
	return $actions;
}
add_filter( 'post_row_actions', 'sdp_duplicate_post_row_action', 10, 2 );
add_filter( 'page_row_actions', 'sdp_duplicate_post_row_action', 10, 2 );
