<?php
/**
 * Plugin Name: Smart Download Redirector
 * Plugin URI: https://wordpress.org/plugins/smart-download-redirector/
 * Description: Redirects external links to a countdown page before allowing download or redirection.
 * Version: 1.0.0
 * Requires at least: 6.8
 * Requires PHP: 7.4
 * Author: Tobias Kurth
 * Author URI: https://desk9.design/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: smart-download-redirector
 *
 * @package SmartDownloadRedirector
 */

// If this file is called directly, abort.
if (! defined('ABSPATH')) {
    exit;
}

// Plugin constants.
define('SMARTDR_VERSION', '1.0.0');
define('SMARTDR_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SMARTDR_PLUGIN_URL', plugin_dir_url(__FILE__));

// Include required files.
require_once SMARTDR_PLUGIN_DIR . 'includes/redirect-functions.php';
require_once SMARTDR_PLUGIN_DIR . 'includes/security-enhancements.php';
require_once SMARTDR_PLUGIN_DIR . 'admin/settings-page.php';

/**
 * Plugin activation hook.
 */
function smartdr_activate()
{
    // Create default options.
    $default_options = array(
        'countdown_time' => 10,
        'redirect_delay' => 3,
        'target_domains' => '',
        'custom_css'     => '',
    );

    foreach ($default_options as $option => $value) {
        if (false === get_option('smartdr_' . $option)) {
            add_option('smartdr_' . $option, $value);
        }
    }
}
register_activation_hook(__FILE__, 'smartdr_activate');

/**
 * Plugin deactivation hook.
 */
function smartdr_deactivate()
{
    // Clean up on deactivation
}
register_deactivation_hook(__FILE__, 'smartdr_deactivate');

/**
 * Register and enqueue plugin assets.
 *
 * @since 1.0.0
 * @return void
 */
function smartdr_register_assets()
{
    // Register countdown script
    wp_register_script(
        'smartdr-countdown',
        SMARTDR_PLUGIN_URL . 'assets/countdown.js',
        ['jquery'],
        filemtime(SMARTDR_PLUGIN_DIR . 'assets/countdown.js'),
        true // Load in footer for better performance
    );

    // Register main stylesheet
    wp_register_style(
        'smartdr-styles',
        SMARTDR_PLUGIN_URL . 'assets/styles.css',
        [],
        filemtime(SMARTDR_PLUGIN_DIR . 'assets/styles.css')
    );
}
add_action('wp_enqueue_scripts', 'smartdr_register_assets');

/**
 * Check if we need to enqueue countdown assets based on content
 */
function smartdr_maybe_enqueue_assets()
{
    // Check if we're on a page with the shortcode
    global $post;
    $should_enqueue = false;
    
    if (is_object($post) && has_shortcode($post->post_content, 'smartdr_download_countdown')) {
        $should_enqueue = true;
    }
    
    // Check if we're on a download URL (with URL parameters)
    if (isset($_GET['url']) && isset($_GET['smartdr_nonce'])) {
        $should_enqueue = true;
    }
    
    if ($should_enqueue) {
        wp_enqueue_script('smartdr-countdown');
        wp_enqueue_style('smartdr-styles');
        
        // Prepare localization data
        $url_param = isset($_GET['url']) ? sanitize_text_field(wp_unslash($_GET['url'])) : '';
        $nonce = isset($_GET['smartdr_nonce']) ? sanitize_key(wp_unslash($_GET['smartdr_nonce'])) : '';
        $url = null;
        
        if (!empty($url_param) && !empty($nonce)) {
            // Verify nonce
            if (wp_verify_nonce($nonce, 'smartdr_download')) {
                $decoded_url = urldecode($url_param);
                $decoded_url = smartdr_normalize_url($decoded_url);
                
                if (smartdr_validate_redirect_url($decoded_url)) {
                    $url = $decoded_url;
                }
            }
        }
        
        // Always prepare countdown configuration
        $countdown_time = absint(get_option('smartdr_countdown_time', 10));
        $redirect_delay = absint(get_option('smartdr_redirect_delay', 3));
        $redirect_delay = min(max($redirect_delay, 1), 10);
        $redirect_delay_ms = $redirect_delay * 1000;
        
        // Localize script with data
        wp_localize_script(
            'smartdr-countdown',
            'smartdrData',
            [
                'redirectUrl' => $url ? $url : '',
                'countdownTime' => $countdown_time,
                'nonce' => wp_create_nonce('smartdr-countdown'),
                'showDestination' => get_option('smartdr_show_destination', true),
                'destinationStep' => absint(get_option('smartdr_destination_display_step', 1)),
                'showManualLink' => get_option('smartdr_show_manual_link', true),
                'manualLinkStep' => absint(get_option('smartdr_manual_link_display_step', 2)),
                'redirectDelay' => $redirect_delay_ms,
                'secondsText' => __('seconds remaining', 'smart-download-redirector'),
                'downloadReadyText' => __('Download ready!', 'smart-download-redirector'),
                'hasValidUrl' => !empty($url)
            ]
        );
        
        // Add CSS variables for styling
        $styles = get_option('smartdr_styles', array());
        $default_styles = array(
            'heading_color' => '#212121',
            'heading_size' => '32px',
            'counter_color' => '#00897B',
            'counter_size' => '36px',
            'counter_font_weight' => 'bold',
            'background_color' => '#ffffff',
            'border_radius' => '8px',
            'border_size' => '1px',
            'border_color' => '#dddddd',
            'manual_link_size' => '14px',
            'manual_link_color' => '#00897B',
            'manual_link_hover_color' => '#00695C'
        );
        $styles = array_merge($default_styles, $styles);
        
        // Add inline CSS with custom properties
        $inline_css = "
/* CSS Custom Properties for dynamic styling */
:root {
    --smartdr-counter-color: " . esc_attr($styles['counter_color']) . ";
    --smartdr-counter-size: " . esc_attr($styles['counter_size']) . ";
    --smartdr-counter-font-weight: " . (($styles['counter_font_weight'] === 'bold') ? '600' : 'normal') . ";
    --smartdr-heading-color: " . esc_attr($styles['heading_color']) . ";
    --smartdr-heading-size: " . esc_attr($styles['heading_size']) . ";
    --smartdr-manual-link-size: " . esc_attr($styles['manual_link_size']) . ";
    --smartdr-manual-link-color: " . esc_attr($styles['manual_link_color']) . ";
    --smartdr-manual-link-hover-color: " . esc_attr($styles['manual_link_hover_color']) . ";
    --smartdr-bg-color: " . esc_attr($styles['background_color']) . ";
    --smartdr-border-radius: " . esc_attr($styles['border_radius']) . ";
    --smartdr-border-size: " . esc_attr($styles['border_size']) . ";
    --smartdr-border-color: " . esc_attr($styles['border_color']) . ";
}";
        wp_add_inline_style('smartdr-styles', $inline_css);
    }
}
add_action('wp_enqueue_scripts', 'smartdr_maybe_enqueue_assets', 20);
