<?php
/**
 * Security enhancements for Smart Download Redirector
 *
 * @package SmartDownloadRedirector
 */

// If this file is called directly, abort.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Rate limiting for download redirects.
 *
 * @return bool True if within rate limit, false otherwise.
 */
function smartdr_check_rate_limit()
{
    // Get rate limit settings.
    $rate_limit_settings = get_option(
        'smartdr_rate_limit',
        array(
            'enabled'      => true,
            'max_requests' => 100,
            'time_window'  => 3600,
        )
    );

    // Allow developers to modify rate limit settings.
    $rate_limit_settings = apply_filters('smartdr_rate_limit_settings', $rate_limit_settings);

    // If rate limiting is disabled, always return true.
    if (! $rate_limit_settings['enabled']) {
        return true;
    }

    // Use transients instead of sessions for rate limiting
    $user_ip = '';
    if (isset($_SERVER['REMOTE_ADDR'])) {
        $user_ip = sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
    }
    $transient_key = 'smartdr_rate_limit_' . md5($user_ip);
    
    $rate_data = get_transient($transient_key);
    if (false === $rate_data) {
        $rate_data = array(
            'count' => 0,
            'first_download' => time(),
        );
    }

    $time_window  = $rate_limit_settings['time_window'];
    $max_requests = $rate_limit_settings['max_requests'];

    if (time() - $rate_data['first_download'] > $time_window) {
        $rate_data = array(
            'count' => 0,
            'first_download' => time(),
        );
    }

    $rate_data['count']++;
    set_transient($transient_key, $rate_data, $time_window);

    return $rate_data['count'] <= $max_requests;
}

/**
 * Add security headers.
 */
function smartdr_add_security_headers()
{
    if (! is_page_template('download-page.php')) {
        return;
    }

    // Add security headers.
    header('X-Frame-Options: DENY');
    header('X-XSS-Protection: 1; mode=block');
    header('X-Content-Type-Options: nosniff');
    header('Referrer-Policy: strict-origin-when-cross-origin');
    header(
        "Content-Security-Policy: default-src 'self'; " .
        "script-src 'self' 'unsafe-inline'; " .
        "style-src 'self' 'unsafe-inline';"
    );
}
add_action('send_headers', 'smartdr_add_security_headers');

/**
 * Validate domain name.
 *
 * @param string $domain Domain name to validate.
 * @return bool True if valid, false otherwise.
 */
function smartdr_is_valid_domain($domain)
{
    // Basic domain validation.
    if (empty($domain)) {
        return false;
    }

    // Check domain length.
    if (strlen($domain) > 253) {
        return false;
    }

    // Check for valid characters.
    if (! preg_match('/^[a-z0-9]+([\-\.]{1}[a-z0-9]+)*\.[a-z]{2,}$/i', $domain)) {
        return false;
    }

    return true;
}
