<?php
/**
 * Redirect functions for Smart Download Redirector
 *
 * @package SmartDownloadRedirector
 */

// If this file is called directly, abort.
if (! defined('ABSPATH')) {
    exit;
}





/**
 * Filter content to modify matching links
 */
function smartdr_filter_content($content)
{
    // Skip if not in the main loop
    if (!in_the_loop()) {
        return $content;
    }

    // Get target domains
    $target_domains = array_filter(explode("\n", get_option('smartdr_target_domains', '')));
    if (empty($target_domains)) {
        return $content;
    }

    // Get the page with the shortcode
    $shortcode_page = get_option('smartdr_shortcode_page');
    
    // If no shortcode page is set, check if current page has the shortcode
    if (!$shortcode_page) {
        $current_post = get_post();
        if ($current_post && has_shortcode($current_post->post_content, 'smartdr_download_countdown')) {
            $shortcode_page = $current_post->ID;
            update_option('smartdr_shortcode_page', $shortcode_page);
        }
    }
    
    // If still no shortcode page found, try to find it in all pages
    if (!$shortcode_page) {
        $pages = get_posts([
            'post_type' => 'page',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ]);
        
        foreach ($pages as $page) {
            if (has_shortcode($page->post_content, 'smartdr_download_countdown')) {
                $shortcode_page = $page->ID;
                update_option('smartdr_shortcode_page', $shortcode_page);
                break;
            }
        }
    }
    
    // If no page with shortcode found, return original content
    if (!$shortcode_page) {
        return $content;
    }

    // Regular expression pattern for links
    $pattern = '/<a([^>]*?)href=[\'"]([^\'"]*)[\'"](.*?)>(.*?)<\/a>/i';

    return preg_replace_callback($pattern, function ($matches) use ($target_domains, $shortcode_page) {
        $attrs = $matches[1];
        $url = $matches[2];
        $remaining_attrs = $matches[3];
        $link_text = $matches[4];

        // Skip empty or invalid URLs
        if (empty($url) || strpos($url, '#') === 0) {
            return $matches[0];
        }

        // Check if URL matches any target domain
        $parsed_url = wp_parse_url($url);
        if (!isset($parsed_url['host'])) {
            return $matches[0]; // Return original if no host
        }

        $domain = strtolower($parsed_url['host']);
        $domain = preg_replace('/^www\./', '', $domain);

        foreach ($target_domains as $target_domain) {
            $target_domain = trim(strtolower($target_domain));
            if (empty($target_domain)) {
                continue;
            }

            // Remove www., http://, or https:// if present in target domain
            $target_domain = preg_replace('/^(https?:\/\/)?(www\.)?/', '', $target_domain);

            // Check if domain matches
            if ($domain === $target_domain) {
                // Create redirect URL using query parameters to the shortcode page
                $redirect_url = add_query_arg([
                    'url' => urlencode($url),
                    'smartdr_nonce' => wp_create_nonce('smartdr_download')
                ], get_permalink($shortcode_page));
                
                // Build new link
                return sprintf(
                    '<a%shref="%s"%s>%s</a>',
                    $attrs ? ' ' . trim($attrs) : '',
                    esc_url($redirect_url),
                    $remaining_attrs ? ' ' . trim($remaining_attrs) : '',
                    $link_text
                );
            }
        }

        return $matches[0]; // Return original if no match
    }, $content);
}
add_filter('the_content', 'smartdr_filter_content', 20);
add_filter('widget_text_content', 'smartdr_filter_content', 20);

/**
 * Validate and sanitize redirect URL.
 *
 * @param string $url The URL to validate.
 * @return bool True if valid, false otherwise.
 */
function smartdr_validate_redirect_url($url)
{
    // Basic URL validation
    if (empty($url)) {
        // Remove error_log for production
        // error_log('SMARTDR: Empty URL provided');
        return false;
    }

    // Get target domains
    $target_domains = array_filter(explode("\n", get_option('smartdr_target_domains', '')));
    if (empty($target_domains)) {
        // Remove error_log for production
        // error_log('SMARTDR: No target domains set, allowing all domains');
        return true; // If no domains are set, allow all
    }

    // Parse URL and get host
    $parsed_url = wp_parse_url($url);
    if (!isset($parsed_url['host'])) {
        // Remove error_log for production
        // error_log('SMARTDR: No host found in URL: ' . $url);
        return false;
    }

    $domain = strtolower($parsed_url['host']);
    // Remove www. if present
    $domain = preg_replace('/^www\./', '', $domain);

    // Remove error_log for production
    // error_log('SMARTDR: Checking domain: ' . $domain);

    // Check if domain matches any target domain
    foreach ($target_domains as $target_domain) {
        $target_domain = trim(strtolower($target_domain));
        if (empty($target_domain)) {
            continue;
        }

        // Remove www., http://, or https:// if present in target domain
        $target_domain = preg_replace('/^(https?:\/\/)?(www\.)?/', '', $target_domain);
        
        // Remove error_log for production
        // error_log('SMARTDR: Comparing with target domain: ' . $target_domain);

        // Check exact match
        if ($domain === $target_domain) {
            // Remove error_log for production
            // error_log('SMARTDR: Match found for domain: ' . $domain);
            return true;
        }
    }

    // Remove error_log for production
    // error_log('SMARTDR: No matching domain found for: ' . $domain);
    return false;
}

/**
 * Generate download URL.
 *
 * @param string $url The original download URL.
 * @return string The encoded download URL.
 */
function smartdr_generate_download_url($url)
{
    if (! smartdr_validate_redirect_url($url)) {
        return $url;
    }

    // Get the shortcode page
    $shortcode_page = get_option('smartdr_shortcode_page');
    
    // If no shortcode page found, try to find it
    if (!$shortcode_page) {
        $pages = get_posts([
            'post_type' => 'page',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ]);
        
        foreach ($pages as $page) {
            if (has_shortcode($page->post_content, 'smartdr_download_countdown')) {
                $shortcode_page = $page->ID;
                update_option('smartdr_shortcode_page', $shortcode_page);
                break;
            }
        }
    }
    
    // If still no shortcode page found, return original URL
    if (!$shortcode_page) {
        return $url;
    }

    // Create redirect URL using query parameters to the shortcode page
    $redirect_url = add_query_arg([
        'url' => urlencode($url),
        'smartdr_nonce' => wp_create_nonce('smartdr_download')
    ], get_permalink($shortcode_page));
    
    return $redirect_url;
}

/**
 * Shortcode for download countdown
 */
function smartdr_countdown_shortcode($atts)
{
    // Store the current page ID as the shortcode page
    $current_post = get_post();
    if ($current_post) {
        update_option('smartdr_shortcode_page', $current_post->ID);
    }

    // Check for URL parameters
    $url_param = isset($_GET['url']) ? sanitize_text_field(wp_unslash($_GET['url'])) : '';
    $nonce = isset($_GET['smartdr_nonce']) ? sanitize_key(wp_unslash($_GET['smartdr_nonce'])) : '';
    $url = null;
    
    if (!empty($url_param) && !empty($nonce)) {
        // Verify nonce
        if (wp_verify_nonce($nonce, 'smartdr_download')) {
            $decoded_url = urldecode($url_param);
            $decoded_url = smartdr_normalize_url($decoded_url);
            
            if (smartdr_validate_redirect_url($decoded_url)) {
                $url = $decoded_url;
            }
        }
    }
    
    // Get countdown configuration for template
    $countdown_time = absint(get_option('smartdr_countdown_time', 10));
    $redirect_delay = absint(get_option('smartdr_redirect_delay', 3));
    $redirect_delay = min(max($redirect_delay, 1), 10);
    
    // If still no URL, show placeholder
    if (!$url) {
        return '<div class="smartdr-info">' . esc_html__('Download countdown will appear here when accessing a download link.', 'smart-download-redirector') . '</div>';
    }

    // We have a valid URL, proceed with countdown
    
    // Start output buffering to capture template content
    ob_start();
    
    // Load the template
    $template_path = locate_template('smart-download-redirector/download-page.php');
    if (!$template_path) {
        $template_path = SMARTDR_PLUGIN_DIR . 'templates/download-page.php';
    }
    
    // Include template with proper variable scope
    $template_vars = [
        'url' => $url,
        'countdown_time' => $countdown_time,
        'show_destination' => get_option('smartdr_show_destination', true),
        'destination_display_step' => absint(get_option('smartdr_destination_display_step', 1)),
        'show_manual_link' => get_option('smartdr_show_manual_link', true),
        'manual_link_display_step' => absint(get_option('smartdr_manual_link_display_step', 2)),
        'redirect_delay' => $redirect_delay,
        'custom_heading' => get_option('smartdr_custom_heading', esc_html__('Preparing Your Download', 'smart-download-redirector')),
        'show_title' => get_option('smartdr_show_title', true),
        'manual_link_display_timing' => 'at_end',
        'styles' => get_option('smartdr_styles', [])
    ];
    
    // Extract variables for template use
    extract($template_vars);
    
    // Include template
    include $template_path;
    
    // Get buffered content
    $content = ob_get_clean();
    
    // Get custom styles for CSS variables
    $style_attrs = '';
    $styles = get_option('smartdr_styles', []);
    if (!empty($styles)) {
        $style_vars = [];
        if (isset($styles['background_color'])) $style_vars[] = '--smartdr-bg-color: ' . esc_attr($styles['background_color']);
        if (isset($styles['border_radius'])) $style_vars[] = '--smartdr-border-radius: ' . esc_attr($styles['border_radius']);
        if (isset($styles['border_size'])) $style_vars[] = '--smartdr-border-size: ' . esc_attr($styles['border_size']);
        if (isset($styles['border_color'])) $style_vars[] = '--smartdr-border-color: ' . esc_attr($styles['border_color']);
        
        if (!empty($style_vars)) {
            $style_attrs = ' style="' . implode('; ', $style_vars) . ';"';
        }
    }
    
    // Wrap content in theme-independent container
    return sprintf(
        '<div class="smartdr-container" data-smartdr-version="%s" data-style-mode="custom"%s>%s</div>',
        esc_attr(SMARTDR_VERSION),
        $style_attrs,
        $content
    );
}
add_shortcode('smartdr_download_countdown', 'smartdr_countdown_shortcode');

/**
 * Get WPML translated string or fallback to original
 *
 * @param string $name The string name
 * @param string $original The original string
 * @return string The translated string or original
 */
function smartdr_get_wpml_string($name, $original) {
    // Try newer WPML API first (WPML 4.0+)
    if (function_exists('wpml_translate_single_string')) {
        return wpml_translate_single_string('smart-download-redirector', $name, $original);
    }
    
    // Fallback to older WPML API (WPML 3.x)
    if (function_exists('icl_t')) {
        return icl_t('smart-download-redirector', $name, $original);
    }
    
    // Fallback to original string if WPML is not available
    return $original;
}

/**
 * Normalize URL format
 *
 * @param string $url The URL to normalize
 * @return string Normalized URL
 */
function smartdr_normalize_url($url) {
    // Remove whitespace
    $url = trim($url);
    
    // Add scheme if missing
    if (!preg_match('~^(?:f|ht)tps?://~i', $url)) {
        // Check if starts with www.
        if (stripos($url, 'www.') === 0) {
            $url = 'https://' . $url;
        } else {
            $url = 'https://www.' . $url;
        }
    }
    
    // Ensure www. is present after scheme
    if (!preg_match('~^https?://www\.~i', $url)) {
        $url = preg_replace('~^(https?://)~i', '$1www.', $url);
    }
    
    return $url;
}
