/**
 * Simple and reliable countdown timer
 */
(function($) {
    'use strict';
    
    // Function to run when DOM is ready
    function initCountdown() {
        console.log('[SmartDR] Countdown script starting...');
        
        // Check if jQuery is available
        if (typeof $ === 'undefined') {
            console.error('[SmartDR] jQuery not available');
            return;
        }
        
        // Check if smartdrData is available
        if (typeof smartdrData === 'undefined') {
            console.error('[SmartDR] smartdrData not available');
            return;
        }
        
        console.log('[SmartDR] smartdrData:', smartdrData);
        
        // Check if we have a valid URL to countdown for
        if (!smartdrData.hasValidUrl) {
            console.log('[SmartDR] No valid URL - countdown not started');
            return;
        }
        
        // Get countdown time
        var countdownTime = parseInt(smartdrData.countdownTime) || 10;
        var redirectUrl = smartdrData.redirectUrl || '#';
        var redirectDelay = parseInt(smartdrData.redirectDelay) || 3000;
        
        console.log('[SmartDR] Configuration:', {
            countdownTime: countdownTime,
            redirectUrl: redirectUrl,
            redirectDelay: redirectDelay
        });
        
        // Find countdown elements
        var $counterNumber = $('#smartdr-countdown-number');
        var $counterText = $('#smartdr-countdown-text');
        var $progressBar = $('#smartdr-progress-bar');
        var $manualRedirect = $('.smartdr-manual-redirect');
        var $progressRing = $('.smartdr-progress-ring-circle');
        
        console.log('[SmartDR] Elements found:', {
            counterNumber: $counterNumber.length,
            counterText: $counterText.length,
            progressBar: $progressBar.length,
            manualRedirect: $manualRedirect.length,
            progressRing: $progressRing.length
        });
        
        // Determine which element to use
        var $activeElement = null;
        var layoutType = 'unknown';
        
        if ($counterNumber.length > 0) {
            $activeElement = $counterNumber;
            layoutType = 'circular';
            console.log('[SmartDR] Using circular layout');
        } else if ($counterText.length > 0) {
            $activeElement = $counterText;
            layoutType = 'progressbar';
            console.log('[SmartDR] Using progressbar layout');
        } else {
            console.error('[SmartDR] No countdown element found!');
            return;
        }
        
        // Initialize progress ring if available
        var circumference = 0;
        if ($progressRing.length > 0 && layoutType === 'circular') {
            try {
                var radius = $progressRing[0].r.baseVal.value;
                circumference = radius * 2 * Math.PI;
                $progressRing[0].style.strokeDasharray = circumference;
                $progressRing[0].style.strokeDashoffset = circumference;
                console.log('[SmartDR] Progress ring initialized, circumference:', circumference);
            } catch (e) {
                console.warn('[SmartDR] Progress ring initialization failed:', e);
            }
        }
        
        // Start countdown
        var timeLeft = countdownTime;
        var initialTime = countdownTime;
        var isComplete = false;
        
        console.log('[SmartDR] Starting countdown from', timeLeft);
        
        function updateCountdown() {
            if (isComplete) return;
            
            console.log('[SmartDR] Countdown tick:', timeLeft);
            
            // Update display based on layout type
            if (layoutType === 'circular') {
                $activeElement.text(timeLeft);
                
                // Update progress ring
                if ($progressRing.length > 0 && circumference > 0) {
                    var progress = (initialTime - timeLeft) / initialTime;
                    var offset = circumference - (progress * circumference);
                    $progressRing[0].style.strokeDashoffset = offset;
                }
            } else if (layoutType === 'progressbar') {
                var secondsText = smartdrData.secondsText || 'seconds remaining';
                $activeElement.text(timeLeft + ' ' + secondsText);
                
                // Update progress bar
                if ($progressBar.length > 0) {
                    var progress = (initialTime - timeLeft) / initialTime * 100;
                    $progressBar.css('width', progress + '%');
                }
            }
            
            // Check if countdown is complete
            if (timeLeft <= 0) {
                completeCountdown();
            } else {
                timeLeft--;
            }
        }
        
        function completeCountdown() {
            if (isComplete) return;
            isComplete = true;
            
            console.log('[SmartDR] Countdown complete!');
            
            // Clear interval
            clearInterval(countdownInterval);
            
            // Update display for completion
            if (layoutType === 'circular') {
                $activeElement.text('→');
                
                // Complete progress ring
                if ($progressRing.length > 0) {
                    $progressRing[0].style.strokeDashoffset = 0;
                }
            } else if (layoutType === 'progressbar') {
                var downloadReadyText = smartdrData.downloadReadyText || 'Download ready!';
                $activeElement.text(downloadReadyText);
                
                // Complete progress bar
                if ($progressBar.length > 0) {
                    $progressBar.css('width', '100%');
                }
            }
            
            // Show manual redirect link
            if ($manualRedirect.length > 0) {
                $manualRedirect.show();
            }
            
            // Start redirect timer
            console.log('[SmartDR] Starting redirect timer, delay:', redirectDelay + 'ms');
            setTimeout(function() {
                if (redirectUrl && redirectUrl !== '#') {
                    console.log('[SmartDR] Redirecting to:', redirectUrl);
                    window.location.href = redirectUrl;
                } else {
                    console.log('[SmartDR] No redirect URL, staying on page');
                }
            }, redirectDelay);
        }
        
        // Start the countdown
        updateCountdown(); // Initial update
        var countdownInterval = setInterval(updateCountdown, 1000);
        
        // Manual redirect handler
        $manualRedirect.find('a').on('click', function(e) {
            if (redirectUrl && redirectUrl !== '#') {
                console.log('[SmartDR] Manual redirect clicked');
                clearInterval(countdownInterval);
                // Let the link work normally
            } else {
                e.preventDefault();
                console.log('[SmartDR] Manual redirect clicked but no valid URL');
            }
        });
        
        console.log('[SmartDR] Countdown initialized successfully');
    }
    
    // Initialize when DOM is ready
    $(document).ready(function() {
        initCountdown();
    });
    
    // Fallback for cases where jQuery might not be loaded immediately
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(initCountdown, 100);
        });
    } else {
        setTimeout(initCountdown, 100);
    }
    
})(jQuery); 