<?php
/**
 * Admin settings page for Smart Download Redirector
 *
 * @package SmartDownloadRedirector
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add the admin menu item
 */
function smartdr_add_admin_menu(): void
{
    $page = add_options_page(
        __('Download Redirector Settings', 'smart-download-redirector'),
        __('Download Redirector', 'smart-download-redirector'),
        'manage_options',
        'smartdr-settings',
        'smartdr_render_settings_page'
    );

    // Add admin CSS only on our settings page
    add_action('admin_print_styles-' . $page, 'smartdr_admin_styles');
}
add_action('admin_menu', 'smartdr_add_admin_menu');

/**
 * AJAX handler for resetting styles to default
 */
function smartdr_ajax_reset_styles() {
    // Verify nonce
    if (!wp_verify_nonce(sanitize_key($_POST['nonce'] ?? ''), 'smartdr_reset_styles')) {
        wp_die('Security check failed');
    }

    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_die('Insufficient permissions');
    }

    // Default style values
    $default_styles = array(
        'heading_color' => '#212121',
        'heading_size' => '32px',
        'counter_color' => '#00897B',
        'counter_size' => '36px',
        'counter_font_weight' => 'bold',
        'background_color' => '#ffffff',
        'border_radius' => '8px',
        'border_size' => '1px',
        'border_color' => '#dddddd',
        'manual_link_size' => '14px',
        'manual_link_color' => '#00897B',
        'manual_link_hover_color' => '#00695C'
    );

    $default_destination_styles = array(
        'background_color' => '#f8f9fa',
        'border_size' => '1px',
        'border_color' => '#dee2e6',
        'border_radius' => '4px',
        'text_size' => '14px'
    );

    // Update options in database
    update_option('smartdr_styles', $default_styles);
    update_option('smartdr_destination_styles', $default_destination_styles);

    wp_send_json_success(array(
        'message' => __('Style settings have been reset to default values.', 'smart-download-redirector'),
        'styles' => $default_styles,
        'destination_styles' => $default_destination_styles
    ));
}
add_action('wp_ajax_smartdr_reset_styles', 'smartdr_ajax_reset_styles');

/**
 * AJAX handler for resetting all settings to default
 */
function smartdr_ajax_reset_everything() {
    // Verify nonce
    if (!wp_verify_nonce(sanitize_key($_POST['nonce'] ?? ''), 'smartdr_reset_everything')) {
        wp_die('Security check failed');
    }

    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_die('Insufficient permissions');
    }

    // Default values for all settings
    $default_options = array(
        'smartdr_target_domains' => "google.com",
        'smartdr_custom_heading' => __('Preparing Your Download', 'smart-download-redirector'),
        'smartdr_manual_link_text' => __('Click here if not redirected automatically', 'smart-download-redirector'),
        'smartdr_show_title' => true,
        'smartdr_countdown_time' => 10,
        'smartdr_redirect_delay' => 3,
        'smartdr_show_manual_link' => true,
        'smartdr_layout_style' => 'standard',
    );

    // Default rate limiting settings
    $default_rate_limit = array(
        'enabled' => true,
        'max_requests' => 100,
        'time_window' => 3600,
    );

    // Default style values
    $default_styles = array(
        'heading_color' => '#212121',
        'heading_size' => '32px',
        'counter_color' => '#00897B',
        'counter_size' => '36px',
        'counter_font_weight' => 'bold',
        'background_color' => '#ffffff',
        'border_radius' => '8px',
        'border_size' => '1px',
        'border_color' => '#dddddd',
        'manual_link_size' => '14px',
        'manual_link_color' => '#00897B',
        'manual_link_hover_color' => '#00695C'
    );

    $default_destination_styles = array(
        'background_color' => '#f8f9fa',
        'border_size' => '1px',
        'border_color' => '#dee2e6',
        'border_radius' => '4px',
        'text_size' => '14px'
    );

    // Update all options in database
    foreach ($default_options as $option => $value) {
        update_option($option, $value);
    }
    
    update_option('smartdr_rate_limit', $default_rate_limit);
    update_option('smartdr_styles', $default_styles);
    update_option('smartdr_destination_styles', $default_destination_styles);

    wp_send_json_success(array(
        'message' => __('All settings have been reset to default values.', 'smart-download-redirector')
    ));
}
add_action('wp_ajax_smartdr_reset_everything', 'smartdr_ajax_reset_everything');

/**
 * Enqueue admin styles and scripts
 */
function smartdr_admin_styles()
{
    // Enqueue admin CSS
    wp_enqueue_style(
        'smartdr-admin-styles',
        SMARTDR_PLUGIN_URL . 'assets/admin-styles.css',
        [],
        filemtime(SMARTDR_PLUGIN_DIR . 'assets/admin-styles.css')
    );
    
    // Enqueue WordPress color picker
    wp_enqueue_style('wp-color-picker');
    
    // Enqueue admin JavaScript
    wp_enqueue_script(
        'smartdr-admin-scripts',
        SMARTDR_PLUGIN_URL . 'assets/admin-scripts.js',
        ['jquery', 'wp-color-picker'],
        filemtime(SMARTDR_PLUGIN_DIR . 'assets/admin-scripts.js'),
        true
    );
    
    // Localize script for translations and nonces
    wp_localize_script('smartdr-admin-scripts', 'smartdrAdmin', [
        'confirmResetStyles' => __('Are you sure you want to reset all style settings to default values? This will immediately save the changes to the database.', 'smart-download-redirector'),
        'confirmResetAll' => __('Are you sure you want to reset ALL settings to default values? This will reset domains, text, styling, display, countdown, and rate limiting settings and immediately save the changes to the database.', 'smart-download-redirector'),
        'resetting' => __('Resetting...', 'smart-download-redirector'),
        'resetStyles' => __('Reset Styles to Default', 'smart-download-redirector'),
        'resetEverything' => __('Reset Everything to Default', 'smart-download-redirector'),
        'errorReset' => __('Error resetting settings. Please try again.', 'smart-download-redirector'),
        'resetStylesNonce' => wp_create_nonce('smartdr_reset_styles'),
        'resetEverythingNonce' => wp_create_nonce('smartdr_reset_everything'),
        'defaultHeading' => __('Preparing Your Download', 'smart-download-redirector'),
        'defaultLinkText' => __('Click here if not redirected automatically', 'smart-download-redirector')
    ]);
}

/**
 * Register plugin settings
 */
function smartdr_register_settings()
{
    // General settings section
    add_settings_section(
        'smartdr_general_section',
        __('General Settings', 'smart-download-redirector'),
        'smartdr_general_section_callback',
        'smartdr_settings'
    );

    // Target domains setting
    register_setting(
        'smartdr_settings',
        'smartdr_target_domains',
        [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
            'default' => "google.com",
        ]
    );

    add_settings_field(
        'smartdr_target_domains',
        __('Target Domains', 'smart-download-redirector'),
        'smartdr_target_domains_callback',
        'smartdr_settings',
        'smartdr_general_section'
    );

    // Text Customization section
    add_settings_section(
        'smartdr_text_section',
        __('Text Customization', 'smart-download-redirector'),
        'smartdr_text_section_callback',
        'smartdr_settings'
    );

    // Custom heading
    register_setting(
        'smartdr_settings',
        'smartdr_custom_heading',
        [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => __('Preparing Your Download', 'smart-download-redirector'),
        ]
    );

    add_settings_field(
        'smartdr_custom_heading',
        __('Custom Heading', 'smart-download-redirector'),
        'smartdr_custom_heading_callback',
        'smartdr_settings',
        'smartdr_text_section'
    );

    // Manual Link Text
    register_setting(
        'smartdr_settings',
        'smartdr_manual_link_text',
        [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => __('Click here if not redirected automatically', 'smart-download-redirector'),
            'show_in_rest' => true,
        ]
    );

    // Show Title
    register_setting(
        'smartdr_settings',
        'smartdr_show_title',
        [
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true,
        ]
    );

    add_settings_field(
        'smartdr_manual_link_text',
        __('Manual Link Text', 'smart-download-redirector'),
        'smartdr_manual_link_text_callback',
        'smartdr_settings',
        'smartdr_text_section'
    );

        add_settings_field(
        'smartdr_show_title',
        __('Display Custom Heading', 'smart-download-redirector'),
        'smartdr_show_title_callback',
            'smartdr_settings',
        'smartdr_text_section'
    );

    // Display Options section
    add_settings_section(
        'smartdr_display_section',
        __('Display Options', 'smart-download-redirector'),
        'smartdr_display_section_callback',
        'smartdr_settings'
    );



    // Show manual link
    register_setting(
        'smartdr_settings',
        'smartdr_show_manual_link',
        [
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true,
        ]
    );



    // Layout style setting
    register_setting(
        'smartdr_settings',
        'smartdr_layout_style',
        [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'standard',
        ]
    );

    add_settings_field(
        'smartdr_show_manual_link',
        __('Show Manual Link', 'smart-download-redirector'),
        'smartdr_show_manual_link_callback',
        'smartdr_settings',
        'smartdr_display_section'
    );



    add_settings_field(
        'smartdr_layout_style',
        __('Layout Style', 'smart-download-redirector'),
        'smartdr_layout_style_callback',
        'smartdr_settings',
        'smartdr_display_section'
    );

    // Countdown settings section
    add_settings_section(
        'smartdr_countdown_section',
        __('Countdown Settings', 'smart-download-redirector'),
        'smartdr_countdown_section_callback',
        'smartdr_settings'
    );

    // Countdown time setting
    register_setting(
        'smartdr_settings',
        'smartdr_countdown_time',
        [
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 10,
        ]
    );

    // Redirection delay setting
    register_setting(
        'smartdr_settings',
        'smartdr_redirect_delay',
        [
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 3,
            'show_in_rest' => true,
        ]
    );

    add_settings_field(
        'smartdr_countdown_time',
        __('Countdown Time (seconds)', 'smart-download-redirector'),
        'smartdr_countdown_time_callback',
        'smartdr_settings',
        'smartdr_countdown_section'
    );

    add_settings_field(
        'smartdr_redirect_delay',
        __('Redirection Delay (seconds)', 'smart-download-redirector'),
        'smartdr_redirect_delay_callback',
        'smartdr_settings',
        'smartdr_countdown_section'
    );

    // Styling settings section
    add_settings_section(
        'smartdr_styling_section',
        __('Styling Settings', 'smart-download-redirector'),
        'smartdr_styling_section_callback',
        'smartdr_settings'
    );



    // Register style settings
    register_setting(
        'smartdr_settings',
        'smartdr_styles',
        [
            'type' => 'object',
            'sanitize_callback' => 'smartdr_sanitize_styles',
            'default' => [
                'heading_color' => '#212121',
                'heading_size' => '32px',
                'counter_color' => '#00897B',
                'counter_size' => '36px',
                'counter_font_weight' => 'bold',
                'background_color' => '#ffffff',
                'border_radius' => '8px',
                'border_size' => '1px',
                'border_color' => '#dddddd',
                'step_label_size' => '14px',
                'manual_link_size' => '14px'
            ],
        ]
    );



    // Add style fields with clear sections
    $style_fields = [
        // Main Heading Section
        '_heading_section' => [
            'label' => __('Main Heading Styles', 'smart-download-redirector'),
            'type' => 'html'
        ],

        'heading_color' => [
            'label' => __('Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the color of the main heading text', 'smart-download-redirector')
        ],
        'heading_size' => [
            'label' => __('Size', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the size of the main heading text', 'smart-download-redirector')
        ],
        
        // Countdown Timer Section
        '_timer_section' => [
            'label' => __('Countdown Timer Styles', 'smart-download-redirector'),
            'type' => 'html'
        ],
        'counter_color' => [
            'label' => __('Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the color of countdown number, progress ring, active step dots and labels', 'smart-download-redirector')
        ],
        'counter_size' => [
            'label' => __('Number Size', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the size of the countdown number', 'smart-download-redirector')
        ],
        'counter_font_weight' => [
            'label' => __('Font Weight', 'smart-download-redirector'),
            'type' => 'select',
            'options' => [
                'normal' => __('Normal', 'smart-download-redirector'),
                'bold' => __('Bold', 'smart-download-redirector')
            ],
            'description' => __('Controls the font weight of the countdown number', 'smart-download-redirector')
        ],
        
        // Container/Background Section
        '_container_section' => [
            'label' => __('Main Container Styles', 'smart-download-redirector'),
            'type' => 'html'
        ],
        'background_color' => [
            'label' => __('Background Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the background color of the entire download page container', 'smart-download-redirector')
        ],
        'border_radius' => [
            'label' => __('Border Radius', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the rounded corners of the main container border', 'smart-download-redirector')
        ],
        'border_size' => [
            'label' => __('Border Size', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the thickness of the main container border outline', 'smart-download-redirector')
        ],
        'border_color' => [
            'label' => __('Border Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the color of the main container border outline', 'smart-download-redirector')
        ],
        
        // Manual Link Section
        '_text_section' => [
            'label' => __('Manual Link Styles', 'smart-download-redirector'),
            'type' => 'html'
        ],
        'manual_link_size' => [
            'label' => __('Manual Link Size', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the size of the manual redirect link text', 'smart-download-redirector')
        ],
        'manual_link_color' => [
            'label' => __('Manual Link Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the color of the manual redirect link', 'smart-download-redirector')
        ],
        'manual_link_hover_color' => [
            'label' => __('Manual Link Hover Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the color of the manual redirect link when hovered', 'smart-download-redirector')
        ]
    ];

    foreach ($style_fields as $key => $field) {
        $args = [
            'key' => $key,
            'type' => $field['type']
        ];
        
        // Add options for select fields
        if (isset($field['options'])) {
            $args['options'] = $field['options'];
        }
        
        // Add description if provided
        if (isset($field['description'])) {
            $args['description'] = $field['description'];
        }
        
        // Pass label for HTML sections
        if ($field['type'] === 'html') {
            $args['label'] = $field['label'];
        }
        
        add_settings_field(
            'smartdr_styles_' . $key,
            ($field['type'] === 'html') ? '' : $field['label'], // Empty label for HTML sections to avoid duplication
            'smartdr_style_field_callback',
            'smartdr_settings',
            'smartdr_styling_section',
            $args
        );
    }

    // Register destination box style settings
    register_setting(
        'smartdr_settings',
        'smartdr_destination_styles',
        [
            'type' => 'object',
            'sanitize_callback' => 'smartdr_sanitize_destination_styles',
            'default' => [
                'background_color' => '#f8f9fa',
                'border_size' => '1px',
                'border_color' => '#dee2e6',
                'border_radius' => '4px',
                'text_size' => '14px'
            ],
        ]
    );

    // Add destination box style fields with section header
    $destination_style_fields = [
        '_destination_section' => [
            'label' => __('Destination URL Box Styles', 'smart-download-redirector'),
            'type' => 'html'
        ],
        'background_color' => [
            'label' => __('Background Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the background color of the destination URL display box', 'smart-download-redirector')
        ],
        'border_size' => [
            'label' => __('Border Size', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the thickness of the destination box border', 'smart-download-redirector')
        ],
        'border_color' => [
            'label' => __('Border Color', 'smart-download-redirector'),
            'type' => 'color',
            'description' => __('Controls the color of the destination box border', 'smart-download-redirector')
        ],
        'border_radius' => [
            'label' => __('Border Radius', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the rounded corners of the destination box', 'smart-download-redirector')
        ],
        'text_size' => [
            'label' => __('Text Size', 'smart-download-redirector'),
            'type' => 'text',
            'description' => __('Controls the text size inside the destination URL box', 'smart-download-redirector')
        ]
    ];

    foreach ($destination_style_fields as $key => $field) {
        $args = [
            'key' => $key,
            'type' => $field['type'],
            'class' => 'smartdr-custom-style-field'
        ];
        
        // Add description if provided
        if (isset($field['description'])) {
            $args['description'] = $field['description'];
        }
        
        // Pass label for HTML sections
        if ($field['type'] === 'html') {
            $args['label'] = $field['label'];
        }
        
        add_settings_field(
            'smartdr_destination_styles_' . $key,
            ($field['type'] === 'html') ? '' : $field['label'], // Empty label for HTML sections to avoid duplication
            'smartdr_destination_style_field_callback',
            'smartdr_settings',
            'smartdr_styling_section',
            $args
        );
    }

    // Rate limiting settings section
    add_settings_section(
        'smartdr_rate_limit_section',
        __('Rate Limiting', 'smart-download-redirector'),
        'smartdr_rate_limit_section_callback',
        'smartdr_settings'
    );

    // Rate limit settings
    register_setting(
        'smartdr_settings',
        'smartdr_rate_limit',
        [
            'type' => 'object',
            'sanitize_callback' => 'smartdr_sanitize_rate_limit',
            'default' => [
                'enabled' => true,
                'max_requests' => 100,
                'time_window' => 3600,
            ],
        ]
    );

    add_settings_field(
        'smartdr_rate_limit_enabled',
        __('Enable Rate Limiting', 'smart-download-redirector'),
        'smartdr_rate_limit_enabled_callback',
        'smartdr_settings',
        'smartdr_rate_limit_section'
    );

    add_settings_field(
        'smartdr_rate_limit_max_requests',
        __('Download Limit', 'smart-download-redirector'),
        'smartdr_rate_limit_max_requests_callback',
        'smartdr_settings',
        'smartdr_rate_limit_section'
    );

    add_settings_field(
        'smartdr_rate_limit_time_window',
        __('Time Period', 'smart-download-redirector'),
        'smartdr_rate_limit_time_window_callback',
        'smartdr_settings',
        'smartdr_rate_limit_section'
    );
}
add_action('admin_init', 'smartdr_register_settings');

/**
 * General section callback
 */
function smartdr_general_section_callback()
{
    echo '<p>' . esc_html__(
        'Configure which domains are allowed for download redirection.',
        'smart-download-redirector'
    ) . '</p>';
}

/**
 * Target domains field callback
 */
function smartdr_target_domains_callback()
{
    $value = get_option('smartdr_target_domains', '');
    ?>
    <textarea
        name="smartdr_target_domains"
        id="smartdr_target_domains"
        rows="5"
        cols="50"
        class="large-text code"
    ><?php echo esc_textarea($value); ?></textarea>
    <p class="description">
        <?php
        esc_html_e(
            'Enter one domain per line. Do not include http:// or https://. Example: example.com',
            'smart-download-redirector'
        );
        ?>
    </p>
    <?php
}

/**
 * Text section callback
 */
function smartdr_text_section_callback()
{
    echo '<p>' . esc_html__(
        'Customize the text displayed on the download countdown page.',
        'smart-download-redirector'
    ) . '</p>';
}

/**
 * Custom heading callback
 */
function smartdr_custom_heading_callback()
{
    $value = get_option('smartdr_custom_heading', '');
    // If empty (after reset or fresh install), use the default value directly in HTML
    if (empty($value)) {
        $value = 'Preparing Your Download';
    }
    ?>
    <input 
        type="text" 
        name="smartdr_custom_heading" 
        id="smartdr_custom_heading" 
        value="<?php echo esc_attr($value); ?>" 
        class="regular-text"
    />
    <p class="description">
        <?php esc_html_e('The main heading displayed on the countdown page. This text can be translated with WPML if enabled.', 'smart-download-redirector'); ?>
    </p>
    <?php
}

/**
 * Manual link text callback
 */
function smartdr_manual_link_text_callback()
{
    $value = get_option('smartdr_manual_link_text', __('Click here if not redirected automatically', 'smart-download-redirector'));
    ?>
    <input 
        type="text" 
        name="smartdr_manual_link_text" 
        id="smartdr_manual_link_text" 
        value="<?php echo esc_attr($value); ?>" 
        class="regular-text"
    />
    <p class="description">
        <?php esc_html_e('Text displayed for the manual redirect link. This text can be translated with WPML if enabled.', 'smart-download-redirector'); ?>
    </p>
    <?php
}

/**
 * Show title callback
 */
function smartdr_show_title_callback()
{
    $value = get_option('smartdr_show_title', true);
    ?>
    <label>
        <input 
            type="checkbox" 
            name="smartdr_show_title" 
            id="smartdr_show_title"
            <?php checked($value); ?>
        >
        <?php esc_html_e('Display the custom heading on the countdown page', 'smart-download-redirector'); ?>
    </label>
    <?php
}

/**
 * Display section callback
 */
function smartdr_display_section_callback()
{
    echo '<p>' . esc_html__(
        'Configure what information is shown on the download countdown page.',
        'smart-download-redirector'
    ) . '</p>';
}



/**
 * Layout style callback
 */
function smartdr_layout_style_callback()
{
    $value = get_option('smartdr_layout_style', 'standard');
    ?>
    <select name="smartdr_layout_style" id="smartdr_layout_style">
        <option value="standard" <?php selected($value, 'standard'); ?>><?php esc_html_e('Layout 1 (Standard)', 'smart-download-redirector'); ?></option>
        <option value="compact" <?php selected($value, 'compact'); ?>><?php esc_html_e('Layout 2 (Compact)', 'smart-download-redirector'); ?></option>
        <option value="progressbar" <?php selected($value, 'progressbar'); ?>><?php esc_html_e('Layout 3 (Progressbar)', 'smart-download-redirector'); ?></option>
    </select>
    <p class="description">
        <?php esc_html_e('Choose the layout style for the download page. Standard: vertical layout with centered elements. Compact: horizontal layout with text on the left and counter on the right. Progressbar: vertical layout with horizontal progress bar instead of circular countdown.', 'smart-download-redirector'); ?>
    </p>
    <?php
}

/**
 * Countdown section callback
 */
function smartdr_countdown_section_callback()
{
    echo '<p>' . esc_html__(
        'Configure the countdown timer settings for the download page.',
        'smart-download-redirector'
    ) . '</p>';
}

/**
 * Countdown time field callback
 */
function smartdr_countdown_time_callback()
{
    $value = get_option('smartdr_countdown_time', 10);
    ?>
    <input
        type="number"
        name="smartdr_countdown_time"
        id="smartdr_countdown_time"
        value="<?php echo esc_attr($value); ?>"
        min="1"
        max="60"
        step="1"
    />
    <p class="description">
        <?php
        esc_html_e(
            'Number of seconds to wait before allowing download (1-60 seconds).',
            'smart-download-redirector'
        );
        ?>
    </p>
    <?php
}

/**
 * Redirection delay field callback
 */
function smartdr_redirect_delay_callback()
{
    $value = get_option('smartdr_redirect_delay', 3);
    ?>
    <input
        type="number"
        name="smartdr_redirect_delay"
        id="smartdr_redirect_delay"
        value="<?php echo esc_attr($value); ?>"
        min="1"
        max="10"
        step="1"
        class="small-text"
    />
    <p class="description">
        <?php
        esc_html_e(
            'Number of seconds to wait before redirecting after countdown reaches zero (1-10 seconds).',
            'smart-download-redirector'
        );
        ?>
    </p>
    <?php
}

/**
 * Rate limit section callback
 */
function smartdr_rate_limit_section_callback()
{
    echo '<p>' . esc_html__(
        'Protect your server from abuse by limiting how many downloads users can request.',
        'smart-download-redirector'
    ) . '</p>';
}

/**
 * Rate limit enabled field callback
 */
function smartdr_rate_limit_enabled_callback()
{
    $options = get_option('smartdr_rate_limit');
    $enabled = isset($options['enabled']) ? $options['enabled'] : true;
    ?>
    <label style="display: flex; align-items: center; gap: 8px;">
        <input
            type="checkbox"
            name="smartdr_rate_limit[enabled]"
            id="smartdr_rate_limit_enabled"
            <?php checked($enabled); ?>
        />
        <strong><?php esc_html_e('Enable Rate Limiting Protection', 'smart-download-redirector'); ?></strong>
    </label>
    
    <div style="background: #f0f6fc; border: 1px solid #c0d3eb; border-radius: 4px; padding: 15px; margin: 15px 0;">
        <strong><?php esc_html_e('What does this do?', 'smart-download-redirector'); ?></strong><br>
        <?php esc_html_e('Rate limiting prevents users from making too many download requests too quickly, which helps protect your server from overload and potential abuse.', 'smart-download-redirector'); ?>
    </div>

    <?php
}

/**
 * Rate limit max requests field callback
 */
function smartdr_rate_limit_max_requests_callback()
{
    $options = get_option('smartdr_rate_limit');
    $max_requests = isset($options['max_requests']) ? $options['max_requests'] : 100;
    ?>
    <div style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
    <input
        type="number"
        name="smartdr_rate_limit[max_requests]"
            id="smartdr_rate_limit_max_requests"
        value="<?php echo esc_attr($max_requests); ?>"
        min="1"
        max="1000"
        step="1"
            style="width: 80px;"
        />
        <span><?php esc_html_e('downloads allowed', 'smart-download-redirector'); ?></span>
        
        <div style="margin-left: 20px;">
            <strong><?php esc_html_e('Quick presets:', 'smart-download-redirector'); ?></strong>
            <button type="button" class="button button-small" onclick="document.getElementById('smartdr_rate_limit_max_requests').value=10">10</button>
            <button type="button" class="button button-small" onclick="document.getElementById('smartdr_rate_limit_max_requests').value=50">50</button>
            <button type="button" class="button button-small" onclick="document.getElementById('smartdr_rate_limit_max_requests').value=100">100</button>
            <button type="button" class="button button-small" onclick="document.getElementById('smartdr_rate_limit_max_requests').value=500">500</button>
        </div>
    </div>
    
    <div style="background: #fff9e6; border: 1px solid #f0d000; border-radius: 4px; padding: 10px; margin: 10px 0;">
        <strong><?php esc_html_e('How to choose:', 'smart-download-redirector'); ?></strong><br>
        <ul style="margin: 5px 0 0 20px;">
            <li><strong>10-25:</strong> <?php esc_html_e('High security (small websites)', 'smart-download-redirector'); ?></li>
            <li><strong>50-100:</strong> <?php esc_html_e('Balanced protection (most websites)', 'smart-download-redirector'); ?></li>
            <li><strong>200-500:</strong> <?php esc_html_e('Light protection (high-traffic websites)', 'smart-download-redirector'); ?></li>
        </ul>
    </div>
    <?php
}

/**
 * Rate limit time window field callback
 */
function smartdr_rate_limit_time_window_callback()
{
    $options = get_option('smartdr_rate_limit');
    $time_window = isset($options['time_window']) ? $options['time_window'] : 3600;
    
    $time_options = [
        300 => __('5 minutes', 'smart-download-redirector'),
        900 => __('15 minutes', 'smart-download-redirector'),
        1800 => __('30 minutes', 'smart-download-redirector'),
        3600 => __('1 hour', 'smart-download-redirector'),
        7200 => __('2 hours', 'smart-download-redirector'),
        21600 => __('6 hours', 'smart-download-redirector'),
        43200 => __('12 hours', 'smart-download-redirector'),
        86400 => __('24 hours', 'smart-download-redirector'),
        'custom' => __('Custom (seconds)', 'smart-download-redirector')
    ];
    
    $is_custom = !array_key_exists($time_window, $time_options);
    ?>
    <div style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
        <label for="smartdr_time_window_preset"><strong><?php esc_html_e('Time period:', 'smart-download-redirector'); ?></strong></label>
        <select id="smartdr_time_window_preset" style="min-width: 150px;">
            <?php foreach ($time_options as $value => $label): ?>
                <option value="<?php echo esc_attr($value); ?>" <?php selected($is_custom ? 'custom' : $time_window, $value); ?>>
                    <?php echo esc_html($label); ?>
                </option>
            <?php endforeach; ?>
        </select>
        
        <div id="smartdr_custom_time_window" style="<?php echo $is_custom ? '' : 'display: none;'; ?>">
    <input
        type="number"
        name="smartdr_rate_limit[time_window]"
                id="smartdr_rate_limit_time_window"
        value="<?php echo esc_attr($time_window); ?>"
        min="60"
        max="86400"
        step="60"
                style="width: 100px;"
            />
            <span><?php esc_html_e('seconds', 'smart-download-redirector'); ?></span>
        </div>
        
        <input type="hidden" name="smartdr_rate_limit[time_window]" id="smartdr_rate_limit_time_window_hidden" value="<?php echo esc_attr($time_window); ?>">
    </div>
    
    <div style="background: #e7f5e7; border: 1px solid #4caf50; border-radius: 4px; padding: 10px; margin: 10px 0;">
        <strong><?php esc_html_e('Examples:', 'smart-download-redirector'); ?></strong><br>
        <ul style="margin: 5px 0 0 20px;">
            <li><strong><?php esc_html_e('100 downloads per 1 hour:', 'smart-download-redirector'); ?></strong> <?php esc_html_e('Good for most websites', 'smart-download-redirector'); ?></li>
            <li><strong><?php esc_html_e('50 downloads per 15 minutes:', 'smart-download-redirector'); ?></strong> <?php esc_html_e('Strict protection', 'smart-download-redirector'); ?></li>
            <li><strong><?php esc_html_e('500 downloads per 24 hours:', 'smart-download-redirector'); ?></strong> <?php esc_html_e('Daily limit approach', 'smart-download-redirector'); ?></li>
        </ul>
    </div>

    <?php
}

/**
 * Sanitize rate limit settings
 * 
 * @param array|null $input The input to sanitize
 * @return array The sanitized rate limit settings
 */
function smartdr_sanitize_rate_limit($input)
{
    $defaults = [
        'enabled' => true,
        'max_requests' => 100,
        'time_window' => 3600,
    ];

    if (!is_array($input)) {
        return $defaults;
    }

    $sanitized = [];

    // Sanitize enabled
    $sanitized['enabled'] = isset($input['enabled']);

    // Sanitize max_requests
    $max_requests = isset($input['max_requests']) ? absint($input['max_requests']) : $defaults['max_requests'];
    $sanitized['max_requests'] = min(max($max_requests, 1), 1000);

    // Sanitize time_window
    $time_window = isset($input['time_window']) ? absint($input['time_window']) : $defaults['time_window'];
    $sanitized['time_window'] = min(max($time_window, 60), 86400);

    return $sanitized;
}

/**
 * Styling section callback
 */
function smartdr_styling_section_callback()
{
    echo '<p>' . esc_html__(
        'Customize the appearance of the download countdown page.',
        'smart-download-redirector'
    ) . '</p>';
}



/**
 * Style field callback
 *
 * @param array $args Field arguments
 */
function smartdr_style_field_callback($args)
{
    // Handle HTML section headers
    if ($args['type'] === 'html') {
        echo '<h3 style="margin: 20px 0 10px 0; color: #23282d; border-bottom: 1px solid #ccd0d4; padding-bottom: 5px; font-size: 14px;">' . esc_html($args['label']) . '</h3>';
        return;
    }

    $styles = get_option('smartdr_styles', []);
    $value = isset($styles[$args['key']]) ? $styles[$args['key']] : '';
    $name = 'smartdr_styles[' . $args['key'] . ']';
    $id = 'smartdr_styles_' . $args['key'];

    if ($args['type'] === 'color') {
        // Set appropriate default based on field
        $default_colors = [
            'heading_color' => '#212121',
            'counter_color' => '#00897B',
            'background_color' => '#ffffff',
            'border_color' => '#dddddd',
            'manual_link_color' => '#00897B',
            'manual_link_hover_color' => '#00695C'
        ];
        $default_color = isset($default_colors[$args['key']]) ? $default_colors[$args['key']] : '#dddddd';
        
        // Set color value or use field-specific default
        $color_value = !empty($value) ? $value : $default_color;
        // Ensure we have a valid hex color value
        if (!preg_match('/^#[a-fA-F0-9]{6}$/', $color_value)) {
            $color_value = $default_color;
        }
        ?>
        <input 
            type="color" 
            id="<?php echo esc_attr($id); ?>" 
            name="<?php echo esc_attr($name); ?>" 
            value="<?php echo esc_attr($color_value); ?>" 
            class="smartdr-color-picker"
        />
        <?php
    } elseif ($args['type'] === 'select') {
        printf('<select id="%s" name="%s">', esc_attr($id), esc_attr($name));
        if (isset($args['options']) && is_array($args['options'])) {
            foreach ($args['options'] as $option_value => $option_label) {
        printf(
                    '<option value="%s" %s>%s</option>',
                    esc_attr($option_value),
                    selected($value, $option_value, false),
                    esc_html($option_label)
                );
            }
        }
        echo '</select>';
    } else {
        printf(
            '<input type="text" id="%s" name="%s" value="%s" class="regular-text">',
            esc_attr($id),
            esc_attr($name),
            esc_attr($value)
        );
        if (strpos($args['key'], 'size') !== false) {
            echo '<p class="description">' . esc_html__('Use CSS units (px, em, rem)', 'smart-download-redirector') . '</p>';
        }
    }
    
    // Add description if provided
    if (isset($args['description']) && !empty($args['description'])) {
        echo '<p class="description">' . esc_html($args['description']) . '</p>';
    }
}

/**
 * Sanitize style settings
 *
 * @param array $input The input array to sanitize
 * @return array Sanitized input
 */
function smartdr_sanitize_styles($input)
{
    if (!is_array($input)) {
        return [];
    }

    $sanitized = [];
    $allowed_keys = [
        'heading_color', 'heading_size', 'counter_color', 'counter_size', 'counter_font_weight',
        'background_color', 'border_radius', 'border_size',
        'border_color', 'manual_link_size', 'manual_link_color', 'manual_link_hover_color'
    ];

    foreach ($allowed_keys as $key) {
        if (isset($input[$key])) {
            if (strpos($key, 'color') !== false) {
                $sanitized[$key] = sanitize_hex_color($input[$key]);
            } elseif ($key === 'counter_font_weight') {
                $sanitized[$key] = in_array($input[$key], ['normal', 'bold']) ? $input[$key] : 'bold';
        } else {
                $sanitized[$key] = sanitize_text_field($input[$key]);
            }
        }
    }

    return $sanitized;
}



/**
 * Destination box style field callback
 */
function smartdr_destination_style_field_callback($args)
{
    // Handle HTML section headers
    if ($args['type'] === 'html') {
        echo '<h3 style="margin: 20px 0 10px 0; color: #23282d; border-bottom: 1px solid #ccd0d4; padding-bottom: 5px; font-size: 14px;">' . esc_html($args['label']) . '</h3>';
        return;
    }

    $styles = get_option('smartdr_destination_styles', []);
    $value = isset($styles[$args['key']]) ? $styles[$args['key']] : '';
    $name = 'smartdr_destination_styles[' . $args['key'] . ']';
    $id = 'smartdr_destination_styles_' . $args['key'];

    if ($args['type'] === 'color') {
        // Set default color value if empty
        $color_value = !empty($value) ? $value : '#dddddd';
        // Ensure we have a valid hex color value
        if (!preg_match('/^#[a-fA-F0-9]{6}$/', $color_value)) {
            $color_value = '#dddddd';
        }
        ?>
        <input 
            type="color" 
            id="<?php echo esc_attr($id); ?>" 
            name="<?php echo esc_attr($name); ?>" 
            value="<?php echo esc_attr($color_value); ?>" 
            class="smartdr-color-picker"
        />
        <?php
    } else {
        printf(
            '<input type="text" id="%s" name="%s" value="%s" class="regular-text">',
            esc_attr($id),
            esc_attr($name),
            esc_attr($value)
        );
        if (strpos($args['key'], 'size') !== false) {
            echo '<p class="description">' . esc_html__('Use CSS units (px, em, rem)', 'smart-download-redirector') . '</p>';
        }
    }
    
    // Add description if provided
    if (isset($args['description']) && !empty($args['description'])) {
        echo '<p class="description">' . esc_html($args['description']) . '</p>';
    }
}

/**
 * Sanitize destination box styles
 */
function smartdr_sanitize_destination_styles($input)
{
    if (!is_array($input)) {
        return [];
    }

    $sanitized = [];
    $allowed_keys = [
        'background_color',
        'border_size',
        'border_color',
        'border_radius',
        'text_size'
    ];

    foreach ($allowed_keys as $key) {
        if (isset($input[$key])) {
            if (strpos($key, 'color') !== false) {
                $sanitized[$key] = sanitize_hex_color($input[$key]);
            } else {
                $sanitized[$key] = sanitize_text_field($input[$key]);
            }
        }
    }

    return $sanitized;
}

/**
 * Show manual link callback
 */
function smartdr_show_manual_link_callback()
{
    $value = get_option('smartdr_show_manual_link', true);
    ?>
    <label>
        <input 
            type="checkbox" 
            name="smartdr_show_manual_link" 
            id="smartdr_show_manual_link"
            <?php checked($value); ?>
        >
        <?php esc_html_e('Show a manual redirect link on the countdown page when the countdown reaches 0 seconds', 'smart-download-redirector'); ?>
    </label>
    <?php
}



/**
 * Render the settings page
 */
function smartdr_render_settings_page()
{
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'smart-download-redirector'));
    }

    // Check if settings were updated
    // Use WordPress core function to check if options were updated - this is security-safe
    $settings_updated = false;
    if (current_user_can('manage_options')) {
        // Check if any of our options were recently updated
        $check_options = [
            'smartdr_target_domains',
            'smartdr_countdown_time', 
            'smartdr_redirect_delay',
            'smartdr_styles',
            'smartdr_layout_style'
        ];
        
        foreach ($check_options as $option) {
            if (get_transient('smartdr_option_updated_' . $option)) {
                $settings_updated = true;
                delete_transient('smartdr_option_updated_' . $option);
                break;
            }
        }
        
        // Fallback: Check WordPress core parameter (already validated by WordPress)
        if (!$settings_updated && 
            filter_input(INPUT_GET, 'settings-updated', FILTER_SANITIZE_STRING) === 'true' &&
            !get_settings_errors('smartdr_messages')
        ) {
            $settings_updated = true;
        }
    }
    
    if ($settings_updated) {
        add_settings_error(
            'smartdr_messages',
            'smartdr_message',
            __('Settings Saved', 'smart-download-redirector'),
            'updated'
        );
    }

    // Show error/update messages
    settings_errors('smartdr_messages');
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        <form action="options.php" method="post">
            <?php
            settings_fields('smartdr_settings');
            do_settings_sections('smartdr_settings');
            ?>
            <div class="submit-buttons" style="display: flex; gap: 10px; align-items: center; flex-wrap: wrap;">
                <div class="submit" style="margin: 0;">
                    <?php submit_button(__('Save Changes', 'smart-download-redirector'), 'primary', 'submit', false); ?>
                </div>
                <input type="button" id="smartdr-reset-styles" class="button button-secondary" value="<?php esc_attr_e('Reset Styles to Default', 'smart-download-redirector'); ?>" style="margin: 0;">
                <input type="button" id="smartdr-reset-everything" class="button button-secondary" value="<?php esc_attr_e('Reset Everything to Default', 'smart-download-redirector'); ?>" style="margin: 0; background-color: #dc3545; border-color: #dc3545; color: white;">
            </div>
        </form>
    </div>
    <?php
}

/**
 * Simple Color Picker JavaScript (no longer needed - keeping for reference)
 */
function smartdr_color_picker_script() {
    // Color picker inputs now work natively - no additional JavaScript needed
    // Modern browsers handle color input validation and hex conversion automatically
}

// Only load on settings page
add_action('admin_footer', function() {
    $screen = get_current_screen();
    if ($screen && $screen->id === 'toplevel_page_smartdr-settings') {
        smartdr_color_picker_script();
    }
});

/**
 * Register strings for WPML translation
 */
function smartdr_register_wpml_strings() {
    if (function_exists('wpml_register_single_string')) {
        // Register custom heading
        $custom_heading = get_option('smartdr_custom_heading', __('Preparing Your Download', 'smart-download-redirector'));
        wpml_register_single_string('smart-download-redirector', 'Custom Heading', $custom_heading);
        
        // Register manual link text
        $manual_link_text = get_option('smartdr_manual_link_text', __('Click here if not redirected automatically', 'smart-download-redirector'));
        wpml_register_single_string('smart-download-redirector', 'Manual Link Text', $manual_link_text);
    }
}



/**
 * Set transient when options are updated for secure update detection
 */
function smartdr_option_updated($option_name) {
    if (strpos($option_name, 'smartdr_') === 0) {
        set_transient('smartdr_option_updated_' . $option_name, true, 60); // 1 minute
    }
}

// Hook to track option updates
add_action('updated_option', 'smartdr_option_updated');

// Hook to register WPML strings when options are updated
add_action('update_option_smartdr_custom_heading', 'smartdr_register_wpml_strings');
add_action('update_option_smartdr_manual_link_text', 'smartdr_register_wpml_strings');

// Register strings on plugin activation
add_action('admin_init', 'smartdr_register_wpml_strings'); 
