import {
  Tr,
  Td,
  Button,
  ButtonGroup,
  HStack,
  VStack,
  Text,
  Badge,
  IconButton,
  Tooltip,
  useColorModeValue,
  Menu,
  MenuButton,
  MenuList,
  MenuItem,
  useDisclosure,
  AlertDialog,
  AlertDialogBody,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogContent,
  AlertDialogOverlay,
} from '@chakra-ui/react';
import { useRef } from '@wordpress/element';
import { EditIcon, DeleteIcon, ChevronDownIcon } from '@chakra-ui/icons';
import ShortcodeCell from './ShortcodeCell';
import { formatCondition, formatDiscount } from '../../../utils/formatters';

const TableRow = ({ rule, onEdit, onDelete }) => {
  const { isOpen, onOpen, onClose } = useDisclosure();
  const cancelRef = useRef();
  const hoverBg = useColorModeValue('gray.50', 'gray.700');

  const getConditionBadgeColor = (type) => {
    return type === 'cart_total' ? 'green' : 'purple';
  };

  const getDiscountBadgeColor = (type) => {
    return type === 'percentage' ? 'blue' : 'orange';
  };

  const handleDelete = () => {
    onClose();
    onDelete(rule.id);
  };

  return (
    <>
      <Tr _hover={{ bg: hoverBg }} transition="background-color 0.2s">
        <Td>
          <VStack align="start" spacing={1}>
            <Text fontWeight="medium">{rule.title.rendered}</Text>
            <Badge size="sm" colorScheme={getConditionBadgeColor(rule.meta.dlift_condition_type)}>
              {rule.meta.dlift_condition_type === 'cart_total' ? 'Cart Total' : 'Quantity'}
            </Badge>
          </VStack>
        </Td>
        <Td>
          <VStack align="start" spacing={1}>
            <Text>{formatCondition(rule.meta.dlift_condition_type, rule.meta.dlift_condition_value)}</Text>
            <Text fontSize="sm" color="gray.500">
              Minimum requirement
            </Text>
          </VStack>
        </Td>
        <Td>
          <VStack align="start" spacing={1}>
            <HStack>
              <Badge colorScheme={getDiscountBadgeColor(rule.meta.dlift_discount_type)}>
                {formatDiscount(rule.meta.dlift_discount_type, rule.meta.dlift_discount_value)}
              </Badge>
            </HStack>
            <Text fontSize="sm" color="gray.500">
              {rule.meta.dlift_discount_type === 'percentage' ? 'Percentage off' : 'Fixed amount off'}
            </Text>
          </VStack>
        </Td>
        <Td>
          <ShortcodeCell ruleId={rule.id} />
        </Td>
        <Td>
          <ButtonGroup size="sm" spacing={2}>
            <Tooltip label="Edit Rule" hasArrow>
              <IconButton
                icon={<EditIcon />}
                colorScheme="blue"
                variant="ghost"
                onClick={() => onEdit(rule)}
                aria-label="Edit rule"
              />
            </Tooltip>
            <Tooltip label="Delete Rule" hasArrow>
              <IconButton
                icon={<DeleteIcon />}
                colorScheme="red"
                variant="ghost"
                onClick={onOpen}
                aria-label="Delete rule"
              />
            </Tooltip>
          </ButtonGroup>
        </Td>
      </Tr>

      <AlertDialog
        isOpen={isOpen}
        leastDestructiveRef={cancelRef}
        onClose={onClose}
      >
        <AlertDialogOverlay>
          <AlertDialogContent>
            <AlertDialogHeader fontSize="lg" fontWeight="bold">
              Delete Rule
            </AlertDialogHeader>

            <AlertDialogBody>
              Are you sure you want to delete this rule? This action cannot be undone.
            </AlertDialogBody>

            <AlertDialogFooter>
              <Button ref={cancelRef} onClick={onClose}>
                Cancel
              </Button>
              <Button colorScheme="red" onClick={handleDelete} ml={3}>
                Delete
              </Button>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialogOverlay>
      </AlertDialog>
    </>
  );
};

export default TableRow;