import { useState } from '@wordpress/element';
import {
  Button,
  Code,
  HStack,
  useToast,
  VStack,
  Text,
  IconButton,
  Tooltip,
  useClipboard,
  useColorModeValue,
  Box,
} from '@chakra-ui/react';
import { CopyIcon, CheckIcon } from '@chakra-ui/icons';
import { getShortcode } from '../../../utils/formatters';

const ShortcodeCell = ({ ruleId }) => {
  const shortcode = getShortcode(ruleId);
  const { hasCopied, onCopy } = useClipboard(shortcode);
  const toast = useToast();

  // Use brand colors
  const codeBg = useColorModeValue('secondary.50', 'secondary.900');
  const codeBorder = useColorModeValue('secondary.200', 'secondary.700');
  const codeColor = useColorModeValue('secondary.700', 'secondary.200');

  const handleCopy = () => {
    onCopy();
    // Close any existing toasts first
    toast.closeAll();
    // Show new toast
    toast({
      title: "✨ Shortcode Copied!",
      description: (
        <Box>
          <Text fontWeight="medium" mb={1}>
            {shortcode}
          </Text>
          <Text fontSize="sm">
            You can now paste it anywhere in your content
          </Text>
        </Box>
      ),
      status: "success",
      duration: 3000,
      isClosable: true,
      position: "bottom",
      variant: "solid",
      colorScheme: "primary",
      containerStyle: {
        width: '100%',
        maxWidth: '400px',
        margin: '0 auto'
      }
    });
  };

  return (
    <VStack align="start" spacing={2}>
      <Box
        position="relative"
        width="100%"
        borderRadius="md"
        overflow="hidden"
        borderWidth="1px"
        borderColor={codeBorder}
        transition="all 0.2s"
        _hover={{
          borderColor: 'primary.500',
          shadow: 'sm'
        }}
      >
        <HStack spacing={0} width="100%">
          <Code
            p={3}
            bg={codeBg}
            color={codeColor}
            flex="1"
            fontSize="sm"
            borderRadius="0"
            border="none"
            whiteSpace="nowrap"
            overflow="hidden"
            textOverflow="ellipsis"
          >
            {shortcode}
          </Code>
          <Tooltip
            label={hasCopied ? "Copied!" : "Copy shortcode"}
            hasArrow
            placement="top"
            closeOnClick={false}
            bg={hasCopied ? 'primary.500' : 'secondary.500'}
          >
            <IconButton
              icon={hasCopied ? <CheckIcon /> : <CopyIcon />}
              size="md"
              variant="ghost"
              colorScheme={hasCopied ? "primary" : "secondary"}
              onClick={handleCopy}
              aria-label="Copy shortcode"
              borderRadius="0"
              height="100%"
              _hover={{
                bg: hasCopied ? 'primary.50' : 'secondary.50'
              }}
            />
          </Tooltip>
        </HStack>
      </Box>
      <Text fontSize="xs" color="gray.600">
        Use this shortcode to display the discount message
      </Text>
    </VStack>
  );
};

export default ShortcodeCell;