import {
  Box,
  VStack,
  HStack,
  Text,
  Badge,
  IconButton,
  Menu,
  MenuButton,
  MenuList,
  MenuItem,
  useColorModeValue,
  Flex,
  Collapse,
  Button,
  Tooltip,
} from '@chakra-ui/react';
import { useState } from '@wordpress/element';
import {
  ChevronDownIcon,
  ChevronUpIcon,
  EditIcon,
  DeleteIcon,
  InfoIcon
} from '@chakra-ui/icons';
import { formatCondition, formatDiscount } from '../../../utils/formatters';
import ShortcodeCell from './ShortcodeCell';

const RuleListItem = ({ rule, onEdit, onDelete }) => {
  const [isExpanded, setIsExpanded] = useState(false);
  const borderColor = useColorModeValue('gray.200', 'gray.600');
  const hoverBg = useColorModeValue('gray.50', 'gray.700');

  const getConditionBadgeColor = (type) => {
    return type === 'cart_total' ? 'green' : 'purple';
  };

  const getDiscountBadgeColor = (type) => {
    return type === 'percentage' ? 'primary' : 'accent';
  };

  return (
    <Box
      borderWidth="1px"
      borderColor={borderColor}
      borderRadius="lg"
      overflow="hidden"
      transition="all 0.2s"
      _hover={{
        borderColor: 'primary.200',
        shadow: 'sm',
      }}
    >
      {/* Main Row */}
      <Flex
        p={4}
        justify="space-between"
        align="center"
        onClick={() => setIsExpanded(!isExpanded)}
        cursor="pointer"
        bg={isExpanded ? hoverBg : 'transparent'}
        _hover={{ bg: hoverBg }}
      >
        <HStack spacing={4} flex={1}>
          <VStack align="start" spacing={1} flex={1}>
            <Text fontWeight="medium" color="secondary.500">
              {rule.title.rendered}
            </Text>
            <HStack spacing={2}>
              <Badge size="sm" colorScheme={getConditionBadgeColor(rule.meta.dlift_condition_type)}>
                {formatCondition(rule.meta.dlift_condition_type, rule.meta.dlift_condition_value)}
              </Badge>
              <Badge size="sm" colorScheme={getDiscountBadgeColor(rule.meta.dlift_discount_type)}>
                {formatDiscount(rule.meta.dlift_discount_type, rule.meta.dlift_discount_value)}
              </Badge>
            </HStack>
          </VStack>

          <HStack spacing={2}>
            <Tooltip label="Edit Rule" hasArrow>
              <IconButton
                icon={<EditIcon />}
                variant="ghost"
                colorScheme="primary"
                size="sm"
                onClick={(e) => {
                  e.stopPropagation();
                  onEdit(rule);
                }}
              />
            </Tooltip>
            <Tooltip label="Delete Rule" hasArrow>
              <IconButton
                icon={<DeleteIcon />}
                variant="ghost"
                colorScheme="red"
                size="sm"
                onClick={(e) => {
                  e.stopPropagation();
                  onDelete(rule.id);
                }}
              />
            </Tooltip>
            <IconButton
              icon={isExpanded ? <ChevronUpIcon /> : <ChevronDownIcon />}
              variant="ghost"
              size="sm"
              onClick={() => setIsExpanded(!isExpanded)}
            />
          </HStack>
        </HStack>
      </Flex>

      {/* Expanded Content */}
      <Collapse in={isExpanded}>
        <Box p={4} pt={0}>
          <VStack spacing={4} align="stretch" divider={<Box borderColor={borderColor} />}>
            {/* Messages */}
            <Box>
              <HStack spacing={2} mb={2}>
                <Text fontSize="sm" color="gray.500">Progress Message</Text>
                <Tooltip label="Shown when condition is not met" hasArrow>
                  <InfoIcon color="gray.400" />
                </Tooltip>
              </HStack>
              <Text fontSize="sm">{rule.meta.dlift_progress_message}</Text>
            </Box>

            <Box>
              <HStack spacing={2} mb={2}>
                <Text fontSize="sm" color="gray.500">Success Message</Text>
                <Tooltip label="Shown when condition is met" hasArrow>
                  <InfoIcon color="gray.400" />
                </Tooltip>
              </HStack>
              <Text fontSize="sm">{rule.meta.dlift_success_message}</Text>
            </Box>

            {/* Shortcode */}
            <Box>
              <Text fontSize="sm" color="gray.500" mb={2}>Shortcode</Text>
              <ShortcodeCell ruleId={rule.id} />
            </Box>
          </VStack>
        </Box>
      </Collapse>
    </Box>
  );
};

export default RuleListItem;