import {
  FormControl,
  FormLabel,
  NumberInput,
  NumberInputField,
  NumberInputStepper,
  NumberIncrementStepper,
  NumberDecrementStepper,
  VStack,
  Text,
  InputGroup,
  InputLeftAddon,
  Box,
  useColorModeValue,
  Icon,
  Tooltip,
  Badge,
  FormHelperText
} from '@chakra-ui/react';
import { InfoIcon } from '@chakra-ui/icons';

const ConditionSettings = ({ formData, onChange }) => {
  const helperTextColor = useColorModeValue('gray.600', 'gray.400');
  const borderColor = useColorModeValue('gray.200', 'gray.600');
  const bgHover = useColorModeValue('gray.50', 'gray.700');
  const isValidCondition = formData.conditionValue > 0;

  return (
    <Box
      borderWidth="1px"
      borderColor={borderColor}
      borderRadius="lg"
      p={6}
      _hover={{ bg: bgHover }}
      transition="background 0.2s"
    >
      <VStack spacing={6} align="stretch">
        <Box>
          <Text fontSize="lg" fontWeight="medium" mb={4}>
            Condition Settings
            <Badge ml={2} colorScheme={isValidCondition ? 'green' : 'orange'} variant="subtle">
              {isValidCondition ? 'Active' : 'Set Minimum'}
            </Badge>
          </Text>
        </Box>

        <FormControl>
          <FormLabel fontWeight="medium">
            Minimum Cart Total
            <Tooltip
              label="The minimum cart amount required to activate this discount"
              placement="top"
              hasArrow
            >
              <Icon as={InfoIcon} ml={2} color={helperTextColor} w={4} h={4} cursor="help" />
            </Tooltip>
          </FormLabel>
          <InputGroup>
            <InputLeftAddon
              children="$"
              bg={useColorModeValue('gray.50', 'gray.700')}
              borderColor={borderColor}
            />
            <NumberInput
              value={formData.conditionValue}
              onChange={value => onChange('conditionValue', parseFloat(value) || 0)}
              min={0}
              precision={2}
              step={0.01}
              w="full"
            >
              <NumberInputField
                borderLeftRadius={0}
                borderColor={borderColor}
                _hover={{ borderColor: 'primary.500' }}
                _focus={{
                  borderColor: 'primary.500',
                  boxShadow: '0 0 0 1px var(--chakra-colors-primary-500)'
                }}
              />
              <NumberInputStepper>
                <NumberIncrementStepper
                  borderColor={borderColor}
                  color={helperTextColor}
                />
                <NumberDecrementStepper
                  borderColor={borderColor}
                  color={helperTextColor}
                />
              </NumberInputStepper>
            </NumberInput>
          </InputGroup>
          <FormHelperText color={helperTextColor}>
            {formData.conditionValue === 0
              ? 'Set the minimum cart total required to activate the discount'
              : `Discount will apply when cart total is $${formData.conditionValue} or more`}
          </FormHelperText>
        </FormControl>

        <Box
          mt={4}
          p={4}
          bg={useColorModeValue('primary.50', 'gray.700')}
          borderRadius="md"
          borderWidth="1px"
          borderColor={useColorModeValue('primary.100', 'gray.600')}
        >
          <Text fontSize="sm" color={helperTextColor}>
            <Icon as={InfoIcon} mr={2} color="primary.500" w={4} h={4} />
            Customers will only see the discount message when their cart total meets or exceeds this amount
          </Text>
        </Box>
      </VStack>
    </Box>
  );
};

export default ConditionSettings;