import {
  Box,
  VStack,
  FormControl,
  FormLabel,
  Input,
  Select,
  NumberInput,
  NumberInputField,
  NumberInputStepper,
  NumberIncrementStepper,
  NumberDecrementStepper,
  Button,
  HStack,
  Card,
  CardBody,
  CardHeader,
  InputGroup,
  InputLeftAddon,
  useColorModeValue,
  Text,
  Flex,
  Badge,
  IconButton,
  Tooltip,
  InputRightElement,
  Spacer
} from '@chakra-ui/react';
import { CheckIcon, CloseIcon } from '@chakra-ui/icons';
import { getConditionOptions, getDiscountOptions } from './options';

const EditorForm = ({ formData, onChange, onSave, onCancel }) => {
  const bgColor = useColorModeValue('white', 'gray.800');
  const borderColor = useColorModeValue('gray.200', 'gray.700');
  const headerBg = useColorModeValue('primary.50', 'gray.700');
  const isValidTitle = formData.title && formData.title.length >= 3;

  return (
    <Card
      bg={bgColor}
      borderColor={borderColor}
      borderWidth="1px"
      borderRadius="lg"
      shadow="md"
      maxW="700px"
      mx="auto"
      overflow="hidden"
    >
      <CardHeader bg={headerBg} py={4} px={6}>
        <Flex align="center" gap={3}>
          <Box>
            <Text color="gray.600" fontSize="sm" fontWeight="medium">
              {isValidTitle ? 'Editing Rule' : 'New Rule'}
            </Text>
            <Text fontSize="lg" fontWeight="semibold" mt={1}>
              {formData.title || 'Untitled Rule'}
            </Text>
          </Box>
          <Spacer />
          <Badge
            colorScheme={isValidTitle ? 'green' : 'orange'}
            variant="subtle"
            px={2}
            py={1}
            borderRadius="full"
          >
            {isValidTitle ? 'Ready to Save' : 'Draft'}
          </Badge>
        </Flex>
      </CardHeader>

      <CardBody p={6}>
        <VStack spacing={6} align="stretch">
          <FormControl isRequired>
            <FormLabel>Rule Name</FormLabel>
            <InputGroup>
              <Input
                value={formData.title}
                onChange={e => onChange('title', e.target.value)}
                placeholder="Enter rule name"
                size="md"
                pr="4.5rem"
              />
              <InputRightElement>
                {formData.title && (
                  <IconButton
                    size="sm"
                    variant="ghost"
                    icon={isValidTitle ? <CheckIcon color="green.500" /> : <CloseIcon color="red.500" />}
                    aria-label={isValidTitle ? 'Valid title' : 'Invalid title'}
                  />
                )}
              </InputRightElement>
            </InputGroup>
          </FormControl>

          <FormControl>
            <FormLabel>Minimum Cart Total</FormLabel>
            <InputGroup>
              <InputLeftAddon>
                <Text>$</Text>
              </InputLeftAddon>
              <NumberInput
                value={formData.conditionValue}
                onChange={value => onChange('conditionValue', parseFloat(value) || 0)}
                min={0}
                precision={2}
                step={0.01}
                w="full"
              >
                <NumberInputField borderLeftRadius={0} />
                <NumberInputStepper>
                  <NumberIncrementStepper />
                  <NumberDecrementStepper />
                </NumberInputStepper>
              </NumberInput>
            </InputGroup>
          </FormControl>

          <FormControl>
            <FormLabel>Discount Type</FormLabel>
            <Select
              value={formData.discountType}
              onChange={e => onChange('discountType', e.target.value)}
              size="md"
            >
              {getDiscountOptions().map(option => (
                <option key={option.value} value={option.value}>
                  {option.label}
                </option>
              ))}
            </Select>
          </FormControl>

          <FormControl>
            <FormLabel>
              {formData.discountType === 'percentage' ? 'Discount Percentage' : 'Discount Amount'}
            </FormLabel>
            <InputGroup>
              <InputLeftAddon>
                <Text>{formData.discountType === 'percentage' ? '%' : '$'}</Text>
              </InputLeftAddon>
              <NumberInput
                value={formData.discountValue}
                onChange={value => onChange('discountValue', parseFloat(value) || 0)}
                min={0}
                max={formData.discountType === 'percentage' ? 100 : undefined}
                precision={formData.discountType === 'percentage' ? 0 : 2}
                step={formData.discountType === 'percentage' ? 1 : 0.01}
                w="full"
              >
                <NumberInputField borderLeftRadius={0} />
                <NumberInputStepper>
                  <NumberIncrementStepper />
                  <NumberDecrementStepper />
                </NumberInputStepper>
              </NumberInput>
            </InputGroup>
          </FormControl>

          <HStack spacing={4} justify="flex-end" pt={4}>
            <Button
              key="cancel-button"
              variant="ghost"
              onClick={onCancel}
              size="md"
              leftIcon={<CloseIcon />}
            >
              Cancel
            </Button>
            <Button
              key="save-button"
              colorScheme="primary"
              onClick={onSave}
              size="md"
              isDisabled={!isValidTitle}
              leftIcon={<CheckIcon />}
              shadow="sm"
              _hover={{
                shadow: 'md',
                transform: 'translateY(-1px)'
              }}
            >
              Save Rule
            </Button>
          </HStack>
        </VStack>
      </CardBody>
    </Card>
  );
};

export default EditorForm;