<?php
if ( ! defined( 'ABSPATH' ) ) exit;
global $wpdb;

add_action('admin_menu', 's4rm_admin_menu');
add_action('admin_init', 's4rm_admin_init');
add_action('admin_post_s4rm_save_settings', 's4rm_handle_save_settings');
add_action('admin_post_s4rm_export_logs', 's4rm_handle_export_logs');
add_action('admin_post_s4rm_create_rule', 's4rm_handle_create_rule');
add_action('admin_post_s4rm_delete_rule', 's4rm_handle_delete_rule');
add_action('admin_post_s4rm_clear_logs', 's4rm_handle_clear_logs');

function s4rm_admin_menu() {
    add_options_page('Smart 404 Redirect', 'Smart 404 Redirect', 'manage_options', 's4rm', 's4rm_settings_page');
}

function s4rm_admin_init() {
    // reserved for future enhancements
}

function s4rm_settings_page() {
    if ( ! current_user_can('manage_options') ) return;
    $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'settings';
    ?>
    <div class="wrap">
        <h1>Smart 404 Redirect Manager</h1>
        <h2 class="nav-tab-wrapper">
            <a class="nav-tab <?php echo $tab === 'settings' ? 'nav-tab-active' : ''; ?>" href="?page=s4rm&tab=settings">Settings</a>
            <a class="nav-tab <?php echo $tab === 'logs' ? 'nav-tab-active' : ''; ?>" href="?page=s4rm&tab=logs">404 Logs</a>
            <a class="nav-tab <?php echo $tab === 'rules' ? 'nav-tab-active' : ''; ?>" href="?page=s4rm&tab=rules">Redirect Rules</a>
        </h2>
        <div style="margin-top:20px;">
            <?php
            if ( $tab === 'settings' ) {
                s4rm_render_settings();
            } elseif ( $tab === 'logs' ) {
                s4rm_render_logs();
            } else {
                s4rm_render_rules();
            }
            ?>
        </div>
    </div>
    <?php
}

function s4rm_render_settings() {
    $enabled = get_option('s4rm_global_redirect_enabled', '0');
    $redirect_url = get_option('s4rm_redirect_url', home_url('/'));
    $redirect_type = get_option('s4rm_redirect_type', '301');
    $excluded = get_option('s4rm_excluded_urls', '');
    $loop_threshold = get_option('s4rm_loop_threshold', 5);
    ?>
    <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
        <?php wp_nonce_field('s4rm_save_settings'); ?>
        <input type="hidden" name="action" value="s4rm_save_settings">
        <table class="form-table">
            <tr>
                <th scope="row">Enable Global 404 Redirect</th>
                <td>
                    <input type="checkbox" name="s4rm_global_redirect_enabled" value="1" <?php checked($enabled, '1'); ?> />
                    Redirect 404s to the URL below (unless excluded)
                </td>
            </tr>
            <tr>
                <th scope="row">Redirect Target URL</th>
                <td><input type="text" name="s4rm_redirect_url" value="<?php echo esc_attr($redirect_url); ?>" size="60" /></td>
            </tr>
            <tr>
                <th scope="row">Redirect Type</th>
                <td>
                    <select name="s4rm_redirect_type">
                        <option value="301" <?php selected($redirect_type, '301'); ?>>301 - Permanent</option>
                        <option value="302" <?php selected($redirect_type, '302'); ?>>302 - Temporary</option>
                    </select>
                </td>
            </tr>
            <tr>
                <th scope="row">Excluded URLs / Patterns</th>
                <td>
                    <textarea name="s4rm_excluded_urls" rows="5" cols="60" placeholder="/wp-login.php"><?php echo esc_textarea($excluded); ?></textarea>
                    <p class="description">Add one pattern per line or comma-separated. Simple 'contains' matching is used.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">Loop Protection Threshold</th>
                <td><input type="number" name="s4rm_loop_threshold" value="<?php echo esc_attr($loop_threshold); ?>" min="1" max="20" /></td>
            </tr>
        </table>
        <?php submit_button('Save Settings'); ?>
    </form>
    <?php
}

function s4rm_render_logs() {
    global $wpdb;
    $table = $wpdb->prefix . 's4rm_404_logs';
    $rows = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY hits DESC, last_seen DESC LIMIT 200" );
    ?>
    <p>
        <a class="button" href="<?php echo esc_url( admin_url('admin-post.php?action=s4rm_export_logs') ); ?>">Export CSV</a>
        <a class="button" href="<?php echo esc_url( admin_url('admin-post.php?action=s4rm_clear_logs') ); ?>" onclick="return confirm('Clear all logs?');">Clear Logs</a>
    </p>
    <table class="widefat fixed striped">
        <thead><tr><th>URI</th><th>Hits</th><th>First Seen</th><th>Last Seen</th><th>Referrer</th><th>Action</th></tr></thead>
        <tbody>
        <?php if ( $rows ) {
            foreach ( $rows as $r ) { ?>
                <tr>
                    <td><code><?php echo esc_html($r->request_uri); ?></code></td>
                    <td><?php echo intval($r->hits); ?></td>
                    <td><?php echo esc_html($r->first_seen); ?></td>
                    <td><?php echo esc_html($r->last_seen); ?></td>
                    <td><?php echo esc_html($r->referrer); ?></td>
                    <td>
                        <form style="display:inline-block;margin:0;" method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
                            <?php wp_nonce_field('s4rm_create_rule'); ?>
                            <input type="hidden" name="action" value="s4rm_create_rule" />
                            <input type="hidden" name="old_path" value="<?php echo esc_attr($r->request_uri); ?>" />
                            <input type="text" name="new_url" placeholder="Target URL (required)" size="20" />
                            <select name="type"><option value="301">301</option><option value="302">302</option></select>
                            <input class="button redirect" type="submit" value="Create Redirect" />
                        </form>
                    </td>
                </tr>
            <?php }
        } else {
            echo '<tr><td colspan="6">No logs yet.</td></tr>';
        } ?>
        </tbody>
    </table>
    <?php
}


function s4rm_admin_notice() {
    // Success notice for rule creation
    if ( isset($_GET['created']) && $_GET['created'] == '1' ) {
        echo '<div class="notice notice-success is-dismissible"><p><strong>Redirect created successfully!</strong></p></div>';
    }
    
    // Notice for duplicate rule (NEW)
    if ( isset($_GET['duplicate']) && $_GET['duplicate'] == '1' ) {
        echo '<div class="notice notice-error is-dismissible"><p><strong>Redirect already exists for that old path!</strong> Please delete the existing rule under the "Redirect Rules" tab to set a new one.</p></div>';
    }

    // You might also want a notice for deletion
    if ( isset($_GET['deleted']) && $_GET['deleted'] == '1' ) {
        echo '<div class="notice notice-success is-dismissible"><p><strong>Redirect rule deleted successfully!</strong></p></div>';
    }
    // You might also want a notice for clearing logs
    if ( isset($_GET['cleared']) && $_GET['cleared'] == '1' ) {
        echo '<div class="notice notice-success is-dismissible"><p><strong>404 logs cleared successfully!</strong></p></div>';
    }
}
// Remove the old s4rm_admin_notice_success if it exists, and replace with this hook:
add_action('admin_notices', 's4rm_admin_notice');

function s4rm_render_rules() {
    global $wpdb;
    $table = $wpdb->prefix . 's4rm_redirect_rules';
    $rows = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY created_at DESC" );
    ?>
    <p>Manual redirect rules created from logs or imported.</p>
    <table class="widefat fixed striped">
        <thead><tr><th>Old Path</th><th>New URL</th><th>Type</th><th>Created</th><th>Action</th></tr></thead>
        <tbody>
        <?php if ( $rows ) {
            foreach ( $rows as $r ) { ?>
                <tr>
                    <td><code><?php echo esc_html($r->old_path); ?></code></td>
                    <td><?php echo esc_html($r->new_url); ?></td>
                    <td><?php echo esc_html($r->type); ?></td>
                    <td><?php echo esc_html($r->created_at); ?></td>
                    <td>
                        <form style="display:inline-block;" method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
                            <?php wp_nonce_field('s4rm_delete_rule'); ?>
                            <input type="hidden" name="action" value="s4rm_delete_rule" />
                            <input type="hidden" name="id" value="<?php echo intval($r->id); ?>" />
                            <input class="button" type="submit" value="Delete" onclick="return confirm('Delete this redirect rule?');" />
                        </form>
                    </td>
                </tr>
            <?php }
        } else {
            echo '<tr><td colspan="5">No rules yet.</td></tr>';
        } ?>
        </tbody>
    </table>
    <?php
}

// === Handlers === //

function s4rm_handle_save_settings() {
    if ( ! current_user_can('manage_options') ) wp_die('Permission denied');
    check_admin_referer('s4rm_save_settings');

    $enabled = isset($_POST['s4rm_global_redirect_enabled']) ? '1' : '0';
    update_option('s4rm_global_redirect_enabled', $enabled);

    $url = isset($_POST['s4rm_redirect_url']) ? esc_url_raw(trim($_POST['s4rm_redirect_url'])) : home_url('/');
    update_option('s4rm_redirect_url', $url);

    $type = (isset($_POST['s4rm_redirect_type']) && $_POST['s4rm_redirect_type'] === '302') ? '302' : '301';
    update_option('s4rm_redirect_type', $type);

    $excluded = isset($_POST['s4rm_excluded_urls']) ? sanitize_textarea_field($_POST['s4rm_excluded_urls']) : '';
    update_option('s4rm_excluded_urls', $excluded);

    $threshold = isset($_POST['s4rm_loop_threshold']) ? intval($_POST['s4rm_loop_threshold']) : 5;
    update_option('s4rm_loop_threshold', $threshold);

    wp_redirect( admin_url('options-general.php?page=s4rm&tab=settings&updated=1') );
    exit;
}

function s4rm_handle_export_logs() {
    if ( ! current_user_can('manage_options') ) wp_die('Permission denied');
    global $wpdb;
    $table = $wpdb->prefix . 's4rm_404_logs';
    $rows = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY hits DESC" );

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=s4rm-404-logs.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, array('request_uri','hits','first_seen','last_seen','referrer','ip','user_agent'));
    foreach ( $rows as $r ) {
        fputcsv($out, array($r->request_uri,intval($r->hits),$r->first_seen,$r->last_seen,$r->referrer,$r->ip,$r->user_agent));
    }
    fclose($out);
    exit;
}

function s4rm_handle_create_rule() {
    global $wpdb;
    if ( ! current_user_can('manage_options') ) wp_die('Permission denied');
    check_admin_referer('s4rm_create_rule');

    $old = isset($_POST['old_path']) ? sanitize_text_field($_POST['old_path']) : '';
    $new = isset($_POST['new_url']) ? esc_url_raw(trim($_POST['new_url'])) : '';
    $type = (isset($_POST['type']) && $_POST['type'] === '302') ? '302' : '301';

    if ( empty($old) || empty($new) ) {
        wp_redirect( admin_url('options-general.php?page=s4rm&tab=logs&error=1') );
        exit;
    }

    $table = $wpdb->prefix . 's4rm_redirect_rules';
    
    // **NEW CHECK FOR DUPLICATES**
    $existing = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$table} WHERE old_path = %s LIMIT 1", $old ) );
    
    if ( $existing ) {
        // Rule already exists for this path, redirect back with a notice
        wp_redirect( admin_url('options-general.php?page=s4rm&tab=logs&duplicate=1') );
        exit;
    }
    // **END NEW CHECK**

    $wpdb->insert(
        $table,
        array(
            'old_path' => $old,
            'new_url'  => $new,
            'type'     => $type,
            'created_at' => current_time('mysql')
        ),
        array('%s','%s','%s','%s')
    );

    wp_redirect( admin_url('options-general.php?page=s4rm&tab=rules&created=1') );
    exit;
}

function s4rm_handle_delete_rule() {
    global $wpdb;
    if ( ! current_user_can('manage_options') ) wp_die('Permission denied');
    check_admin_referer('s4rm_delete_rule');

    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    if ( $id ) {
        $table = $wpdb->prefix . 's4rm_redirect_rules';
        $wpdb->delete( $table, array( 'id' => $id ), array('%d') );
    }

    wp_redirect( admin_url('options-general.php?page=s4rm&tab=rules&deleted=1') );
    exit;
}

function s4rm_handle_clear_logs() {
    global $wpdb;
    if ( ! current_user_can('manage_options') ) wp_die('Permission denied');
    $table = $wpdb->prefix . 's4rm_404_logs';
    $wpdb->query( "TRUNCATE TABLE {$table}" );
    wp_redirect( admin_url('options-general.php?page=s4rm&tab=logs&cleared=1') );
    exit;
}
