<?php
/**
 * Plugin Name: Small Business Chatbot
 * Description: Small Business Chatbot - Conversational AI for your website.
 * Version: 1.0.0
 * Author: Small Business Chatbot
 * Author URI: https://smallbusinesschatbot.com
 * License: GPL-2.0+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: small-business-chatbot
 */
// <-- CHANGED: Removed Domain Path header to avoid "nonexistent folder" warning.

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class SmallBusinessChatbotIntegration {

    // Updated these two private property names to be more unique
    private $option_name = 'smallbusinesschatbot_url';
    private $registered_option_name = 'smallbusinesschatbot_registered';

    private $api_signup_url = 'https://smallbusinesschatbot.com/onboarding_create_chatbot';
    private $signup_page = 'https://smallbusinesschatbot.com/signup';
    private $login_page = 'https://smallbusinesschatbot.com/login';

    public function __construct() {
        register_activation_hook( __FILE__, array( $this, 'on_activate' ) );
        register_deactivation_hook( __FILE__, array( $this, 'on_deactivate' ) );

        add_action('admin_menu', array($this, 'add_admin_page'));
        add_action('admin_init', array($this, 'handle_redirect'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));

        // Enqueue chatbot script properly on the front end
        // <-- CHANGED: Instead of injecting <script> in wp_head, we enqueue it.
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_script'));

        add_filter('safe_style_css', array($this, 'allow_custom_inline_styles'));
    }

    public function allow_custom_inline_styles( $allowed_properties ) {
        // List any CSS properties you need (e.g. for flex layouts)
        $extra_properties = array(
            'background',
            'background-color',
            'border',
            'border-radius',
            'color',
            'display',
            'flex',
            'flex-direction',
            'flex-wrap',
            'font-family',
            'font-size',
            'font-weight',
            'gap',
            'height',
            'justify-content',
            'line-height',
            'list-style',
            'margin',
            'margin-bottom',
            'margin-right',
            'margin-top',
            'min-height',
            'padding',
            'text-align',
            'text-decoration',
            'vertical-align',
            'width',
            'align-items',
            'align-self',
        );

        return array_merge( $allowed_properties, $extra_properties );
    }

    public function on_activate() {
        $chatbot_url = get_option($this->option_name);
        if ( empty($chatbot_url) ) {
            $chatbot_url = 'chatbot_' . uniqid();
            update_option($this->option_name, $chatbot_url);
            update_option($this->registered_option_name, false);

            $domain = home_url();
            $response = wp_remote_post($this->api_signup_url, array(
                'headers' => array(
                    'Content-Type' => 'application/json; charset=utf-8'
                ),
                // <-- CHANGED: Use wp_json_encode() instead of json_encode()
                'body' => wp_json_encode(array(
                    'Chatbot_URL'  => $chatbot_url,
                    'Domain'       => $domain,
                    'Signup_Type'  => 'Wordpress'
                ))
            ));
            // Optional: Check response if needed
            // $response_body = wp_remote_retrieve_body($response);
        } else {
            if ( get_option($this->registered_option_name) === false ) {
                update_option($this->registered_option_name, false);
            }
        }
    }

    public function on_deactivate() {
        // *** TEMPORARY CODE TO RESET THE CHATBOT_URL ON DEACTIVATION - REMOVE THIS LATER ***
        delete_option($this->option_name);
        delete_option($this->registered_option_name);
        // *** END TEMPORARY CODE ***
    }

    public function add_admin_page() {
        add_menu_page(
            'Small Business Chatbot',
            'SB Chatbot',
            'manage_options',
            'sb-chatbot',
            array($this, 'admin_page_content'),
            'dashicons-buddicons-buddypress-logo',
            100
        );
    }

    public function handle_redirect() {
        // Check for page=sb-chatbot
        if ( isset($_GET['page']) && $_GET['page'] === 'sb-chatbot' ) {

            // If there's no chatbot_url in the URL, this is likely just a normal page load.
            if ( ! isset($_GET['chatbot_url']) ) {
                return;
            }

            // CHANGED: Sanitize _wpnonce right away to satisfy PHPCS
            $nonce_in = '';
            if ( isset( $_GET['_wpnonce'] ) ) {
                $nonce_in = sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) );
            }

            if ( ! wp_verify_nonce( $nonce_in, 'sb_chatbot_integration' ) ) {
                wp_die( 'Security check failed.', 'Error', array('response' => 403) );
            }

            // CHANGED: Also sanitize chatbot_url right away to satisfy PHPCS
            $new_chatbot_url = '';
            if ( isset( $_GET['chatbot_url'] ) ) {
                $new_chatbot_url = sanitize_text_field( wp_unslash( $_GET['chatbot_url'] ) );
            }

            if ( ! empty( $new_chatbot_url ) ) {
                update_option( $this->option_name, $new_chatbot_url );
                update_option( $this->registered_option_name, true );
            }
        }
    }

    public function enqueue_assets($hook) {
        // Only enqueue on our plugin page
        if ($hook === 'toplevel_page_sb-chatbot') {
            wp_enqueue_style('sb-chatbot-css', plugin_dir_url(__FILE__) . 'assets/style.css', array(), '1.0.0');
            wp_enqueue_script('sb-chatbot-js', plugin_dir_url(__FILE__) . 'assets/script.js', array(), '1.0.0', true);
        }
    }

    public function admin_page_content() {
        $domain = home_url();
        $chatbot_url = get_option($this->option_name);
        $already_registered = get_option($this->registered_option_name, false);
        $nonce = wp_create_nonce('sb_chatbot_integration');

        $signup_url = add_query_arg(
            array(
                'chatbot_url'  => $chatbot_url,
                'domain'       => $domain,
                'signup_type'  => 'Wordpress',
                '_wpnonce'     => $nonce
            ),
            $this->signup_page
        );

        $login_redirect_uri = admin_url('admin.php?page=sb-chatbot');
        $login_url = add_query_arg(
            array(
                'chatbot_url'  => $chatbot_url,
                'domain'       => $domain,
                'signup_type'  => 'Wordpress',
                '_wpnonce'     => $nonce,
            ),
            $this->login_page
        );

        echo '<div class="wrap">';
        echo '<h1>Small Business Chatbot Integration</h1>';

        $template_path = plugin_dir_path(__FILE__) . 'assets/view.html';
        if ( file_exists($template_path) ) {
            // <-- CHANGED: Use WP_Filesystem instead of file_get_contents
            if ( ! function_exists('WP_Filesystem') ) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            WP_Filesystem();
            global $wp_filesystem;

            if ( $wp_filesystem->exists( $template_path ) ) {
                $html = $wp_filesystem->get_contents( $template_path );
                $assets_url = plugin_dir_url(__FILE__) . 'assets/';
                $html = str_replace('{{PLUGIN_ASSETS}}', esc_url($assets_url), $html);

            } else {
                $html = '';
            }

            // Replace placeholders in the HTML for signup/login URLs
            $html = str_replace('{{SIGNUP_URL}}', esc_url($signup_url), $html);
            $html = str_replace('{{LOGIN_URL}}', esc_url($login_url), $html);

            // If user is already registered, override text/buttons
            if ( $already_registered ) {
                // 1) Change the big header text
                $html = str_replace(
                    'Welcome! Your chatbot is being activated.',
                    'Congratulations! Your chatbot is set up.',
                    $html
                );

                // 2) Change the paragraph
                $html = str_replace(
                    'Please sign up or log in to add extra configurations. No credit card required.',
                    'You can now manage your chatbot configurations.',
                    $html
                );

                // 3) Remove the Sign Up button entirely
                $html = preg_replace(
                    '#<a[^>]+href="\{\{SIGNUP_URL\}\}"[^>]*>Sign Up</a>#i',
                    '',
                    $html
                );

                // 4) Rename "Log In" to "Open Platform"
                $html = str_replace('Log In', 'Open Platform', $html);
            }

            $allowed_tags = array(
                'html' => array(
                    'lang' => true,
                ),
                'head' => array(),
                'meta' => array(
                    'charset' => true,
                    'name'    => true,
                    'content' => true,
                ),
                'title' => array(),
                'link' => array(
                    'href' => true,
                    'rel'  => true,
                ),
                'body' => array(
                    'style' => true,
                ),
                'div' => array(
                    'style' => true,
                ),
                'nav' => array(
                    'style' => true,
                ),
                'ul' => array(
                    'style' => true,
                ),
                'li' => array(
                    'style' => true,
                ),
                'h1' => array(
                    'style' => true,
                ),
                'p' => array(
                    'style' => true,
                ),
                'span' => array(
                    'style' => true,
                ),
                'img' => array(
                    'src'   => true,
                    'alt'   => true,
                    'style' => true,
                ),
                'a' => array(
                    'href'   => true,
                    'target' => true,
                    'rel'    => true,
                    'style'  => true,
                ),
            );
            $final_html   = wp_kses( $html, $allowed_tags );
            
            echo $final_html; 

        } else {
            echo '<p>Template not found. Please ensure view.html exists in assets/ directory.</p>';
        }

        echo '</div>';
    }

    // <-- CHANGED: Instead of injecting <script> directly, enqueue it properly.
    public function enqueue_frontend_script() {
        $chatbot_url = get_option($this->option_name);

        // Bail if no chatbot URL
        if ( empty($chatbot_url) ) {
            return;
        }

        // Register the external script
        wp_register_script(
            'sb-chatbot-external',
            'https://smallbusinesschatbot.com/static/install-chatbot-iframe.js',
            array(),
            '1.0.0',
            false // or true to load in the footer
        );

        // Use the script_loader_tag filter to add data-chatbot-url
        add_filter('script_loader_tag', function ( $tag, $handle, $src ) use ( $chatbot_url ) {
            if ( 'sb-chatbot-external' === $handle ) {
                // Rebuild the <script> tag with data-chatbot-url
                $tag = sprintf(
                    '<script src="%s" data-chatbot-url="%s"></script>',
                    esc_url( $src ),
                    esc_attr( $chatbot_url )
                );
            }
            return $tag;
        }, 10, 3);

        // Enqueue it so WP outputs it
        wp_enqueue_script('sb-chatbot-external');

        // (Optional) Remove ?ver=1.0.0 from the final script URL
        add_filter( 'script_loader_src', function ( $src, $handle ) {
            if ( 'sb-chatbot-external' === $handle ) {
                $src = remove_query_arg( 'ver', $src );
            }
            return $src;
        }, 10, 2 );
    }
}

new SmallBusinessChatbotIntegration();
