<?php
/*
Plugin Name: SkySEOManager PRO
Plugin URI: https://siteskyline.com/
Description: A powerful SEO plugin for bulk editing meta titles, descriptions, and featured images
Version: 1.6
Author: Siteskyline
Author URI: https://siteskyline.com/
License: GPL v2 or later
*/

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin file constant for hooks
define('SKYSEOMANAGER_PLUGIN_FILE', __FILE__);

// Define plugin version
define('SKYSEOMANAGER_VERSION', '1.6');

// Include the Media Optimizer functionality
require_once plugin_dir_path(__FILE__) . 'includes/class-sky-media-optimizer.php';


class SkySEOManager {
    private $active_seo_plugin = null;
    private $meta_keys = array();
    private $posts_per_page = 20; // Number of posts to show per page
    private $options = array();
    private $seo_limits = array(
        'title' => array(
            'min' => 30,
            'max' => 60
        ),
        'description' => array(
            'min' => 120,
            'max' => 160
        )
    );
    
    public function __construct() {
        // Then set up admin menus with the appropriate visibility
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_save_bulk_seo_data', array($this, 'save_bulk_seo_data'));
        add_action('wp_ajax_auto_save_seo_data', array($this, 'auto_save_seo_data'));
        add_action('wp_ajax_start_seo_import', array($this, 'start_seo_import'));
        add_action('wp_ajax_ai_optimize_content', array($this, 'ai_optimize_content'));
        
        add_action('add_meta_boxes', array($this, 'add_seo_meta_box'));
        add_action('save_post', array($this, 'save_seo_meta_box'));
        add_action('wp_head', array($this, 'output_head_meta'));
        
        // Add a lower priority to ensure this runs after all submenus are registered
        add_action('admin_menu', array($this, 'reorder_submenus'), 999);
        
        // Load options
        $this->options = get_option('skyseomanager_options', array(
            'post_types' => array('post', 'page'),
            'twitter_site' => '',
            'facebook_page' => '',
            'company_name' => '',
            'logo_url' => '',
            'posts_per_page' => 20,
            'debug_mode' => false,
            'manual_seo_plugin' => '',
            'gemini_api_key' => '',
            'ai_model' => 'gemini-2.0-flash-lite',
            'ai_timeout' => 60,
            'ai_system_prompt' => 'You are an SEO expert. Analyze the following content and generate an optimized title and meta description. The title should be 50-60 characters, and the description should be 150-160 characters. Both should be compelling and include relevant keywords from the content.',
            'media_system_prompt' => 'Analyze this image and generate an appropriate title and alt text description. The title should be concise (3-7 words) and the alt text should be descriptive (10-20 words) for accessibility purposes.'
        ));
        
        // Set posts per page from options
        $this->posts_per_page = isset($this->options['posts_per_page']) ? absint($this->options['posts_per_page']) : 20;
        
        // Try to preload Yoast classes
        $this->ensure_yoast_classes_loaded();
        
        // Check if a manual override is set
        if (!empty($this->options['manual_seo_plugin'])) {
            $this->set_seo_plugin_by_name($this->options['manual_seo_plugin']);
        } else {
            // Detect active SEO plugin and set meta keys based on direct class detection only
            $this->detect_seo_plugin();
        }
        
        // Handle the deletion of other plugin data and manual override
        add_action('admin_init', function() {
            // Handle data deletion
            if (isset($_POST['action']) && $_POST['action'] === 'sky_seo_delete_other_data') {
                $this->handle_delete_other_plugin_data();
            }
            
            // Handle manual override
            if (isset($_POST['action']) && $_POST['action'] === 'sky_seo_manual_override') {
                $this->handle_manual_override();
            }
            
            // Handle force detection
            if (isset($_POST['action']) && $_POST['action'] === 'sky_seo_force_detection') {
                $this->handle_force_detection();
            }
            
            // Handle data deletion for SkySEOManager
            if (isset($_POST['action']) && $_POST['action'] === 'sky_seo_delete_data') {
                $this->handle_data_deletion();
            }
        });
    }
    
    /**
     * Ensures Yoast classes are loaded before detection
     */
    private function ensure_yoast_classes_loaded() {
        // Check if Yoast plugin file exists and try to load it
        $yoast_plugin_path = WP_PLUGIN_DIR . '/wordpress-seo/wp-seo.php';
        $yoast_premium_path = WP_PLUGIN_DIR . '/wordpress-seo-premium/wp-seo-premium.php';
        
        // Check if the constant is defined but classes aren't loaded
        if (defined('WPSEO_VERSION') && !class_exists('WPSEO_Admin', false)) {
            error_log('WPSEO_VERSION constant exists but WPSEO_Admin class not loaded yet.');
            
            if (file_exists($yoast_plugin_path)) {
                error_log('Loading Yoast SEO classes from: ' . $yoast_plugin_path);
                include_once(ABSPATH . 'wp-admin/includes/plugin.php');
                
                // Only check if the plugin is active, don't try to include it directly
                // as this can cause issues with autoloaders
                if (is_plugin_active('wordpress-seo/wp-seo.php')) {
                    error_log('Yoast SEO plugin is active');
                }
            }
        }
    }

    private function detect_seo_plugin() {
        // Default meta keys (our own)
        $this->meta_keys = array(
            'title' => '_sky_seo_title',
            'description' => '_sky_seo_description'
        );
        
        // Make sure plugin.php is loaded for is_plugin_active function
        if (!function_exists('is_plugin_active')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        
        // Check for Yoast SEO
        if (is_plugin_active('wordpress-seo/wp-seo.php') || is_plugin_active('wordpress-seo-premium/wp-seo-premium.php')) {
            error_log('Yoast SEO detected via is_plugin_active');
            $this->active_seo_plugin = 'yoast';
            $this->meta_keys = array(
                'title' => '_yoast_wpseo_title',
                'description' => '_yoast_wpseo_metadesc'
            );
            return;
        }
        
        // Check for All in One SEO
        if (is_plugin_active('all-in-one-seo-pack/all_in_one_seo_pack.php') || is_plugin_active('all-in-one-seo-pack-pro/all_in_one_seo_pack.php')) {
            error_log('All in One SEO detected via is_plugin_active');
            $this->active_seo_plugin = 'aioseo';
            $this->meta_keys = array(
                'title' => '_aioseo_title',
                'description' => '_aioseo_description'
            );
            return;
        }
        
        // Check for Rank Math
        if (is_plugin_active('seo-by-rank-math/rank-math.php') || is_plugin_active('seo-by-rank-math-pro/rank-math-pro.php')) {
            error_log('Rank Math detected via is_plugin_active');
            $this->active_seo_plugin = 'rank_math';
            $this->meta_keys = array(
                'title' => 'rank_math_title',
                'description' => 'rank_math_description'
            );
            return;
        }
        
        // Check for SEOPress
        if (is_plugin_active('wp-seopress/seopress.php') || is_plugin_active('wp-seopress-pro/seopress-pro.php')) {
            error_log('SEOPress detected via is_plugin_active');
            $this->active_seo_plugin = 'seopress';
            $this->meta_keys = array(
                'title' => '_seopress_titles_title',
                'description' => '_seopress_titles_desc'
            );
            return;
        }
        
        // Check for The SEO Framework
        if (is_plugin_active('autodescription/autodescription.php')) {
            error_log('The SEO Framework detected via is_plugin_active');
            $this->active_seo_plugin = 'seoframework';
            $this->meta_keys = array(
                'title' => '_genesis_title',
                'description' => '_genesis_description'
            );
            return;
        }
        
        // Check for SmartCrawl
        if (is_plugin_active('smartcrawl-seo/wpmu-dev-seo.php')) {
            error_log('SmartCrawl detected via is_plugin_active');
            $this->active_seo_plugin = 'smartcrawl';
            $this->meta_keys = array(
                'title' => '_wds_title',
                'description' => '_wds_metadesc'
            );
            return;
        }
        
        // Fallback to constants if no plugin is active
        if (defined('WPSEO_VERSION')) {
            error_log('Yoast SEO detected via WPSEO_VERSION constant');
            $this->active_seo_plugin = 'yoast';
            $this->meta_keys = array(
                'title' => '_yoast_wpseo_title',
                'description' => '_yoast_wpseo_metadesc'
            );
            return;
        } else if (defined('AIOSEO_VERSION')) {
            error_log('All in One SEO detected via AIOSEO_VERSION constant');
            $this->active_seo_plugin = 'aioseo';
            $this->meta_keys = array(
                'title' => '_aioseo_title',
                'description' => '_aioseo_description'
            );
            return;
        } else if (defined('RANK_MATH_VERSION')) {
            error_log('Rank Math detected via RANK_MATH_VERSION constant');
            $this->active_seo_plugin = 'rank_math';
            $this->meta_keys = array(
                'title' => 'rank_math_title',
                'description' => 'rank_math_description'
            );
            return;
        } else if (defined('SEOPRESS_VERSION')) {
            error_log('SEOPress detected via SEOPRESS_VERSION constant');
            $this->active_seo_plugin = 'seopress';
            $this->meta_keys = array(
                'title' => '_seopress_titles_title',
                'description' => '_seopress_titles_desc'
            );
            return;
        } else if (defined('THE_SEO_FRAMEWORK_VERSION')) {
            error_log('The SEO Framework detected via THE_SEO_FRAMEWORK_VERSION constant');
            $this->active_seo_plugin = 'seoframework';
            $this->meta_keys = array(
                'title' => '_genesis_title',
                'description' => '_genesis_description'
            );
            return;
        } else if (defined('SMARTCRAWL_VERSION')) {
            error_log('SmartCrawl detected via SMARTCRAWL_VERSION constant');
            $this->active_seo_plugin = 'smartcrawl';
            $this->meta_keys = array(
                'title' => '_wds_title',
                'description' => '_wds_metadesc'
            );
            return;
        }
        
        // If we get here, no SEO plugin was detected
        error_log('No SEO plugin detected - using default meta keys');
        $this->active_seo_plugin = 'skyseomanager';
    }
    
    /**
     * Check if specific meta keys exist in the database
     * 
     * @param string $meta_key The meta key to check for
     * @return bool True if meta key exists, false otherwise
     */
    private function check_meta_keys_exist($meta_key) {
        global $wpdb;
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $wpdb->postmeta WHERE meta_key = %s LIMIT 1",
            $meta_key
        ));
        return !empty($count);
    }

    /**
     * Check if Rank Math files exist
     * 
     * @return bool True if Rank Math files are found
     */
    private function check_rank_math_files() {
        // Check standard plugin path
        $plugin_path = WP_PLUGIN_DIR . '/seo-by-rank-math/rank-math.php';
        $pro_plugin_path = WP_PLUGIN_DIR . '/seo-by-rank-math-pro/rank-math-pro.php';
        
        if (file_exists($plugin_path) || file_exists($pro_plugin_path)) {
            return true;
        }
        
        // Check for active Rank Math tables in database
        global $wpdb;
        $rank_math_tables = array(
            $wpdb->prefix . 'rank_math_redirections',
            $wpdb->prefix . 'rank_math_redirections_cache'
        );
        
        foreach ($rank_math_tables as $table) {
            $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));
            if ($table_exists) {
                return true;
            }
        }
        
        return false;
    }

    public function register_settings() {
        register_setting(
            'skyseomanager_options',
            'skyseomanager_options',
            array($this, 'sanitize_options')
        );

        add_settings_section(
            'skyseomanager_general_section',
            'General Settings',
            array($this, 'settings_section_callback'),
            'skyseomanager_settings'
        );
        
        add_settings_section(
            'skyseomanager_ai_section',
            'AI Settings',
            array($this, 'ai_settings_section_callback'),
            'skyseomanager_settings'
        );

        add_settings_field(
            'post_types',
            'Post Types',
            array($this, 'post_types_callback'),
            'skyseomanager_settings',
            'skyseomanager_general_section'
        );

        add_settings_field(
            'posts_per_page',
            'Posts Per Page',
            array($this, 'posts_per_page_callback'),
            'skyseomanager_settings',
            'skyseomanager_general_section'
        );
        
        add_settings_field(
            'social_settings',
            'Social Media Handles',
            array($this, 'social_settings_callback'),
            'skyseomanager_settings',
            'skyseomanager_general_section'
        );
        
        add_settings_field(
            'company_settings',
            'Company Information',
            array($this, 'company_settings_callback'),
            'skyseomanager_settings',
            'skyseomanager_general_section'
        );
        
        add_settings_field(
            'debug_mode',
            'Debug Mode',
            array($this, 'debug_mode_callback'),
            'skyseomanager_settings',
            'skyseomanager_general_section'
        );
        
        add_settings_field(
            'gemini_api_key',
            'Gemini API Key',
            array($this, 'gemini_api_key_callback'),
            'skyseomanager_settings',
            'skyseomanager_ai_section'
        );
        
        add_settings_field(
            'ai_model',
            'AI Model',
            array($this, 'ai_model_callback'),
            'skyseomanager_settings',
            'skyseomanager_ai_section'
        );
        
        add_settings_field(
            'ai_timeout',
            'AI Request Timeout (seconds)',
            array($this, 'ai_timeout_callback'),
            'skyseomanager_settings',
            'skyseomanager_ai_section'
        );
        
        add_settings_field(
            'ai_system_prompt',
            'AI Content Prompt',
            array($this, 'ai_system_prompt_callback'),
            'skyseomanager_settings',
            'skyseomanager_ai_section'
        );
        
        add_settings_field(
            'media_system_prompt',
            'AI Media Prompt',
            array($this, 'media_system_prompt_callback'),
            'skyseomanager_settings',
            'skyseomanager_ai_section'
        );
    }

    public function sanitize_options($input) {
        $sanitized = array();
        
        // Post types
        $sanitized['post_types'] = !empty($input['post_types']) ? array_map('sanitize_text_field', $input['post_types']) : array('post', 'page');
        
        // Posts per page
        $sanitized['posts_per_page'] = isset($input['posts_per_page']) ? absint($input['posts_per_page']) : 20;
        if ($sanitized['posts_per_page'] < 1) {
            $sanitized['posts_per_page'] = 20;
        }
        
        // Social settings
        $sanitized['twitter_site'] = isset($input['twitter_site']) ? sanitize_text_field($input['twitter_site']) : '';
        $sanitized['facebook_page'] = isset($input['facebook_page']) ? esc_url_raw($input['facebook_page']) : '';
        
        // Company settings
        $sanitized['company_name'] = isset($input['company_name']) ? sanitize_text_field($input['company_name']) : '';
        $sanitized['logo_url'] = isset($input['logo_url']) ? esc_url_raw($input['logo_url']) : '';
        
        // Debug mode
        $sanitized['debug_mode'] = !empty($input['debug_mode']) ? true : false;
        
        // Manual SEO plugin override
        $sanitized['manual_seo_plugin'] = !empty($input['manual_seo_plugin']) ? sanitize_text_field($input['manual_seo_plugin']) : '';
        
        // AI settings
        $sanitized['gemini_api_key'] = !empty($input['gemini_api_key']) ? sanitize_text_field($input['gemini_api_key']) : '';
        
        // AI Model - accept any model name
        $sanitized['ai_model'] = !empty($input['ai_model']) ? sanitize_text_field($input['ai_model']) : 'gemini-2.0-flash-lite';
        
        // AI Timeout
        $sanitized['ai_timeout'] = isset($input['ai_timeout']) ? absint($input['ai_timeout']) : 60;
        if ($sanitized['ai_timeout'] < 5 || $sanitized['ai_timeout'] > 120) {
            $sanitized['ai_timeout'] = 60; // Ensure timeout is reasonable
        }
        
        // AI Prompts
        $sanitized['ai_system_prompt'] = !empty($input['ai_system_prompt']) 
            ? sanitize_textarea_field($input['ai_system_prompt']) 
            : 'You are an SEO expert. Analyze the following content and generate an optimized title and meta description. The title should be 50-60 characters, and the description should be 150-160 characters. Both should be compelling and include relevant keywords from the content.';
            
        $sanitized['media_system_prompt'] = !empty($input['media_system_prompt']) 
            ? sanitize_textarea_field($input['media_system_prompt']) 
            : 'Analyze this image and generate an appropriate title and alt text description. The title should be concise (3-7 words) and the alt text should be descriptive (10-20 words) for accessibility purposes.';
        
        return $sanitized;
    }

    public function settings_section_callback() {
        echo '<p>Configure your SEO settings below:</p>';
    }
    
    public function ai_settings_section_callback() {
        echo '<p>Configure AI settings for SEO optimization below:</p>';
    }

    public function post_types_callback() {
        $post_types = get_post_types(array('public' => true), 'objects');
        $selected = isset($this->options['post_types']) ? $this->options['post_types'] : array('post', 'page');
        
        foreach ($post_types as $post_type) {
            printf(
                '<label><input type="checkbox" name="skyseomanager_options[post_types][]" value="%s" %s> %s</label><br>',
                esc_attr($post_type->name),
                in_array($post_type->name, $selected) ? 'checked' : '',
                esc_html($post_type->label)
            );
        }
    }

    public function posts_per_page_callback() {
        ?>
        <input type="number" name="skyseomanager_options[posts_per_page]" value="<?php echo esc_attr($this->options['posts_per_page']); ?>" min="1" max="999" step="1">
        <p class="description">Choose how many posts to display per page in the bulk editor. This affects pagination. Default is 20. Maximum is 999.</p>
        <p class="description"><strong>Note:</strong> Higher values may impact performance on large websites.</p>
        <?php
    }

    public function social_settings_callback() {
        ?>
        <p>
            <label>Twitter @username:
                <input type="text" name="skyseomanager_options[twitter_site]" value="<?php echo esc_attr($this->options['twitter_site']); ?>">
            </label>
        </p>
        <p>
            <label>Facebook Page URL:
                <input type="url" name="skyseomanager_options[facebook_page]" value="<?php echo esc_url($this->options['facebook_page']); ?>">
            </label>
        </p>
        <?php
    }

    public function company_settings_callback() {
        ?>
        <p>
            <label>Company Name:
                <input type="text" name="skyseomanager_options[company_name]" value="<?php echo esc_attr($this->options['company_name']); ?>">
            </label>
        </p>
        <p>
            <label>Company Logo URL:
                <input type="url" name="skyseomanager_options[logo_url]" value="<?php echo esc_url($this->options['logo_url']); ?>">
            </label>
            <button type="button" class="button select-logo">Select Logo</button>
        </p>
        <?php
    }

    public function debug_mode_callback() {
        $debug_mode = isset($this->options['debug_mode']) ? $this->options['debug_mode'] : false;
        ?>
        <label>
            <input type="checkbox" name="skyseomanager_options[debug_mode]" value="1" <?php checked($debug_mode, true); ?>>
            Enable debug mode
        </label>
        <p class="description">When enabled, additional debug information will be displayed below in the settings page.</p>
        <?php
    }

    public function gemini_api_key_callback() {
        ?>
        <input type="text" name="skyseomanager_options[gemini_api_key]" value="<?php echo esc_attr($this->options['gemini_api_key']); ?>">
        <p class="description">Enter your Gemini API key here.</p>
        <?php
    }

    public function ai_model_callback() {
        $current_model = isset($this->options['ai_model']) ? $this->options['ai_model'] : 'gemini-2.0-flash-lite';
        ?>
        <input type="text" name="skyseomanager_options[ai_model]" value="<?php echo esc_attr($current_model); ?>" class="regular-text">
        <p class="description">Enter the AI model to use (e.g., gemini-2.0-flash-lite, gemini-2.0-flash, gemini-2.5-flash-preview-04-17)</p>
        <?php
    }

    public function ai_timeout_callback() {
        ?>
        <input type="number" name="skyseomanager_options[ai_timeout]" value="<?php echo esc_attr($this->options['ai_timeout']); ?>" min="1" max="60" step="1">
        <p class="description">Enter your AI request timeout (in seconds) here. Default is 60 seconds.</p>
        <?php
    }

    public function ai_system_prompt_callback() {
        $default_prompt = 'You are an SEO expert. Analyze the following content and generate an optimized title and meta description. The title should be 50-60 characters, and the description should be 150-160 characters. Both should be compelling and include relevant keywords from the content.';
        $system_prompt = isset($this->options['ai_system_prompt']) && !empty($this->options['ai_system_prompt']) ? $this->options['ai_system_prompt'] : $default_prompt;
        ?>
        <textarea name="skyseomanager_options[ai_system_prompt]" rows="4" cols="50" style="width: 100%;"><?php echo esc_textarea($system_prompt); ?></textarea>
        <p class="description">Enter your AI system prompt here. This prompt will be used to generate optimized titles and meta descriptions.</p>
        <?php
    }

    public function media_system_prompt_callback() {
        $default_prompt = 'Analyze this image and generate an appropriate title and alt text description. The title should be concise (3-7 words) and the alt text should be descriptive (10-20 words) for accessibility purposes.';
        $system_prompt = isset($this->options['media_system_prompt']) && !empty($this->options['media_system_prompt']) ? $this->options['media_system_prompt'] : $default_prompt;
        ?>
        <textarea name="skyseomanager_options[media_system_prompt]" rows="4" cols="50" style="width: 100%;"><?php echo esc_textarea($system_prompt); ?></textarea>
        <p class="description">Enter your media optimization prompt here. This prompt will be used to generate optimized titles and alt text descriptions for images.</p>
        <?php
    }

    public function import_seo_data() {
        check_ajax_referer('skyseomanager_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }

        $plugin = isset($_POST['plugin']) ? sanitize_text_field($_POST['plugin']) : '';
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $batch_size = 10; // Process 10 posts at a time
        
        if (empty($plugin)) {
            wp_send_json_error('Invalid plugin specified');
            return;
        }

        global $wpdb;

        // Get meta key mapping based on plugin
        switch ($plugin) {
            case 'yoast':
                $meta_mapping = array(
                    '_yoast_wpseo_title' => $this->meta_keys['title'],
                    '_yoast_wpseo_metadesc' => $this->meta_keys['description']
                );
                break;
            case 'rank_math':
                $meta_mapping = array(
                    'rank_math_title' => $this->meta_keys['title'],
                    'rank_math_description' => $this->meta_keys['description']
                );
                break;
            case 'aioseo':
                $meta_mapping = array(
                    '_aioseo_title' => $this->meta_keys['title'],
                    '_aioseo_description' => $this->meta_keys['description']
                );
                break;
            case 'seopress':
                $meta_mapping = array(
                    '_seopress_titles_title' => $this->meta_keys['title'],
                    '_seopress_titles_desc' => $this->meta_keys['description']
                );
                break;
            case 'seoframework':
                $meta_mapping = array(
                    '_genesis_title' => $this->meta_keys['title'],
                    '_genesis_description' => $this->meta_keys['description']
                );
                break;
            case 'smartcrawl':
                $meta_mapping = array(
                    '_wds_title' => $this->meta_keys['title'],
                    '_wds_metadesc' => $this->meta_keys['description']
                );
                break;
            case 'premium_seo':
                $meta_mapping = array(
                    '_premium_seo_pack_title' => $this->meta_keys['title'],
                    '_premium_seo_pack_description' => $this->meta_keys['description']
                );
                break;
            case 'seo_ultimate':
                $meta_mapping = array(
                    '_su_title' => $this->meta_keys['title'],
                    '_su_description' => $this->meta_keys['description']
                );
                break;
            default:
                wp_send_json_error('Invalid plugin specified');
                return;
        }

        // Get total count for progress calculation
        $total_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT pm.post_id) 
            FROM $wpdb->postmeta pm 
            JOIN $wpdb->posts p ON p.ID = pm.post_id 
            WHERE pm.meta_key IN (" . implode(',', array_fill(0, count($meta_mapping), '%s')) . ")
            AND p.post_status = 'publish'",
            array_keys($meta_mapping)
        ));

        // Get batch of posts to process
        $posts = $wpdb->get_results($wpdb->prepare(
            "SELECT DISTINCT pm.post_id, p.post_title, p.post_type 
            FROM $wpdb->postmeta pm 
            JOIN $wpdb->posts p ON p.ID = pm.post_id 
            WHERE pm.meta_key IN (" . implode(',', array_fill(0, count($meta_mapping), '%s')) . ")
            AND p.post_status = 'publish'
            LIMIT %d, %d",
            array_merge(array_keys($meta_mapping), array($offset, $batch_size))
        ));

        $imported = array();
        foreach ($posts as $post) {
            $post_data = array(
                'title' => $post->post_title,
                'type' => $post->post_type,
                'status' => array()
            );

            foreach ($meta_mapping as $old_key => $new_key) {
                $old_value = get_post_meta($post->post_id, $old_key, true);
                if (!empty($old_value)) {
                    update_post_meta($post->post_id, $new_key, $old_value);
                    $post_data['status'][] = 'Imported ' . (strpos($old_key, 'title') !== false ? 'title' : 'description');
                }
            }

            if (empty($post_data['status'])) {
                $post_data['status'][] = 'No data to import';
            }
            $imported[] = $post_data;
        }

        $progress = min(100, round(($offset + count($posts)) / $total_count * 100));
        $is_complete = ($offset + $batch_size) >= $total_count;

        wp_send_json_success(array(
            'imported' => $imported,
            'progress' => $progress,
            'offset' => $offset + $batch_size,
            'is_complete' => $is_complete,
            'total' => $total_count
        ));
    }

    public function add_admin_menu() {
        // Main menu page callback depends on license status
        $main_callback = array($this, 'render_bulk_edit_page');
            
        // Always add the main menu item
        add_menu_page(
            'SkySEOManager',
            'SkySEOManager',
            'manage_options',
            'skyseomanager',
            $main_callback,
            'dashicons-chart-area',
            30
        );
        
        // Bulk Edit page will be the default submenu
        add_submenu_page(
            'skyseomanager',
            'Bulk Edit',
            'Bulk Edit',
            'manage_options',
            'skyseomanager',
            array($this, 'render_bulk_edit_page')
        );
        
        // Settings submenu
        add_submenu_page(
            'skyseomanager',
            'Settings',
            'Settings',
            'manage_options',
            'skyseomanager_settings',
            array($this, 'render_settings_page')
        );
    }

    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>SkySEOManager Settings</h1>
            
            <?php settings_errors('sky_seo_messages'); ?>
            
            <?php
            // Display SEO plugin detection status
            $seo_plugin_name = $this->get_seo_plugin_name();
            ?>
            <div class="notice notice-info">
                <p><strong>Active SEO Plugin:</strong> <?php echo esc_html($seo_plugin_name); ?></p>
                <p><strong>Meta Keys Being Used:</strong> Title: <code><?php echo esc_html($this->meta_keys['title']); ?></code>, Description: <code><?php echo esc_html($this->meta_keys['description']); ?></code></p>
            </div>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('skyseomanager_options');
                do_settings_sections('skyseomanager_settings');
                submit_button();
                ?>
            </form>
            
            <hr>
            
            <div class="sky-seo-data-cleanup">
                <h2>Data Management</h2>
                <p>Use this button to delete all SEO data created by SkySEOManager. This will not affect data from other SEO plugins like Yoast SEO, Rank Math or All in One SEO.</p>
                
                <form method="post" action="">
                    <?php wp_nonce_field('sky_seo_delete_data', 'sky_seo_delete_nonce'); ?>
                    <input type="hidden" name="action" value="sky_seo_delete_data">
                    <p class="submit">
                        <input type="submit" name="sky_seo_delete_button" class="button button-secondary" value="Delete SkySEOManager Data" onclick="return confirm('Are you sure you want to delete all SEO data created by SkySEOManager? This action cannot be undone.');">
                    </p>
                </form>
            </div>
            
            <hr>
            
            <div class="sky-seo-import-export">
                <h2>Import/Export SEO Data</h2>
                
                <div class="sky-seo-import-section">
                    <h3>Import Data From Other SEO Plugins</h3>
                    <p>Import SEO data from other plugins into SkySEOManager. This will not delete the original data.</p>
                    
                    <div class="import-options">
                        <form id="import-seo-data-form">
                            <?php wp_nonce_field('skyseomanager_nonce', 'import_nonce'); ?>
                            <select name="import_plugin" id="import-plugin">
                                <option value="">Select SEO Plugin</option>
                                <option value="yoast">Yoast SEO</option>
                                <option value="rank_math">Rank Math</option>
                                <option value="aioseo">All In One SEO</option>
                                <option value="seopress">SEOPress</option>
                                <option value="seoframework">The SEO Framework</option>
                                <option value="smartcrawl">SmartCrawl</option>
                                <option value="premium_seo">Premium SEO Pack</option>
                                <option value="seo_ultimate">SEO Ultimate</option>
                            </select>
                            <button type="button" id="start-import" class="button button-primary">Start Import</button>
                        </form>
                        
                        <div id="import-progress-container" style="display: none; margin-top: 10px;">
                            <div class="progress-bar-container" style="background-color: #f0f0f0; height: 20px; width: 100%; border-radius: 4px; overflow: hidden;">
                                <div id="import-progress-bar" style="background-color: #0073aa; height: 100%; width: 0%;"></div>
                            </div>
                            <p id="import-status">Ready to import...</p>
                            <div id="import-results"></div>
                        </div>
                    </div>
                </div>
                
                <div class="sky-seo-cleanup-section" style="margin-top: 20px;">
                    <h3>Delete Data From Other SEO Plugins</h3>
                    <p><strong>Warning:</strong> This will permanently delete SEO data from the selected plugin. Make sure you have a backup before proceeding.</p>
                    
                    <form method="post" action="">
                        <?php wp_nonce_field('sky_seo_delete_other_data', 'sky_seo_delete_other_nonce'); ?>
                        <input type="hidden" name="action" value="sky_seo_delete_other_data">
                        <select name="delete_plugin" id="delete-plugin">
                            <option value="">Select SEO Plugin</option>
                            <option value="yoast">Yoast SEO</option>
                            <option value="rank_math">Rank Math</option>
                            <option value="aioseo">All In One SEO</option>
                            <option value="seopress">SEOPress</option>
                            <option value="seoframework">The SEO Framework</option>
                            <option value="smartcrawl">SmartCrawl</option>
                            <option value="premium_seo">Premium SEO Pack</option>
                            <option value="seo_ultimate">SEO Ultimate</option>
                        </select>
                        <input type="submit" name="sky_seo_delete_other_button" class="button button-secondary" value="Delete Plugin Data" onclick="return confirm('WARNING: This will permanently delete SEO data from the selected plugin. This action cannot be undone. Are you sure you want to proceed?');">
                    </form>
                </div>
            </div>
            
            <hr>
            
            <h3>Manual SEO Plugin Override</h3>
            <p>If automatic detection is not working, you can manually set the SEO plugin to use.</p>
            <form method="post">
                <?php wp_nonce_field('sky_seo_manual_override', 'sky_seo_override_nonce'); ?>
                <input type="hidden" name="action" value="sky_seo_manual_override">
                <select name="seo_plugin" id="seo_plugin">
                    <option value="skyseomanager" <?php selected($this->active_seo_plugin, 'skyseomanager'); ?>>SkySEOManager (Built-in)</option>
                    <option value="rank_math" <?php selected($this->active_seo_plugin, 'rank_math'); ?>>Rank Math SEO</option>
                    <option value="yoast" <?php selected($this->active_seo_plugin, 'yoast'); ?>>Yoast SEO</option>
                    <option value="aioseo" <?php selected($this->active_seo_plugin, 'aioseo'); ?>>All In One SEO</option>
                    <option value="seopress" <?php selected($this->active_seo_plugin, 'seopress'); ?>>SEOPress</option>
                    <option value="seoframework" <?php selected($this->active_seo_plugin, 'seoframework'); ?>>The SEO Framework</option>
                    <option value="smartcrawl" <?php selected($this->active_seo_plugin, 'smartcrawl'); ?>>SmartCrawl</option>
                    <option value="premium_seo" <?php selected($this->active_seo_plugin, 'premium_seo'); ?>>Premium SEO Pack</option>
                    <option value="seo_ultimate" <?php selected($this->active_seo_plugin, 'seo_ultimate'); ?>>SEO Ultimate</option>
                </select>
                <p class="submit">
                    <input type="submit" name="sky_seo_override_button" class="button button-primary" value="Set SEO Plugin Manually">
                </p>
            </form>
            
            <form method="post" action="">
                <?php wp_nonce_field('sky_seo_force_detection', 'sky_seo_force_nonce'); ?>
                <input type="hidden" name="action" value="sky_seo_force_detection">
                <p class="submit">
                    <input type="submit" name="sky_seo_force_button" class="button button-secondary" value="Force Re-detection of SEO Plugins">
                </p>
            </form>
            
            <?php if (isset($this->options['debug_mode']) && $this->options['debug_mode']): ?>
            <div class="sky-seo-debug">
                <h2>Debug Information</h2>
                <p>This information can help troubleshoot plugin detection issues:</p>
                
                <table class="widefat" style="margin-top: 10px;">
                    <tbody>
                        <!-- Yoast SEO Detection -->
                        <tr>
                            <th colspan="2" style="background-color: #f0f0f0;"><h3>Yoast SEO Detection</h3></th>
                        </tr>
                        <tr>
                            <th>WPSEO_VERSION Constant</th>
                            <td><?php echo defined('WPSEO_VERSION') ? esc_html(WPSEO_VERSION) : 'Not defined'; ?></td>
                        </tr>
                        <tr>
                            <th>WPSEO_Admin Class</th>
                            <td><?php echo class_exists('WPSEO_Admin') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>Yoast\WP\SEO\Main Class</th>
                            <td><?php echo class_exists('Yoast\WP\SEO\Main') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>wpseo_init() Function</th>
                            <td><?php echo function_exists('wpseo_init') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>Yoast SEO Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('wordpress-seo/wp-seo.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>Yoast SEO Premium Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('wordpress-seo-premium/wp-seo-premium.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>'_yoast_wpseo_title' Meta Keys</th>
                            <td><?php echo $this->check_meta_keys_exist('_yoast_wpseo_title') ? 'Exist in database' : 'Not found in database'; ?></td>
                        </tr>
                        <tr>
                            <th>Yoast SEO Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/wordpress-seo/wp-seo.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>Yoast SEO Premium Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/wordpress-seo-premium/wp-seo-premium.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        
                        <!-- Rank Math Detection -->
                        <tr>
                            <th colspan="2" style="background-color: #f0f0f0;"><h3>Rank Math SEO Detection</h3></th>
                        </tr>
                        <tr>
                            <th>RANK_MATH_VERSION Constant</th>
                            <td><?php echo defined('RANK_MATH_VERSION') ? esc_html(RANK_MATH_VERSION) : 'Not defined'; ?></td>
                        </tr>
                        <tr>
                            <th>RankMath Class</th>
                            <td><?php echo class_exists('RankMath') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>RankMath\Helper Class</th>
                            <td><?php echo class_exists('RankMath\Helper') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>RankMath\Plugin Class</th>
                            <td><?php echo class_exists('RankMath\\Plugin') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>rank_math() Function</th>
                            <td><?php echo function_exists('rank_math') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>RANK_MATH_FILE Constant</th>
                            <td><?php echo defined('RANK_MATH_FILE') ? esc_html(RANK_MATH_FILE) : 'Not defined'; ?></td>
                        </tr>
                        <tr>
                            <th>SEO by Rank Math Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('seo-by-rank-math/rank-math.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>SEO by Rank Math Pro Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('seo-by-rank-math-pro/rank-math-pro.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>'rank_math_title' Meta Keys</th>
                            <td><?php echo $this->check_meta_keys_exist('rank_math_title') ? 'Exist in database' : 'Not found in database'; ?></td>
                        </tr>
                        <tr>
                            <th>Rank Math Plugin File Exists</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/seo-by-rank-math/rank-math.php') ? 'Yes' : 'No'; ?></td>
                        </tr>
                        <tr>
                            <th>Rank Math Pro Plugin File Exists</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/seo-by-rank-math-pro/rank-math-pro.php') ? 'Yes' : 'No'; ?></td>
                        </tr>
                        <?php
                        // Check for Rank Math tables
                        global $wpdb;
                        $rank_math_tables = array(
                            $wpdb->prefix . 'rank_math_redirections',
                            $wpdb->prefix . 'rank_math_redirections_cache'
                        );
                        
                        foreach ($rank_math_tables as $table): 
                            $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));
                        ?>
                        <tr>
                            <th>Database Table: <?php echo esc_html($table); ?></th>
                            <td><?php echo $table_exists ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <?php endforeach; ?>
                        
                        <!-- All in One SEO Detection -->
                        <tr>
                            <th colspan="2" style="background-color: #f0f0f0;"><h3>All in One SEO Detection</h3></th>
                        </tr>
                        <tr>
                            <th>AIOSEO\Plugin\AIOSEO Class</th>
                            <td><?php echo class_exists('AIOSEO\Plugin\AIOSEO') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>AIOSEO_VERSION Constant</th>
                            <td><?php echo defined('AIOSEO_VERSION') ? esc_html(AIOSEO_VERSION) : 'Not defined'; ?></td>
                        </tr>
                        <tr>
                            <th>All_in_One_SEO_Pack Class</th>
                            <td><?php echo class_exists('All_in_One_SEO_Pack') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>aioseo() Function</th>
                            <td><?php echo function_exists('aioseo') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>All in One SEO Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('all-in-one-seo-pack/all_in_one_seo_pack.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>All in One SEO Pro Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('all-in-one-seo-pack-pro/all_in_one_seo_pack.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>'_aioseo_title' Meta Keys</th>
                            <td><?php echo $this->check_meta_keys_exist('_aioseo_title') ? 'Exist in database' : 'Not found in database'; ?></td>
                        </tr>
                        <tr>
                            <th>All in One SEO Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/all-in-one-seo-pack/all_in_one_seo_pack.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>All in One SEO Pro Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/all-in-one-seo-pack-pro/all_in_one_seo_pack.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        
                        <!-- SEOPress Detection -->
                        <tr>
                            <th colspan="2" style="background-color: #f0f0f0;"><h3>SEOPress Detection</h3></th>
                        </tr>
                        <tr>
                            <th>SEOPRESS_VERSION Constant</th>
                            <td><?php echo defined('SEOPRESS_VERSION') ? esc_html(SEOPRESS_VERSION) : 'Not defined'; ?></td>
                        </tr>
                        <tr>
                            <th>SEOPRESS\Core\Kernel Class</th>
                            <td><?php echo class_exists('SEOPRESS\Core\Kernel') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>seopress_activation() Function</th>
                            <td><?php echo function_exists('seopress_activation') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>SEOPress Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('wp-seopress/seopress.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>SEOPress Pro Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('wp-seopress-pro/seopress-pro.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>'_seopress_titles_title' Meta Keys</th>
                            <td><?php echo $this->check_meta_keys_exist('_seopress_titles_title') ? 'Exist in database' : 'Not found in database'; ?></td>
                        </tr>
                        <tr>
                            <th>SEOPress Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/wp-seopress/seopress.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>SEOPress Pro Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/wp-seopress-pro/seopress-pro.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        
                        <!-- The SEO Framework Detection -->
                        <tr>
                            <th colspan="2" style="background-color: #f0f0f0;"><h3>The SEO Framework Detection</h3></th>
                        </tr>
                        <tr>
                            <th>THE_SEO_FRAMEWORK_VERSION Constant</th>
                            <td><?php echo defined('THE_SEO_FRAMEWORK_VERSION') ? esc_html(THE_SEO_FRAMEWORK_VERSION) : 'Not defined'; ?></td>
                        </tr>
                        <tr>
                            <th>the_seo_framework() Function</th>
                            <td><?php echo function_exists('the_seo_framework') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>The SEO Framework Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('autodescription/autodescription.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>'_genesis_title' Meta Keys</th>
                            <td><?php echo $this->check_meta_keys_exist('_genesis_title') ? 'Exist in database' : 'Not found in database'; ?></td>
                        </tr>
                        <tr>
                            <th>'_tsf_title_no_blogname' Meta Keys</th>
                            <td><?php echo $this->check_meta_keys_exist('_tsf_title_no_blogname') ? 'Exist in database' : 'Not found in database'; ?></td>
                        </tr>
                        <tr>
                            <th>The SEO Framework Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/autodescription/autodescription.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        
                        <!-- SmartCrawl Detection -->
                        <tr>
                            <th colspan="2" style="background-color: #f0f0f0;"><h3>SmartCrawl Detection</h3></th>
                        </tr>
                        <tr>
                            <th>SMARTCRAWL_VERSION Constant</th>
                            <td><?php echo defined('SMARTCRAWL_VERSION') ? esc_html(SMARTCRAWL_VERSION) : 'Not defined'; ?></td>
                        </tr>
                        <tr>
                            <th>SmartCrawl Class</th>
                            <td><?php echo class_exists('SmartCrawl') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        <tr>
                            <th>SmartCrawl Plugin</th>
                            <td><?php echo function_exists('is_plugin_active') && is_plugin_active('smartcrawl-seo/wpmu-dev-seo.php') ? 'Active' : 'Not active'; ?></td>
                        </tr>
                        <tr>
                            <th>'_wds_title' Meta Keys</th>
                            <td><?php echo $this->check_meta_keys_exist('_wds_title') ? 'Exist in database' : 'Not found in database'; ?></td>
                        </tr>
                        <tr>
                            <th>SmartCrawl Plugin File</th>
                            <td><?php echo file_exists(WP_PLUGIN_DIR . '/smartcrawl-seo/wpmu-dev-seo.php') ? 'Exists' : 'Not found'; ?></td>
                        </tr>
                        
                        <tr>
                            <th>Plugin Directory Path</th>
                            <td><?php echo esc_html(WP_PLUGIN_DIR); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>
        <?php
        
        // Handle data deletion if the form was submitted
        if (isset($_POST['action']) && $_POST['action'] === 'sky_seo_delete_data') {
            $this->handle_data_deletion();
        }
        
        // Handle force detection if the form was submitted
        if (isset($_POST['action']) && $_POST['action'] === 'sky_seo_force_detection') {
            $this->handle_force_detection();
        }
        
        // Handle manual override if the form was submitted
        if (isset($_POST['action']) && $_POST['action'] === 'sky_seo_manual_override') {
            $this->handle_manual_override();
        }
    }

    private function has_plugin_data($plugin) {
        global $wpdb;
        
        $meta_key = '';
        switch ($plugin) {
            case 'yoast':
                $meta_key = '_yoast_wpseo_title';
                break;
            case 'rank_math':
                $meta_key = 'rank_math_title';
                break;
            case 'aioseo':
                $meta_key = '_aioseo_title';
                break;
            case 'seopress':
                $meta_key = '_seopress_titles_title';
                break;
            case 'seoframework':
                $meta_key = '_genesis_title';
                break;
            case 'smartcrawl':
                $meta_key = '_wds_title';
                break;
            case 'premium_seo':
                $meta_key = '_premium_seo_pack_title';
                break;
            case 'seo_ultimate':
                $meta_key = '_su_title';
                break;
        }
        
        if (empty($meta_key)) return false;
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $wpdb->postmeta WHERE meta_key = %s",
            $meta_key
        ));
        
        return $count > 0;
    }

    public function enqueue_admin_assets($hook) {
        // Debug information
        $debug_mode = isset($this->options['debug_mode']) && $this->options['debug_mode'];
        $version = $debug_mode ? time() :mt_rand(1000, 9999999);
        
        // Common scripts and styles for all plugin pages
        if (strpos($hook, 'skyseomanager') !== false || in_array($hook, array('post.php', 'post-new.php'))) {
            wp_enqueue_media();
            wp_enqueue_style('skyseomanager-style', plugin_dir_url(__FILE__) . 'assets/css/style.css', array(), $version);
            wp_enqueue_script('skyseomanager-script', plugin_dir_url(__FILE__) . 'assets/js/script.js', array('jquery'), $version, true);
            
            wp_localize_script('skyseomanager-script', 'skyseomanager_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('skyseomanager_nonce'),
                'active_seo_plugin' => $this->active_seo_plugin,
                'debug_mode' => $debug_mode ? 'true' : 'false',
                'import_text' => array(
                    'starting' => __('Starting import process...', 'skyseomanager'),
                    'processing' => __('Processing: ', 'skyseomanager'),
                    'completed' => __('Import completed successfully!', 'skyseomanager'),
                    'error' => __('Error during import.', 'skyseomanager')
                )
            ));
        }
        
        // Media optimizer specific assets
        if ($hook === 'skyseomanager_page_skyseomanager-media') {
            wp_enqueue_style('skyseomanager-media-style', plugin_dir_url(__FILE__) . 'assets/css/media-optimizer.css', array(), $version);
            wp_enqueue_script('skyseomanager-media-script', plugin_dir_url(__FILE__) . 'assets/js/media-optimizer.js', array('jquery'), $version, true);
            
            wp_localize_script('skyseomanager-media-script', 'skyseomanager_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('skyseomanager_nonce'),
                'debug_mode' => $debug_mode ? 'true' : 'false',
                'version' => $version
            ));
            
            // Add console log for debugging - attached directly to the media script
            if ($debug_mode) {
                $debug_script = "
                    console.log('Debug mode enabled for SkySEOManager');
                    console.log('Current hook: skyseomanager_page_skyseomanager-media');
                    console.log('AJAX URL: ' + skyseomanager_ajax.ajax_url);
                    console.log('Nonce: ' + skyseomanager_ajax.nonce);
                ";
                wp_add_inline_script('skyseomanager-media-script', $debug_script);
            }
        }
        
        // Add JavaScript for the import functionality on settings page
        if ($hook === 'skyseomanager_page_skyseomanager_settings') {
            ob_start();
            ?>
            <script type="text/javascript">
            jQuery(document).ready(function($) {
                // Handle import button click
                $('#start-import').on('click', function() {
                    var selectedPlugin = $('#import-plugin').val();
                    if (!selectedPlugin) {
                        alert('Please select an SEO plugin to import from.');
                        return;
                    }
                    
                    // Show progress container
                    $('#import-progress-container').show();
                    $('#import-status').text(skyseomanager_ajax.import_text.starting);
                    $('#import-progress-bar').css('width', '0%');
                    $('#import-results').empty();
                    
                    // Start the import process
                    startImport(selectedPlugin, 0);
                });
                
                function startImport(plugin, offset) {
                    $.ajax({
                        url: skyseomanager_ajax.ajax_url,
                        type: 'POST',
                        data: {
                            action: 'start_seo_import',
                            plugin: plugin,
                            offset: offset,
                            nonce: skyseomanager_ajax.nonce
                        },
                        success: function(response) {
                            if (response.success) {
                                // Update progress bar
                                $('#import-progress-bar').css('width', response.data.progress + '%');
                                $('#import-status').text(skyseomanager_ajax.import_text.processing + ' ' + response.data.progress + '%');
                                
                                // Add imported items to results
                                if (response.data.imported && response.data.imported.length > 0) {
                                    var resultHtml = '';
                                    $.each(response.data.imported, function(index, item) {
                                        resultHtml += '<div class="import-result-item">';
                                        resultHtml += '<strong>' + item.title + '</strong> (' + item.type + '): ';
                                        resultHtml += item.status.join(', ');
                                        resultHtml += '</div>';
                                    });
                                    $('#import-results').prepend(resultHtml);
                                }
                                
                                // Continue import if not complete
                                if (!response.data.is_complete) {
                                    startImport(plugin, response.data.offset);
                                } else {
                                    // Import completed
                                    $('#import-status').text(skyseomanager_ajax.import_text.completed);
                                }
                            } else {
                                $('#import-status').text(skyseomanager_ajax.import_text.error);
                                if (response.data && response.data.message) {
                                    alert(response.data.message);
                                }
                            }
                        },
                        error: function() {
                            $('#import-status').text(skyseomanager_ajax.import_text.error);
                            alert('An error occurred during the import process. Please try again.');
                        }
                    });
                }
            });
            </script>
            <?php
            $script = ob_get_clean();
            wp_add_inline_script('skyseomanager-script', $script);
        }
    }

    public function get_seo_meta_title($post_id) {
        $title = get_post_meta($post_id, $this->meta_keys['title'], true);
        
        // Special handling for different SEO plugins
        switch ($this->active_seo_plugin) {
            case 'yoast':
                if (empty($title)) {
                    $title = get_post_meta($post_id, '_yoast_wpseo_title', true);
                }
                break;
            case 'rank_math':
                if (empty($title)) {
                    $title = get_post_meta($post_id, 'rank_math_title', true);
                }
                break;
            case 'aioseo':
                if (empty($title)) {
                    $title = get_post_meta($post_id, '_aioseo_title', true);
                }
                break;
            case 'seopress':
                if (empty($title)) {
                    $title = get_post_meta($post_id, '_seopress_titles_title', true);
                }
                break;
            case 'seoframework':
                if (empty($title)) {
                    $title = get_post_meta($post_id, '_genesis_title', true);
                    // Try alternative meta key if empty
                    if (empty($title)) {
                        $title = get_post_meta($post_id, '_tsf_title_no_blogname', true);
                    }
                }
                break;
            case 'smartcrawl':
                if (empty($title)) {
                    $title = get_post_meta($post_id, '_wds_title', true);
                }
                break;
            case 'premium_seo':
                if (empty($title)) {
                    $title = get_post_meta($post_id, '_premium_seo_pack_title', true);
                }
                break;
            case 'seo_ultimate':
                if (empty($title)) {
                    $title = get_post_meta($post_id, '_su_title', true);
                }
                break;
        }
        
        return $title;
    }

    public function get_seo_meta_description($post_id) {
        $description = get_post_meta($post_id, $this->meta_keys['description'], true);
        
        // Special handling for different SEO plugins
        switch ($this->active_seo_plugin) {
            case 'yoast':
                if (empty($description)) {
                    $description = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
                }
                break;
            case 'rank_math':
                if (empty($description)) {
                    $description = get_post_meta($post_id, 'rank_math_description', true);
                }
                break;
            case 'aioseo':
                if (empty($description)) {
                    $description = get_post_meta($post_id, '_aioseo_description', true);
                }
                break;
            case 'seopress':
                if (empty($description)) {
                    $description = get_post_meta($post_id, '_seopress_titles_desc', true);
                }
                break;
            case 'seoframework':
                if (empty($description)) {
                    $description = get_post_meta($post_id, '_genesis_description', true);
                    // Try alternative meta key if empty
                    if (empty($description)) {
                        $description = get_post_meta($post_id, '_tsf_description_no_blogname', true);
                    }
                }
                break;
            case 'smartcrawl':
                if (empty($description)) {
                    $description = get_post_meta($post_id, '_wds_metadesc', true);
                }
                break;
            case 'premium_seo':
                if (empty($description)) {
                    $description = get_post_meta($post_id, '_premium_seo_pack_description', true);
                }
                break;
            case 'seo_ultimate':
                if (empty($description)) {
                    $description = get_post_meta($post_id, '_su_description', true);
                }
                break;
        }
        
        return $description;
    }

    public function render_bulk_edit_page() {
        $post_types = get_post_types(array('public' => true), 'objects');
        $current_post_type = isset($_GET['post_type']) ? sanitize_text_field($_GET['post_type']) : 'post';
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $current_filter = isset($_GET['seo_filter']) ? sanitize_text_field($_GET['seo_filter']) : '';
        
        // Get total number of posts
        $total_posts = wp_count_posts($current_post_type)->publish;
        $total_pages = ceil($total_posts / $this->posts_per_page);
        
        // Get posts of the selected type with pagination
        $args = array(
            'post_type' => $current_post_type,
            'posts_per_page' => $this->posts_per_page,
            'offset' => ($current_page - 1) * $this->posts_per_page,
            'post_status' => 'publish',
            'suppress_filters' => false // Important for our custom WHERE clause to work
        );

        // Apply filters
        if ($current_filter) {
            switch ($current_filter) {
                case 'title_too_long':
                    add_filter('posts_where', array($this, 'filter_title_too_long'));
                    break;
                case 'desc_too_long':
                    add_filter('posts_where', array($this, 'filter_desc_too_long'));
                    break;
                default:
                    $meta_query = array('relation' => 'AND');
                    
                    switch ($current_filter) {
                        case 'no_title':
                            $meta_query[] = array(
                                'relation' => 'OR',
                                array(
                                    'key' => $this->meta_keys['title'],
                                    'compare' => 'NOT EXISTS'
                                ),
                                array(
                                    'key' => $this->meta_keys['title'],
                                    'value' => '',
                                    'compare' => '='
                                )
                            );
                            break;
                        case 'no_desc':
                            $meta_query[] = array(
                                'relation' => 'OR',
                                array(
                                    'key' => $this->meta_keys['description'],
                                    'compare' => 'NOT EXISTS'
                                ),
                                array(
                                    'key' => $this->meta_keys['description'],
                                    'value' => '',
                                    'compare' => '='
                                )
                            );
                            break;
                        case 'no_image':
                            $meta_query[] = array(
                                'key' => '_thumbnail_id',
                                'compare' => 'NOT EXISTS'
                            );
                            break;
                        case 'title_too_short':
                            $meta_query[] = array(
                                'key' => $this->meta_keys['title'],
                                'value' => '^.{1,' . ($this->seo_limits['title']['min'] - 1) . '}$',
                                'compare' => 'REGEXP'
                            );
                            break;
                        case 'desc_too_short':
                            $meta_query[] = array(
                                'key' => $this->meta_keys['description'],
                                'value' => '^.{1,' . ($this->seo_limits['description']['min'] - 1) . '}$',
                                'compare' => 'REGEXP'
                            );
                            break;
                    }
                    
                    if (!empty($meta_query)) {
                        $args['meta_query'] = $meta_query;
                    }
            }
        }

        $posts = get_posts($args);

        // Remove our custom filters after the query
        if ($current_filter === 'title_too_long') {
            remove_filter('posts_where', array($this, 'filter_title_too_long'));
        } elseif ($current_filter === 'desc_too_long') {
            remove_filter('posts_where', array($this, 'filter_desc_too_long'));
        }

        $seo_plugin_name = $this->get_seo_plugin_name();
        
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">SkySEOManager - Bulk Edit</h1>
            
            <div class="notice notice-info">
                <p>Currently using meta fields from: <strong><?php echo esc_html($seo_plugin_name); ?></strong></p>
            </div>

            <!-- Toast Message Container -->
            <div id="sky-seo-toast" class="sky-seo-toast"></div>
            
            <div class="tablenav top">
                <form method="get" class="seo-filters alignleft actions">
                    <input type="hidden" name="page" value="skyseomanager">
                    
                    <select name="post_type" id="post_type" class="postform">
                        <?php foreach ($post_types as $type): ?>
                            <option value="<?php echo esc_attr($type->name); ?>" <?php selected($current_post_type, $type->name); ?>>
                                <?php echo esc_html($type->labels->name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>

                    <select name="seo_filter" id="seo_filter" class="postform">
                        <option value="">All Posts</option>
                        <optgroup label="Missing Content">
                            <option value="no_title" <?php selected($current_filter, 'no_title'); ?>>No Meta Title</option>
                            <option value="no_desc" <?php selected($current_filter, 'no_desc'); ?>>No Meta Description</option>
                            <option value="no_image" <?php selected($current_filter, 'no_image'); ?>>No Featured Image</option>
                        </optgroup>
                        <optgroup label="Title Issues">
                            <option value="title_too_short" <?php selected($current_filter, 'title_too_short'); ?>>Title Too Short (< <?php echo $this->seo_limits['title']['min']; ?> chars)</option>
                            <option value="title_too_long" <?php selected($current_filter, 'title_too_long'); ?>>Title Too Long (> <?php echo $this->seo_limits['title']['max']; ?> chars)</option>
                        </optgroup>
                        <optgroup label="Description Issues">
                            <option value="desc_too_short" <?php selected($current_filter, 'desc_too_short'); ?>>Description Too Short (< <?php echo $this->seo_limits['description']['min']; ?> chars)</option>
                            <option value="desc_too_long" <?php selected($current_filter, 'desc_too_long'); ?>>Description Too Long (> <?php echo $this->seo_limits['description']['max']; ?> chars)</option>
                        </optgroup>
                    </select>

                    <input type="submit" class="button action" value="Apply Filters">
                </form>
                
                <div class="alignright">
                    <button id="ai-optimize-button" class="button button-primary">AI Optimize</button>
                    <span id="ai-optimize-status" style="display:none;margin-left:10px;"></span>
                </div>
            </div>

            <div class="wp-list-table-wrap">
                <table class="wp-list-table widefat fixed striped table-view-list">
                    <thead>
                        <tr>
                            <th scope="col" class="column-id" style="width: 60px;">ID</th>
                            <th scope="col" class="column-title" style="width: 15%;">Post Title</th>
                            <th scope="col" class="column-meta-title">Meta Title <span class="help-tip" title="Recommended: <?php echo $this->seo_limits['title']['min']; ?>-<?php echo $this->seo_limits['title']['max']; ?> characters">?</span></th>
                            <th scope="col" class="column-meta-desc">Meta Description <span class="help-tip" title="Recommended: <?php echo $this->seo_limits['description']['min']; ?>-<?php echo $this->seo_limits['description']['max']; ?> characters">?</span></th>
                            <th scope="col" class="column-featured-image" style="width: 120px;">Featured Image</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($posts as $post): ?>
                            <?php 
                            $meta_title = $this->get_seo_meta_title($post->ID);
                            $meta_desc = $this->get_seo_meta_description($post->ID);
                            $featured_image = get_the_post_thumbnail_url($post->ID);
                            $featured_image_id = get_post_thumbnail_id($post->ID);
                            
                            // Calculate SEO scores
                            $title_length = mb_strlen($meta_title);
                            $desc_length = mb_strlen($meta_desc);
                            
                            $title_score = $this->get_seo_score($title_length, 'title');
                            $desc_score = $this->get_seo_score($desc_length, 'description');
                            ?>
                            <tr data-post-id="<?php echo esc_attr($post->ID); ?>">
                                <td><?php echo esc_html($post->ID); ?></td>
                                <td><?php echo esc_html($post->post_title); ?></td>
                                <td>
                                    <div class="seo-field-container">
                                        <input type="text" class="seo-title" value="<?php echo esc_attr($meta_title); ?>" placeholder="Enter SEO title">
                                        <span class="seo-score <?php echo esc_attr($title_score); ?>" title="Character count: <?php echo $title_length; ?>"></span>
                                        <span class="character-count"><?php echo $title_length; ?>/<?php echo $this->seo_limits['title']['max']; ?></span>
                                    </div>
                                </td>
                                <td>
                                    <div class="seo-field-container">
                                        <textarea class="seo-description" placeholder="Enter meta description"><?php echo esc_textarea($meta_desc); ?></textarea>
                                        <span class="seo-score <?php echo esc_attr($desc_score); ?>" title="Character count: <?php echo $desc_length; ?>"></span>
                                        <span class="character-count"><?php echo $desc_length; ?>/<?php echo $this->seo_limits['description']['max']; ?></span>
                                    </div>
                                </td>
                                <td>
                                    <div class="featured-image-container">
                                        <div class="image-preview" data-image-id="<?php echo esc_attr($featured_image_id); ?>">
                                            <?php if ($featured_image): ?>
                                                <img src="<?php echo esc_url($featured_image); ?>" alt="Featured image" style="max-width: 100px;">
                                            <?php endif; ?>
                                        </div>
                                        <button class="button select-image" data-post-id="<?php echo esc_attr($post->ID); ?>">
                                            <?php echo $featured_image ? 'Change Image' : 'Select Image'; ?>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <span class="displaying-num"><?php echo esc_html($total_posts); ?> items</span>
                            <span class="pagination-links">
                                <?php
                                echo paginate_links(array(
                                    'base' => add_query_arg('paged', '%#%'),
                                    'format' => '',
                                    'prev_text' => __('&laquo;'),
                                    'next_text' => __('&raquo;'),
                                    'total' => $total_pages,
                                    'current' => $current_page
                                ));
                                ?>
                            </span>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    public function filter_title_too_long($where) {
        global $wpdb;
        $where .= $wpdb->prepare(
            " AND EXISTS (
                SELECT 1 FROM $wpdb->postmeta 
                WHERE $wpdb->postmeta.post_id = $wpdb->posts.ID 
                AND $wpdb->postmeta.meta_key = %s 
                AND LENGTH($wpdb->postmeta.meta_value) > %d
            )",
            $this->meta_keys['title'],
            $this->seo_limits['title']['max']
        );
        return $where;
    }

    public function filter_desc_too_long($where) {
        global $wpdb;
        $where .= $wpdb->prepare(
            " AND EXISTS (
                SELECT 1 FROM $wpdb->postmeta 
                WHERE $wpdb->postmeta.post_id = $wpdb->posts.ID 
                AND $wpdb->postmeta.meta_key = %s 
                AND LENGTH($wpdb->postmeta.meta_value) > %d
            )",
            $this->meta_keys['description'],
            $this->seo_limits['description']['max']
        );
        return $where;
    }

    private function get_seo_plugin_name() {
        switch ($this->active_seo_plugin) {
            case 'yoast':
                return 'Yoast SEO';
            case 'rank_math':
                return 'Rank Math SEO';
            case 'aioseo':
                return 'All in One SEO';
            case 'seopress':
                return 'SEOPress';
            case 'seoframework':
                return 'The SEO Framework';
            case 'smartcrawl':
                return 'SmartCrawl';
            case 'premium_seo':
                return 'Premium SEO Pack';
            case 'seo_ultimate':
                return 'SEO Ultimate';
            default:
                return 'SkySEOManager (Built-in)';
        }
    }

    public function save_bulk_seo_data() {
        check_ajax_referer('skyseomanager_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }

        $posts_data = isset($_POST['posts_data']) ? $_POST['posts_data'] : array();

        foreach ($posts_data as $post_data) {
            $post_id = absint($post_data['post_id']);
            
            if ($post_id) {
                // Update meta title
                if (isset($post_data['seo_title'])) {
                    update_post_meta($post_id, $this->meta_keys['title'], sanitize_text_field($post_data['seo_title']));
                }
                
                // Update meta description
                if (isset($post_data['seo_description'])) {
                    update_post_meta($post_id, $this->meta_keys['description'], sanitize_textarea_field($post_data['seo_description']));
                }
                
                // Update featured image
                if (isset($post_data['featured_image_id'])) {
                    set_post_thumbnail($post_id, absint($post_data['featured_image_id']));
                }
            }
        }

        wp_send_json_success('Changes saved successfully');
    }

    public function auto_save_seo_data() {
        // Check nonce and permissions
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'skyseomanager_nonce')) {
            wp_send_json_error('Security check failed');
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }

        $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
        $field_type = isset($_POST['field_type']) ? sanitize_text_field($_POST['field_type']) : '';
        $value = isset($_POST['value']) ? $_POST['value'] : '';

        if (!$post_id || !$field_type) {
            wp_send_json_error('Missing required data');
            return;
        }

        // Debug - Log meta keys and values
        error_log('Auto save called for post_id: ' . $post_id . ', field_type: ' . $field_type);
        error_log('Current meta_keys in class: ' . print_r($this->meta_keys, true));

        switch ($field_type) {
            case 'title':
                $meta_key = $this->meta_keys['title'];
                $value = sanitize_text_field($value);
                
                // Debug - Previous value
                $prev_value = get_post_meta($post_id, $meta_key, true);
                error_log('Previous title value: ' . $prev_value);
                
                $result = update_post_meta($post_id, $meta_key, $value);
                
                // Debug - New value and result
                error_log('New title value: ' . $value . ', Update result: ' . var_export($result, true));
                break;
                
            case 'description':
                $meta_key = $this->meta_keys['description'];
                $value = sanitize_textarea_field($value);
                
                // Debug - Previous value
                $prev_value = get_post_meta($post_id, $meta_key, true);
                error_log('Previous description value: ' . $prev_value);
                
                $result = update_post_meta($post_id, $meta_key, $value);
                
                // Debug - New value and result
                error_log('New description value: ' . $value . ', Update result: ' . var_export($result, true));
                break;
                
            case 'featured_image_id':
                // Debug - Previous featured image
                $prev_image_id = get_post_thumbnail_id($post_id);
                error_log('Previous featured image ID: ' . $prev_image_id);
                
                $result = set_post_thumbnail($post_id, absint($value));
                
                // Debug - New featured image and result
                error_log('New featured image ID: ' . absint($value) . ', Update result: ' . var_export($result, true));
                break;
                
            default:
                wp_send_json_error('Invalid field type');
                return;
        }

        if ($result !== false) {
            // Double check if the data was actually saved
            $verification = '';
            if ($field_type === 'title' || $field_type === 'description') {
                $verification = get_post_meta($post_id, $meta_key, true);
                error_log('Verification - ' . $field_type . ' was saved as: ' . $verification);
            } else if ($field_type === 'featured_image_id') {
                $verification = get_post_thumbnail_id($post_id);
                error_log('Verification - featured image was saved as ID: ' . $verification);
            }
            
            wp_send_json_success(array(
                'message' => 'Data saved successfully',
                'post_id' => $post_id,
                'field' => $field_type,
                'value' => $value,
                'verification' => $verification
            ));
        } else {
            error_log('Failed to save data for post_id: ' . $post_id . ', field_type: ' . $field_type);
            wp_send_json_error('Failed to save data');
        }
    }

    private function get_seo_score($length, $type) {
        if ($length === 0) {
            return 'empty';
        }
        
        $min = $this->seo_limits[$type]['min'];
        $max = $this->seo_limits[$type]['max'];
        
        if ($length < $min) {
            return 'warning';
        } elseif ($length > $max) {
            return 'error';
        } else {
            return 'success';
        }
    }

    public function add_seo_meta_box() {
        // Only add our own meta boxes if no other SEO plugin is active
        if ($this->has_active_seo_plugin()) {
            return;
        }
        
        $enabled_post_types = isset($this->options['post_types']) ? $this->options['post_types'] : array('post', 'page');
        
        foreach ($enabled_post_types as $post_type) {
            add_meta_box(
                'skyseomanager_seo_box',
                'SEO Settings',
                array($this, 'render_seo_meta_box'),
                $post_type,
                'normal',
                'low'
            );
        }
    }

    public function render_seo_meta_box($post) {
        wp_nonce_field('skyseomanager_meta_box', 'skyseomanager_meta_box_nonce');

        $meta_title = $this->get_seo_meta_title($post->ID);
        $meta_desc = $this->get_seo_meta_description($post->ID);
        
        $title_length = mb_strlen($meta_title);
        $desc_length = mb_strlen($meta_desc);
        
        $title_score = $this->get_seo_score($title_length, 'title');
        $desc_score = $this->get_seo_score($desc_length, 'description');
        ?>
        <div class="sky-seo-meta-box">
            <p>
                <label for="sky_seo_title">Meta Title:</label>
                <div class="seo-field-container">
                    <input type="text" id="sky_seo_title" name="sky_seo_title" value="<?php echo esc_attr($meta_title); ?>" class="widefat">
                    <span class="seo-score <?php echo esc_attr($title_score); ?>"></span>
                    <span class="character-count"><?php echo $title_length; ?>/<?php echo $this->seo_limits['title']['max']; ?></span>
                </div>
            </p>
            <p>
                <label for="sky_seo_description">Meta Description:</label>
                <div class="seo-field-container">
                    <textarea id="sky_seo_description" name="sky_seo_description" class="widefat" rows="3"><?php echo esc_textarea($meta_desc); ?></textarea>
                    <span class="seo-score <?php echo esc_attr($desc_score); ?>"></span>
                    <span class="character-count"><?php echo $desc_length; ?>/<?php echo $this->seo_limits['description']['max']; ?></span>
                </div>
            </p>
        </div>
        <?php
    }

    public function save_seo_meta_box($post_id) {
        if (!isset($_POST['skyseomanager_meta_box_nonce'])) {
            return;
        }

        if (!wp_verify_nonce($_POST['skyseomanager_meta_box_nonce'], 'skyseomanager_meta_box')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        if (isset($_POST['sky_seo_title'])) {
            update_post_meta($post_id, $this->meta_keys['title'], sanitize_text_field($_POST['sky_seo_title']));
        }

        if (isset($_POST['sky_seo_description'])) {
            update_post_meta($post_id, $this->meta_keys['description'], sanitize_textarea_field($_POST['sky_seo_description']));
        }
    }

    public function output_head_meta() {
        if (!is_singular()) {
            return;
        }

        $post_id = get_queried_object_id();
        $meta_title = $this->get_seo_meta_title($post_id);
        $meta_desc = $this->get_seo_meta_description($post_id);
        $featured_image = get_the_post_thumbnail_url($post_id, 'large');

        // Basic meta tags
        if ($meta_title) {
            echo '<title>' . esc_html($meta_title) . '</title>' . "\n";
            echo '<meta name="title" content="' . esc_attr($meta_title) . '">' . "\n";
        }
        if ($meta_desc) {
            echo '<meta name="description" content="' . esc_attr($meta_desc) . '">' . "\n";
        }

        // OpenGraph tags
        echo '<meta property="og:type" content="' . (is_front_page() ? 'website' : 'article') . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url(get_permalink($post_id)) . '">' . "\n";
        if ($meta_title) {
            echo '<meta property="og:title" content="' . esc_attr($meta_title) . '">' . "\n";
        }
        if ($meta_desc) {
            echo '<meta property="og:description" content="' . esc_attr($meta_desc) . '">' . "\n";
        }
        if ($featured_image) {
            echo '<meta property="og:image" content="' . esc_url($featured_image) . '">' . "\n";
        }
        if (!empty($this->options['facebook_page'])) {
            echo '<meta property="article:publisher" content="' . esc_url($this->options['facebook_page']) . '">' . "\n";
        }

        // Twitter Card tags
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        if (!empty($this->options['twitter_site'])) {
            echo '<meta name="twitter:site" content="@' . esc_attr(ltrim($this->options['twitter_site'], '@')) . '">' . "\n";
        }
        if ($meta_title) {
            echo '<meta name="twitter:title" content="' . esc_attr($meta_title) . '">' . "\n";
        }
        if ($meta_desc) {
            echo '<meta name="twitter:description" content="' . esc_attr($meta_desc) . '">' . "\n";
        }
        if ($featured_image) {
            echo '<meta name="twitter:image" content="' . esc_url($featured_image) . '">' . "\n";
        }

        // Schema.org JSON-LD
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => is_front_page() ? 'WebSite' : 'Article',
            'headline' => $meta_title ?: get_the_title($post_id),
            'description' => $meta_desc,
            'url' => get_permalink($post_id),
            'datePublished' => get_the_date('c', $post_id),
            'dateModified' => get_the_modified_date('c', $post_id)
        );

        if ($featured_image) {
            $schema['image'] = array(
                '@type' => 'ImageObject',
                'url' => $featured_image
            );
        }

        if (!empty($this->options['company_name'])) {
            $schema['publisher'] = array(
                '@type' => 'Organization',
                'name' => $this->options['company_name']
            );

            if (!empty($this->options['logo_url'])) {
                $schema['publisher']['logo'] = array(
                    '@type' => 'ImageObject',
                    'url' => $this->options['logo_url']
                );
            }
        }

        echo '<script type="application/ld+json">' . PHP_EOL;
        echo json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        echo PHP_EOL . '</script>' . PHP_EOL;
    }

    // Helper function to check if any supported SEO plugin is active
    private function has_active_seo_plugin() {
        return $this->active_seo_plugin !== 'skyseomanager';
    }

    private function handle_data_deletion() {
        // Verify nonce
        if (!isset($_POST['sky_seo_delete_nonce']) || !wp_verify_nonce($_POST['sky_seo_delete_nonce'], 'sky_seo_delete_data')) {
            add_settings_error('sky_seo_messages', 'sky_seo_nonce_error', 'Security verification failed. Please try again.', 'error');
            return;
        }
        
        global $wpdb;
        
        // Delete all posts' meta data created by SkySEOManager
        $deleted_title = $wpdb->query("DELETE FROM $wpdb->postmeta WHERE meta_key = '_sky_seo_title'");
        $deleted_desc = $wpdb->query("DELETE FROM $wpdb->postmeta WHERE meta_key = '_sky_seo_description'");
        
        $total_deleted = $deleted_title + $deleted_desc;
        
        // Add success message
        add_settings_error(
            'sky_seo_messages',
            'sky_seo_data_deleted',
            sprintf('SkySEOManager data deleted successfully. %d meta fields removed.', $total_deleted),
            'success'
        );
        
        // Make sure the messages are displayed
        add_action('admin_notices', function() {
            settings_errors('sky_seo_messages');
        });
    }

    private function handle_force_detection() {
        // Verify nonce
        if (!isset($_POST['sky_seo_force_nonce']) || !wp_verify_nonce($_POST['sky_seo_force_nonce'], 'sky_seo_force_detection')) {
            add_settings_error('sky_seo_messages', 'sky_seo_force_error', 'Security verification failed.', 'error');
            return;
        }

        if (!current_user_can('manage_options')) {
            add_settings_error('sky_seo_messages', 'sky_seo_permission_error', 'You do not have sufficient permissions to perform this action.', 'error');
            return;
        }

        // Reset the detection logic
        $this->active_seo_plugin = null;
        $this->meta_keys = array();
        
        // Clear manual override setting
        $this->options['manual_seo_plugin'] = '';
        update_option('skyseomanager_options', $this->options);
        
        // Run detection again
        $this->detect_seo_plugin();

        add_settings_error('sky_seo_messages', 'sky_seo_force_success', 'SEO plugin detection completed. Detected: ' . $this->get_seo_plugin_name(), 'success');
    }

    private function handle_manual_override() {
        // Verify nonce
        if (!isset($_POST['sky_seo_override_nonce']) || !wp_verify_nonce($_POST['sky_seo_override_nonce'], 'sky_seo_manual_override')) {
            add_settings_error('sky_seo_messages', 'sky_seo_security_error', 'Security verification failed. Please try again.', 'error');
            return;
        }

        if (!current_user_can('manage_options')) {
            add_settings_error('sky_seo_messages', 'sky_seo_permission_error', 'You do not have sufficient permissions to perform this action.', 'error');
            return;
        }

        $seo_plugin = isset($_POST['seo_plugin']) ? sanitize_text_field($_POST['seo_plugin']) : '';

        if (empty($seo_plugin)) {
            add_settings_error('sky_seo_messages', 'sky_seo_selection_error', 'Invalid SEO plugin selected.', 'error');
            return;
        }

        // Set active SEO plugin and update the options
        $this->set_seo_plugin_by_name($seo_plugin);
        
        // Update the option in the database
        $options = get_option('skyseomanager_options', array());
        $options['manual_seo_plugin'] = $seo_plugin;
        update_option('skyseomanager_options', $options);
        
        // Update the local copy too
        $this->options['manual_seo_plugin'] = $seo_plugin;
        
        // Add success message
        add_settings_error('sky_seo_messages', 'sky_seo_override_success', 'SEO plugin set to ' . $this->get_seo_plugin_name() . ' successfully.', 'success');
    }

    /**
     * Set SEO plugin meta keys based on plugin name
     */
    private function set_seo_plugin_by_name($plugin_name) {
        $this->active_seo_plugin = $plugin_name;
        
        // Set meta keys based on selected plugin
        switch ($plugin_name) {
            case 'yoast':
                $this->meta_keys = array(
                    'title' => '_yoast_wpseo_title',
                    'description' => '_yoast_wpseo_metadesc'
                );
                break;
            case 'rank_math':
                $this->meta_keys = array(
                    'title' => 'rank_math_title',
                    'description' => 'rank_math_description'
                );
                break;
            case 'aioseo':
                $this->meta_keys = array(
                    'title' => '_aioseo_title',
                    'description' => '_aioseo_description'
                );
                break;
            case 'seopress':
                $this->meta_keys = array(
                    'title' => '_seopress_titles_title',
                    'description' => '_seopress_titles_desc'
                );
                break;
            case 'seoframework':
                $this->meta_keys = array(
                    'title' => '_genesis_title',
                    'description' => '_genesis_description'
                );
                break;
            case 'smartcrawl':
                $this->meta_keys = array(
                    'title' => '_wds_title',
                    'description' => '_wds_metadesc'
                );
                break;
            case 'premium_seo':
                $this->meta_keys = array(
                    'title' => '_premium_seo_pack_title',
                    'description' => '_premium_seo_pack_description'
                );
                break;
            case 'seo_ultimate':
                $this->meta_keys = array(
                    'title' => '_su_title',
                    'description' => '_su_description'
                );
                break;
            default:
                $this->meta_keys = array(
                    'title' => '_sky_seo_title',
                    'description' => '_sky_seo_description'
                );
                break;
        }
    }
    
    private function handle_delete_other_plugin_data() {
        // Verify nonce
        if (!isset($_POST['sky_seo_delete_other_nonce']) || !wp_verify_nonce($_POST['sky_seo_delete_other_nonce'], 'sky_seo_delete_other_data')) {
            add_settings_error('sky_seo_messages', 'sky_seo_nonce_error', 'Security verification failed. Please try again.', 'error');
            return;
        }
        
        if (!current_user_can('manage_options')) {
            add_settings_error('sky_seo_messages', 'sky_seo_permission_error', 'You do not have sufficient permissions to perform this action.', 'error');
            return;
        }
        
        $plugin = isset($_POST['delete_plugin']) ? sanitize_text_field($_POST['delete_plugin']) : '';
        
        if (empty($plugin)) {
            add_settings_error('sky_seo_messages', 'sky_seo_plugin_error', 'No SEO plugin selected.', 'error');
            return;
        }
        
        global $wpdb;
        $meta_keys = $this->get_plugin_meta_keys($plugin);
        
        if (empty($meta_keys)) {
            add_settings_error('sky_seo_messages', 'sky_seo_plugin_error', 'Invalid SEO plugin selected.', 'error');
            return;
        }
        
        $total_deleted = 0;
        foreach ($meta_keys as $meta_key) {
            $deleted = $wpdb->query($wpdb->prepare("DELETE FROM $wpdb->postmeta WHERE meta_key = %s", $meta_key));
            $total_deleted += $deleted;
        }
        
        // Add success message
        add_settings_error(
            'sky_seo_messages',
            'sky_seo_data_deleted',
            sprintf('SEO data for %s deleted successfully. %d meta fields removed.', $this->get_plugin_display_name($plugin), $total_deleted),
            'success'
        );
    }
    
    private function get_plugin_meta_keys($plugin) {
        switch ($plugin) {
            case 'yoast':
                return array('_yoast_wpseo_title', '_yoast_wpseo_metadesc', '_yoast_wpseo_focuskw', '_yoast_wpseo_linkdex', '_yoast_wpseo_content_score');
            case 'rank_math':
                return array('rank_math_title', 'rank_math_description', 'rank_math_focus_keyword', 'rank_math_seo_score');
            case 'aioseo':
                return array('_aioseo_title', '_aioseo_description', '_aioseo_keywords', '_aioseo_og_title', '_aioseo_og_description');
            case 'seopress':
                return array('_seopress_titles_title', '_seopress_titles_desc', '_seopress_analysis_target_kw', '_seopress_robots_canonical');
            case 'seoframework':
                return array('_genesis_title', '_genesis_description', '_tsf_title_no_blogname', '_tsf_description_no_blogname');
            case 'smartcrawl':
                return array('_wds_title', '_wds_metadesc', '_wds_focus-keywords', '_wds_canonical');
            case 'premium_seo':
                return array('_premium_seo_pack_title', '_premium_seo_pack_description', '_premium_seo_pack_keywords');
            case 'seo_ultimate':
                return array('_su_title', '_su_description', '_su_keywords', '_su_og_title');
            default:
                return array();
        }
    }
    
    private function get_plugin_display_name($plugin) {
        switch ($plugin) {
            case 'yoast':
                return 'Yoast SEO';
            case 'rank_math':
                return 'Rank Math SEO';
            case 'aioseo':
                return 'All in One SEO';
            case 'seopress':
                return 'SEOPress';
            case 'seoframework':
                return 'The SEO Framework';
            case 'smartcrawl':
                return 'SmartCrawl';
            case 'premium_seo':
                return 'Premium SEO Pack';
            case 'seo_ultimate':
                return 'SEO Ultimate';
            default:
                return 'Unknown Plugin';
        }
    }

    public function start_seo_import() {
        check_ajax_referer('skyseomanager_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
            return;
        }

        $plugin = isset($_POST['plugin']) ? sanitize_text_field($_POST['plugin']) : '';
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $batch_size = 10; // Process 10 posts at a time
        
        if (empty($plugin)) {
            wp_send_json_error(array('message' => 'Invalid plugin specified'));
            return;
        }

        global $wpdb;

        // Get meta key mapping based on plugin
        $source_meta_keys = $this->get_plugin_meta_keys($plugin);
        
        if (empty($source_meta_keys)) {
            wp_send_json_error(array('message' => 'No valid meta keys found for the selected plugin'));
            return;
        }

        // Create a mapping of source meta keys to target meta keys
        $meta_mapping = array();
        foreach ($source_meta_keys as $meta_key) {
            if (strpos($meta_key, 'title') !== false) {
                $meta_mapping[$meta_key] = $this->meta_keys['title'];
            } else if (strpos($meta_key, 'desc') !== false || strpos($meta_key, 'description') !== false) {
                $meta_mapping[$meta_key] = $this->meta_keys['description'];
            }
        }

        // Get total count for progress calculation
        $placeholders = implode(',', array_fill(0, count($source_meta_keys), '%s'));
        $total_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT pm.post_id) 
            FROM $wpdb->postmeta pm 
            JOIN $wpdb->posts p ON p.ID = pm.post_id 
            WHERE pm.meta_key IN ($placeholders)
            AND p.post_status = 'publish'",
            $source_meta_keys
        ));

        // Get batch of posts to process
        $posts = $wpdb->get_results($wpdb->prepare(
            "SELECT DISTINCT pm.post_id, p.post_title, p.post_type 
            FROM $wpdb->postmeta pm 
            JOIN $wpdb->posts p ON p.ID = pm.post_id 
            WHERE pm.meta_key IN ($placeholders)
            AND p.post_status = 'publish'
            LIMIT %d, %d",
            array_merge($source_meta_keys, array($offset, $batch_size))
        ));

        $imported = array();
        foreach ($posts as $post) {
            $post_data = array(
                'title' => $post->post_title,
                'type' => $post->post_type,
                'status' => array()
            );

            foreach ($meta_mapping as $source_key => $target_key) {
                $old_value = get_post_meta($post->post_id, $source_key, true);
                if (!empty($old_value)) {
                    update_post_meta($post->post_id, $target_key, $old_value);
                    $post_data['status'][] = 'Imported ' . (strpos($source_key, 'title') !== false ? 'title' : 'description');
                }
            }

            if (empty($post_data['status'])) {
                $post_data['status'][] = 'No data to import';
            }
            $imported[] = $post_data;
        }

        $progress = min(100, round(($offset + count($posts)) / max(1, $total_count) * 100));
        $is_complete = ($offset + $batch_size) >= $total_count;

        wp_send_json_success(array(
            'imported' => $imported,
            'progress' => $progress,
            'offset' => $offset + $batch_size,
            'is_complete' => $is_complete,
            'total' => $total_count
        ));
    }
    
    /**
     * Process AI optimization requests for content
     * 
     * @since 1.5
     * @return void
     */
    public function ai_optimize_content() {
        // Check nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'skyseomanager_nonce')) {
            wp_send_json_error('Invalid security token');
        }
        
        // Check if API key exists
        if (empty($this->options['gemini_api_key'])) {
            wp_send_json_error('Gemini API key is missing. Please add it in the plugin settings.');
        }
        
        // Get post ID from request
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error('Invalid post ID');
        }
        
        // Debug mode logging
        if (!empty($this->options['debug_mode'])) {
            error_log('SkySEOManager: Starting AI optimization for post ID ' . $post_id);
        }
        
        // Get post content
        $post = get_post($post_id);
        
        if (!$post) {
            wp_send_json_error('Post not found');
        }
        
        // Extract clean text from title and content using WordPress functions
        // This is the proper way to handle content in WordPress
        $post_title = '';
        $post_content = '';
        
        // Process post title - use WordPress sanitization functions
        if (!empty($post->post_title)) {
            // wp_strip_all_tags properly removes all HTML tags
            $post_title = wp_strip_all_tags($post->post_title);
            // sanitize_text_field removes invalid UTF-8 characters and normalizes spacing
            $post_title = sanitize_text_field($post_title);
        }
        
        // Process post content - properly extract text using WordPress functions
        if (!empty($post->post_content)) {
            // First properly remove all shortcodes
            $stripped_content = strip_shortcodes($post->post_content);
            // Then strip all HTML tags, including scripts and styles
            $stripped_content = wp_strip_all_tags($stripped_content, true);
            // Finally sanitize to handle any remaining issues
            $post_content = sanitize_text_field($stripped_content);
        }
        
        // Additional debugging for problematic posts
        if (!empty($this->options['debug_mode'])) {
            $content_sample = substr($post_content, 0, 100);
            error_log("SkySEOManager: Post $post_id processed title: $post_title");
            error_log("SkySEOManager: Post $post_id processed content sample: $content_sample");
        }
        
        // Verify we have content to work with
        if (empty($post_title) && empty($post_content)) {
            wp_send_json_error('Post has no title or content to analyze');
        }
        
        // Format content to emphasize the title
        $content = "TITLE: " . $post_title . "\n\nCONTENT: " . $post_content;
        
        // Truncate content if it's too long (Gemini has token limits)
        if (strlen($content) > 10000) {
            // Keep the title but truncate the content
            $content = "TITLE: " . $post_title . "\n\nCONTENT: " . substr($post_content, 0, 9500) . "...";
        }
        
        // Get the system prompt from options
        $instruction = !empty($this->options['ai_system_prompt']) 
            ? $this->options['ai_system_prompt'] 
            : 'You are an SEO expert. Analyze the following content and generate an optimized title and meta description. The title should be 50-60 characters, and the description should be 150-160 characters. Both should be compelling and include relevant keywords from the content.';
        
        // Combine instruction and content
        $combined_content = $instruction . "\n\nContent to analyze:\n" . $content;
        
        // Get AI model from options or use default
        $model = !empty($this->options['ai_model']) 
            ? $this->options['ai_model'] 
            : 'gemini-2.0-flash-lite';
            
        // Debug logging
        if (!empty($this->options['debug_mode'])) {
            error_log('SkySEOManager: Using AI model: ' . $model);
        }
        
        // Prepare the request payload
        $request_data = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => [
                        [
                            'text' => $combined_content
                        ]
                    ]
                ]
            ],
            'generationConfig' => [
                'responseMimeType' => 'application/json',
                'responseSchema' => [
                    'type' => 'object',
                    'properties' => [
                        'Title' => [
                            'type' => 'string'
                        ],
                        'Description' => [
                            'type' => 'string'
                        ]
                    ],
                    'required' => [
                        'Title',
                        'Description'
                    ]
                ]
            ]
        ];
        
        // Debug logging - request payload
        if (!empty($this->options['debug_mode'])) {
            error_log('SkySEOManager: API request payload');
        }
        
        // Set longer timeout (60 seconds instead of 30)
        $timeout = !empty($this->options['ai_timeout']) ? intval($this->options['ai_timeout']) : 60;
        
        // Validate that request_data is properly formatted
        if (!is_array($request_data) || empty($request_data)) {
            wp_send_json_error('Invalid request data format');
            return;
        }
        
        // Use a try-catch block for JSON encoding with safe encoding options
        try {
            // Use encoding options that handle Unicode and invalid UTF-8
            $json_options = JSON_THROW_ON_ERROR | JSON_UNESCAPED_UNICODE | JSON_INVALID_UTF8_SUBSTITUTE;
            $request_body = json_encode($request_data, $json_options);
            
            // Additional verification of JSON encoding
            if ($request_body === false) {
                throw new Exception('JSON encoding returned false');
            }
            
            // Log success for debugging
            if (!empty($this->options['debug_mode'])) {
                error_log("SkySEOManager: JSON encoding successful for post $post_id");
            }
        } catch (Exception $e) {
            // If JSON encoding still fails, try ultra-safe fallback
            $error_msg = 'Failed to encode request data: ' . $e->getMessage();
            if (!empty($this->options['debug_mode'])) {
                error_log('SkySEOManager: ' . $error_msg);
            }
            
            // Try one more time with ultra-safe content
            try {
                // Create a simplified request with only basic alphanumeric characters
                $ultra_safe_title = preg_replace('/[^a-zA-Z0-9\s]/', '', $post_title);
                $ultra_safe_content = "Please generate an SEO title and description for content about: $ultra_safe_title";
                
                $simple_request = [
                    'contents' => [
                        [
                            'role' => 'user',
                            'parts' => [['text' => $ultra_safe_content]]
                        ]
                    ],
                    'generationConfig' => $request_data['generationConfig']
                ];
                
                $request_body = json_encode($simple_request, JSON_THROW_ON_ERROR);
                
                if (!empty($this->options['debug_mode'])) {
                    error_log("SkySEOManager: Using ultra-safe fallback for post $post_id");
                }
            } catch (Exception $e2) {
                if (!empty($this->options['debug_mode'])) {
                    error_log('SkySEOManager: Even ultra-safe encoding failed: ' . $e2->getMessage());
                }
                wp_send_json_error('Failed to encode content for API request. Please try again or contact support.');
                return;
            }
        }
        
        // Make API call to Gemini
        $response = wp_remote_post(
            'https://generativelanguage.googleapis.com/v1beta/models/' . $model . ':generateContent?key=' . $this->options['gemini_api_key'],
            [
                'headers' => [
                    'Content-Type' => 'application/json'
                ],
                'body' => $request_body,
                'timeout' => $timeout
            ]
        );
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            if (!empty($this->options['debug_mode'])) {
                error_log('SkySEOManager: API request failed: ' . $error_message);
            }
            wp_send_json_error('API request failed: ' . $error_message);
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 200) {
            $error_message = wp_remote_retrieve_body($response);
            if (!empty($this->options['debug_mode'])) {
                error_log('SkySEOManager: API error (code ' . $response_code . '): ' . $error_message);
            }
            wp_send_json_error('API error (code ' . $response_code . '): ' . $error_message);
        }
        
        $response_body = wp_remote_retrieve_body($response);
        
        // Debug logging - response
        if (!empty($this->options['debug_mode'])) {
            error_log('SkySEOManager: API response: ' . $response_body);
        }
        
        $result = json_decode($response_body, true);
        
        // Extract the generated title and description
        if (isset($result['candidates'][0]['content']['parts'][0]['text'])) {
            $ai_text_response = $result['candidates'][0]['content']['parts'][0]['text'];
            
            // Try to parse as JSON
            $ai_response = json_decode($ai_text_response, true);
            
            // Debug logging - parsed response
            if (!empty($this->options['debug_mode'])) {
                error_log('SkySEOManager: Parsed AI response: ' . json_encode($ai_response));
                error_log('SkySEOManager: JSON parse error (if any): ' . json_last_error_msg());
            }
            
            // Handle both potential formats - direct JSON or text that needs extraction
            if (json_last_error() !== JSON_ERROR_NONE || !isset($ai_response['Title']) || !isset($ai_response['Description'])) {
                // Check if we have a partial response (only title or only description)
                if (json_last_error() === JSON_ERROR_NONE && isset($ai_response['Title']) && !isset($ai_response['Description'])) {
                    if (!empty($this->options['debug_mode'])) {
                        error_log('SkySEOManager: Partial response received - missing Description field');
                    }
                    wp_send_json_error('Incomplete AI response: Description field is missing. Try again or use a different model.');
                }
                
                if (json_last_error() === JSON_ERROR_NONE && !isset($ai_response['Title']) && isset($ai_response['Description'])) {
                    if (!empty($this->options['debug_mode'])) {
                        error_log('SkySEOManager: Partial response received - missing Title field');
                    }
                    wp_send_json_error('Incomplete AI response: Title field is missing. Try again or use a different model.');
                }
                
                // Try to extract title and description from text using regex if JSON parsing failed
                $title = '';
                $description = '';
                
                // Look for patterns like "Title: Some title" and "Description: Some description"
                if (preg_match('/title[:\s]+([^\n]+)/i', $ai_text_response, $title_matches)) {
                    $title = trim($title_matches[1]);
                }
                if (preg_match('/description[:\s]+([^\n]+)/i', $ai_text_response, $desc_matches)) {
                    $description = trim($desc_matches[1]);
                }
                
                // Debug logging - regex extraction
                if (!empty($this->options['debug_mode'])) {
                    error_log('SkySEOManager: Extracted title via regex: ' . $title);
                    error_log('SkySEOManager: Extracted description via regex: ' . $description);
                }
                
                // If still empty, fail with detailed message
                if (empty($title) || empty($description)) {
                    if (!empty($this->options['debug_mode'])) {
                        error_log('SkySEOManager: Failed to extract title/description from response: ' . $ai_text_response);
                    }
                    wp_send_json_error('Failed to parse AI response: ' . json_last_error_msg() . '. Raw response: ' . substr($ai_text_response, 0, 100));
                    return;
                }
            } else {
                $title = $ai_response['Title'];
                $description = $ai_response['Description'];
            }
            
            // Update post meta with the generated title and description
            update_post_meta($post_id, $this->meta_keys['title'], $title);
            update_post_meta($post_id, $this->meta_keys['description'], $description);
            
            // Debug logging - success
            if (!empty($this->options['debug_mode'])) {
                error_log('SkySEOManager: Successfully updated post ' . $post_id . ' with AI-generated title and description');
            }
            
            wp_send_json_success([
                'title' => $title,
                'description' => $description
            ]);
        } else {
            if (!empty($this->options['debug_mode'])) {
                error_log('SkySEOManager: Invalid API response structure: ' . json_encode($result));
            }
            wp_send_json_error('Invalid API response structure');
        }
    }
    
    /**
     * Helper function to sanitize UTF-8 content
     * 
     * @param string $text The text to sanitize
     * @return string Sanitized text with valid UTF-8 characters
     */
    private function sanitize_utf8($text) {
        if (empty($text)) {
            return '';
        }
        
        // First, try regular sanitization
        $sanitized = sanitize_text_field($text);
        
        // If the text contains HTML entities, decode them first
        $sanitized = html_entity_decode($sanitized, ENT_QUOTES, 'UTF-8');
        
        // Remove any potentially problematic control characters
        $sanitized = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $sanitized);
        
        // Handle mojibake (double-encoded UTF-8)
        if (preg_match('/%[0-9A-F]{2}/i', $sanitized)) {
            $sanitized = urldecode($sanitized);
        }
        
        // Ensure we're working with valid UTF-8
        if (!mb_check_encoding($sanitized, 'UTF-8')) {
            // Try to convert from common encodings to UTF-8
            $encodings = array('Windows-1252', 'ISO-8859-1', 'ASCII');
            
            foreach ($encodings as $encoding) {
                $converted = @mb_convert_encoding($text, 'UTF-8', $encoding);
                if (mb_check_encoding($converted, 'UTF-8')) {
                    $sanitized = $converted;
                    break;
                }
            }
            
            // If all else fails, force UTF-8 by removing invalid bytes
            if (!mb_check_encoding($sanitized, 'UTF-8')) {
                $sanitized = mb_convert_encoding($sanitized, 'UTF-8', 'UTF-8');
            }
            
            // Final fallback - strip non-UTF8 characters entirely as a last resort
            $sanitized = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}\x{10000}-\x{10FFFF}]/u', '', $sanitized);
        }
        
        return $sanitized;
    }

    public function reorder_submenus() {
        global $submenu;
        
        if (!isset($submenu['skyseomanager'])) {
            return;
        }
        
        // If license is valid, reorder menus (Media, Bulk Edit, Settings, License)
        // Get the Media Optimizer and License menu items
        $media_item = null;
        $settings_item = null;
        
        foreach ($submenu['skyseomanager'] as $key => $item) {
            if ($item[2] === 'skyseomanager-media') {
                $media_item = $item;
                unset($submenu['skyseomanager'][$key]);
            } elseif ($item[2] === 'skyseomanager_settings') {
                $settings_item = $item;
                unset($submenu['skyseomanager'][$key]);
            }
        }
        
        // Create a new ordered array
        $new_submenu = array();
        
        // First menu item is always the main menu
        if (isset($submenu['skyseomanager'][0])) {
            $new_submenu[] = $submenu['skyseomanager'][0];
        }
        
        // Add media item after main menu
        if ($media_item !== null) {
            $new_submenu[] = $media_item;
        }
        
        // Add all remaining items except settings and license
        foreach ($submenu['skyseomanager'] as $key => $item) {
            if ($key === 0) {
                continue; // Skip the main menu item as it's already added
            }
            if ($item[2] !== 'skyseomanager_settings') {
                $new_submenu[] = $item;
            }
        }
        
        // Add settings item
        if ($settings_item !== null) {
            $new_submenu[] = $settings_item;
        }
        
        // Replace the submenu array
        $submenu['skyseomanager'] = $new_submenu;
    }
}

// Initialize the plugin
new SkySEOManager();
