<?php
/**
 * Media Optimizer class for the SkySEOManager plugin.
 *
 * @package SkySEOManager
 * @since 1.5
 */

if (!defined('ABSPATH')) {
    exit;
}

// Make sure we have access to WordPress error handling


/**
 * Class Sky_Media_Optimizer
 * 
 * Handles the functionality for optimizing media Title and Alt attributes
 * with AI assistance.
 */
class Sky_Media_Optimizer {
    /**
     * Plugin options array
     *
     * @var array
     */
    private $options = array();

    /**
     * Number of items to show per page
     *
     * @var int
     */
    private $items_per_page = 20;

    /**
     * License status
     *
     * @var bool
     */
    private $is_license_active = false;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct() {
        $this->items_per_page = 20;
        
        // Get the license manager to check status
        $this->check_license_status();
        
        // Register admin menu only if license is active
        if ($this->is_license_active) {
            add_action('admin_menu', array($this, 'add_admin_menu'));
        }
        
        // Add AJAX handlers
        add_action('wp_ajax_save_media_data', array($this, 'save_media_data'));
        add_action('wp_ajax_ai_optimize_media', array($this, 'ai_optimize_media'));
    }

    /**
     * Check license status to determine if features should be active
     */
    private function check_license_status() {
        // Get main plugin options
        $this->options = get_option('skyseomanager_options', array());
        
        // Check if license is active by checking if the license_status option exists and is valid
        $license_status = get_option('skyseomanager_license_status', 'inactive');
        $this->is_license_active = ($license_status === 'valid');
    }

    /**
     * Add submenu for Media Optimizer
     */
    public function add_admin_menu() {
        // Only add menu if license is active
        if (!$this->is_license_active) {
            return;
        }
        
        // Add menu with a position before Settings (which typically has position 10)
        add_submenu_page(
            'skyseomanager',            // Parent slug
            'Optimize Media',           // Page title
            'Optimize Media',           // Menu title
            'manage_options',           // Capability
            'skyseomanager-media',      // Menu slug
            array($this, 'render_media_page')
        );
    }

    /**
     * Render the media optimization page
     */
    public function render_media_page() {
        // Current page and filter settings
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $current_filter = isset($_GET['media_filter']) ? sanitize_text_field($_GET['media_filter']) : '';

        // Get total number of attachments
        $total_attachments = wp_count_posts('attachment')->inherit;
        $total_pages = ceil($total_attachments / $this->items_per_page);

        // Build query args
        $args = array(
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'posts_per_page' => $this->items_per_page,
            'offset'         => ($current_page - 1) * $this->items_per_page,
            'orderby'        => 'date',
            'order'          => 'DESC'
        );

        // Add mime type filter
        $args['post_mime_type'] = 'image';
        
        // Add filter logic
        if (!empty($current_filter)) {
            $meta_query = array('relation' => 'AND');
            
            switch ($current_filter) {
                case 'no_title':
                    $meta_query[] = array(
                        'relation' => 'OR',
                        array(
                            'key'     => '_wp_attachment_image_alt',
                            'compare' => 'NOT EXISTS'
                        ),
                        array(
                            'key'     => '_wp_attachment_image_alt',
                            'value'   => '',
                            'compare' => '='
                        )
                    );
                    break;
                case 'no_alt':
                    $args['meta_query'] = array(
                        array(
                            'relation' => 'OR',
                            array(
                                'key'     => '_wp_attachment_image_alt',
                                'compare' => 'NOT EXISTS'
                            ),
                            array(
                                'key'     => '_wp_attachment_image_alt',
                                'value'   => '',
                                'compare' => '='
                            )
                        )
                    );
                    break;
                case 'no_both':
                    $args['meta_query'] = array(
                        'relation' => 'AND',
                        array(
                            'relation' => 'OR',
                            array(
                                'key'     => '_wp_attachment_image_alt',
                                'compare' => 'NOT EXISTS'
                            ),
                            array(
                                'key'     => '_wp_attachment_image_alt',
                                'value'   => '',
                                'compare' => '='
                            )
                        ),
                        array(
                            'relation' => 'OR',
                            array(
                                'key'     => 'post_title',
                                'compare' => 'NOT EXISTS'
                            ),
                            array(
                                'key'     => 'post_title',
                                'value'   => '',
                                'compare' => '='
                            )
                        )
                    );
                    break;
            }
            
            if (!empty($meta_query) && count($meta_query) > 1) {
                $args['meta_query'] = $meta_query;
            }
        }

        // Get attachments
        $attachments = get_posts($args);
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">SkySEOManager - Optimize Media</h1>
            
            <!-- Toast Message Container -->
            <div id="sky-media-toast" class="sky-seo-toast"></div>
            
            <div class="tablenav top">
                <form method="get" class="media-filters alignleft actions">
                    <input type="hidden" name="page" value="skyseomanager-media">
                    
                    <select name="media_filter" id="media_filter" class="postform">
                        <option value="">All Media</option>
                        <optgroup label="Missing Content">
                            <option value="no_title" <?php selected($current_filter, 'no_title'); ?>>No Title</option>
                            <option value="no_alt" <?php selected($current_filter, 'no_alt'); ?>>No Alt Text</option>
                            <option value="no_both" <?php selected($current_filter, 'no_both'); ?>>No Title or Alt Text</option>
                        </optgroup>
                    </select>

                    <input type="submit" class="button action" value="Apply Filters">
                </form>
                
                <div class="alignright">
                    <button id="media-ai-optimize-button" class="button button-primary">AI Optimize</button>
                    <span id="media-ai-optimize-status" style="display:none;margin-left:10px;"></span>
                </div>
            </div>

            <div class="wp-list-table-wrap">
                <table class="wp-list-table widefat fixed striped table-view-list">
                    <thead>
                        <tr>
                            <th scope="col" class="column-id" style="width: 60px;">ID</th>
                            <th scope="col" class="column-image" style="width: 120px;">Image</th>
                            <th scope="col" class="column-title">Image Title</th>
                            <th scope="col" class="column-alt">Alt Text</th>
                            <th scope="col" class="column-filename" style="width: 15%;">Filename</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($attachments as $attachment): 
                            $image_src = wp_get_attachment_image_src($attachment->ID, 'thumbnail');
                            $image_alt = get_post_meta($attachment->ID, '_wp_attachment_image_alt', true);
                            $file_name = basename(get_attached_file($attachment->ID));
                        ?>
                            <tr data-media-id="<?php echo esc_attr($attachment->ID); ?>">
                                <td><?php echo esc_html($attachment->ID); ?></td>
                                <td>
                                    <?php if ($image_src): ?>
                                        <img src="<?php echo esc_url($image_src[0]); ?>" alt="Thumbnail" style="max-width: 100px; max-height: 100px;">
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="media-field-container">
                                        <input type="text" class="media-title" value="<?php echo esc_attr($attachment->post_title); ?>" placeholder="Enter image title">
                                    </div>
                                </td>
                                <td>
                                    <div class="media-field-container">
                                        <input type="text" class="media-alt" value="<?php echo esc_attr($image_alt); ?>" placeholder="Enter alt text">
                                    </div>
                                </td>
                                <td><?php echo esc_html($file_name); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <span class="displaying-num"><?php echo esc_html($total_attachments); ?> items</span>
                            <span class="pagination-links">
                                <?php
                                echo paginate_links(array(
                                    'base' => add_query_arg('paged', '%#%'),
                                    'format' => '',
                                    'prev_text' => __('&laquo;'),
                                    'next_text' => __('&raquo;'),
                                    'total' => $total_pages,
                                    'current' => $current_page
                                ));
                                ?>
                            </span>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Save media title and alt text data via AJAX
     */
    public function save_media_data() {
        // Check nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'skyseomanager_nonce')) {
            wp_send_json_error('Invalid security token sent: ' . (isset($_POST['nonce']) ? $_POST['nonce'] : 'none'));
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }

        $attachment_id = isset($_POST['attachment_id']) ? absint($_POST['attachment_id']) : 0;
        $field_type = isset($_POST['field_type']) ? sanitize_text_field($_POST['field_type']) : '';
        $value = isset($_POST['value']) ? sanitize_text_field($_POST['value']) : '';

        if (!$attachment_id || !$field_type) {
            wp_send_json_error('Missing required data');
            return;
        }

        switch ($field_type) {
            case 'title':
                // Update the attachment title by updating the post
                $attachment = array(
                    'ID' => $attachment_id,
                    'post_title' => $value
                );
                $result = wp_update_post($attachment);
                break;
                
            case 'alt':
                // Update the alt text in the post meta
                $result = update_post_meta($attachment_id, '_wp_attachment_image_alt', $value);
                break;
                
            default:
                wp_send_json_error('Invalid field type');
                return;
        }

        if ($result !== false) {
            wp_send_json_success(array(
                'message' => 'Data saved successfully',
                'attachment_id' => $attachment_id,
                'field' => $field_type,
                'value' => $value
            ));
        } else {
            wp_send_json_error('Failed to save data');
        }
    }

    /**
     * Process AI optimization for media image title and alt text
     */
    public function ai_optimize_media() {
        // Check nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'skyseomanager_nonce')) {
            wp_send_json_error('Invalid security token');
            return;
        }
        
        // Check if API key exists
        if (empty($this->options['gemini_api_key'])) {
            wp_send_json_error('Gemini API key is missing. Please add it in the plugin settings.');
            return;
        }
        
        // Get attachment ID from request
        $attachment_id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;
        
        if (!$attachment_id) {
            wp_send_json_error('Invalid attachment ID');
            return;
        }
        
        // Debug - Log start of optimization
        if (!empty($this->options['debug_mode'])) {
            error_log("Media Optimizer: Starting optimization for attachment ID: {$attachment_id}");
        }
        
        // Get attachment data
        $attachment = get_post($attachment_id);
        
        if (!$attachment) {
            wp_send_json_error('Attachment not found');
            return;
        }
        
        // Get image file path
        $file_path = get_attached_file($attachment_id);
        // If original file missing, attempt fallback to common extensions
        if (!file_exists($file_path)) {
            $fallback_path = $this->find_existing_file($file_path);
            if ($fallback_path !== false) {
                $file_path = $fallback_path;
            } else {
                wp_send_json_error('Image file not found: ' . $file_path);
                return;
            }
        }
        
        // Get file info
        $file_name = basename($file_path);
        $mime_type = get_post_mime_type($attachment_id);
        $file_size = filesize($file_path);
        
        if (!$mime_type) {
            $mime_type = mime_content_type($file_path);
        }
        
        // Debug - Log file info
        if (!empty($this->options['debug_mode'])) {
            error_log("Media Optimizer: File info - Path: {$file_path}, Name: {$file_name}, MIME: {$mime_type}, Size: {$file_size}");
        }
        
        // Step 1: Upload the file to Gemini API
        $upload_result = $this->upload_file_to_gemini($file_path, $file_name, $mime_type, $file_size);
        
        if (is_wp_error($upload_result)) {
            $error_message = $upload_result->get_error_message();
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: File upload failed - {$error_message}");
            }
            wp_send_json_error('Failed to upload image: ' . $error_message);
            return;
        }
        
        if (empty($upload_result['file_uri'])) {
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: Upload succeeded but no file_uri returned");
                error_log("Media Optimizer: Upload result: " . print_r($upload_result, true));
            }
            wp_send_json_error('Failed to get file URI from Gemini API');
            return;
        }
        
        $file_uri = $upload_result['file_uri'];
        
        // Debug - Log successful upload
        if (!empty($this->options['debug_mode'])) {
            error_log("Media Optimizer: File successfully uploaded, got URI: {$file_uri}");
        }
        
        // Step 2: Send the file URI to Gemini for analysis
        $default_prompt = 'Analyze this image and generate an appropriate title and alt text description. The title should be concise (3-7 words) and the alt text should be descriptive (10-20 words) for accessibility purposes.';
        $instruction = isset($this->options['media_system_prompt']) && !empty($this->options['media_system_prompt']) 
            ? $this->options['media_system_prompt'] 
            : $default_prompt;
        
        // Debug - Log prompt
        if (!empty($this->options['debug_mode'])) {
            error_log("Media Optimizer: Using prompt: {$instruction}");
        }
        
        // Prepare the request payload with the file and instruction
        $request_data = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => [
                        [
                            'fileData' => [
                                'mimeType' => $mime_type,
                                'fileUri' => $file_uri
                            ]
                        ],
                        [
                            'text' => $instruction
                        ]
                    ]
                ]
            ],
            'generationConfig' => [
                'responseMimeType' => 'application/json',
                'responseSchema' => [
                    'type' => 'object',
                    'properties' => [
                        'title' => [
                            'type' => 'string'
                        ],
                        'alttxt' => [
                            'type' => 'string'
                        ]
                    ],
                    'required' => [
                        'title',
                        'alttxt'
                    ]
                ]
            ]
        ];
        
        // Debug - Log request data
        if (!empty($this->options['debug_mode'])) {
            $sanitized_request = $request_data;
            if (isset($sanitized_request['contents'][0]['parts'][0]['fileData']['fileUri'])) {
                $sanitized_request['contents'][0]['parts'][0]['fileData']['fileUri'] = '[REDACTED]';
            }
            error_log("Media Optimizer: Request payload: " . json_encode($sanitized_request, JSON_PRETTY_PRINT));
            error_log("Media Optimizer: Using API endpoint: https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash-lite:generateContent");
        }
        
        // Add JSON encoding error handling
        $request_body = '';
        try {
            $request_body = json_encode($request_data);
            if ($request_body === false) {
                throw new Exception(json_last_error_msg());
            }
        } catch (Exception $e) {
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: JSON encoding error - " . $e->getMessage());
            }
            wp_send_json_error('Failed to encode request: ' . $e->getMessage());
            return;
        }
        
        // Make API call to Gemini
        if (!empty($this->options['debug_mode'])) {
            error_log("Media Optimizer: Sending API request...");
        }
        
        $response = wp_remote_post(
            'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash-lite:generateContent?key=' . $this->options['gemini_api_key'],
            [
                'headers' => [
                    'Content-Type' => 'application/json'
                ],
                'body' => $request_body,
                'timeout' => 30
            ]
        );
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: API request WP error - " . $error_message);
                error_log("Media Optimizer: Response error details: " . print_r($response, true));
            }
            wp_send_json_error('API request failed: ' . $error_message);
            return;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Debug - Log API response
        if (!empty($this->options['debug_mode'])) {
            error_log("Media Optimizer: API response code: {$response_code}");
            error_log("Media Optimizer: API response body: {$response_body}");
        }
        
        if ($response_code !== 200) {
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: API error response (code {$response_code}): {$response_body}");
                // Log headers for more context
                $headers = wp_remote_retrieve_headers($response);
                if ($headers) {
                    error_log("Media Optimizer: Response headers: " . print_r($headers->getAll(), true));
                }
                // Check if this is a rate limit or quota issue
                if ($response_code === 429 || $response_code === 503) {
                    error_log("Media Optimizer: This appears to be a rate limit or service overload issue.");
                }
            }
            wp_send_json_error('API error (code ' . $response_code . '): ' . $response_body);
            return;
        }
        
        $result = json_decode($response_body, true);
        
        // Debug - Log parsed result
        if (!empty($this->options['debug_mode'])) {
            error_log("Media Optimizer: Parsed API response: " . json_encode($result, JSON_PRETTY_PRINT));
        }
        
        // Extract the generated title and alt text
        if (isset($result['candidates'][0]['content']['parts'][0]['text'])) {
            $ai_text_response = $result['candidates'][0]['content']['parts'][0]['text'];
            
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: Raw AI text response: {$ai_text_response}");
            }
            
            $ai_response = json_decode($ai_text_response, true);
            
            // Log parsing results if debug mode is enabled
            if (!empty($this->options['debug_mode'])) {
                error_log('Media Optimizer: Parsed AI JSON response: ' . json_encode($ai_response));
                error_log('Media Optimizer: JSON parse error (if any): ' . json_last_error_msg());
            }
            
            if (json_last_error() !== JSON_ERROR_NONE || !isset($ai_response['title']) || !isset($ai_response['alttxt'])) {
                // Check for partial response
                if (json_last_error() === JSON_ERROR_NONE && isset($ai_response['title']) && !isset($ai_response['alttxt'])) {
                    if (!empty($this->options['debug_mode'])) {
                        error_log('Media Optimizer: Partial response received - missing alttxt field');
                    }
                    wp_send_json_error('Incomplete AI response: Alt text field is missing. Try again or use a different model.');
                    return;
                }
                
                if (json_last_error() === JSON_ERROR_NONE && !isset($ai_response['title']) && isset($ai_response['alttxt'])) {
                    if (!empty($this->options['debug_mode'])) {
                        error_log('Media Optimizer: Partial response received - missing title field');
                    }
                    wp_send_json_error('Incomplete AI response: Title field is missing. Try again or use a different model.');
                    return;
                }
                
                // Provide detailed error message with raw response for debugging
                if (!empty($this->options['debug_mode'])) {
                    error_log('Media Optimizer: Failed to parse AI response: ' . $ai_text_response);
                }
                wp_send_json_error('Failed to parse AI response: ' . json_last_error_msg() . '. Raw response: ' . substr($ai_text_response, 0, 100) . '...');
                return;
            }
            
            $title = $ai_response['title'];
            $alt_text = $ai_response['alttxt'];
            
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: Successfully extracted title: '{$title}' and alt text: '{$alt_text}'");
            }
            
            // Update attachment title
            $attachment_data = array(
                'ID' => $attachment_id,
                'post_title' => $title
            );
            $update_result = wp_update_post($attachment_data);
            
            if (is_wp_error($update_result)) {
                if (!empty($this->options['debug_mode'])) {
                    error_log("Media Optimizer: Failed to update post title: " . $update_result->get_error_message());
                }
                wp_send_json_error('Failed to update title: ' . $update_result->get_error_message());
                return;
            }
            
            // Update alt text
            $alt_result = update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
            
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: Title update result: " . ($update_result ? "Success" : "Failed"));
                error_log("Media Optimizer: Alt text update result: " . ($alt_result ? "Success" : "Failed"));
            }
            
            wp_send_json_success([
                'title' => $title,
                'alt_text' => $alt_text
            ]);
        } else {
            if (!empty($this->options['debug_mode'])) {
                error_log("Media Optimizer: Invalid API response structure - missing parts[0].text field");
                error_log("Media Optimizer: Full response: " . json_encode($result));
            }
            wp_send_json_error('Invalid API response structure');
        }
    }
    
    /**
     * Upload a file to Gemini API for processing
     * 
     * @param string $file_path Full path to the file
     * @param string $display_name Name to display for the file
     * @param string $mime_type MIME type of the file
     * @param int $file_size Size of the file in bytes
     * @return array|WP_Error Array with file_uri on success, WP_Error on failure
     */
    private function upload_file_to_gemini($file_path, $display_name, $mime_type, $file_size) {
        // Verify the file exists and is readable before proceeding
        if (!file_exists($file_path) || !is_readable($file_path)) {
            return new WP_Error('file_access_error', sprintf('File does not exist or is not readable: %s', $file_path));
        }

        // Step 1: Start the upload process
        $start_upload_response = wp_remote_post(
            'https://generativelanguage.googleapis.com/upload/v1beta/files?key=' . $this->options['gemini_api_key'],
            [
                'headers' => [
                    'X-Goog-Upload-Protocol' => 'resumable',
                    'X-Goog-Upload-Command' => 'start',
                    'X-Goog-Upload-Header-Content-Length' => $file_size,
                    'X-Goog-Upload-Header-Content-Type' => $mime_type,
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode([
                    'file' => [
                        'display_name' => $display_name
                    ]
                ])
            ]
        );
        
        if (is_wp_error($start_upload_response)) {
            return $start_upload_response;
        }
        
        if (wp_remote_retrieve_response_code($start_upload_response) !== 200) {
            return new WP_Error('upload_failed', 'Failed to start upload: ' . wp_remote_retrieve_body($start_upload_response));
        }
        
        // Extract the upload URL from headers
        $headers = wp_remote_retrieve_headers($start_upload_response);
        $upload_url = '';
        
        foreach ($headers as $name => $value) {
            if (strtolower($name) === 'x-goog-upload-url') {
                $upload_url = $value;
                break;
            }
        }
        
        if (empty($upload_url)) {
            return new WP_Error('upload_failed', 'Failed to get upload URL from response headers');
        }
        
        // Step 2: Upload the file content
        $file_content = @file_get_contents($file_path);
        if ($file_content === false) {
            return new WP_Error('file_read_error', sprintf('Failed to read file content from: %s', $file_path));
        }
        
        // Ensure we have content to upload
        if (empty($file_content)) {
            return new WP_Error('empty_file', sprintf('File is empty: %s', $file_path));
        }
        
        $upload_response = wp_remote_post(
            $upload_url,
            [
                'headers' => [
                    'Content-Length' => $file_size,
                    'X-Goog-Upload-Offset' => 0,
                    'X-Goog-Upload-Command' => 'upload, finalize'
                ],
                'body' => $file_content
            ]
        );
        
        if (is_wp_error($upload_response)) {
            return $upload_response;
        }
        
        if (wp_remote_retrieve_response_code($upload_response) !== 200) {
            return new WP_Error('upload_failed', 'Failed to upload file content: ' . wp_remote_retrieve_body($upload_response));
        }
        
        // Parse the response to get the file URI
        $upload_result = json_decode(wp_remote_retrieve_body($upload_response), true);
        
        if (empty($upload_result['file']['uri'])) {
            return new WP_Error('parse_error', 'Failed to parse file URI from upload response');
        }
        
        return [
            'file_uri' => $upload_result['file']['uri']
        ];
    }

    /**
     * Attempt to find an existing file, including appended webp extensions.
     *
     * @param string $file_path The original file path.
     * @return string|false The existing file path or false if none found.
     */
    private function find_existing_file($file_path) {
        // 1. Check if webp was appended to the original filename (e.g., .png.webp)
        $appended = $file_path . '.webp';
        if (file_exists($appended) && is_readable($appended)) {
            return $appended;
        }
        // 2. Try swapping the extension (replace original with common ones)
        $ext = pathinfo($file_path, PATHINFO_EXTENSION);
        $base = substr($file_path, 0, -strlen($ext) - 1);
        $candidates = array('webp', 'jpg', 'jpeg', 'png');
        foreach ($candidates as $candidate) {
            if ($candidate === $ext) {
                continue;
            }
            $new_path = $base . '.' . $candidate;
            if (file_exists($new_path) && is_readable($new_path)) {
                return $new_path;
            }
        }
        return false;
    }
}

// Initialize the Media Optimizer
add_action('plugins_loaded', function() {
    new Sky_Media_Optimizer();
}); 