jQuery(document).ready(function($) {
    // SEO limits
    const seoLimits = {
        title: { min: 30, max: 60 },
        description: { min: 120, max: 160 }
    };

    // Function to get SEO score class
    function getSeoScoreClass(length, type) {
        if (length === 0) return 'empty';
        if (length < seoLimits[type].min) return 'warning';
        if (length > seoLimits[type].max) return 'error';
        return 'success';
    }

    // Function to update SEO indicators in real-time
    function updateSeoIndicators($field) {
        const container = $field.closest('.seo-field-container');
        const scoreIndicator = container.find('.seo-score');
        const charCount = container.find('.character-count');
        const length = $field.val().length;
        const type = $field.hasClass('seo-title') || $field.attr('id') === 'sky_seo_title' ? 'title' : 'description';
        const max = seoLimits[type].max;

        // Update character count
        charCount.text(length + '/' + max);

        // Update score indicator
        scoreIndicator
            .removeClass('empty warning error success')
            .addClass(getSeoScoreClass(length, type))
            .attr('title', 'Character count: ' + length);
    }

    // Initialize WordPress media uploader
    var mediaUploader;
    
    // Function to show toast message
    function showToast(message, type = 'success') {
        const toast = $('#sky-seo-toast');
        toast.text(message)
            .removeClass('success error')
            .addClass(type)
            .fadeIn();
        
        setTimeout(function() {
            toast.fadeOut();
        }, 3000);
    }

    // Function to auto-save data
    function autoSave(postId, fieldType, value, field) {
        field.addClass('saving');
        
        // Debug info - Start
        console.log('Auto-save triggered for:', {
            postId: postId,
            fieldType: fieldType,
            value: value
        });
        
        return $.ajax({
            url: skyseomanager_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'auto_save_seo_data',
                nonce: skyseomanager_ajax.nonce,
                post_id: postId,
                field_type: fieldType,
                value: value
            }
        })
        .done(function(response) {
            // Debug response
            console.log('Auto-save response:', response);
            
            if (response.success) {
                showToast('Changes saved successfully');
                // Debug - verify what's being saved
                console.log('Successfully saved data:', {
                    postId: postId,
                    fieldType: fieldType,
                    value: value,
                    responseData: response.data
                });
            } else {
                showToast('Error saving changes: ' + (response.data || 'Unknown error'), 'error');
                console.error('Auto-save error response:', response);
            }
        })
        .fail(function(xhr, status, error) {
            showToast('Error saving changes: ' + error, 'error');
            console.error('Save error:', {
                error: error,
                status: status,
                xhr: xhr
            });
        })
        .always(function() {
            field.removeClass('saving');
        });
    }

    // Initialize SEO indicators
    $('.seo-title, .seo-description, #sky_seo_title, #sky_seo_description').each(function() {
        updateSeoIndicators($(this));
    });

    // Update indicators and save on input
    $(document).on('input', '.seo-title, .seo-description, #sky_seo_title, #sky_seo_description', function() {
        const $field = $(this);
        const $row = $field.closest('tr');
        const postId = $row.data('post-id');
        const fieldType = $field.hasClass('seo-title') || $field.attr('id') === 'sky_seo_title' ? 'title' : 'description';
        
        // Update indicators immediately
        updateSeoIndicators($field);
        
        // Save immediately without delay
        autoSave(postId, fieldType, $field.val(), $field);
    });

    // Handle image selection button click
    $('.select-image').on('click', function(e) {
        e.preventDefault();
        
        var button = $(this);
        var postId = button.data('post-id');
        var imagePreview = button.closest('.featured-image-container').find('.image-preview');
        
        // Create a new media uploader instance
        mediaUploader = wp.media({
            title: 'Select or Upload Featured Image',
            button: {
                text: 'Use this image'
            },
            multiple: false
        });
        
        // When an image is selected
        mediaUploader.on('select', function() {
            var attachment = mediaUploader.state().get('selection').first().toJSON();
            
            // Update the image preview
            var img = imagePreview.find('img');
            
            if (img.length === 0) {
                img = $('<img>').attr('alt', 'Featured image').css('max-width', '100px');
                imagePreview.html(img);
            }
            
            img.attr('src', attachment.url);
            button.text('Change Image');
            
            // Store the attachment ID and auto-save with correct field type
            imagePreview.attr('data-image-id', attachment.id);
            
            // Debug - Log before saving image
            console.log('Saving featured image:', {
                postId: postId,
                imageId: attachment.id
            });
            
            autoSave(postId, 'featured_image_id', attachment.id, button);
        });
        
        mediaUploader.open();
    });

    // Media uploader for company logo
    $('.select-logo').on('click', function(e) {
        e.preventDefault();
        const logoUrlInput = $('input[name="skyseomanager_options[logo_url]"]');
        
        const frame = wp.media({
            title: 'Select or Upload Company Logo',
            button: {
                text: 'Use this image'
            },
            multiple: false
        });

        frame.on('select', function() {
            const attachment = frame.state().get('selection').first().toJSON();
            logoUrlInput.val(attachment.url);
        });

        frame.open();
    });

    // Initialize tooltips
    $('.help-tip').hover(function() {
        var $this = $(this);
        if (!$this.attr('title')) return;
        
        $this.data('title', $this.attr('title'))
            .removeAttr('title');
        
        $('<div class="tooltip"></div>')
            .text($this.data('title'))
            .appendTo('body')
            .css({
                top: $this.offset().top + 20,
                left: $this.offset().left - 100
            })
            .fadeIn('fast');
    }, function() {
        var $this = $(this);
        $this.attr('title', $this.data('title'));
        $('.tooltip').remove();
    });

    // Handle AI optimization
    $('#ai-optimize-button').on('click', function() {
        const $button = $(this);
        const $status = $('#ai-optimize-status');
        const originalButtonText = $button.text();
        const postRows = $('tr[data-post-id]');
        
        if (postRows.length === 0) {
            showToast('No posts found to optimize', 'error');
            return;
        }
        
        // Disable button
        $button.prop('disabled', true).text('Processing...');
        $status.text('0/' + postRows.length + ' posts processed').show();
        
        let processed = 0;
        let successCount = 0;
        let errorCount = 0;
        
        // Process posts one by one
        function processNextPost(index) {
            if (index >= postRows.length) {
                // All posts processed
                $button.prop('disabled', false).text(originalButtonText);
                $status.text(`Completed: ${successCount} successful, ${errorCount} failed`);
                setTimeout(() => {
                    $status.fadeOut();
                }, 5000);
                return;
            }
            
            const $row = $(postRows[index]);
            const postId = $row.data('post-id');
            
            // Call AI optimization endpoint
            $.ajax({
                url: skyseomanager_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'ai_optimize_content',
                    nonce: skyseomanager_ajax.nonce,
                    post_id: postId
                },
                beforeSend: function() {
                    $row.addClass('processing');
                },
                success: function(response) {
                    if (response.success) {
                        // Update the UI with the new title and description
                        const $titleField = $row.find('.seo-title');
                        const $descField = $row.find('.seo-description');
                        
                        $titleField.val(response.data.title);
                        $descField.val(response.data.description);
                        
                        // Update indicators
                        updateSeoIndicators($titleField);
                        updateSeoIndicators($descField);
                        
                        successCount++;
                    } else {
                        console.error('Error optimizing post ' + postId + ':', response.data);
                        errorCount++;
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error:', error);
                    errorCount++;
                },
                complete: function() {
                    $row.removeClass('processing');
                    processed++;
                    $status.text(processed + '/' + postRows.length + ' posts processed');
                    
                    // Process next post
                    processNextPost(index + 1);
                }
            });
        }
        
        // Start processing posts
        processNextPost(0);
    });
}); 