jQuery(document).ready(function($) {
    console.log('Media optimizer script loaded');
    
    // Log AJAX parameters for debugging
    console.log('AJAX URL:', skyseomanager_ajax.ajax_url);
    console.log('Nonce:', skyseomanager_ajax.nonce);
    
    // Function to show toast message
    function showToast(message, type = 'success') {
        const toast = $('#sky-media-toast');
        toast.text(message)
            .removeClass('success error')
            .addClass(type)
            .fadeIn();
        
        setTimeout(function() {
            toast.fadeOut();
        }, 3000);
    }

    // Function to auto-save media data
    function saveMediaData(attachmentId, fieldType, value, field) {
        field.addClass('saving');
        console.log('Saving media data:', attachmentId, fieldType, value);
        
        return $.ajax({
            url: skyseomanager_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'save_media_data',
                nonce: skyseomanager_ajax.nonce,
                attachment_id: attachmentId,
                field_type: fieldType,
                value: value
            }
        })
        .done(function(response) {
            console.log('Save response:', response);
            if (response.success) {
                console.log('Successfully saved media data:', response.data);
                showToast('Saved successfully', 'success');
            } else {
                showToast('Error saving changes: ' + (response.data || 'Unknown error'), 'error');
                console.error('Auto-save error response:', response);
            }
        })
        .fail(function(xhr, status, error) {
            showToast('Error saving changes: ' + error, 'error');
            console.error('Save error:', {
                error: error,
                status: status,
                xhr: xhr
            });
        })
        .always(function() {
            field.removeClass('saving');
        });
    }

    // Auto-save on input change
    $(document).on('change', '.media-title, .media-alt', function() {
        const $field = $(this);
        const $row = $field.closest('tr');
        const attachmentId = $row.data('media-id');
        const fieldType = $field.hasClass('media-title') ? 'title' : 'alt';
        
        // Save the data
        saveMediaData(attachmentId, fieldType, $field.val(), $field);
    });

    // Handle AI optimization
    $('#media-ai-optimize-button').on('click', function() {
        console.log('AI Optimize button clicked');
        const $button = $(this);
        const $status = $('#media-ai-optimize-status');
        const mediaRows = $('tr[data-media-id]');
        
        if (mediaRows.length === 0) {
            showToast('No media found to optimize', 'error');
            return;
        }
        
        // Disable button
        $button.prop('disabled', true).text('Processing...');
        $status.text('0/' + mediaRows.length + ' images processed').show();
        
        let processed = 0;
        let successCount = 0;
        let errorCount = 0;
        
        // Process media one by one
        function processNextMedia(index) {
            if (index >= mediaRows.length) {
                // All media processed
                $button.prop('disabled', false).text('AI Optimize');
                $status.text(`Completed: ${successCount} successful, ${errorCount} failed`);
                setTimeout(() => {
                    $status.fadeOut();
                }, 5000);
                return;
            }
            
            const $row = $(mediaRows[index]);
            const attachmentId = $row.data('media-id');
            console.log('Processing media ID:', attachmentId);
            // Add debug: log the AI optimize request payload
            console.log('AI optimize request data for attachment ID ' + attachmentId + ':', {
                action: 'ai_optimize_media',
                nonce: skyseomanager_ajax.nonce,
                attachment_id: attachmentId
            });
            
            // Call AI optimization endpoint
            $.ajax({
                url: skyseomanager_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'ai_optimize_media',
                    nonce: skyseomanager_ajax.nonce,
                    attachment_id: attachmentId
                },
                beforeSend: function() {
                    $row.addClass('processing');
                },
                success: function(response) {
                    console.log('AI optimize response:', response);
                    if (response.success) {
                        // Update the UI with the new title and alt text
                        const $titleField = $row.find('.media-title');
                        const $altField = $row.find('.media-alt');
                        
                        $titleField.val(response.data.title);
                        $altField.val(response.data.alt_text);
                        
                        successCount++;
                        showToast('Optimized: ' + response.data.title, 'success');
                    } else {
                        console.error('Error optimizing media ' + attachmentId + ':', response.data);
                        showToast('Error: ' + (response.data || 'Unknown error'), 'error');
                        errorCount++;
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error:', error, xhr.responseText);
                    showToast('Error: ' + error, 'error');
                    errorCount++;
                },
                complete: function() {
                    $row.removeClass('processing');
                    processed++;
                    $status.text(processed + '/' + mediaRows.length + ' images processed');
                    
                    // Process next media
                    processNextMedia(index + 1);
                }
            });
        }
        
        // Start processing media
        processNextMedia(0);
    });
}); 