<?php
/**
 * Uninstall script for Site Update Logger.
 *
 * This file is called automatically when the plugin is deleted from the WordPress admin.
 * It removes all plugin data including log files and directories.
 *
 * @package Site_Update_Logger
 */

// Exit if accessed directly.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

// Define log directory constant if not already defined.
if ( ! defined( 'SULOG_LOG_DIR' ) ) {
	$sulog_uploads = wp_upload_dir();
	define( 'SULOG_LOG_DIR', trailingslashit( $sulog_uploads['basedir'] ) . 'site-update-logger/logs/' );
}

/**
 * Recursively delete a directory and all its contents.
 *
 * @param string $dir Directory path to delete.
 * @return bool True on success, false on failure.
 */
function sulog_delete_directory( $dir ) {
	if ( ! file_exists( $dir ) ) {
		return true;
	}

	// Prefer WP_Filesystem for file operations.
	require_once ABSPATH . 'wp-admin/includes/file.php';
	global $wp_filesystem;
	if ( empty( $wp_filesystem ) ) {
		WP_Filesystem();
	}

	if ( ! is_dir( $dir ) ) {
		if ( ! empty( $wp_filesystem ) && method_exists( $wp_filesystem, 'delete' ) ) {
			return $wp_filesystem->delete( $dir );
		}
		if ( function_exists( 'wp_delete_file' ) ) {
			return wp_delete_file( $dir );
		}
        // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- fallback for non-WP context
		return @unlink( $dir );
	}

	// Directory: attempt recursive delete via WP_Filesystem.
	if ( ! empty( $wp_filesystem ) && method_exists( $wp_filesystem, 'delete' ) ) {
		return $wp_filesystem->delete( $dir, true );
	}

	// Fallback to manual recursion.
	foreach ( scandir( $dir ) as $item ) {
		if ( '.' === $item || '..' === $item ) {
			continue;
		}

		if ( ! sulog_delete_directory( $dir . DIRECTORY_SEPARATOR . $item ) ) {
			return false;
		}
	}

	// Prefer WP_Filesystem rmdir if available.
	if ( ! empty( $wp_filesystem ) && method_exists( $wp_filesystem, 'rmdir' ) ) {
		return $wp_filesystem->rmdir( $dir, true );
	}

    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir -- fallback for non-WP context
	return @rmdir( $dir );
}

// Clean up any plugin options from the database (for future use if we add options).
$sulog_delete_data_on_uninstall = get_option( 'sulog_delete_data_on_uninstall', '0' );

// Only delete data if the user has chosen to do so.
if ( '1' === $sulog_delete_data_on_uninstall ) {
	// Delete all log files and the log directory.
	if ( file_exists( SULOG_LOG_DIR ) ) {
		sulog_delete_directory( SULOG_LOG_DIR );
	}

	// Clean up plugin options.
	delete_option( 'sulog_delete_data_on_uninstall' );
	delete_option( 'sulog_version' );
	delete_option( 'sulog_settings' );

	// For multisite, clean up site options as well.
	if ( is_multisite() ) {
		delete_site_option( 'sulog_delete_data_on_uninstall' );
		delete_site_option( 'sulog_version' );
		delete_site_option( 'sulog_settings' );
	}
}
