<?php
/**
 * Plugin Name: Site Update Logger
 * Description: Comprehensive WordPress Site Update Logger logs core, plugin, and theme updates, installations, activations, deactivations, deletions, and theme switches. Bilingual admin interface and configurable data management.
 * Version: 1.0.0
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Author: David Francisco
 * Author URI: https://davidfrancisco.me
 * Text Domain: site-update-logger
 * Domain Path: /languages
 *
 * @package Site_Update_Logger
 */

/**
 * Plugin bootstrap and initialization.
 *
 * @package Site_Update_Logger
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants.
if ( ! defined( 'SULOG_PLUGIN_DIR' ) ) {
	define( 'SULOG_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'SULOG_PLUGIN_URL' ) ) {
	define( 'SULOG_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}
if ( ! defined( 'SULOG_LOG_DIR' ) ) {
	// Store logs under the uploads directory in a plugin-specific folder.
	$sulog_uploads = wp_upload_dir();
	define( 'SULOG_LOG_DIR', trailingslashit( $sulog_uploads['basedir'] ) . 'site-update-logger/logs/' );
}
if ( ! defined( 'SULOG_VERSION' ) ) {
	define( 'SULOG_VERSION', '1.0.0' );
}

// Note: WordPress.org automatically loads translations from translate.wordpress.org.

// Load translations early so plugin headers and admin UI are translated.
// Try WP_LANG_DIR first (wp-content/languages/plugins/), then plugin's languages.
$sulog_locale = null;
if ( function_exists( 'determine_locale' ) ) {
	$sulog_locale = determine_locale();
} elseif ( function_exists( 'get_locale' ) ) {
	$sulog_locale = get_locale();
}
if ( $sulog_locale ) {
	$sulog_wp_mo = WP_LANG_DIR . '/plugins/site-update-logger-' . $sulog_locale . '.mo';
	if ( file_exists( $sulog_wp_mo ) ) {
		load_textdomain( 'site-update-logger', $sulog_wp_mo );
	}
	$sulog_plugin_mo = SULOG_PLUGIN_DIR . 'languages/site-update-logger-' . $sulog_locale . '.mo';
	if ( file_exists( $sulog_plugin_mo ) ) {
		load_textdomain( 'site-update-logger', $sulog_plugin_mo );
	}
}

// Admin notice when compiled .mo is missing for the current locale.
add_action(
	'admin_notices',
	function () {
		if ( ! is_admin() ) {
			return;
		}
		$sulog_locale = get_locale();
		if ( empty( $sulog_locale ) || strpos( $sulog_locale, 'en' ) === 0 ) {
			return; // English or undefined, no notice.
		}

		$plugin_mo = plugin_dir_path( __FILE__ ) . 'languages/site-update-logger-' . $sulog_locale . '.mo';
		$wp_mo = WP_LANG_DIR . '/plugins/site-update-logger-' . $sulog_locale . '.mo';
		if ( ! file_exists( $plugin_mo ) && ! file_exists( $wp_mo ) ) {
			$class = 'notice notice-warning';
			$mo_filename = 'languages/site-update-logger-' . $sulog_locale . '.mo';
			$languages_path = 'wp-content/languages/plugins/';
			/* translators: 1: .mo filename, 2: WP languages/plugins path */
			$message = sprintf( __( 'Site Update Logger: translations for your site language are not compiled. Run the provided tools/compile-translations.ps1 to generate %1$s, or upload the .mo to %2$s.', 'site-update-logger' ), $mo_filename, $languages_path );
			printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), esc_html( $message ) );
		}
	}
);

// Include required files.
require_once SULOG_PLUGIN_DIR . 'includes/class-sulog-plugin.php';
require_once SULOG_PLUGIN_DIR . 'includes/class-sulog-logger.php';
require_once SULOG_PLUGIN_DIR . 'includes/class-sulog-update-tracker.php';
require_once SULOG_PLUGIN_DIR . 'includes/class-sulog-admin.php';
// Compatibility aliases removed; plugin uses `site-update-logger` class names only.

// Initialize plugin components.
add_action(
	'init',
	function () {
		// Prefer new class names; aliases ensure old names still work.
		if ( class_exists( 'SULOG_Logger' ) ) {
			SULOG_Logger::init();
		} elseif ( class_exists( 'Site_Update_Logger_Logger' ) ) {
			Site_Update_Logger_Logger::init();
		} else {
			SULOG_Logger::init();
		}

		if ( class_exists( 'SULOG_Update_Tracker' ) ) {
			SULOG_Update_Tracker::init();
		} elseif ( class_exists( 'Site_Update_Logger_Update_Tracker' ) ) {
			Site_Update_Logger_Update_Tracker::init();
		} else {
			SULOG_Update_Tracker::init();
		}

		if ( class_exists( 'SULOG_Admin' ) ) {
			SULOG_Admin::init();
		} elseif ( class_exists( 'Site_Update_Logger_Admin' ) ) {
			Site_Update_Logger_Admin::init();
		} else {
			SULOG_Admin::init();
		}
	}
);

// Register activation/deactivation hooks.
register_activation_hook(
	__FILE__,
	function () {
		if ( class_exists( 'SULOG_Logger' ) ) {
			SULOG_Logger::activate();
		} elseif ( class_exists( 'Site_Update_Logger_Logger' ) ) {
			Site_Update_Logger_Logger::activate();
		}
		if ( function_exists( 'sulog_migrate_options' ) ) {
			sulog_migrate_options();
		}
	}
);
register_deactivation_hook(
	__FILE__,
	function () {
		if ( class_exists( 'SULOG_Logger' ) ) {
			SULOG_Logger::deactivate();
		} elseif ( class_exists( 'Site_Update_Logger_Logger' ) ) {
			Site_Update_Logger_Logger::deactivate();
		}
	}
);

/**
 * Migrate legacy options from the old prefix `wp_update_tracker_*` to
 * the new `site_update_logger_*` keys. Runs once on activation (and
 * on admin_init as a fallback) and is safe to call repeatedly.
 */
function sulog_migrate_options() {
	if ( defined( 'SULOG_MIGRATED' ) ) {
		return;
	}

	// Avoid running repeatedly; check both legacy flag and new flag.
	if ( get_option( 'sulog_migration_done', '0' ) === '1' || get_option( 'site_update_logger_migration_done', '0' ) === '1' ) {
		return;
	}

	global $wpdb;

	// Static option name mappings.
	$mappings = array(
		'wp_update_tracker_delete_data_on_uninstall' => 'site_update_logger_delete_data_on_uninstall',
		'wp_update_tracker_version' => 'site_update_logger_version',
		'wp_update_tracker_settings' => 'site_update_logger_settings',
		'wp_update_tracker_last_version' => 'site_update_logger_last_version',
		'wp_update_tracker_pre_update_wp_version' => 'site_update_logger_pre_update_wp_version',
	);

	foreach ( $mappings as $old => $new ) {
		$val = get_option( $old );
		if ( false !== $val && get_option( $new ) === false ) {
			update_option( $new, $val );
		}
	}

	// Also migrate current site_update_logger_* keys to new sulog_* keys.
	$mappings2 = array(
		'site_update_logger_delete_data_on_uninstall' => 'sulog_delete_data_on_uninstall',
		'site_update_logger_version' => 'sulog_version',
		'site_update_logger_settings' => 'sulog_settings',
		'site_update_logger_last_version' => 'sulog_last_version',
		'site_update_logger_pre_update_wp_version' => 'sulog_pre_update_wp_version',
	);
	foreach ( $mappings2 as $old => $new ) {
		$val = get_option( $old );
		if ( false !== $val && get_option( $new ) === false ) {
			update_option( $new, $val );
		}
	}

	// Migrate pattern options (plugin_*, theme_*, pre_update_plugin_*, pre_update_theme_*).
	$like_patterns = array(
		'wp_update_tracker_plugin_%',
		'wp_update_tracker_theme_%',
		'wp_update_tracker_pre_update_plugin_%',
		'wp_update_tracker_pre_update_theme_%',
		'wp_update_tracker_pre_update_plugin_%',
		'wp_update_tracker_pre_update_theme_%',
	);

	foreach ( $like_patterns as $pattern ) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- needed to locate option rows by pattern for migration.
		$rows = $wpdb->get_results( $wpdb->prepare( "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE %s", $pattern ) );
		if ( $rows ) {
			foreach ( $rows as $row ) {
				$old_name = $row->option_name;
				$new_name = str_replace( 'wp_update_tracker_', 'site_update_logger_', $old_name );
				// Use maybe_unserialize to restore original PHP value.
				$value = maybe_unserialize( $row->option_value );
				if ( get_option( $new_name ) === false ) {
					update_option( $new_name, $value );
				}
			}
		}
	}

	// Migrate network/site options if present (multisite).
	if ( is_multisite() ) {
		$site_keys = array(
			'wp_update_tracker_delete_data_on_uninstall',
			'wp_update_tracker_version',
			'wp_update_tracker_settings',
		);
		foreach ( $site_keys as $old_site_key ) {
			$val = get_site_option( $old_site_key );
			if ( false !== $val && get_site_option( str_replace( 'wp_update_tracker_', 'site_update_logger_', $old_site_key ) ) === false ) {
				update_site_option( str_replace( 'wp_update_tracker_', 'site_update_logger_', $old_site_key ), $val );
			}
		}
	}

	// Mark migration complete.
	update_option( 'sulog_migration_done', '1' );
}

// Run migration on admin_init as a fallback in case activation hook was missed.
add_action(
	'admin_init',
	function () {
		if ( get_option( 'sulog_migration_done', '0' ) !== '1' ) {
			sulog_migrate_options();
		}
	}
);
