<?php
/**
 * Update tracker class for Site Update Logger
 *
 * Hooks into update processes and logs changes/errors.
 *
 * @package Site_Update_Logger
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Primary class is prefixed; provide aliases for older class names below.
/**
 * SULOG_Update_Tracker class
 *
 * Tracks update lifecycle events and logs them.
 */
class SULOG_Update_Tracker {
	/**
	 * Initialize hooks.
	 */
	public static function init() {
		// Hook into core and plugin updates.
		add_action( 'upgrader_process_complete', array( __CLASS__, 'track_update' ), 10, 2 );
		add_action( 'admin_init', array( __CLASS__, 'setup_error_handling' ) );
		add_action( 'admin_init', array( __CLASS__, 'store_current_versions' ) );

		// Store versions on various hooks to ensure they're always current.
		add_action( 'plugins_loaded', array( __CLASS__, 'store_current_versions' ) );
		add_action( 'init', array( __CLASS__, 'store_current_versions' ) );

		// Specific hooks for core updates.
		add_action( '_core_updated_successfully', array( __CLASS__, 'track_core_update' ), 10, 1 );

		// Store pre-update versions.
		add_action( 'upgrader_pre_install', array( __CLASS__, 'store_pre_update_versions' ), 9, 2 );

		// Track plugin lifecycle events.
		add_action( 'activated_plugin', array( __CLASS__, 'track_plugin_activation' ), 10, 2 );
		add_action( 'deactivated_plugin', array( __CLASS__, 'track_plugin_deactivation' ), 10, 2 );
		add_action( 'delete_plugin', array( __CLASS__, 'track_plugin_deletion' ), 10, 1 );
		add_action( 'deleted_plugin', array( __CLASS__, 'track_plugin_deleted' ), 10, 2 );

		// Track theme lifecycle events.
		add_action( 'switch_theme', array( __CLASS__, 'track_theme_switch' ), 10, 3 );
		add_action( 'delete_theme', array( __CLASS__, 'track_theme_deletion' ), 10, 1 );
		add_action( 'deleted_theme', array( __CLASS__, 'track_theme_deleted' ), 10, 2 );
	}

	/**
	 * Track updates (core, plugin).
	 *
	 * @param object $upgrader Upgrader instance.
	 * @param array  $options  Options passed by upgrader.
	 */
	public static function track_update( $upgrader, $options ) {
		if ( ! isset( $options['type'] ) || ! isset( $options['action'] ) ) {
			return;
		}
		if ( 'update' !== $options['action'] ) {
			// Track plugin installations.
			if ( 'install' === $options['action'] && 'plugin' === $options['type'] && ! empty( $options['plugins'] ) ) {
				foreach ( $options['plugins'] as $plugin ) {
					if ( ! function_exists( 'get_plugin_data' ) ) {
						require_once ABSPATH . 'wp-admin/includes/plugin.php';
					}
					$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );
					SULOG_Logger::info( sprintf( '%s: %s (Version: %s)', __( 'Plugin installed', 'site-update-logger' ), $plugin_data['Name'], $plugin_data['Version'] ) );
				}
			}
			return;
		}

		// Core update.
		if ( 'core' === $options['type'] ) {
			$old_version = get_option( 'sulog_pre_update_wp_version', get_bloginfo( 'version' ) );
			$new_version = get_bloginfo( 'version' );
			if ( $old_version !== $new_version ) {
				SULOG_Logger::info( sprintf( '%s %s %s %s', __( 'WordPress updated from', 'site-update-logger' ), $old_version, __( 'to', 'site-update-logger' ), $new_version ) );
				update_option( 'sulog_last_version', $new_version );
				delete_option( 'sulog_pre_update_wp_version' ); // Clean up.
			}
		}

		// Plugin update.
		if ( 'plugin' === $options['type'] && ! empty( $options['plugins'] ) ) {
			foreach ( $options['plugins'] as $plugin ) {
				$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );
				$old_version = get_option( 'sulog_pre_update_plugin_' . md5( $plugin ), $plugin_data['Version'] );
				$new_version = $plugin_data['Version'];
				if ( $old_version !== $new_version ) {
					SULOG_Logger::info( sprintf( '%s: %s (%s > %s)', __( 'Plugin updated', 'site-update-logger' ), $plugin_data['Name'], $old_version, $new_version ) );
					update_option( 'sulog_plugin_' . md5( $plugin ), $new_version );
					delete_option( 'sulog_pre_update_plugin_' . md5( $plugin ) ); // Clean up.
				}
			}
		}

		// Theme update.
		if ( 'theme' === $options['type'] && ! empty( $options['themes'] ) ) {
			foreach ( $options['themes'] as $theme_stylesheet ) {
				$theme = wp_get_theme( $theme_stylesheet );
				$old_version = get_option( 'sulog_pre_update_theme_' . md5( $theme_stylesheet ), $theme->get( 'Version' ) );
				$new_version = $theme->get( 'Version' );
				if ( $old_version !== $new_version ) {
					SULOG_Logger::info( sprintf( '%s: %s (%s > %s)', __( 'Theme updated', 'site-update-logger' ), $theme->get( 'Name' ), $old_version, $new_version ) );
					update_option( 'sulog_theme_' . md5( $theme_stylesheet ), $new_version );
					delete_option( 'sulog_pre_update_theme_' . md5( $theme_stylesheet ) ); // Clean up.
				}
			}
		}

		// Theme installation.
		if ( 'install' === $options['action'] && 'theme' === $options['type'] && ! empty( $options['themes'] ) ) {
			foreach ( $options['themes'] as $theme_stylesheet ) {
				$theme = wp_get_theme( $theme_stylesheet );
				SULOG_Logger::info( sprintf( '%s: %s (Version: %s)', __( 'Theme installed', 'site-update-logger' ), $theme->get( 'Name' ), $theme->get( 'Version' ) ) );
			}
		}
	}

	/**
	 * Track core updates using the _core_updated_successfully hook.
	 *
	 * @param string $new_version New WP version.
	 */
	public static function track_core_update( $new_version ) {
		$old_version = get_option( 'site_update_logger_pre_update_wp_version', get_bloginfo( 'version' ) );
		if ( $old_version !== $new_version ) {
			SULOG_Logger::info( sprintf( '%s %s %s %s', __( 'WordPress updated from', 'site-update-logger' ), $old_version, __( 'to', 'site-update-logger' ), $new_version ) );
			update_option( 'sulog_last_version', $new_version );
			delete_option( 'sulog_pre_update_wp_version' ); // Clean up.
		}
	}

	/**
	 * Store current versions before updates.
	 */
	public static function store_current_versions() {
		// Store current WordPress version.
		update_option( 'sulog_last_version', get_bloginfo( 'version' ) );

		// Store current plugin versions.
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugins = get_plugins();
		foreach ( $plugins as $plugin_file => $plugin_data ) {
			update_option( 'sulog_plugin_' . md5( $plugin_file ), $plugin_data['Version'] );
		}

		// Store current theme versions.
		$themes = wp_get_themes();
		foreach ( $themes as $theme_stylesheet => $theme ) {
			update_option( 'sulog_theme_' . md5( $theme_stylesheet ), $theme->get( 'Version' ) );
		}
	}

	/**
	 * Store versions before update starts.
	 *
	 * @param mixed $response  Upgrader response.
	 * @param mixed $hook_extra Hook extra data.
	 */
	public static function store_pre_update_versions( $response, $hook_extra ) {
		// Store pre-update versions in separate options.
		update_option( 'sulog_pre_update_wp_version', get_bloginfo( 'version' ) );

		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugins = get_plugins();
		foreach ( $plugins as $plugin_file => $plugin_data ) {
			update_option( 'sulog_pre_update_plugin_' . md5( $plugin_file ), $plugin_data['Version'] );
		}

		// Store pre-update theme versions.
		$themes = wp_get_themes();
		foreach ( $themes as $theme_stylesheet => $theme ) {
			update_option( 'sulog_pre_update_theme_' . md5( $theme_stylesheet ), $theme->get( 'Version' ) );
		}
	}

	/**
	 * Setup error handling for update process.
	 */
	public static function setup_error_handling() {
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_set_error_handler -- intentional runtime error handler for update lifecycle.
		set_error_handler( array( __CLASS__, 'handle_php_error' ) );
	}

	/**
	 * Track plugin activation.
	 *
	 * @param string $plugin       Plugin file path.
	 * @param bool   $network_wide Network activation flag.
	 */
	public static function track_plugin_activation( $plugin, $network_wide ) {
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );
		$scope = $network_wide ? __( 'network-wide', 'site-update-logger' ) : __( 'single site', 'site-update-logger' );
		SULOG_Logger::info( sprintf( '%s: %s (Version: %s, Scope: %s)', __( 'Plugin activated', 'site-update-logger' ), $plugin_data['Name'], $plugin_data['Version'], $scope ) );
	}

	/**
	 * Track plugin deactivation.
	 *
	 * @param string $plugin       Plugin file path.
	 * @param bool   $network_wide Network activation flag.
	 */
	public static function track_plugin_deactivation( $plugin, $network_wide ) {
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );
		$scope = $network_wide ? __( 'network-wide', 'site-update-logger' ) : __( 'single site', 'site-update-logger' );
		SULOG_Logger::info( sprintf( '%s: %s (Version: %s, Scope: %s)', __( 'Plugin deactivated', 'site-update-logger' ), $plugin_data['Name'], $plugin_data['Version'], $scope ) );
	}

	/**
	 * Track plugin deletion (before deletion).
	 *
	 * @param string $plugin_file Plugin file path.
	 */
	public static function track_plugin_deletion( $plugin_file ) {
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_file );
		SULOG_Logger::info( sprintf( '%s: %s (Version: %s)', __( 'Plugin deletion initiated', 'site-update-logger' ), $plugin_data['Name'], $plugin_data['Version'] ) );
	}

	/**
	 * Track plugin deleted (after deletion).
	 *
	 * @param string $plugin_file Plugin file path.
	 * @param bool   $deleted     Deletion success flag.
	 */
	public static function track_plugin_deleted( $plugin_file, $deleted ) {
		if ( $deleted ) {
			SULOG_Logger::info( sprintf( '%s: %s', __( 'Plugin deleted successfully', 'site-update-logger' ), basename( $plugin_file ) ) );
		} else {
			SULOG_Logger::error( sprintf( '%s: %s', __( 'Plugin deletion failed', 'site-update-logger' ), basename( $plugin_file ) ) );
		}
	}

	/**
	 * Handle PHP errors during update.
	 *
	 * @param int    $errno   Error number.
	 * @param string $errstr  Error message.
	 * @param string $errfile File where error occurred.
	 * @param int    $errline Line number.
	 */
	public static function handle_php_error( $errno, $errstr, $errfile, $errline ) {
		$types = array(
			E_ERROR => 'ERROR',
			E_WARNING => 'WARNING',
			E_NOTICE => 'NOTICE',
			E_USER_ERROR => 'ERROR',
			E_USER_WARNING => 'WARNING',
			E_USER_NOTICE => 'NOTICE',
		);
		$type = isset( $types[ $errno ] ) ? $types[ $errno ] : 'ERROR';
		/* translators: 1: error type, 2: error message, 3: file, 4: line number */
		$msg = sprintf( __( 'PHP %1$s: %2$s in %3$s on line %4$d', 'site-update-logger' ), $type, $errstr, $errfile, $errline );
		SULOG_Logger::error( $msg );
	}

	/**
	 * Track theme switch.
	 *
	 * @param WP_Theme $new_name  New theme object.
	 * @param WP_Theme $new_theme New theme object.
	 * @param WP_Theme $old_theme Old theme object.
	 */
	public static function track_theme_switch( $new_name, $new_theme, $old_theme ) {
		if ( $old_theme && $old_theme->get_stylesheet() !== $new_theme->get_stylesheet() ) {
			SULOG_Logger::info( sprintf( '%s %s %s %s', __( 'Theme switched from', 'site-update-logger' ), $old_theme->get( 'Name' ), __( 'to', 'site-update-logger' ), $new_theme->get( 'Name' ) ) );
		} else {
			SULOG_Logger::info( sprintf( '%s: %s', __( 'Theme activated', 'site-update-logger' ), $new_theme->get( 'Name' ) ) );
		}
	}

	/**
	 * Track theme deletion (before deletion).
	 *
	 * @param string $stylesheet Theme stylesheet.
	 */
	public static function track_theme_deletion( $stylesheet ) {
		$theme = wp_get_theme( $stylesheet );
		if ( $theme->exists() ) {
			SULOG_Logger::info( sprintf( '%s: %s (Version: %s)', __( 'Theme deletion initiated', 'site-update-logger' ), $theme->get( 'Name' ), $theme->get( 'Version' ) ) );
		}
	}

	/**
	 * Track theme deleted (after deletion).
	 *
	 * @param string $stylesheet Theme stylesheet.
	 * @param bool   $deleted    Deletion success flag.
	 */
	public static function track_theme_deleted( $stylesheet, $deleted ) {
		if ( $deleted ) {
			SULOG_Logger::info( sprintf( '%s: %s', __( 'Theme deleted successfully', 'site-update-logger' ), $stylesheet ) );
		} else {
			SULOG_Logger::error( sprintf( '%s: %s', __( 'Theme deletion failed', 'site-update-logger' ), $stylesheet ) );
		}
	}
}

// Backwards-compatible aliases for older installs.
if ( ! class_exists( 'WP_Update_Tracker_Update_Tracker' ) ) {
	class_alias( 'SULOG_Update_Tracker', 'WP_Update_Tracker_Update_Tracker' );
}
if ( ! class_exists( 'Site_Update_Logger_Update_Tracker' ) ) {
	class_alias( 'SULOG_Update_Tracker', 'Site_Update_Logger_Update_Tracker' );

}
