<?php
/**
 * Site Update Logger logging utilities
 *
 * Handles logging of updates and errors.
 *
 * @package Site_Update_Logger
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Primary class is prefixed; provide aliases for older class names below.
/**
 * Logging utilities
 *
 * Logger for Site Update Logger.
 */
class SULOG_Logger {
	/**
	 * Log directory path.
	 *
	 * @var string
	 */
	private static $log_dir;
	/**
	 * Initialization flag.
	 *
	 * @var bool
	 */
	private static $initialized = false;

	/**
	 * Initialize logger: create log directory if needed.
	 */
	public static function init() {
		if ( self::$initialized ) {
			return;
		}

		self::$log_dir = rtrim( SULOG_LOG_DIR, '/\\' ) . '/';

		if ( ! is_dir( self::$log_dir ) ) {
			wp_mkdir_p( self::$log_dir );
		}

		self::$initialized = true;
	}

	/**
	 * Get log file path for a specific day (default: today).
	 *
	 * @param string|null $date Date in Y-m-d format.
	 * @return string
	 */
	public static function get_log_file_path( $date = null ) {
		if ( null === $date ) {
			$date = current_time( 'Y-m-d' );
		}
		return self::$log_dir . $date . '.log';
	}

	/**
	 * Write a log entry (prepends to file so newest entries appear first).
	 *
	 * @param string $type    Entry type.
	 * @param string $message Message to log.
	 */
	public static function log( $type, $message ) {
		self::init();
		$datetime = current_time( 'Y-m-d H:i:s' );
		$entry    = '[' . $datetime . '] ' . strtoupper( $type ) . ': ' . $message . "\n";
		$file     = self::get_log_file_path();

		// Prepend to file so newest entries are at the top.
		require_once ABSPATH . 'wp-admin/includes/file.php';
		global $wp_filesystem;
		if ( empty( $wp_filesystem ) ) {
			WP_Filesystem();
		}
		$existing = '';
		if ( ! empty( $wp_filesystem ) && method_exists( $wp_filesystem, 'exists' ) && $wp_filesystem->exists( $file ) ) {
			$existing = $wp_filesystem->get_contents( $file );
		} elseif ( file_exists( $file ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- fallback when WP_Filesystem not available
			$existing = file_get_contents( $file );
		}

		if ( ! empty( $wp_filesystem ) && method_exists( $wp_filesystem, 'put_contents' ) ) {
			$success = $wp_filesystem->put_contents( $file, $entry . $existing );
			if ( false === $success ) {
				self::error( sprintf( 'Failed to write log file: %s', $file ) );
			}
		} elseif ( false === file_put_contents( $file, $entry . $existing, LOCK_EX ) ) { // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- fallback when WP_Filesystem not available
			self::error( sprintf( 'Failed to write log file: %s', $file ) );
		}

		self::rotate_logs( $file );
	}

	/**
	 * Log info.
	 *
	 * @param string $message Message to log.
	 */
	public static function info( $message ) {
		self::log( 'INFO', $message );
	}

	/**
	 * Log error.
	 *
	 * @param string $message Message to log.
	 */
	public static function error( $message ) {
		self::log( 'ERROR', $message );
	}

	/**
	 * Log warning.
	 *
	 * @param string $message Message to log.
	 */
	public static function warning( $message ) {
		self::log( 'WARNING', $message );
	}

	/**
	 * Log user note.
	 *
	 * @param string $message Message to log.
	 */
	public static function user_note( $message ) {
		self::log( 'USER NOTE', $message );
	}

	/**
	 * Retrieve logs for a specific month or all months (reverse chronological).
	 *
	 * @param int|null $year  Year to filter.
	 * @param int|null $month Month to filter.
	 * @return string
	 */
	public static function get_logs( $year = null, $month = null ) {
		self::init();

		$files = glob( self::$log_dir . '/*.log' );
		// Sort files by date inferred from filename (YYYY-MM-DD.log or log-YYYY-MM.log).
		if ( $files ) {
			usort(
				$files,
				function ( $a, $b ) {
					$extract = function ( $file ) {
						$base = basename( $file );
						// Daily pattern.
						if ( preg_match( '/(\d{4})-(\d{2})-(\d{2})/', $base, $m ) ) {
							return (int) sprintf( '%04d%02d%02d', $m[1], $m[2], $m[3] );
						}
						// Monthly pattern.
						if ( preg_match( '/log-(\d{4})-(\d{2})/', $base, $m ) ) {
							return (int) sprintf( '%04d%02d%02d', $m[1], $m[2], 1 );
						}
						// Fallback to file modification time as YYYYMMDD integer.
						$ts = filemtime( $file );
						// Use gmdate to avoid timezone-dependent date conversions.
						return (int) gmdate( 'Ymd', $ts );
					};

					$va = $extract( $a );
					$vb = $extract( $b );

					if ( $va === $vb ) {
						// tiebreaker: newest modification time first.
						$ma = filemtime( $a );
						$mb = filemtime( $b );
						if ( $ma === $mb ) {
							// final tiebreaker: filename descending.
							return strcmp( basename( $b ), basename( $a ) );
						}
						return $mb <=> $ma;
					}

					// Descending by inferred date.
					return $vb <=> $va;
				}
			);
		}
		$logs = '';

		foreach ( $files as $file ) {
			$basename = basename( $file );

			// Support both daily Y-m-d.log and monthly log-YYYY-MM.log for backward compatibility.
			$file_year  = null;
			$file_month = null;
			if ( preg_match( '/^(\d{4})-(\d{2})-\d{2}\.log$/', $basename, $m ) ) {
				$file_year  = $m[1];
				$file_month = $m[2];
			} elseif ( preg_match( '/^log-(\d{4})-(\d{2})\.log$/', $basename, $m ) ) {
				$file_year  = $m[1];
				$file_month = $m[2];
			}

			// Filter by year/month if provided.
			if ( null !== $year || null !== $month ) {
				if ( null === $file_year || null === $file_month ) {
					continue;
				}
				if ( null !== $year && sprintf( '%04d', $year ) !== $file_year ) {
					continue;
				}
				if ( null !== $month && sprintf( '%02d', $month ) !== $file_month ) {
					continue;
				}
			}

			require_once ABSPATH . 'wp-admin/includes/file.php';
			global $wp_filesystem;
			if ( empty( $wp_filesystem ) ) {
				WP_Filesystem();
			}
			$content = '';
			if ( ! empty( $wp_filesystem ) && method_exists( $wp_filesystem, 'get_contents' ) ) {
				$content = $wp_filesystem->get_contents( $file );
			} else {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- fallback when WP_Filesystem not available
				$content = file_get_contents( $file );
			}
			$lines = array_filter( explode( "\n", $content ) );
			// Don't reverse - entries are already in descending order (newest first).
			$logs .= implode( "\n", $lines ) . "\n\n";
		}

		return $logs;
	}

	/**
	 * Get list of available log months.
	 *
	 * @return array
	 */
	public static function get_available_months() {
		self::init();
		$files  = glob( self::$log_dir . '/*.log' );
		$months = array();
		foreach ( $files as $file ) {
			$matches = array();
			if ( preg_match( '/(\d{4})-(\d{2})-\d{2}\.log$/', basename( $file ), $matches ) ||
				preg_match( '/log-(\d{4})-(\d{2})\.log$/', basename( $file ), $matches ) ) {
				$months[] = array(
					'year'  => $matches[1],
					'month' => $matches[2],
					'label' => date_i18n( 'F Y', mktime( 0, 0, 0, $matches[2], 1, $matches[1] ) ),
				);
			}
		}
		// Deduplicate and sort descending.
		$unique = array();
		foreach ( $months as $m ) {
			$key            = $m['year'] . '-' . $m['month'];
			$unique[ $key ] = $m;
		}
		usort(
			$unique,
			function ( $a, $b ) {
				$a_key = $a['year'] . $a['month'];
				$b_key = $b['year'] . $b['month'];
				return strcmp( $b_key, $a_key ); // Descending.
			}
		);
		return array_values( $unique );
	}

	/**
	 * Rotate logs if file exceeds 5MB.
	 *
	 * @param string $file File path.
	 */
	public static function rotate_logs( $file ) {
		if ( file_exists( $file ) && filesize( $file ) > 5 * 1024 * 1024 ) {
			$archive = $file . '.bak_' . time();
			// Use WP_Filesystem when available for safer file operations.
			require_once ABSPATH . 'wp-admin/includes/file.php';
			global $wp_filesystem;
			if ( empty( $wp_filesystem ) ) {
				WP_Filesystem();
			}
			$renamed = false;
			if ( ! empty( $wp_filesystem ) && method_exists( $wp_filesystem, 'move' ) ) {
				$renamed = (bool) $wp_filesystem->move( $file, $archive, true );
			} else {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- fallback when WP_Filesystem not available
				$renamed = rename( $file, $archive );
			}
			if ( ! $renamed ) {
				self::error( sprintf( 'Failed to archive log file: %s', $file ) );
			}
		}
	}

	/**
	 * Activation: create log directory.
	 */
	public static function activate() {
		if ( ! is_dir( SULOG_LOG_DIR ) ) {
			wp_mkdir_p( SULOG_LOG_DIR );
		}
	}

	/**
	 * Deactivation: no action needed.
	 */
	public static function deactivate() {
		// No action.
	}
}

// Backwards-compatible aliases for older installs.
if ( ! class_exists( 'WP_Update_Tracker_Logger' ) ) {
	class_alias( 'SULOG_Logger', 'WP_Update_Tracker_Logger' );
}
if ( ! class_exists( 'Site_Update_Logger_Logger' ) ) {
	class_alias( 'SULOG_Logger', 'Site_Update_Logger_Logger' );

}
