<?php
/**
 * Site Update Logger Admin Class
 * Handles admin menu, log display, download, and notes.
 *
 * @package Site_Update_Logger
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Primary class is prefixed; provide aliases for older class names below.
/**
 * Class SULOG_Admin.
 * Admin controller for Site Update Logger.
 */
class SULOG_Admin {
	/**
	 * Initialize hooks.
	 */
	public static function init() {
		add_action( 'admin_menu', array( __CLASS__, 'add_menu' ) );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_assets' ) );
		add_action( 'admin_post_site_update_logger_download', array( __CLASS__, 'handle_download' ) );
		add_action( 'admin_post_site_update_logger_add_note', array( __CLASS__, 'handle_add_note' ) );
		add_action( 'admin_post_site_update_logger_save_settings', array( __CLASS__, 'handle_save_settings' ) );
	}

	/**
	 * Add admin menu page.
	 */
	public static function add_menu() {
		add_menu_page(
			esc_html__( 'Site Update Logger', 'site-update-logger' ),
			esc_html__( 'Site Update Logger', 'site-update-logger' ),
			'manage_options',
			'site-update-logger',
			array( __CLASS__, 'render_page' ),
			'dashicons-update',
			80
		);

		add_submenu_page(
			'site-update-logger',
			esc_html__( 'Settings', 'site-update-logger' ),
			esc_html__( 'Settings', 'site-update-logger' ),
			'manage_options',
			'site-update-logger-settings',
			array( __CLASS__, 'render_settings_page' )
		);
	}

	/**
	 * Enqueue admin CSS.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public static function enqueue_assets( $hook ) {
		if ( 'toplevel_page_site-update-logger' !== $hook ) {
			return;
		}
		wp_enqueue_style( 'site-update-logger-admin', SULOG_PLUGIN_URL . 'assets/css/admin.css', array(), SULOG_VERSION );
	}

	/**
	 * Render admin page.
	 */
	public static function render_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to access this page.', 'site-update-logger' ) );
		}

		// Get selected month or show all.
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET filter for UI; no action performed.
		$year_month = isset( $_GET['year_month'] ) ? sanitize_text_field( wp_unslash( $_GET['year_month'] ) ) : '';
		$selected_year = ! empty( $year_month ) ? substr( $year_month, 0, 4 ) : null;
		$selected_month = ! empty( $year_month ) ? substr( $year_month, 5, 2 ) : null;

		// Get logs.
		$logs = SULOG_Logger::get_logs( $selected_year, $selected_month );
		$available_months = SULOG_Logger::get_available_months();

		$nonce_download = wp_create_nonce( 'site_update_logger_download' );
		$nonce_note = wp_create_nonce( 'site_update_logger_add_note' );
		?>
		<div class="wrap site-update-logger-admin">
			<h1><?php esc_html_e( 'Site Update Logger Logs', 'site-update-logger' ); ?></h1>

			<!-- Month Selector -->
			<form method="get" action="" style="margin-bottom: 20px;">
				<input type="hidden" name="page" value="site-update-logger">
				<label for="log-month"><?php esc_html_e( 'Select Month:', 'site-update-logger' ); ?></label>
				<select name="year_month" id="log-month" onchange="this.form.submit()">
					<option value=""><?php esc_html_e( 'All Months', 'site-update-logger' ); ?></option>
					<?php foreach ( $available_months as $month_data ) : ?>
						<?php $value = $month_data['year'] . '-' . $month_data['month']; ?>
						<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $year_month, $value ); ?>>
							<?php echo esc_html( $month_data['label'] ); ?>
						</option>
					<?php endforeach; ?>
				</select>
				<noscript><button type="submit" class="button"><?php esc_html_e( 'Filter', 'site-update-logger' ); ?></button></noscript>
			</form>

			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<input type="hidden" name="action" value="site_update_logger_download">
				<input type="hidden" name="_wpnonce" value="<?php echo esc_attr( $nonce_download ); ?>">
				<?php if ( $selected_year && $selected_month ) : ?>
					<input type="hidden" name="year" value="<?php echo esc_attr( $selected_year ); ?>">
					<input type="hidden" name="month" value="<?php echo esc_attr( $selected_month ); ?>">
				<?php endif; ?>
				<button type="submit" class="button button-primary">
					<?php esc_html_e( 'Download Log File', 'site-update-logger' ); ?>
				</button>
			</form>

			<h2><?php esc_html_e( 'Add Note/Observation', 'site-update-logger' ); ?></h2>
			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<input type="hidden" name="action" value="site_update_logger_add_note">
				<input type="hidden" name="_wpnonce" value="<?php echo esc_attr( $nonce_note ); ?>">
				<textarea name="note" rows="3" cols="60" required></textarea><br>
				<button type="submit" class="button">
					<?php esc_html_e( 'Add Note', 'site-update-logger' ); ?>
				</button>
			</form>
			<h2><?php esc_html_e( 'Log Entries', 'site-update-logger' ); ?></h2>
			<pre class="site-update-logger-log" style="background:#fff; border:1px solid #ccc; padding:10px; max-height:400px; overflow:auto;"><?php echo esc_html( $logs ); ?></pre>
		</div>
		<?php
	}

	/**
	 * Handle log file download.
	 */
	public static function handle_download() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Permission denied.', 'site-update-logger' ) );
		}
		check_admin_referer( 'site_update_logger_download' );

		// Optional month filter.
		$year = isset( $_POST['year'] ) ? sanitize_text_field( wp_unslash( $_POST['year'] ) ) : null;
		$month = isset( $_POST['month'] ) ? sanitize_text_field( wp_unslash( $_POST['month'] ) ) : null;

		// Aggregate logs (already filtered in getter).
		$content = SULOG_Logger::get_logs( $year, $month );
		if ( empty( $content ) ) {
			wp_die( esc_html__( 'Log file not found.', 'site-update-logger' ) );
		}

		$filename = $year && $month
			? sprintf( 'logs-%s-%s.txt', $year, $month )
			: 'logs-all.txt';

		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: text/plain' );
		header( 'Content-Disposition: attachment; filename="' . rawurlencode( $filename ) . '"' );
		header( 'Content-Length: ' . strlen( $content ) );
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- intentionally outputting file contents for download
		echo $content;
		exit;
	}

	/**
	 * Handle adding user note.
	 */
	public static function handle_add_note() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Permission denied.', 'site-update-logger' ) );
		}
		check_admin_referer( 'site_update_logger_add_note' );
		$note = isset( $_POST['note'] ) ? sanitize_text_field( wp_unslash( $_POST['note'] ) ) : '';
		if ( ! empty( $note ) ) {
			SULOG_Logger::user_note( $note );
		}
		wp_safe_redirect( admin_url( 'admin.php?page=site-update-logger' ) );
		exit;
	}

	/**
	 * Render settings page.
	 */
	public static function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to access this page.', 'site-update-logger' ) );
		}

		$delete_data_on_uninstall = get_option( 'sulog_delete_data_on_uninstall', '0' );
		$nonce = wp_create_nonce( 'site_update_logger_save_settings' );

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- UI feedback via GET param.
		if ( isset( $_GET['settings-updated'] ) && wp_unslash( $_GET['settings-updated'] ) == 'true' ) {
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved successfully.', 'site-update-logger' ) . '</p></div>';
		}
		?>
		<div class="wrap site-update-logger-settings">
			<h1><?php esc_html_e( 'Site Update Logger Settings', 'site-update-logger' ); ?></h1>

			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<input type="hidden" name="action" value="site_update_logger_save_settings">
				<input type="hidden" name="_wpnonce" value="<?php echo esc_attr( $nonce ); ?>">

				<table class="form-table">
					<tr>
						<th scope="row"><?php esc_html_e( 'Data Management', 'site-update-logger' ); ?></th>
						<td>
							<fieldset>
								<legend class="screen-reader-text">
									<span><?php esc_html_e( 'Delete plugin data on uninstall', 'site-update-logger' ); ?></span>
								</legend>
								<label for="delete_data_on_uninstall">
									<input name="delete_data_on_uninstall" type="checkbox" id="delete_data_on_uninstall" value="1" <?php checked( $delete_data_on_uninstall, '1' ); ?>>
									<?php esc_html_e( 'Delete all plugin data (logs and settings) when the plugin is uninstalled', 'site-update-logger' ); ?>
								</label>
								<p class="description">
									<?php esc_html_e( 'If unchecked, log files will be preserved when you delete the plugin. You can manually delete them later if needed.', 'site-update-logger' ); ?>
								</p>
							</fieldset>
						</td>
					</tr>
				</table>

				<?php submit_button( esc_html__( 'Save Settings', 'site-update-logger' ) ); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Handle saving settings.
	 */
	public static function handle_save_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Permission denied.', 'site-update-logger' ) );
		}
		check_admin_referer( 'site_update_logger_save_settings' );

		$delete_data = isset( $_POST['delete_data_on_uninstall'] ) ? '1' : '0';
		update_option( 'sulog_delete_data_on_uninstall', $delete_data );

		wp_safe_redirect( add_query_arg( 'settings-updated', 'true', admin_url( 'admin.php?page=site-update-logger-settings' ) ) );
		exit;
	}
}

// Backwards-compatible aliases for older installs.
if ( ! class_exists( 'WP_Update_Tracker_Admin' ) ) {
	class_alias( 'SULOG_Admin', 'WP_Update_Tracker_Admin' );
}
if ( ! class_exists( 'Site_Update_Logger_Admin' ) ) {
	class_alias( 'SULOG_Admin', 'Site_Update_Logger_Admin' );
}
