<?php
/**
 * Plugin Name:       Site Security Auditor
 * Description:       Audits plugins, file integrity, and hardening to improve site security.
 * Version:           1.3.3
 * Author:            Gopal Bogati
 * Author URI:        https://wordpress.org/
 * Text Domain:       site-security-auditor
 * Domain Path:       /languages
 * Requires at least: 5.8
 * Tested up to:      6.8
 * Requires PHP:      7.4
 * License:           GPL-2.0+
 */
define( 'SITESEAU_PLUGIN_FILE', __FILE__ );
define( 'SITESEAU_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SITESEAU_PLUGIN_URL', plugin_dir_url( __FILE__ ) );


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Siteseau_Site_Security_Auditor' ) ) :

/**
 * Main plugin class.
 */
class Siteseau_Site_Security_Auditor {

	const OPT_HASHES      = 'siteseau_hashes';
	const OPT_HASHES_TIME = 'siteseau_hashes_time';

	/**
	 * Bootstrap.
	 */
	public static function init() {
		add_action( 'admin_menu', array( __CLASS__, 'add_menu' ) );
		add_action( 'admin_post_ssa_make_baseline', array( __CLASS__, 'handle_baseline' ) );
		add_action( 'admin_post_ssa_run_scan', array( __CLASS__, 'handle_scan' ) );
	}

	/**
	 * Add Tools -> Security Audit page.
	 */
	public static function add_menu() {
		add_management_page(
			__( 'Security Audit', 'site-security-auditor' ),
			__( 'Security Audit', 'site-security-auditor' ),
			'manage_options',
			'site-security-auditor',
			array( __CLASS__, 'render_page' )
		);
	}

	/**
	 * Create/update baseline handler.
	 */
	public static function handle_baseline() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Sorry, you are not allowed to do that.', 'site-security-auditor' ) );
		}
		check_admin_referer( 'ssa_make_baseline' );

		$paths  = self::get_scan_paths();
		$hashes = self::hash_paths( $paths );
		update_option( self::OPT_HASHES, $hashes, false );
		update_option( self::OPT_HASHES_TIME, time(), false );

		wp_safe_redirect( add_query_arg( array( 'page' => 'site-security-auditor', 'ssa_msg' => 'baseline_done' ), admin_url( 'tools.php' ) ) );
		exit;
	}

	/**
	 * Run scan handler.
	 */
	public static function handle_scan() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Sorry, you are not allowed to do that.', 'site-security-auditor' ) );
		}
		check_admin_referer( 'ssa_run_scan' );

		wp_safe_redirect( add_query_arg( array( 'page' => 'site-security-auditor', 'ssa_scan' => '1' ), admin_url( 'tools.php' ) ) );
		exit;
	}

	/**
	 * Render admin page.
	 */
	public static function render_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$plugins_info = self::collect_plugins_info();
		$hardening    = self::hardening_checks();
		$baseline_ts  = intval( get_option( self::OPT_HASHES_TIME, 0 ) );
		$hashes       = get_option( self::OPT_HASHES, array() );
		$scan_results = array();

		if ( isset( $_GET['ssa_scan'] ) && '1' === $_GET['ssa_scan'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$scan_results = self::compare_hashes( is_array( $hashes ) ? $hashes : array(), self::hash_paths( self::get_scan_paths() ) );
		}

		echo '<div class="wrap">';
		echo '<h1>' . esc_html__( 'Security Audit Dashboard', 'site-security-auditor' ) . '</h1>';

		if ( isset( $_GET['ssa_msg'] ) && 'baseline_done' === $_GET['ssa_msg'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			echo '<div class="notice notice-success"><p>' . esc_html__( 'File integrity baseline created/updated.', 'site-security-auditor' ) . '</p></div>';
		}

		// Plugins Risk Overview.
		echo '<h2>' . esc_html__( 'Plugins Risk Overview', 'site-security-auditor' ) . '</h2>';
		echo '<p>' . esc_html__( 'Flags plugins with updates available, last updated over a year ago, or inactive status.', 'site-security-auditor' ) . '</p>';
		echo '<table class="widefat striped"><thead><tr>';
		echo '<th>' . esc_html__( 'Plugin', 'site-security-auditor' ) . '</th>';
		echo '<th>' . esc_html__( 'Version', 'site-security-auditor' ) . '</th>';
		echo '<th>' . esc_html__( 'Status', 'site-security-auditor' ) . '</th>';
		echo '<th>' . esc_html__( 'Update Available', 'site-security-auditor' ) . '</th>';
		echo '<th>' . esc_html__( 'Last Updated', 'site-security-auditor' ) . '</th>';
		echo '<th>' . esc_html__( 'Risk', 'site-security-auditor' ) . '</th>';
		echo '</tr></thead><tbody>';

		if ( empty( $plugins_info ) ) {
			echo '<tr><td colspan="6">' . esc_html__( 'No plugins found.', 'site-security-auditor' ) . '</td></tr>';
		} else {
			foreach ( $plugins_info as $row ) {
				$risk_labels = array();
				if ( ! empty( $row['update_available'] ) ) {
					$risk_labels[] = __( 'Update available', 'site-security-auditor' );
				}
				if ( ! empty( $row['stale'] ) ) {
					$risk_labels[] = __( 'Stale (> 1 year)', 'site-security-auditor' );
				}
				if ( empty( $row['active'] ) ) {
					$risk_labels[] = __( 'Inactive', 'site-security-auditor' );
				}
				echo '<tr>';
				echo '<td>' . esc_html( $row['name'] ) . '</td>';
				echo '<td>' . esc_html( $row['version'] ) . '</td>';
				echo '<td>' . ( ! empty( $row['active'] ) ? esc_html__( 'Active', 'site-security-auditor' ) : esc_html__( 'Inactive', 'site-security-auditor' ) ) . '</td>';
				echo '<td>' . ( ! empty( $row['update_available'] ) ? esc_html__( 'Yes', 'site-security-auditor' ) : esc_html__( 'No', 'site-security-auditor' ) ) . '</td>';
				echo '<td>' . esc_html( $row['last_updated'] ) . '</td>';
				echo '<td>' . ( $risk_labels ? esc_html( implode( ', ', $risk_labels ) ) : esc_html__( 'OK', 'site-security-auditor' ) ) . '</td>';
				echo '</tr>';
			}
		}
		echo '</tbody></table>';

		// File Integrity Monitor.
		echo '<hr /><h2>' . esc_html__( 'File Integrity Monitor', 'site-security-auditor' ) . '</h2>';
		echo '<p>' . esc_html__( 'Create a hash baseline for plugins/themes and scan to detect added, modified or removed files.', 'site-security-auditor' ) . '</p>';

		if ( $baseline_ts ) {
			/* translators: %s: date/time. */
			$baseline_text = sprintf( esc_html__( 'Baseline last updated: %s', 'site-security-auditor' ), wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $baseline_ts ) );
			echo '<p>' . esc_html( $baseline_text ) . '</p>';
		} else {
			echo '<p>' . esc_html__( 'No baseline yet.', 'site-security-auditor' ) . '</p>';
		}

		echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="display:inline-block;margin-right:10px;">';
		wp_nonce_field( 'ssa_make_baseline' );
		echo '<input type="hidden" name="action" value="ssa_make_baseline" />';
		echo '<button type="submit" class="button button-primary" onclick="return confirm(\'' . esc_js( __( 'Create/Update baseline now?', 'site-security-auditor' ) ) . '\');">' . esc_html__( 'Create/Update Baseline', 'site-security-auditor' ) . '</button>';
		echo '</form>';

		echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="display:inline-block;">';
		wp_nonce_field( 'ssa_run_scan' );
		echo '<input type="hidden" name="action" value="ssa_run_scan" />';
		echo '<button type="submit" class="button">' . esc_html__( 'Scan Now', 'site-security-auditor' ) . '</button>';
		echo '</form>';

		if ( ! empty( $scan_results ) ) {
			echo '<h3 style="margin-top:20px;">' . esc_html__( 'Scan Results', 'site-security-auditor' ) . '</h3>';
			echo '<table class="widefat striped"><thead><tr><th>' . esc_html__( 'Change', 'site-security-auditor' ) . '</th><th>' . esc_html__( 'File', 'site-security-auditor' ) . '</th></tr></thead><tbody>';
			$printed = false;
			foreach ( array( 'added', 'modified', 'removed' ) as $key ) {
				if ( ! empty( $scan_results[ $key ] ) ) {
					foreach ( $scan_results[ $key ] as $file ) {
						echo '<tr><td>' . esc_html( ucfirst( $key ) ) . '</td><td>' . esc_html( $file ) . '</td></tr>';
						$printed = true;
					}
				}
			}
			if ( ! $printed ) {
				echo '<tr><td colspan="2">' . esc_html__( 'No changes detected.', 'site-security-auditor' ) . '</td></tr>';
			}
			echo '</tbody></table>';
		}

		// Hardening checks.
		echo '<hr /><h2>' . esc_html__( 'Hardening Checks', 'site-security-auditor' ) . '</h2>';
		echo '<table class="widefat striped"><thead><tr><th>' . esc_html__( 'Check', 'site-security-auditor' ) . '</th><th>' . esc_html__( 'Status', 'site-security-auditor' ) . '</th></tr></thead><tbody>';

		echo '<tr><td>' . esc_html__( 'DISALLOW_FILE_EDIT', 'site-security-auditor' ) . '</td><td>' . ( ! empty( $hardening['disallow_file_edit'] ) ? esc_html__( 'Enabled', 'site-security-auditor' ) : esc_html__( 'Disabled', 'site-security-auditor' ) ) . '</td></tr>';
		echo '<tr><td>' . esc_html__( 'FORCE_SSL_ADMIN', 'site-security-auditor' ) . '</td><td>' . ( ! empty( $hardening['force_ssl_admin'] ) ? esc_html__( 'Enabled', 'site-security-auditor' ) : esc_html__( 'Disabled', 'site-security-auditor' ) ) . '</td></tr>';
		echo '<tr><td>' . esc_html__( 'Plugin auto-updates (general)', 'site-security-auditor' ) . '</td><td>' . ( ! empty( $hardening['plugin_updates_enabled'] ) ? esc_html__( 'Enabled', 'site-security-auditor' ) : esc_html__( 'Disabled', 'site-security-auditor' ) ) . '</td></tr>';
		echo '<tr><td>' . esc_html__( 'XML-RPC availability', 'site-security-auditor' ) . '</td><td>' . ( ! empty( $hardening['xmlrpc_enabled'] ) ? esc_html__( 'Enabled', 'site-security-auditor' ) : esc_html__( 'Disabled', 'site-security-auditor' ) ) . '</td></tr>';

		echo '</tbody></table>';
		echo '</div>';
	}

	/**
	 * Collect plugin info and risk signals.
	 *
	 * @return array
	 */
	protected static function collect_plugins_info() {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$all_plugins = get_plugins();
		$active      = get_option( 'active_plugins', array() );
		$active_map  = array_fill_keys( $active, true );

		$updates     = get_site_transient( 'update_plugins' );
		$updates_map = array();
		if ( isset( $updates->response ) && is_array( $updates->response ) ) {
			foreach ( $updates->response as $file => $obj ) {
				$updates_map[ $file ] = true;
			}
		}

		$results = array();
		foreach ( $all_plugins as $file => $data ) {
			$slug = dirname( $file );
			$info = array(
				'name'            => isset( $data['Name'] ) ? $data['Name'] : $file,
				'version'         => isset( $data['Version'] ) ? $data['Version'] : '',
				'active'          => isset( $active_map[ $file ] ),
				'update_available'=> isset( $updates_map[ $file ] ),
				'last_updated'    => __( 'Unknown', 'site-security-auditor' ),
				'stale'           => false,
			);

			// Try wp.org API for last_updated.
			if ( ! function_exists( 'plugins_api' ) ) {
				
			}
			$args = array(
				'slug'   => $slug,
				'fields' => array( 'last_updated' => true ),
			);
if ( is_admin() && ! function_exists( 'plugins_api' ) ) {
	require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
}
			$api = @plugins_api( 'plugin_information', $args ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
			if ( ! is_wp_error( $api ) && ! empty( $api->last_updated ) ) {
				$info['last_updated'] = mysql2date( get_option( 'date_format' ), $api->last_updated );
				$ts = strtotime( $api->last_updated );
				if ( $ts && ( time() - $ts ) > YEAR_IN_SECONDS ) {
					$info['stale'] = true;
				}
			}

			$results[] = $info;
		}

		return $results;
	}

	/**
	 * Directories to scan.
	 *
	 * @return array
	 */
	protected static function get_scan_paths() {
		return array(
			WP_CONTENT_DIR . '/plugins',
			WP_CONTENT_DIR . '/themes',
		);
	}

	/**
	 * Hash files in given paths.
	 *
	 * @param array $paths Paths.
	 * @return array Map file => hash.
	 */
	protected static function hash_paths( $paths ) {
		$map = array();

		foreach ( $paths as $path ) {
			if ( ! is_dir( $path ) ) {
				continue;
			}
			$iterator = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $path, FilesystemIterator::SKIP_DOTS ) );
			foreach ( $iterator as $file ) {
				/** @var SplFileInfo $file */
				$ext = strtolower( pathinfo( $file->getFilename(), PATHINFO_EXTENSION ) );
				if ( in_array( $ext, array( 'php', 'php7', 'phtml', 'inc', 'twig', 'json' ), true ) ) {
					$real = $file->getPathname();
					$rel  = ltrim( str_replace( WP_CONTENT_DIR, 'wp-content', $real ), '/' );
					$hash = @hash_file( 'sha256', $real ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
					if ( $hash ) {
						$map[ $rel ] = $hash;
					}
				}
			}
		}
		ksort( $map );
		return $map;
	}

	/**
	 * Compare old and new hash maps.
	 *
	 * @param array $old Old.
	 * @param array $new New.
	 * @return array
	 */
	protected static function compare_hashes( $old, $new ) {
		$old = is_array( $old ) ? $old : array();
		$new = is_array( $new ) ? $new : array();

		$added    = array_diff_key( $new, $old );
		$removed  = array_diff_key( $old, $new );
		$modified = array();

		$common = array_intersect_key( $new, $old );
		foreach ( $common as $file => $hash ) {
			if ( $old[ $file ] !== $hash ) {
				$modified[ $file ] = $hash;
			}
		}

		return array(
			'added'    => array_keys( $added ),
			'modified' => array_keys( $modified ),
			'removed'  => array_keys( $removed ),
		);
	}

	/**
	 * Hardening checks.
	 *
	 * @return array
	 */
	protected static function hardening_checks() {
		// Generic on/off for auto-updates by type (does not hook into filters).
		$auto_plugins = function_exists( 'wp_is_auto_update_enabled_for_type' ) ? (bool) wp_is_auto_update_enabled_for_type( 'plugin' ) : false;
		$xmlrpc_on    = apply_filters( 'xmlrpc_enabled', true );

		return array(
			'disallow_file_edit' => defined( 'DISALLOW_FILE_EDIT' ) && DISALLOW_FILE_EDIT,
			'force_ssl_admin'    => defined( 'FORCE_SSL_ADMIN' ) && FORCE_SSL_ADMIN,
			'plugin_updates_enabled'=> $auto_plugins,
			'xmlrpc_enabled'     => (bool) $xmlrpc_on,
		);
	}
}

Siteseau_Site_Security_Auditor::init();

endif;