<?php
/**
 * Plugin bootstrap.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Constants
if ( ! defined( 'SIRAPIX_WC_POS_VERSION' ) ) {
    define( 'SIRAPIX_WC_POS_VERSION', '0.9.0' );
}
if ( ! defined( 'SIRAPIX_WC_POS_PLUGIN_FILE' ) ) {
    define( 'SIRAPIX_WC_POS_PLUGIN_FILE', __FILE__ );
}
if ( ! defined( 'SIRAPIX_WC_POS_PLUGIN_DIR' ) ) {
    define( 'SIRAPIX_WC_POS_PLUGIN_DIR', plugin_dir_path( SIRAPIX_WC_POS_PLUGIN_FILE ) );
}
if ( ! defined( 'SIRAPIX_WC_POS_PLUGIN_URL' ) ) {
    define( 'SIRAPIX_WC_POS_PLUGIN_URL', plugin_dir_url( SIRAPIX_WC_POS_PLUGIN_FILE ) );
}

// WooCommerce dependency check
add_action( 'plugins_loaded', function() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        add_action( 'admin_notices', function() {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'SIRAPIX WooCommerce POS requires WooCommerce to be installed and active.', 'sirapix-pos-for-woocommerce' ) . '</p></div>';
        } );
        // Stop here if WooCommerce is missing; keep the rest of the plugin hooks active otherwise
        return;
    }

// Serve a Service Worker at /sirapix-pos/sw.js for offline POS
add_action( 'init', function(){
    add_rewrite_rule( '^sirapix-pos/sw.js$', 'index.php?sirapix_pos_sw=1', 'top' );
} );
add_filter( 'query_vars', function( $vars ){ $vars[] = 'sirapix_pos_sw'; return $vars; } );
// Prevent canonical redirects from interfering with the Service Worker script URL
add_filter( 'redirect_canonical', function( $redirect_url, $request ) {
    if ( false !== strpos( $request, 'sirapix-pos/sw.js' ) ) {
        return false;
    }
    return $redirect_url;
}, 10, 2 );
add_action( 'template_redirect', function(){
    $page_id = (int) get_option( 'sirapix_wc_pos_page_id' );
    $pos_url = $page_id ? get_permalink( $page_id ) : home_url( '/sirapix-pos/' );
    $pos_path = trailingslashit( wp_parse_url( $pos_url, PHP_URL_PATH ) );
    $host     = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
    $uri      = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
    $req_path = $uri && $host ? wp_parse_url( ( is_ssl() ? 'https://' : 'http://' ) . $host . $uri, PHP_URL_PATH ) : '';
    $is_pos_scoped_sw = $req_path && rtrim( $req_path, '/' ) === rtrim( $pos_path . 'sw.js', '/' );
    // Serve JS only for the pretty /sirapix-pos/sw.js path or when the rewrite-added query_var is present
    if ( get_query_var( 'sirapix_pos_sw' ) || $is_pos_scoped_sw ) {
        $assets_ver = function_exists( 'sirapix_wc_pos_get_assets_version' ) ? sirapix_wc_pos_get_assets_version() : SIRAPIX_WC_POS_VERSION;
        $css = SIRAPIX_WC_POS_PLUGIN_URL . 'assets/css/pos.css?ver=' . $assets_ver;
        $js  = SIRAPIX_WC_POS_PLUGIN_URL . 'assets/js/pos.js?ver=' . $assets_ver;
        $up = wp_upload_dir();
        $static_base = trailingslashit( $up['baseurl'] ) . 'sirapix-pos/';
        $products_file = get_option( 'sirapix_wc_pos_products_file', 'products.json' );
        $products_json = $static_base . ltrim( (string) $products_file, '/' );
        $static_json_urls = [ $products_json ];
        if ( function_exists( 'sirapix_wc_pos_get_settings_snapshot_info' ) ) {
            $snapshots = sirapix_wc_pos_get_settings_snapshot_info();
            if ( is_array( $snapshots ) && ! empty( $snapshots['current']['filename'] ) ) {
                $settings_file = (string) $snapshots['current']['filename'];
                $static_json_urls[] = $static_base . ltrim( $settings_file, '/' );
            }
        }
        header( 'Content-Type: application/javascript; charset=UTF-8' );
        header( 'Service-Worker-Allowed: /' );
        header( 'Cache-Control: no-store' );
        // App-shell SW: cache POS page, CSS/JS, static products/settings JSON; network-first for REST/POST, cache-first for assets/products with offline navigation fallback
        echo "(function(){\n". // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
             "var CACHE='sirapix-pos-v" . esc_js( $assets_ver ) . "';\n".
             "var ASSETS=".wp_json_encode( array_unique( array_filter( [ $pos_url, $css, $js ] ) ) ).
             ";\n".
             "var STATIC_JSON=".wp_json_encode( array_values( array_unique( array_filter( $static_json_urls ) ) ) ).";\n".
             "self.addEventListener('install', function(e){ e.waitUntil(\n".
             "  caches.open(CACHE).then(function(c){ return c.addAll(ASSETS.concat(STATIC_JSON)); }).then(function(){ return self.skipWaiting(); })\n".
             "); });\n".
             "self.addEventListener('activate', function(e){ e.waitUntil(\n".
             "  caches.keys().then(function(keys){\n".
             "    return Promise.all(keys.map(function(k){ if(k!==CACHE){ return caches.delete(k); } }));\n".
             "  }).then(function(){ return self.clients.claim(); })\n".
             "); });\n".
             "function isApi(req){ return req.url.indexOf('/wp-json/')!==-1; }\n".
             "function isPost(req){ return req.method && req.method.toUpperCase()==='POST'; }\n".
             "self.addEventListener('fetch', function(e){\n".
             "  var req=e.request; var url=req.url; if(isPost(req)||isApi(req)){ return; }\n".
             "  if(req.mode==='navigate'){\n".
             "    e.respondWith(\n".
             "      fetch(req).then(function(resp){\n".
             "        if(resp && resp.status===200){\n".
             "          return caches.open(CACHE).then(function(cache){ cache.put(req, resp.clone()); return resp; });\n".
             "        }\n".
             "        return resp;\n".
             "      }).catch(function(){\n".
             "        return caches.open(CACHE).then(function(cache){\n".
             "          return cache.match(ASSETS[0], { ignoreSearch:true }).then(function(shell){\n".
             "            if(shell){ return shell; }\n".
             "            return Response.error();\n".
             "          });\n".
             "        });\n".
             "      })\n".
             "    );\n".
             "    return;\n".
             "  }\n".
             "  e.respondWith(\n".
             "    caches.open(CACHE).then(function(cache){\n".
             "      return cache.match(req, { ignoreSearch:true }).then(function(cached){\n".
             "        if(cached){ return cached; }\n".
             "        return fetch(req).then(function(resp){\n".
             "          if(resp && resp.status===200 && (\n".
             "              url.endsWith('.css') ||\n".
             "              url.endsWith('.js') ||\n".
             "              url.indexOf('/sirapix-pos/products.json')!==-1 ||\n".
             "              url.indexOf('/sirapix-pos/products-')!==-1 ||\n".
             "              url.indexOf('/sirapix-pos/settings-')!==-1 ||\n".
             "              url.indexOf('/wp-content/uploads/')!==-1\n".
             "          )){\n".
             "            cache.put(req, resp.clone());\n".
             "          }\n".
             "          return resp;\n".
             "        }).catch(function(){\n".
             "          if(cached){ return cached; }\n".
             "          return Response.error();\n".
             "        });\n".
             "      });\n".
             "    })\n".
             "  );\n".
             "});\n".
             "})();";
        exit;
    }
} );

// Nightly auto-refresh for static products JSON
add_action( 'init', function(){
    if ( ! wp_next_scheduled( 'sirapix_wc_pos_cron_nightly' ) ) {
        // Schedule roughly nightly; use current time + 1 hour to avoid immediate run on first page load
        wp_schedule_event( time() + HOUR_IN_SECONDS, 'daily', 'sirapix_wc_pos_cron_nightly' );
    }
} );

add_action( 'sirapix_wc_pos_cron_nightly', function(){
    if ( ! class_exists( 'Sirapix_WC_POS_REST' ) ) { return; }
    try {
        // Products & categories
        if ( is_callable( [ 'Sirapix_WC_POS_REST', 'generate_static_products' ] ) ) {
            $req = new WP_REST_Request( 'POST', '/sirapix-pos/v1/static/products' );
            Sirapix_WC_POS_REST::generate_static_products( $req );
        }
    } catch ( \Throwable $e ) {
        // Fail quietly; admins can re-run from settings
    }
} );

// -------- Enforce real email for POS-fabricated users --------
// 1) Register account endpoint /my-account/update-email
add_action( 'init', function(){
    if ( function_exists('add_rewrite_endpoint') ) {
        add_rewrite_endpoint( 'update-email', EP_ROOT | EP_PAGES );
    }
} );

// 2) Redirect gate: if user has fabricated email flag, force them to update on My Account
add_action( 'template_redirect', function(){
    if ( ! function_exists('is_account_page') || ! is_user_logged_in() ) { return; }
    if ( ! is_account_page() ) { return; }
    $user_id = get_current_user_id();
    if ( user_can( $user_id, 'manage_woocommerce' ) ) { return; }
    $flag = get_user_meta( $user_id, 'spx_fabricated_email', true );
    // Also detect by pattern as fallback (digits@domain)
    $user   = get_user_by( 'id', $user_id );
    $domain = function_exists('sirapix_wc_pos_get_site_domain') ? sirapix_wc_pos_get_site_domain() : wp_parse_url( home_url(), PHP_URL_HOST );
    $is_pattern = $user && $domain ? (bool) preg_match( '/^\d+@' . preg_quote( preg_replace('/^www\./i','',$domain), '/' ) . '$/i', (string) $user->user_email ) : false;
    $on_update_endpoint = isset( $GLOBALS['wp']->query_vars['update-email'] );
    if ( ( $flag || $is_pattern ) && ! $on_update_endpoint ) {
        wp_safe_redirect( wc_get_account_endpoint_url( 'update-email' ) );
        exit;
    }
} );

// 3) Render the Update Email form on the endpoint
add_action( 'woocommerce_account_update-email_endpoint', function(){
    if ( ! is_user_logged_in() ) { return; }
    $user = wp_get_current_user();
    echo '<div class="woocommerce">';
    wc_print_notices();
    echo '<h3>' . esc_html__( 'Update your email', 'sirapix-pos-for-woocommerce' ) . '</h3>';
    echo '<p>' . esc_html__( 'Please provide a valid email address to access your account.', 'sirapix-pos-for-woocommerce' ) . '</p>';
    echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" class="woocommerce-EditAccountForm edit-account">';
    echo '<p class="form-row form-row-first"><label for="spx_name">' . esc_html__( 'Full name', 'sirapix-pos-for-woocommerce' ) . ' <span class="required">*</span></label>';
    echo '<input type="text" class="input-text" name="spx_name" id="spx_name" value="' . esc_attr( $user->display_name ) . '" required /></p>';
    echo '<p class="form-row form-row-last"><label for="spx_email">' . esc_html__( 'Email address', 'sirapix-pos-for-woocommerce' ) . ' <span class="required">*</span></label>';
    echo '<input type="email" class="input-text" name="spx_email" id="spx_email" value="" placeholder="name@example.com" required /></p>';
    echo '<div class="clear"></div>';
    wp_nonce_field( 'sirapix_update_email', 'sirapix_update_email_nonce' );
    echo '<input type="hidden" name="action" value="sirapix_update_email" />';
    echo '<p><button type="submit" class="button" style="min-width:180px;">' . esc_html__( 'Save email', 'sirapix-pos-for-woocommerce' ) . '</button></p>';
    echo '</form></div>';
} );

// 4) Handle the form submission securely
add_action( 'admin_post_sirapix_update_email', function(){
    if ( ! is_user_logged_in() ) { wp_safe_redirect( wc_get_page_permalink( 'myaccount' ) ); exit; }
    if ( ! isset( $_POST['sirapix_update_email_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['sirapix_update_email_nonce'] ) ), 'sirapix_update_email' ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
        wc_add_notice( __( 'Security check failed. Please try again.', 'sirapix-pos-for-woocommerce' ), 'error' );
        wp_safe_redirect( wc_get_account_endpoint_url( 'update-email' ) );
        exit;
    }
    $uid  = get_current_user_id();
    $name = isset($_POST['spx_name']) ? sanitize_text_field( wp_unslash( $_POST['spx_name'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
    $mail = isset($_POST['spx_email']) ? sanitize_email( wp_unslash( $_POST['spx_email'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
    if ( ! $name ) { wc_add_notice( __( 'Name is required.', 'sirapix-pos-for-woocommerce' ), 'error' ); wp_safe_redirect( wc_get_account_endpoint_url( 'update-email' ) ); exit; }
    if ( ! $mail || ! is_email( $mail ) ) { wc_add_notice( __( 'Please enter a valid email address.', 'sirapix-pos-for-woocommerce' ), 'error' ); wp_safe_redirect( wc_get_account_endpoint_url( 'update-email' ) ); exit; }
    $exists = email_exists( $mail );
    if ( $exists && (int) $exists !== (int) $uid ) { wc_add_notice( __( 'This email is already in use.', 'sirapix-pos-for-woocommerce' ), 'error' ); wp_safe_redirect( wc_get_account_endpoint_url( 'update-email' ) ); exit; }

    wp_update_user( [ 'ID' => $uid, 'user_email' => $mail, 'display_name' => $name ] );
    update_user_meta( $uid, 'first_name', $name );
    update_user_meta( $uid, 'billing_email', $mail );
    delete_user_meta( $uid, 'spx_fabricated_email' );
    wc_add_notice( __( 'Your email has been updated.', 'sirapix-pos-for-woocommerce' ), 'success' );
    wp_safe_redirect( wc_get_page_permalink( 'myaccount' ) );
    exit;
} );
} );

// Declare compatibility with WooCommerce HPOS (Custom Order Tables)
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( '\\Automattic\\WooCommerce\\Utilities\\FeaturesUtil' ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', SIRAPIX_WC_POS_PLUGIN_FILE, true );
    }
} );

// Includes
require_once SIRAPIX_WC_POS_PLUGIN_DIR . 'includes/helpers.php';
require_once SIRAPIX_WC_POS_PLUGIN_DIR . 'includes/class-sirapix-pos-activator.php';
require_once SIRAPIX_WC_POS_PLUGIN_DIR . 'includes/class-sirapix-pos-settings.php';
require_once SIRAPIX_WC_POS_PLUGIN_DIR . 'includes/class-sirapix-pos-rest.php';
require_once SIRAPIX_WC_POS_PLUGIN_DIR . 'includes/class-sirapix-pos-shortcode.php';
require_once SIRAPIX_WC_POS_PLUGIN_DIR . 'includes/class-sirapix-pos-onboarding.php';

// Compatibility: on POS page or POS settings page, avoid cache layers that can trigger shutdown DB queries
add_action( 'init', function(){
    if ( function_exists( 'sirapix_wc_pos_is_pos_context' ) && sirapix_wc_pos_is_pos_context() ) {
        if ( ! defined( 'DONOTCACHEPAGE' ) ) { define( 'DONOTCACHEPAGE', true ); } // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound
        if ( ! defined( 'DONOTCACHEOBJECT' ) ) { define( 'DONOTCACHEOBJECT', true ); } // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound
        if ( ! defined( 'DONOTCACHEDB' ) ) { define( 'DONOTCACHEDB', true ); } // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound
        // Remove LiteSpeed finalize callbacks for this request only
        add_action( 'shutdown', function(){
            if ( function_exists( 'remove_all_actions' ) ) {
                remove_all_actions( 'litespeed_control_finalize' );
            }
        }, 0 );
    }
} );

// Activation / Deactivation
register_activation_hook( SIRAPIX_WC_POS_PLUGIN_FILE, [ 'Sirapix_WC_POS_Activator', 'activate' ] );
register_deactivation_hook( SIRAPIX_WC_POS_PLUGIN_FILE, [ 'Sirapix_WC_POS_Activator', 'deactivate' ] );

// Display a friendly origin label for POS orders in admin (e.g., Created via)
add_filter( 'woocommerce_order_get_created_via', function( $via, $order ) {
    if ( $via === 'sirapix_pos' ) {
        return __( 'In-Store Purchase', 'sirapix-pos-for-woocommerce' );
    }
    if ( $order instanceof WC_Order && $order->get_meta( '_sirapix_pos_origin' ) ) {
        return __( 'In-Store Purchase', 'sirapix-pos-for-woocommerce' );
    }
    return $via;
}, 10, 2 );

// Some WooCommerce UIs call a helper to render a human label; provide a fallback label for our POS origin
add_filter( 'woocommerce_admin_order_created_via', function( $label, $order ) {
    if ( $order instanceof WC_Order ) {
        $via = method_exists( $order, 'get_created_via' ) ? $order->get_created_via() : '';
        if ( $via === 'sirapix_pos' || $order->get_meta( '_sirapix_pos_origin' ) ) {
            return __( 'In-Store Purchase', 'sirapix-pos-for-woocommerce' );
        }
    }
    return $label;
}, 10, 2 );

add_action( 'admin_init', function() {
    if ( ! is_admin() ) { return; }
    if ( ! current_user_can( 'activate_plugins' ) ) { return; }
    if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) { return; }
    if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) { return; }
    if ( ! get_option( 'sirapix_wc_pos_do_activation_redirect' ) ) { return; }
    delete_option( 'sirapix_wc_pos_do_activation_redirect' );
    if ( isset( $_GET['activate-multi'] ) ) { return; } // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    $target = add_query_arg( 'sirapix_pos_onboarding', '1', home_url( '/' ) );
    wp_safe_redirect( $target );
    exit;
} );

add_filter( 'body_class', function( $classes ) {
    if ( is_page( (int) get_option( 'sirapix_wc_pos_page_id' ) ) ) {
        $classes[] = 'sirapix-pos-page';
    }
    return $classes;
} );

add_filter( 'manage_edit-shop_order_columns', function( $columns ) {
    $new = [];
    foreach ( $columns as $key => $label ) {
        $new[ $key ] = $label;
        if ( 'order_total' === $key ) {
            $new['sirapix_pos_cashier'] = __( 'POS Cashier', 'sirapix-pos-for-woocommerce' );
        }
    }
    if ( ! isset( $new['sirapix_pos_cashier'] ) ) {
        $new['sirapix_pos_cashier'] = __( 'POS Cashier', 'sirapix-pos-for-woocommerce' );
    }
    return $new;
} );

add_action( 'manage_shop_order_posts_custom_column', function( $column, $post_id ) {
    if ( 'sirapix_pos_cashier' !== $column ) {
        return;
    }
    $order = wc_get_order( $post_id );
    if ( ! $order ) {
        return;
    }
    $name = (string) $order->get_meta( '_sirapix_pos_cashier_name' );
    if ( $name === '' ) {
        return;
    }
    echo esc_html( $name );
}, 10, 2 );

add_filter( 'manage_edit-shop_order_sortable_columns', function( $columns ) {
    $columns['sirapix_pos_cashier'] = 'sirapix_pos_cashier';
    return $columns;
} );

add_action( 'pre_get_posts', function( $query ) {
    if ( ! is_admin() || ! $query->is_main_query() ) {
        return;
    }
    $post_type = $query->get( 'post_type' );
    if ( $post_type !== 'shop_order' && $post_type !== 'shop_order_refund' ) {
        return;
    }

    $orderby = $query->get( 'orderby' );
    if ( $orderby === 'sirapix_pos_cashier' ) {
        $query->set( 'meta_key', '_sirapix_pos_cashier_id' );
        $query->set( 'orderby', 'meta_value_num' );
    }

    if ( isset( $_GET['sirapix_pos_cashier'] ) && $_GET['sirapix_pos_cashier'] !== '' ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $cashier = sanitize_text_field( wp_unslash( $_GET['sirapix_pos_cashier'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( $cashier !== '' ) {
            $meta_query   = (array) $query->get( 'meta_query' );
            $meta_query[] = [
                'key'   => '_sirapix_pos_cashier_id',
                'value' => (int) $cashier,
            ];
            $query->set( 'meta_query', $meta_query );
        }
    }
} );

add_action( 'restrict_manage_posts', function( $post_type ) {
    if ( $post_type !== 'shop_order' ) {
        return;
    }
    global $wpdb;
    $meta_key = '_sirapix_pos_cashier_id';
    $cache_key = 'sirapix_wc_pos_cashier_ids';
    $values    = get_transient( $cache_key );
    if ( false === $values ) {
        // Admin-only helper: fetch distinct cashier IDs for the filter dropdown.
        $values = $wpdb->get_col( $wpdb->prepare( "SELECT DISTINCT pm.meta_value FROM {$wpdb->postmeta} pm INNER JOIN {$wpdb->posts} p ON p.ID = pm.post_id WHERE pm.meta_key = %s AND pm.meta_value <> '' ORDER BY CAST(pm.meta_value AS UNSIGNED) ASC", $meta_key ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        set_transient( $cache_key, $values, HOUR_IN_SECONDS );
    }
    if ( empty( $values ) ) {
        return;
    }
    $current = isset( $_GET['sirapix_pos_cashier'] ) ? sanitize_text_field( wp_unslash( $_GET['sirapix_pos_cashier'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    echo '<select name="sirapix_pos_cashier" id="sirapix_pos_cashier" class="wc-enhanced-select">';
    echo '<option value="">' . esc_html__( 'All POS cashiers', 'sirapix-pos-for-woocommerce' ) . '</option>';
    foreach ( $values as $val ) {
        $user_id = (int) $val;
        if ( $user_id <= 0 ) {
            continue;
        }
        $user = get_user_by( 'id', $user_id );
        if ( ! $user ) {
            continue;
        }
        $label = $user->display_name ? $user->display_name : ( 'User #' . $user_id );
        $value = (string) $user_id;
        echo '<option value="' . esc_attr( $value ) . '"' . selected( $current, $value, false ) . '>' . esc_html( $label ) . '</option>';
    }
    echo '</select>';
} );

add_filter( 'plugin_action_links_' . plugin_basename( defined( 'SIRAPIX_WC_POS_PLUGIN_FILE' ) ? SIRAPIX_WC_POS_PLUGIN_FILE : __FILE__ ), function( $links ) {
    $settings_url  = admin_url( 'admin.php?page=sirapix-wc-pos' );
    $settings_link = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'sirapix-pos-for-woocommerce' ) . '</a>';
    array_unshift( $links, $settings_link );
    return $links;
} );

// Admin icon styling for POS (toolbar + WooCommerce submenu)
$sirapix_wc_pos_enqueue_admin_icons = function() {
    $handle = 'sirapix-wc-pos-admin-icons';
    if ( ! wp_style_is( $handle, 'registered' ) ) {
        wp_register_style( $handle, false, [], null );
    }
    wp_enqueue_style( $handle );

    // SVG cursor icon (stroke-based) as data URI (properly encoded)
    $svg = "data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none'%3E%3Cpath d='M17.07 20.61H9.79a2 2 0 0 1-2-2 2 2 0 0 1 2-2h1.87L5 9.86a2 2 0 0 1-.19-2.65 1.88 1.88 0 0 1 1.47-.68 1.84 1.84 0 0 1 1.35.55l4.06 4.06 4.08-3.06a1.91 1.91 0 0 1 2.5.18h0A17.18 17.18 0 0 1 22.42 15l.06.19' stroke='%23020202' stroke-width='1.91' stroke-miterlimit='10'/%3E%3Cpath d='M10.63 10.12A4.73 4.73 0 0 0 11 8.17 4.78 4.78 0 1 0 6.26 13a4.67 4.67 0 0 0 1.55-.26' stroke='%23020202' stroke-width='1.91' stroke-miterlimit='10'/%3E%3C/svg%3E";
    $svg_url = esc_url( $svg, array_merge( wp_allowed_protocols(), [ 'data' ] ) );

    $css = "" .
        "#wpadminbar .sirapix-wc-pos-adminbar{position:relative;padding-left:28px;}" .
        "#wpadminbar .sirapix-wc-pos-adminbar:before{content:'';position:absolute;left:6px;top:50%;width:18px;height:18px;margin-top:-9px;display:inline-block;background-image:url('" . $svg_url . "');background-repeat:no-repeat;background-size:18px 18px;}" .
        "#adminmenu a[href*='page=sirapix-wc-pos']{position:relative;padding-left:28px;}" .
        "#adminmenu a[href*='page=sirapix-wc-pos']::before{content:'';position:absolute;left:6px;top:7px;width:16px;height:16px;display:inline-block;background-image:url('" . $svg_url . "');background-repeat:no-repeat;background-size:16px 16px;}";

    wp_add_inline_style( $handle, $css );
};

add_action( 'admin_enqueue_scripts', function() use ( $sirapix_wc_pos_enqueue_admin_icons ) {
    $sirapix_wc_pos_enqueue_admin_icons();
} );

add_action( 'wp_enqueue_scripts', function() use ( $sirapix_wc_pos_enqueue_admin_icons ) {
    if ( ! is_admin_bar_showing() ) {
        return;
    }
    $sirapix_wc_pos_enqueue_admin_icons();
} );

add_action( 'admin_bar_menu', function( $wp_admin_bar ) {
    if ( ! is_user_logged_in() ) { return; }
    if ( ! ( current_user_can( 'manage_woocommerce' ) || sirapix_wc_pos_user_has_access() ) ) { return; }
    $page_id = (int) get_option( 'sirapix_wc_pos_page_id' );
    if ( ! $page_id ) { return; }
    $pos_url = get_permalink( $page_id );
    if ( $pos_url ) {
        // Build a friendly hashed path /sirapix-pos/{hash}/. The hash is
        // stored in sirapix_wc_pos_pos_hash and bumped on each settings save
        // so that the POS link changes whenever configuration changes.
        $path = wp_parse_url( $pos_url, PHP_URL_PATH );
        if ( $path ) {
            $hash = get_option( 'sirapix_wc_pos_pos_hash', '' );
            if ( ! $hash ) {
                $hash = substr( md5( microtime( true ) . wp_rand() ), 0, 8 );
                update_option( 'sirapix_wc_pos_pos_hash', $hash );
            }
            $hashed_path = trailingslashit( trailingslashit( $path ) . $hash );
            $pos_url = home_url( $hashed_path );
        }
    }
    if ( ! $pos_url ) { return; }

    $on_pos_page = function_exists('is_page') && is_page( $page_id );
    if ( $on_pos_page ) {
        // On the POS page: show a link to POS Settings as the last item on the right
        $settings_url = admin_url( 'admin.php?page=sirapix-wc-pos' );
        $wp_admin_bar->add_node( [
            'id'     => 'sirapix-wc-pos-settings-link',
            'title'  => __( 'POS Settings', 'sirapix-pos-for-woocommerce' ),
            'href'   => esc_url( $settings_url ),
            'parent' => 'top-secondary',
            'meta'   => [ 'class' => 'sirapix-wc-pos-adminbar' ],
        ] );
        // Register Service Worker for offline POS using the pretty /sirapix-pos/sw.js path
        $pos_scope = wp_make_link_relative( get_permalink( (int) get_option( 'sirapix_wc_pos_page_id' ) ) );
        if ( ! $pos_scope ) { $pos_scope = '/sirapix-pos/'; }
        $sw_js = "(function(){ try { if('serviceWorker' in navigator){ var scope='" . esc_js( trailingslashit( $pos_scope ) ) . "'; navigator.serviceWorker.register('/sirapix-pos/sw.js', { scope: scope }).catch(function(){}); } } catch(e){} })();";
        wp_add_inline_script( 'sirapix-wc-pos', $sw_js, 'after' );
    } else {
        // Outside POS page: keep the POS link to open the POS UI
        $wp_admin_bar->add_node( [
            'id'    => 'sirapix-wc-pos-link',
            'title' => __( 'POS', 'sirapix-pos-for-woocommerce' ),
            'href'  => esc_url( $pos_url ),
            'meta'  => [ 'class' => 'sirapix-wc-pos-adminbar' ],
        ] );
    }
}, 999 );

add_action( 'template_redirect', function() {
    if ( is_admin() ) { return; }
    if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) { return; }
    if ( function_exists( 'wp_doing_ajax' ) && wp_doing_ajax() ) { return; }

    $page_id = (int) get_option( 'sirapix_wc_pos_page_id' );
    if ( ! $page_id ) { return; }

    // Detect POS route by URL path so that /sirapix-pos/ and /sirapix-pos/{hash}/
    // are both handled by this renderer, regardless of the main WP query.
    $pos_url  = get_permalink( $page_id );
    $pos_path = $pos_url ? trailingslashit( wp_parse_url( $pos_url, PHP_URL_PATH ) ) : '';
    $host     = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
    $uri      = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
    $req_path = $uri && $host
        ? wp_parse_url( ( is_ssl() ? 'https://' : 'http://' ) . $host . $uri, PHP_URL_PATH )
        : '';

    if ( ! $pos_path || ! $req_path ) { return; }

    $req_slash = trailingslashit( $req_path );
    if ( strpos( $req_slash, $pos_path ) !== 0 ) {
        // Not under the POS path; let WP handle it normally.
        return;
    }

    $assets_ver = function_exists( 'sirapix_wc_pos_get_assets_version' ) ? sirapix_wc_pos_get_assets_version() : SIRAPIX_WC_POS_VERSION;
    $primary   = sanitize_hex_color( (string) get_option( 'sirapix_wc_pos_primary_color', '#0ea5e9' ) );
    $secondary = sanitize_hex_color( (string) get_option( 'sirapix_wc_pos_secondary_color', '#f59e0b' ) );
    $primary   = $primary ? $primary : '#0ea5e9';
    $secondary = $secondary ? $secondary : '#f59e0b';
    $items_per_row = (int) get_option( 'sirapix_wc_pos_items_per_row', 8 );
    if ( $items_per_row < 2 ) { $items_per_row = 2; }
    if ( $items_per_row > 10 ) { $items_per_row = 10; }

    $logo_id  = get_theme_mod( 'custom_logo' );
    $logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'full' ) : '';
    $up = wp_upload_dir();
    $static_url = trailingslashit( $up['baseurl'] ) . 'sirapix-pos/';
    $products_mode = get_option( 'sirapix_wc_pos_products_mode', 'offline' );
    $aggressive_images = (bool) get_option( 'sirapix_wc_pos_aggressive_image_prefetch', 0 );
    $products_file = get_option( 'sirapix_wc_pos_products_file', 'products.json' );
    $settings_snapshot_url = '';
    if ( function_exists( 'sirapix_wc_pos_generate_settings_snapshot' ) ) {
        sirapix_wc_pos_generate_settings_snapshot();
    }
    if ( function_exists( 'sirapix_wc_pos_get_settings_snapshot_info' ) ) {
        $snapshots = sirapix_wc_pos_get_settings_snapshot_info();
        if ( is_array( $snapshots ) && ! empty( $snapshots['current']['filename'] ) ) {
            $settings_snapshot_url = $static_url . ltrim( (string) $snapshots['current']['filename'], '/' );
        }
    }

    $localized = [
        'ajax_url' => admin_url( 'admin-ajax.php' ),
        'rest_url' => esc_url( rest_url() ),
        'nonce'    => wp_create_nonce( 'wp_rest' ),
        'page_id'  => $page_id,
        'domain'   => function_exists( 'sirapix_wc_pos_get_site_domain' ) ? sirapix_wc_pos_get_site_domain() : wp_parse_url( home_url(), PHP_URL_HOST ),
        'items_per_row' => $items_per_row,
        'products_mode' => $products_mode,
        'site_name'=> get_bloginfo( 'name' ),
        'logo'     => $logo_url,
        'static_url'=> esc_url( $static_url ),
        'settings_snapshot_url' => esc_url( $settings_snapshot_url ),
        'static_products_file' => $products_file,
        'aggressive_images' => $aggressive_images,
        'labels'   => [
            'payment_offline' => __( 'Offline Payment', 'sirapix-pos-for-woocommerce' ),
            'shipping_offline_store' => __( 'Offline Store', 'sirapix-pos-for-woocommerce' ),
        ],
        'i18n' => [
            'status_online'          => __( 'Status: Online', 'sirapix-pos-for-woocommerce' ),
            'status_offline'         => __( 'Status: Offline', 'sirapix-pos-for-woocommerce' ),
            'no_items'               => __( 'No items', 'sirapix-pos-for-woocommerce' ),
            'order_prefix'           => __( 'Order #', 'sirapix-pos-for-woocommerce' ),
            'completed_successfully' => __( 'Completed successfully', 'sirapix-pos-for-woocommerce' ),
            'close'                  => __( 'Close', 'sirapix-pos-for-woocommerce' ),
            'print_receipt'          => __( 'Print Receipt', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_requires_online' => __( 'Image prefetch requires online connection.', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_prep'    => __( 'Preparing image prefetch…', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_loading_products' => __( 'Loading product list…', 'sirapix-pos-for-woocommerce' ),
            'offline_images_prefetch' => __( 'Offline Images Prefetch', 'sirapix-pos-for-woocommerce' ),
            'prefetching_images'     => __( 'Prefetching images…', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_products_missing' => __( 'Could not load static products JSON. Make sure offline products are generated.', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_no_images' => __( 'No product images found to prefetch.', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_completed' => __( 'Image prefetch completed.', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_completed_toast' => __( 'Image prefetch completed. Product images should now be available offline.', 'sirapix-pos-for-woocommerce' ),
            'image_prefetch_failed'   => __( 'Image prefetch failed.', 'sirapix-pos-for-woocommerce' ),
            /* translators: %s: synced order number. */
            'toast_queued_order_synced' => __( 'Queued order synced #%s', 'sirapix-pos-for-woocommerce' ),
            /* translators: %s: error message when syncing a queued order fails. */
            'toast_queued_order_failed' => __( 'Queued order failed: %s', 'sirapix-pos-for-woocommerce' ),
            'toast_added_to_cart'       => __( 'Added to cart', 'sirapix-pos-for-woocommerce' ),
            'toast_failed_load_variations' => __( 'Failed to load variations', 'sirapix-pos-for-woocommerce' ),
            'toast_cart_cleared'        => __( 'Cart cleared', 'sirapix-pos-for-woocommerce' ),
            'toast_order_saved_offline' => __( 'Order is saved offline and queued for sync.', 'sirapix-pos-for-woocommerce' ),
            /* translators: %s: completed order number. */
            'toast_order_completed'     => __( 'Order #%s completed', 'sirapix-pos-for-woocommerce' ),
            'error_failed_complete_order' => __( 'Failed to complete order', 'sirapix-pos-for-woocommerce' ),
            'error_customer_phone_required' => __( 'Customer phone is required', 'sirapix-pos-for-woocommerce' ),
            'label_complete_order'     => __( 'Complete Order', 'sirapix-pos-for-woocommerce' ),
        ],
        'currency' => [
            'symbol'   => html_entity_decode( get_woocommerce_currency_symbol(), ENT_QUOTES, 'UTF-8' ),
            'position' => get_option( 'woocommerce_currency_pos' ),
            'thousand' => function_exists( 'wc_get_price_thousand_separator' ) ? wc_get_price_thousand_separator() : ',',
            'decimal'  => function_exists( 'wc_get_price_decimal_separator' ) ? wc_get_price_decimal_separator() : '.',
            'decimals' => function_exists( 'wc_get_price_decimals' ) ? wc_get_price_decimals() : 2,
        ],
    ];

    $css_url = SIRAPIX_WC_POS_PLUGIN_URL . 'assets/css/pos.css?ver=' . $assets_ver;
    $js_url  = SIRAPIX_WC_POS_PLUGIN_URL . 'assets/js/pos.js?ver=' . $assets_ver;
    $jquery_url = includes_url( 'js/jquery/jquery.min.js' );

    $title = get_bloginfo( 'name' ) . ' POS';
    $inline_vars = ":root{--spx-primary: {$primary}; --spx-secondary: {$secondary}; --spx-items-per-row: {$items_per_row}; --spx-bg:#F9FAFB; --spx-text:#111827; --spx-muted:#6B7280; --spx-card:#ffffff; --spx-border:#E5E7EB; --spx-radius:12px; --spx-radius-lg:16px; --spx-shadow:0 8px 24px rgba(17,24,39,.06); --spx-shadow-sm:0 2px 8px rgba(17,24,39,.06); --spx-space:16px; --spx-space-lg:24px;} body.sirapix-pos-page{background: var(--spx-bg);} .sirapix-pos-wrapper{color:var(--spx-text);}";

    $is_logged_in = is_user_logged_in();
    $has_access = $is_logged_in && function_exists( 'sirapix_wc_pos_user_has_access' ) ? sirapix_wc_pos_user_has_access() : false;

    if ( $is_logged_in && ! $has_access ) {
        $content = Sirapix_WC_POS_Shortcode::render_no_permission();
    } elseif ( ! $is_logged_in ) {
        $content = Sirapix_WC_POS_Shortcode::render_login();
    } else {
        $content = Sirapix_WC_POS_Shortcode::render();
    }

    nocache_headers();
    // Ensure the POS HTML page itself is never stored by browser/edge caches.
    // Offline support is handled via the Service Worker cache, not the HTTP cache.
    header( 'Cache-Control: no-store' );
    status_header( 200 );

    // Register + enqueue the POS stylesheet so we can print it via wp_print_styles
    $standalone_handle = 'sirapix-wc-pos-standalone';
    if ( ! wp_style_is( $standalone_handle, 'registered' ) ) {
        wp_register_style( $standalone_handle, $css_url, [], $assets_ver );
    }
    wp_enqueue_style( $standalone_handle );

    echo '<!doctype html>';
    echo '<html '; language_attributes(); echo '>';
    echo '<head>';
    echo '<meta charset="' . esc_attr( get_bloginfo( 'charset' ) ) . '" />';
    echo '<meta name="viewport" content="width=device-width, initial-scale=1" />';
    echo '<title>' . esc_html( $title ) . '</title>';

    // Print only our standalone POS style handle
    ob_start();
    wp_print_styles( $standalone_handle );
    $standalone_styles = ob_get_clean();
    if ( $standalone_styles ) {
        $spx_allowed_tags = [
            'link'   => [
                'rel'   => true,
                'href'  => true,
                'id'    => true,
                'media' => true,
                'type'  => true,
            ],
            'style'  => [
                'id'    => true,
                'media' => true,
                'type'  => true,
            ],
            'script' => [
                'src'            => true,
                'id'             => true,
                'type'           => true,
                'async'          => true,
                'defer'          => true,
                'nomodule'       => true,
                'crossorigin'    => true,
                'integrity'      => true,
                'referrerpolicy' => true,
                'data-*'         => true,
            ],
        ];
        echo wp_kses( $standalone_styles, $spx_allowed_tags );
    }

    wp_add_inline_style( $standalone_handle, $inline_vars );
    echo '</head>';
    echo '<body class="sirapix-pos-page">';
    // $content is generated by our own renderer (login/no-permission/POS markup), not user-submitted HTML.
    // Do not wrap in wp_kses_post() because it can strip required tags/attributes used by the POS app.
    echo $content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

    // Register + enqueue POS scripts and print just those handles for the standalone shell.
    $standalone_js_handle = 'sirapix-wc-pos-standalone-js';
    if ( ! wp_script_is( 'jquery', 'registered' ) ) {
        wp_register_script( 'jquery', $jquery_url, [], $assets_ver, true );
    }
    if ( ! wp_script_is( $standalone_js_handle, 'registered' ) ) {
        wp_register_script( $standalone_js_handle, $js_url, [ 'jquery' ], $assets_ver, true );
    }
    wp_enqueue_script( 'jquery' );
    wp_enqueue_script( $standalone_js_handle );

    wp_add_inline_script( $standalone_js_handle, 'window.SIRAPIX_POS = ' . wp_json_encode( $localized ) . ';', 'before' );

    ob_start();
    wp_print_scripts( [ 'jquery', $standalone_js_handle ] );
    $standalone_scripts = ob_get_clean();
    if ( $standalone_scripts ) {
        $spx_allowed_script_tag = [
            'script' => [
                'src'            => true,
                'id'             => true,
                'type'           => true,
                'async'          => true,
                'defer'          => true,
                'nomodule'       => true,
                'crossorigin'    => true,
                'integrity'      => true,
                'referrerpolicy' => true,
                'data-*'         => true,
            ],
        ];

        $sanitized = preg_replace_callback(
            '/<script\b[^>]*>.*?<\/script>/is',
            function( $m ) use ( $spx_allowed_script_tag ) {
                if ( ! preg_match( '/^(<script\b[^>]*>)(.*?)(<\/script>)$/is', $m[0], $parts ) ) {
                    return '';
                }
                $open  = wp_kses( $parts[1], $spx_allowed_script_tag );
                $body  = $parts[2];
                $close = '</script>';
                return $open . $body . $close;
            },
            $standalone_scripts
        );

        echo $sanitized; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    }

    $pos_scope = wp_make_link_relative( get_permalink( $page_id ) );
    if ( ! $pos_scope ) { $pos_scope = '/sirapix-pos/'; }
    $sw_js = "(function(){try{if('serviceWorker' in navigator){var scope='" . esc_js( trailingslashit( $pos_scope ) ) . "';var pretty='/sirapix-pos/sw.js';var fallback='" . esc_js( home_url( '/?sirapix_pos_sw=1' ) ) . "';navigator.serviceWorker.register(pretty,{scope:scope}).catch(function(){navigator.serviceWorker.register(fallback,{scope:scope}).catch(function(){});});}}catch(e){}})();";

    wp_add_inline_script( $standalone_js_handle, $sw_js, 'after' );

    echo '</body></html>';
    exit;
} );

// Enqueue assets on POS page
add_action( 'wp_enqueue_scripts', function() {
    if ( ! is_page( (int) get_option( 'sirapix_wc_pos_page_id' ) ) ) {
        return;
    }
    $assets_ver = function_exists( 'sirapix_wc_pos_get_assets_version' ) ? sirapix_wc_pos_get_assets_version() : SIRAPIX_WC_POS_VERSION;
    wp_enqueue_style( 'sirapix-wc-pos', SIRAPIX_WC_POS_PLUGIN_URL . 'assets/css/pos.css?ver=' . $assets_ver, [], $assets_ver );
    wp_enqueue_script( 'sirapix-wc-pos', SIRAPIX_WC_POS_PLUGIN_URL . 'assets/js/pos.js?ver=' . $assets_ver, [ 'jquery' ], $assets_ver, true );

    // Expose theme colors and layout tokens as CSS variables for styling
    $primary   = sanitize_hex_color( (string) get_option( 'sirapix_wc_pos_primary_color', '#0ea5e9' ) );
    $secondary = sanitize_hex_color( (string) get_option( 'sirapix_wc_pos_secondary_color', '#f59e0b' ) );
    $primary   = $primary ? $primary : '#0ea5e9';
    $secondary = $secondary ? $secondary : '#f59e0b';
    $items_per_row = (int) get_option( 'sirapix_wc_pos_items_per_row', 8 );
    if ( $items_per_row < 2 ) { $items_per_row = 2; }
    if ( $items_per_row > 10 ) { $items_per_row = 10; }
    $inline_css = ":root{--spx-primary: {$primary}; --spx-secondary: {$secondary}; --spx-items-per-row: {$items_per_row}; --spx-bg:#F9FAFB; --spx-text:#111827; --spx-muted:#6B7280; --spx-card:#ffffff; --spx-border:#E5E7EB; --spx-radius:12px; --spx-radius-lg:16px; --spx-shadow:0 8px 24px rgba(17,24,39,.06); --spx-shadow-sm:0 2px 8px rgba(17,24,39,.06); --spx-space:16px; --spx-space-lg:24px;} body.sirapix-pos-page{background: var(--spx-bg);} .sirapix-pos-wrapper{color:var(--spx-text);}";
    wp_add_inline_style( 'sirapix-wc-pos', $inline_css );

    // Prepare branding info
    $logo_id  = get_theme_mod( 'custom_logo' );
    $logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'full' ) : '';

    $up = wp_upload_dir();
    $static_url = trailingslashit( $up['baseurl'] ) . 'sirapix-pos/';
    $products_mode = get_option( 'sirapix_wc_pos_products_mode', 'offline' );
    $settings_snapshot_url = '';
    if ( function_exists( 'sirapix_wc_pos_generate_settings_snapshot' ) ) {
        sirapix_wc_pos_generate_settings_snapshot();
    }
    if ( function_exists( 'sirapix_wc_pos_get_settings_snapshot_info' ) ) {
        $snapshots = sirapix_wc_pos_get_settings_snapshot_info();
        if ( is_array( $snapshots ) && ! empty( $snapshots['current']['filename'] ) ) {
            $settings_snapshot_url = $static_url . ltrim( (string) $snapshots['current']['filename'], '/' );
        }
    }
    wp_localize_script( 'sirapix-wc-pos', 'SIRAPIX_POS', [
        'ajax_url' => admin_url( 'admin-ajax.php' ),
        'rest_url' => esc_url( rest_url() ),
        'nonce'    => wp_create_nonce( 'wp_rest' ),
        'page_id'  => (int) get_option( 'sirapix_wc_pos_page_id' ),
        'domain'   => sirapix_wc_pos_get_site_domain(),
        'items_per_row' => $items_per_row,
        'products_mode' => $products_mode,
        'site_name'=> get_bloginfo( 'name' ),
        'logo'     => $logo_url,
        'static_url'=> esc_url( $static_url ),
        'labels'   => [
            'payment_offline' => __( 'Offline Payment', 'sirapix-pos-for-woocommerce' ),
            'shipping_offline_store' => __( 'Offline Store', 'sirapix-pos-for-woocommerce' ),
        ],
        'currency' => [
            'symbol'   => html_entity_decode( get_woocommerce_currency_symbol(), ENT_QUOTES, 'UTF-8' ),
            'position' => get_option( 'woocommerce_currency_pos' ),
            'thousand' => wc_get_price_thousand_separator(),
            'decimal'  => wc_get_price_decimal_separator(),
            'decimals' => wc_get_price_decimals(),
        ],
    ] );
    // Register Service Worker for offline POS (only on POS page).
    // Prefer pretty /sirapix-pos/sw.js path but fall back to query URL if needed.
    $pos_scope = wp_make_link_relative( get_permalink( (int) get_option( 'sirapix_wc_pos_page_id' ) ) );
    if ( ! $pos_scope ) { $pos_scope = '/sirapix-pos/'; }
    $sw_js = "(function(){try{if('serviceWorker' in navigator){var scope='" . esc_js( trailingslashit( $pos_scope ) ) . "';var pretty='/sirapix-pos/sw.js';var fallback='" . esc_js( home_url( '/?sirapix_pos_sw=1' ) ) . "';navigator.serviceWorker.register(pretty,{scope:scope}).catch(function(){navigator.serviceWorker.register(fallback,{scope:scope}).catch(function(){});});}}catch(e){}})();";
    wp_add_inline_script( 'sirapix-wc-pos', $sw_js, 'after' );
} );

// On the POS page, keep only admin bar + POS assets; dequeue everything else for a clean canvas
add_action( 'wp_enqueue_scripts', function(){
    $page_id = (int) get_option( 'sirapix_wc_pos_page_id' );
    if ( ! $page_id || ! is_page( $page_id ) ) { return; }

    // Allowlist handles we must keep
    $keep_styles  = [ 'admin-bar', 'dashicons', 'sirapix-wc-pos' ];
    $keep_scripts = [ 'admin-bar', 'jquery', 'jquery-core', 'jquery-migrate', 'sirapix-wc-pos' ];

    global $wp_styles, $wp_scripts;
    if ( $wp_styles instanceof WP_Styles ) {
        foreach ( (array) $wp_styles->queue as $handle ) {
            if ( ! in_array( $handle, $keep_styles, true ) ) {
                wp_dequeue_style( $handle );
                wp_deregister_style( $handle );
            }
        }
    }
    if ( $wp_scripts instanceof WP_Scripts ) {
        foreach ( (array) $wp_scripts->queue as $handle ) {
            if ( ! in_array( $handle, $keep_scripts, true ) ) {
                wp_dequeue_script( $handle );
                wp_deregister_script( $handle );
            }
        }
    }
}, 9999 );
