<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Detect if current request is the POS front-end page or the plugin's settings page.
 */
function sirapix_wc_pos_is_pos_context() {
    // POS front-end shortcode/page
    $page_id = (int) get_option( 'sirapix_wc_pos_page_id' );
    if ( $page_id && function_exists('is_page') && is_page( $page_id ) ) {
        return true;
    }
    // POS settings admin page
    $admin_page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if ( is_admin() && $admin_page === 'sirapix-wc-pos' ) {
        return true;
    }
    return false;
}

function sirapix_wc_pos_get_option_allowed_roles() {
    $roles = get_option( 'sirapix_wc_pos_allowed_roles', [ 'administrator', 'shop_manager' ] );
    if ( ! is_array( $roles ) ) { $roles = [ 'administrator', 'shop_manager' ]; }
    return array_values( array_unique( array_map( 'sanitize_text_field', $roles ) ) );
}

function sirapix_wc_pos_user_has_access( $user_id = 0 ) {
    if ( ! $user_id ) { $user_id = get_current_user_id(); }
    if ( ! $user_id ) { return false; }
    $user = get_userdata( $user_id );
    if ( ! $user ) { return false; }

    // Allow if user can manage WooCommerce regardless of role setting
    if ( user_can( $user_id, 'manage_woocommerce' ) ) { return true; }

    $allowed_roles = sirapix_wc_pos_get_option_allowed_roles();
    foreach ( $user->roles as $role ) {
        if ( in_array( $role, $allowed_roles, true ) ) {
            return true;
        }
    }
    return false;
}

function sirapix_wc_pos_get_site_domain() {
    $host = wp_parse_url( home_url(), PHP_URL_HOST );
    if ( ! $host ) { return 'example.com'; }
    $host = preg_replace( '/^www\./i', '', $host );
    return $host;
}

function sirapix_wc_pos_normalize_phone( $phone ) {
    $digits = preg_replace( '/[^0-9]/', '', (string) $phone );
    return $digits ?: '';
}

function sirapix_wc_pos_email_from_phone_if_missing( $email, $phone ) {
    $email = sanitize_email( (string) $email );
    if ( $email ) { return $email; }
    $digits = sirapix_wc_pos_normalize_phone( $phone );
    if ( ! $digits ) { return ''; }
    $domain = sirapix_wc_pos_get_site_domain();
    return $digits . '@' . $domain;
}

function sirapix_wc_pos_unique_email( $email ) {
    $email = sanitize_email( $email );
    if ( ! $email ) { return ''; }
    if ( ! email_exists( $email ) ) { return $email; }
    // Append +n before @ for uniqueness
    $parts = explode( '@', $email );
    $local = $parts[0];
    $domain = $parts[1] ?? 'example.com';
    $n = 2;
    do {
        $candidate = $local . '+' . $n . '@' . $domain;
        $n++;
    } while ( email_exists( $candidate ) && $n < 1000 );
    return $candidate;
}

function sirapix_wc_pos_find_or_create_customer( $name, $phone, $email ) {
    $name = sanitize_text_field( (string) $name );
    $phone = sirapix_wc_pos_normalize_phone( $phone );
    $original_email = sanitize_email( (string) $email );
    $email = sirapix_wc_pos_email_from_phone_if_missing( $email, $phone );
    $fabricated = empty( $original_email );
    if ( ! $phone || ! $name || ! $email ) {
        return new WP_Error( 'invalid_customer', __( 'Name and phone are required.', 'sirapix-pos-for-woocommerce' ) );
    }
    $email = sirapix_wc_pos_unique_email( $email );

    // 1) Try to find existing by billing_phone first
    $existing_by_phone = get_users( [
        'meta_key'   => 'billing_phone',
        'meta_value' => $phone,
        'number'     => 1,
        'fields'     => 'ID',
    ] );
    if ( ! empty( $existing_by_phone ) ) {
        $uid = (int) $existing_by_phone[0];
        update_user_meta( $uid, 'first_name', $name );
        update_user_meta( $uid, 'billing_email', $email );
        if ( $fabricated ) { update_user_meta( $uid, 'spx_fabricated_email', 1 ); }
        return $uid;
    }

    // 2) Fallback: check by email
    $user_id = email_exists( $email );
    if ( $user_id ) {
        update_user_meta( $user_id, 'billing_phone', $phone );
        update_user_meta( $user_id, 'first_name', $name );
        update_user_meta( $user_id, 'billing_email', $email );
        if ( $fabricated ) { update_user_meta( $user_id, 'spx_fabricated_email', 1 ); }
        return (int) $user_id;
    }

    // 3) Create new user using phone as username and password
    $username_base = sanitize_user( $phone, true );
    if ( ! $username_base ) { $username_base = 'pos' . wp_generate_password( 6, false ); }
    $username = $username_base;
    $i = 1;
    while ( username_exists( $username ) ) { $username = $username_base . $i; $i++; }
    $user_pass = $phone; // as requested: allow login with phone as password
    $new_id = wp_insert_user( [
        'user_login'   => $username,
        'user_email'   => $email,
        'display_name' => $name,
        'first_name'   => $name,
        'role'         => 'customer',
        'user_pass'    => $user_pass,
    ] );
    if ( is_wp_error( $new_id ) ) { return $new_id; }
    update_user_meta( $new_id, 'billing_phone', $phone );
    update_user_meta( $new_id, 'billing_email', $email );
    if ( $fabricated ) { update_user_meta( $new_id, 'spx_fabricated_email', 1 ); }
    return (int) $new_id;
}

function sirapix_wc_pos_get_assets_version() {
    $version = (int) get_option( 'sirapix_wc_pos_assets_version', 1 );
    if ( $version < 1 ) { $version = 1; }
    return SIRAPIX_WC_POS_VERSION . '-' . $version;
}

function sirapix_wc_pos_get_setting( $name, $default = null ) {
    return get_option( $name, $default );
}

function sirapix_wc_pos_generate_settings_snapshot() {
    $up = wp_upload_dir();
    if ( empty( $up['basedir'] ) ) {
        return;
    }
    $dir = trailingslashit( $up['basedir'] ) . 'sirapix-pos';
    if ( ! file_exists( $dir ) ) {
        wp_mkdir_p( $dir );
    }
    if ( ! is_dir( $dir ) || ! wp_is_writable( $dir ) ) {
        return;
    }

    $settings = [];
    $known_keys = [
        'sirapix_wc_pos_page_id',
        'sirapix_wc_pos_allowed_roles',
        'sirapix_wc_pos_products_generated_at',
        'sirapix_wc_pos_products_count',
        'sirapix_wc_pos_items_per_row',
        'sirapix_wc_pos_customers_generated_at',
        'sirapix_wc_pos_customers_count',
        'sirapix_wc_pos_products_mode',
        'sirapix_wc_pos_assets_version',
        'sirapix_wc_pos_products_file',
        'sirapix_wc_pos_sms_api_key',
        'sirapix_wc_pos_primary_color',
        'sirapix_wc_pos_secondary_color',
        // snapshot option itself for debugging/visibility
        'sirapix_wc_pos_settings_snapshots',
    ];
    foreach ( $known_keys as $k ) {
        $settings[ $k ] = sirapix_wc_pos_get_setting( $k );
    }

    $core_payload = [ 'settings' => $settings ];
    $json_for_hash = wp_json_encode( $core_payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES );
    if ( ! is_string( $json_for_hash ) || $json_for_hash === '' ) {
        return;
    }
    $hash = md5( $json_for_hash );
    $filename = 'settings-' . $hash . '.json';
    $file = trailingslashit( $dir ) . $filename;

    $payload = $core_payload;
    $payload['generated_at'] = time();
    $json = wp_json_encode( $payload );
    if ( ! is_string( $json ) || $json === '' ) {
        return;
    }

    file_put_contents( $file, $json );

    $snapshots = get_option( 'sirapix_wc_pos_settings_snapshots', [
        'current'  => null,
        'previous' => null,
    ] );
    if ( ! is_array( $snapshots ) ) {
        $snapshots = [ 'current' => null, 'previous' => null ];
    }

    $current = isset( $snapshots['current'] ) && is_array( $snapshots['current'] ) ? $snapshots['current'] : null;
    $snapshots['previous'] = $current;
    $snapshots['current'] = [
        'hash'        => $hash,
        'filename'    => $filename,
        'generated_at'=> time(),
    ];

    update_option( 'sirapix_wc_pos_settings_snapshots', $snapshots );
}

function sirapix_wc_pos_get_settings_snapshot_info() {
    $snapshots = get_option( 'sirapix_wc_pos_settings_snapshots', [
        'current'  => null,
        'previous' => null,
    ] );
    if ( ! is_array( $snapshots ) ) {
        $snapshots = [ 'current' => null, 'previous' => null ];
    }
    return $snapshots;
}

add_action( 'updated_option', function( $option, $old_value, $value ) {
    if ( strpos( $option, 'sirapix_wc_pos_' ) !== 0 ) {
        return;
    }
    if ( $option === 'sirapix_wc_pos_assets_version' || $option === 'sirapix_wc_pos_settings_snapshots' ) {
        return;
    }
    static $running = false;
    if ( $running ) {
        return;
    }
    $running = true;

    $current = (int) get_option( 'sirapix_wc_pos_assets_version', 1 );
    if ( $current < 1 ) { $current = 1; }
    update_option( 'sirapix_wc_pos_assets_version', $current + 1 );

    // Bump a friendly POS hash used in the POS URL so that each settings
    // save produces a new, cache-busting /sirapix-pos/{hash}/ link.
    $hash = substr( md5( microtime( true ) . wp_rand() ), 0, 8 );
    update_option( 'sirapix_wc_pos_pos_hash', $hash );

    if ( function_exists( 'sirapix_wc_pos_generate_settings_snapshot' ) ) {
        sirapix_wc_pos_generate_settings_snapshot();
    }

    $running = false;
}, 10, 3 );
