<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Sirapix_WC_POS_Settings {

    public static function init() {
        // Use late priority to ensure WooCommerce admin menu is registered first
        add_action( 'admin_menu', [ __CLASS__, 'add_menu' ], 99 );
        add_action( 'admin_init', [ __CLASS__, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue_admin' ] );
    }

    public static function add_menu() {
        $parent = 'woocommerce';
        // If WooCommerce menu does not exist for any reason, attach under Settings as a fallback
        global $admin_page_hooks;
        if ( ! isset( $admin_page_hooks[ $parent ] ) ) {
            $parent = 'options-general.php';
        }
        add_submenu_page(
            $parent,
            __( 'SIRAPIX POS', 'sirapix-pos-for-woocommerce' ),
            __( 'SIRAPIX POS', 'sirapix-pos-for-woocommerce' ),
            'manage_woocommerce',
            'sirapix-wc-pos',
            [ __CLASS__, 'render_page' ]
        );
    }

    public static function register_settings() {
        register_setting( 'sirapix_wc_pos_settings', 'sirapix_wc_pos_allowed_roles', [
            'type' => 'array',
            'sanitize_callback' => function( $value ){
                if ( ! is_array( $value ) ) { return [ 'administrator', 'shop_manager' ]; }
                return array_values( array_unique( array_map( 'sanitize_text_field', $value ) ) );
            }
        ] );

        register_setting( 'sirapix_wc_pos_settings', 'sirapix_wc_pos_items_per_row', [
            'type' => 'integer',
            'default' => 8,
            'sanitize_callback' => function( $val ){
                $n = absint( $val );
                if ( $n < 2 ) { $n = 2; }
                if ( $n > 10 ) { $n = 10; }
                return $n ?: 8;
            }
        ] );

        register_setting( 'sirapix_wc_pos_settings', 'sirapix_wc_pos_products_mode', [
            'type' => 'string',
            'default' => 'offline',
            'sanitize_callback' => function( $val ){
                $val = is_string( $val ) ? trim( $val ) : 'offline';
                return in_array( $val, [ 'offline', 'online' ], true ) ? $val : 'offline';
            }
        ] );

        register_setting( 'sirapix_wc_pos_settings', 'sirapix_wc_pos_sms_api_key', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_text_field',
        ] );

        // Theme color settings
        register_setting( 'sirapix_wc_pos_settings', 'sirapix_wc_pos_primary_color', [
            'type' => 'string',
            'default' => '#0ea5e9',
            'sanitize_callback' => function( $val ){
                $val = trim( (string) $val );
                if ( preg_match( '/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/', $val ) ) { return $val; }
                return '#0ea5e9';
            },
        ] );

        register_setting( 'sirapix_wc_pos_settings', 'sirapix_wc_pos_secondary_color', [
            'type' => 'string',
            'default' => '#f59e0b',
            'sanitize_callback' => function( $val ){
                $val = trim( (string) $val );
                if ( preg_match( '/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/', $val ) ) { return $val; }
                return '#f59e0b';
            },
        ] );

        register_setting( 'sirapix_wc_pos_settings', 'sirapix_wc_pos_aggressive_image_prefetch', [
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => function( $val ){
                return ! empty( $val ) ? 1 : 0;
            },
        ] );
    }

    public static function enqueue_admin( $hook = '' ) {
        $admin_page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( $admin_page !== 'sirapix-wc-pos' ) {
            return;
        }
        wp_enqueue_style( 'wp-color-picker' );
        wp_enqueue_script( 'wp-color-picker' );
        // Add a small inline style for the settings header icon using the color picker handle.
        $svg = "data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none'%3E%3Cpath stroke='%23020202' stroke-width='1.91' stroke-miterlimit='10' d='M17.07,20.61H9.79a2,2,0,0,1-2-2,2,2,0,0,1,2-2h1.87L5,9.86a2,2,0,0,1-.19-2.65,1.88,1.88,0,0,1,1.47-.68,1.84,1.84,0,0,1,1.35.55l4.06,4.06,4.08-3.06a1.91,1.91,0,0,1,2.5.18h0A17.18,17.18,0,0,1,22.42,15l.06.19'/%3E%3Cpath stroke='%23020202' stroke-width='1.91' stroke-miterlimit='10' d='M10.63,10.12A4.73,4.73,0,0,0,11,8.17,4.78,4.78,0,1,0,6.26,13a4.67,4.67,0,0,0,1.55-.26'/%3E%3C/svg%3E";
        $svg_url = esc_url( $svg, array_merge( wp_allowed_protocols(), [ 'data' ] ) );
        $inline_css  = '.sirapix-pos-settings-title{display:flex;align-items:center;gap:8px;}';
        $inline_css .= '.sirapix-pos-settings-title .sirapix-pos-settings-icon{width:24px;height:24px;background-image:url(' . $svg_url . ');background-repeat:no-repeat;background-size:24px 24px;display:inline-block;}';
        wp_add_inline_style( 'wp-color-picker', $inline_css );
        $rest = esc_url( rest_url() );
        $rest_js = esc_js( $rest );
        $nonce = wp_create_nonce( 'wp_rest' );
        $page_id = (int) get_option( 'sirapix_wc_pos_page_id' );
        $pos_url = $page_id ? get_permalink( $page_id ) : home_url( '/sirapix-pos/' );
        $prod_ts = (int) get_option( 'sirapix_wc_pos_products_generated_at', 0 );
        $prod_count = (int) get_option( 'sirapix_wc_pos_products_count', 0 );
        $prod_h = $prod_ts ? esc_js( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $prod_ts ) ) : '';
        $js = "jQuery(function($){ $('.sirapix-color-field').wpColorPicker(); var REST='".$rest_js."'.replace(/\/$/,''); var NONCE='".$nonce."'; var POS_URL='".esc_js( $pos_url )."'; function upd(id, txt){ var el=$('#'+id); if(el.length){ el.text(txt||''); } } function notice(type, msg){ var box=$('#spx-sync-notice'); if(!box.length) return; var cls=(type==='success')?'updated notice-success':'error notice-error'; box.removeClass().addClass('notice '+cls).find('.msg').text(msg||''); box.show(); setTimeout(function(){ box.fadeOut(200); }, 3000); } upd('spx-products-last', '".$prod_h."'); upd('spx-products-count','".$prod_count."'); function bindProducts(){ var btn=$('#spx-sync-products'); var stat=$('#spx-sync-products-stat'); if(!btn.length) return; btn.on('click', function(e){ e.preventDefault(); if(stat.length){ stat.text('Working...'); } btn.prop('disabled', true); fetch(REST+'/sirapix-pos/v1/static/products',{ method:'POST', headers:{ 'X-WP-Nonce': NONCE, 'Content-Type':'application/json' }, body:'{}' }).then(r=>r.json().then(j=>({ ok:r.ok, j }))).then(({ok,j})=>{ if(ok){ var ts=(j&&j.generated_at)? j.generated_at : Math.floor(Date.now()/1000); var d=new Date(ts*1000); $('#spx-products-last').text(d.toLocaleString()); if(j && j.total_products!==undefined){ $('#spx-products-count').text(j.total_products); } if(stat.length){ stat.text('Done'); } notice('success','products.json generated.'); } else { if(stat.length){ stat.text((j&&j.message)||'Failed'); } notice('error','Failed to generate products.json'); } }).catch(()=>{ if(stat.length){ stat.text('Failed'); } notice('error','Failed to generate products.json'); }).finally(()=>{ btn.prop('disabled', false); setTimeout(function(){ if(stat.length){ stat.text(''); } }, 1500); }); }); } function bindAggressiveImages(){ var btn=$('#spx-prefetch-images'); if(!btn.length) return; btn.on('click', function(e){ e.preventDefault(); if(!window.confirm('This will open the POS and prefetch product images for offline use. Continue?')) return; try{ window.localStorage.setItem('sirapix_pos_prefetch_images','1'); }catch(err){} if(POS_URL){ window.open(POS_URL, '_blank'); } }); } bindProducts(); bindAggressiveImages(); });";
        wp_add_inline_script( 'wp-color-picker', $js );
    }

    public static function render_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'You do not have permission to access this page.', 'sirapix-pos-for-woocommerce' ) );
        }
        $roles = wp_roles()->get_names();
        $allowed = sirapix_wc_pos_get_option_allowed_roles();
        $items_per_row = get_option( 'sirapix_wc_pos_items_per_row', 8 );
        if ( $items_per_row < 2 ) { $items_per_row = 2; }
        if ( $items_per_row > 10 ) { $items_per_row = 10; }
        $products_mode = get_option( 'sirapix_wc_pos_products_mode', 'offline' );
        $products_file = get_option( 'sirapix_wc_pos_products_file', 'products.json' );
        $aggressive_images = (bool) get_option( 'sirapix_wc_pos_aggressive_image_prefetch', 0 );
        $sms_key = get_option( 'sirapix_wc_pos_sms_api_key', '' );
        $primary = get_option( 'sirapix_wc_pos_primary_color', '#0ea5e9' );
        $secondary = get_option( 'sirapix_wc_pos_secondary_color', '#f59e0b' );

        $active_tab = 'general';
        ?>
        <div class="wrap">
            <h1 class="sirapix-pos-settings-title"><span class="sirapix-pos-settings-icon" aria-hidden="true"></span><?php esc_html_e( 'SIRAPIX WooCommerce POS Settings', 'sirapix-pos-for-woocommerce' ); ?></h1>
            <div id="spx-sync-notice" class="notice" style="display:none;"><p class="msg" style="margin:.5em 0;"></p></div>
            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=sirapix-wc-pos' ) ); ?>" class="nav-tab nav-tab-active"><?php esc_html_e( 'General', 'sirapix-pos-for-woocommerce' ); ?></a>
            </h2>

            <form method="post" action="options.php">
                <?php settings_fields( 'sirapix_wc_pos_settings' ); ?>
                <?php if ( $active_tab === 'general' ) : ?>
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Allowed Roles', 'sirapix-pos-for-woocommerce' ); ?></th>
                            <td>
                                <?php foreach ( $roles as $key => $label ) : ?>
                                    <label style="display:block; margin-bottom:6px;">
                                        <input type="checkbox" name="sirapix_wc_pos_allowed_roles[]" value="<?php echo esc_attr( $key ); ?>" <?php checked( in_array( $key, $allowed, true ) ); ?> />
                                        <?php echo esc_html( $label ); ?>
                                    </label>
                                <?php endforeach; ?>
                                <p class="description"><?php esc_html_e( 'Users with checked roles can access the POS.', 'sirapix-pos-for-woocommerce' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Items Per Row', 'sirapix-pos-for-woocommerce' ); ?></th>
                            <td>
                                <input type="number" name="sirapix_wc_pos_items_per_row" value="<?php echo esc_attr( $items_per_row ); ?>" min="2" max="8" step="1" />
                                <p class="description"><?php esc_html_e( 'Number of product items per row in the POS grid (2–8).', 'sirapix-pos-for-woocommerce' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Products Source', 'sirapix-pos-for-woocommerce' ); ?></th>
                            <td>
                                <label style="display:block; margin-bottom:4px;">
                                    <input type="radio" name="sirapix_wc_pos_products_mode" value="offline" <?php checked( $products_mode, 'offline' ); ?> />
                                    <?php esc_html_e( 'Offline static JSON (faster, works offline)', 'sirapix-pos-for-woocommerce' ); ?>
                                </label>
                                <label style="display:block; margin-bottom:4px;">
                                    <input type="radio" name="sirapix_wc_pos_products_mode" value="online" <?php checked( $products_mode, 'online' ); ?> />
                                    <?php esc_html_e( 'Live WooCommerce products via REST API (always online)', 'sirapix-pos-for-woocommerce' ); ?>
                                </label>
                                <p class="description"><?php esc_html_e( 'Choose whether the POS loads products from the offline static products.json file or directly from WooCommerce in real time.', 'sirapix-pos-for-woocommerce' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Primary Color', 'sirapix-pos-for-woocommerce' ); ?></th>
                            <td>
                                <input type="text" name="sirapix_wc_pos_primary_color" value="<?php echo esc_attr( $primary ); ?>" class="sirapix-color-field" data-default-color="#0ea5e9" />
                                <p class="description"><?php esc_html_e( 'Primary brand color used for buttons, active states, and accents (e.g., #0ea5e9).', 'sirapix-pos-for-woocommerce' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Secondary Color', 'sirapix-pos-for-woocommerce' ); ?></th>
                            <td>
                                <input type="text" name="sirapix_wc_pos_secondary_color" value="<?php echo esc_attr( $secondary ); ?>" class="sirapix-color-field" data-default-color="#f59e0b" />
                                <p class="description"><?php esc_html_e( 'Secondary accent color for highlights (e.g., #f59e0b).', 'sirapix-pos-for-woocommerce' ); ?></p>
                            </td>
                        </tr>
                    </table>
                    <h2 style="margin-top:24px;"><?php esc_html_e( 'Manual Sync', 'sirapix-pos-for-woocommerce' ); ?></h2>
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Products & Categories', 'sirapix-pos-for-woocommerce' ); ?></th>
                            <td>
                                <button type="button" class="button button-primary" id="spx-sync-products"><?php esc_html_e( 'Generate products.json', 'sirapix-pos-for-woocommerce' ); ?></button>
                                <span id="spx-sync-products-stat" style="margin-left:8px;"></span>
                                <p class="description"><?php esc_html_e( 'Last generated:', 'sirapix-pos-for-woocommerce' ); ?> <span id="spx-products-last"></span> · <?php esc_html_e( 'Total products:', 'sirapix-pos-for-woocommerce' ); ?> <span id="spx-products-count"></span></p>
                                <p class="description"><?php esc_html_e( 'Current offline JSON file:', 'sirapix-pos-for-woocommerce' ); ?> <code><?php echo esc_html( $products_file ); ?></code></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e( 'Offline Images Prefetch', 'sirapix-pos-for-woocommerce' ); ?></th>
                            <td>
                                <label style="display:block; margin-bottom:6px;">
                                    <input type="checkbox" name="sirapix_wc_pos_aggressive_image_prefetch" value="1" <?php checked( $aggressive_images ); ?> />
                                    <?php esc_html_e( 'Enable image prefetch helper (runs from POS to cache product images for offline use).', 'sirapix-pos-for-woocommerce' ); ?>
                                </label>
                                <button type="button" class="button" id="spx-prefetch-images"><?php esc_html_e( 'Run image prefetch now', 'sirapix-pos-for-woocommerce' ); ?></button>
                                <p class="description"><?php esc_html_e( 'Opens the POS and prefetches product images so they are available offline. This may take time and bandwidth depending on how many products you have.', 'sirapix-pos-for-woocommerce' ); ?></p>
                            </td>
                        </tr>
                    </table>
                <?php endif; ?>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}

Sirapix_WC_POS_Settings::init();
