<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Sirapix_WC_POS_REST {

    public static function init() {
        add_action( 'rest_api_init', [ __CLASS__, 'register_routes' ] );
    }

    protected static function get_static_dir() {
        $up = wp_upload_dir();
        $dir = trailingslashit( $up['basedir'] ) . 'sirapix-pos';
        if ( ! file_exists( $dir ) ) { wp_mkdir_p( $dir ); }
        return $dir;
    }

    protected static function get_static_url() {
        $up = wp_upload_dir();
        return trailingslashit( $up['baseurl'] ) . 'sirapix-pos/';
    }

    public static function generate_static_products( WP_REST_Request $request ) {
        // Categories (initial full list; will be filtered later to hide empty ones in static JSON)
        $terms = get_terms( [ 'taxonomy' => 'product_cat', 'hide_empty' => false ] );
        $cats = [];
        if ( ! is_wp_error( $terms ) ) {
            foreach ( $terms as $t ) {
                $thumb_id = get_term_meta( $t->term_id, 'thumbnail_id', true );
                $img = $thumb_id ? wp_get_attachment_image_url( $thumb_id, 'woocommerce_thumbnail' ) : '';
                $cats[] = [ 'id' => (int) $t->term_id, 'name' => $t->name, 'count' => (int) $t->count, 'image' => $img ];
            }
        }

        // Products with variations
        $args = [
            'status' => 'publish',
            'limit'  => -1,
            'return' => 'objects',
            'type'   => [ 'simple', 'variable' ],
        ];
        $results = wc_get_products( $args );
        $products = [];
        $used_cat_counts = [];
        foreach ( $results as $p ) {
            if ( ! $p instanceof WC_Product ) { continue; }
            $img_id = $p->get_image_id();
            $img = $img_id ? wp_get_attachment_image_url( $img_id, 'woocommerce_thumbnail' ) : wc_placeholder_img_src();
            $row = [
                'id'          => $p->get_id(),
                'name'        => $p->get_name(),
                'type'        => $p->get_type(),
                'price'       => (float) wc_get_price_to_display( $p ),
                'price_html'  => wc_price( (float) wc_get_price_to_display( $p ) ),
                'image'       => $img,
                'stock'       => $p->get_stock_quantity(),
                'stock_status'=> $p->get_stock_status(),
                'categories'  => wp_get_post_terms( $p->get_id(), 'product_cat', [ 'fields' => 'ids' ] ),
            ];
            // Track how many products belong to each category ID for this static dataset
            if ( ! empty( $row['categories'] ) && is_array( $row['categories'] ) ) {
                foreach ( $row['categories'] as $cid ) {
                    $cid = (int) $cid;
                    if ( $cid <= 0 ) { continue; }
                    if ( ! isset( $used_cat_counts[ $cid ] ) ) {
                        $used_cat_counts[ $cid ] = 0;
                    }
                    $used_cat_counts[ $cid ]++;
                }
            }
            if ( $p->is_type( 'variable' ) ) {
                /** @var WC_Product_Variable $p */
                $min = (float) wc_get_price_to_display( $p, [ 'price' => $p->get_variation_price( 'min', true ) ] );
                $max = (float) wc_get_price_to_display( $p, [ 'price' => $p->get_variation_price( 'max', true ) ] );
                $row['price_from'] = $min;
                $row['price_to']   = $max;
                $row['has_price_range'] = ( $min !== $max );
                $attrs = [];
                foreach ( $p->get_variation_attributes() as $slug => $opts ) {
                    $attrs[] = [ 'name' => wc_attribute_label( $slug ), 'slug' => $slug, 'options' => array_values( (array) $opts ) ];
                }
                $row['attributes_summary'] = $attrs;
                // Variations list
                $vars = [];
                foreach ( $p->get_children() as $vid ) {
                    $v = wc_get_product( $vid );
                    if ( ! $v || ! $v->exists() ) { continue; }
                    /** @var WC_Product_Variation $v */
                    $atts = [];
                    foreach ( $v->get_attributes() as $k => $val ) { $atts[ $k ] = is_array( $val ) ? implode(',', $val) : (string) $val; }
                    $imgv_id = $v->get_image_id() ?: $p->get_image_id();
                    $imgv = $imgv_id ? wp_get_attachment_image_url( $imgv_id, 'woocommerce_thumbnail' ) : wc_placeholder_img_src();
                    $vars[] = [
                        'variation_id' => (int) $v->get_id(),
                        'attributes'   => $atts,
                        'price'        => (float) wc_get_price_to_display( $v ),
                        'price_html'   => wc_price( (float) wc_get_price_to_display( $v ) ),
                        'stock_status' => $v->get_stock_status(),
                        'stock_qty'    => $v->get_stock_quantity(),
                        'image'        => $imgv,
                    ];
                }
                $row['variations'] = $vars;
            }
            $products[] = $row;
        }

        // Filter categories to only those that have at least one product in the static dataset
        if ( ! empty( $cats ) && ! empty( $used_cat_counts ) ) {
            $cats = array_values( array_filter( $cats, function( $cat ) use ( $used_cat_counts ) {
                $cid = isset( $cat['id'] ) ? (int) $cat['id'] : 0;
                if ( $cid <= 0 ) { return false; }
                return isset( $used_cat_counts[ $cid ] ) && $used_cat_counts[ $cid ] > 0;
            } ) );

            // Override counts with counts based on the static products dataset
            foreach ( $cats as &$cat ) {
                $cid = isset( $cat['id'] ) ? (int) $cat['id'] : 0;
                if ( $cid > 0 && isset( $used_cat_counts[ $cid ] ) ) {
                    $cat['count'] = (int) $used_cat_counts[ $cid ];
                } else {
                    $cat['count'] = 0;
                }
            }
            unset( $cat );

            // Ensure categories are ordered by product count (desc) in the static JSON as well
            usort( $cats, function( $a, $b ) {
                return ( $b['count'] ?? 0 ) <=> ( $a['count'] ?? 0 );
            } );
        }

        $now = time();
        $payload = [ 'generated_at' => $now, 'categories' => $cats, 'products' => $products ];
        $dir = self::get_static_dir();

        // Clean up older static products JSON files to avoid disk bloat, but keep latest 2 for debugging
        $existing = glob( trailingslashit( $dir ) . 'products-*.json' );
        if ( is_array( $existing ) && ! empty( $existing ) ) {
            // Sort by modification time descending (newest first)
            usort( $existing, function( $a, $b ) {
                $ta = @filemtime( $a );
                $tb = @filemtime( $b );
                if ( $ta === $tb ) { return 0; }
                return ( $ta > $tb ) ? -1 : 1;
            } );
            // Keep first 2, delete the rest
            $to_delete = array_slice( $existing, 2 );
            foreach ( $to_delete as $old_file ) {
                if ( is_file( $old_file ) ) {
                    wp_delete_file( $old_file );
                }
            }
        }

        // Use a random hash in the filename to avoid browser caching issues for offline data
        $hash = wp_generate_password( 8, false, false );
        $filename = 'products-' . $hash . '.json';
        $file = trailingslashit( $dir ) . $filename;
        file_put_contents( $file, wp_json_encode( $payload ) );
        update_option( 'sirapix_wc_pos_products_generated_at', $now );
        update_option( 'sirapix_wc_pos_products_count', count( $products ) );
        update_option( 'sirapix_wc_pos_products_file', $filename );
        return new WP_REST_Response( [ 'ok' => true, 'url' => self::get_static_url() . $filename, 'generated_at' => $now, 'total_products' => count( $products ), 'total_categories' => is_array($cats)?count($cats):0 ], 200 );
    }

    public static function get_customer_lookup( WP_REST_Request $request ) {
        $phone = sanitize_text_field( (string) $request->get_param( 'phone' ) );
        $phone = sirapix_wc_pos_normalize_phone( $phone );
        if ( ! $phone ) {
            return new WP_REST_Response( [ 'found' => false ], 200 );
        }
        // Find user by billing_phone meta
        $existing = get_users( [
            'meta_key'   => 'billing_phone',
            'meta_value' => $phone,
            'number'     => 1,
            'fields'     => 'ID',
        ] );
        if ( empty( $existing ) ) {
            return new WP_REST_Response( [ 'found' => false ], 200 );
        }
        $uid   = (int) $existing[0];
        $user  = get_user_by( 'id', $uid );
        $name  = get_user_meta( $uid, 'first_name', true );
        if ( ! $name ) { $name = $user ? $user->display_name : ''; }
        $email = get_user_meta( $uid, 'billing_email', true );
        if ( ! $email && $user ) { $email = $user->user_email; }
        return new WP_REST_Response( [ 'found' => true, 'name' => (string) $name, 'email' => (string) $email ], 200 );
    }

    public static function register_routes() {
        register_rest_route( 'sirapix-pos/v1', '/categories', [
            [
                'methods'  => WP_REST_Server::READABLE,
                'callback' => [ __CLASS__, 'get_categories' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [],
            ],
        ] );

        register_rest_route( 'sirapix-pos/v1', '/products', [
            [
                'methods'  => WP_REST_Server::READABLE,
                'callback' => [ __CLASS__, 'get_products' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'category' => [ 'type' => 'integer', 'required' => false ],
                    'search'   => [ 'type' => 'string', 'required' => false ],
                    'page'     => [ 'type' => 'integer', 'required' => false, 'default' => 1 ],
                    'per_page' => [ 'type' => 'integer', 'required' => false, 'default' => 24 ],
                ],
            ],
        ] );

        // Variations for a specific variable product
        register_rest_route( 'sirapix-pos/v1', '/products/(?P<id>\d+)/variations', [
            [
                'methods'  => WP_REST_Server::READABLE,
                'callback' => [ __CLASS__, 'get_product_variations' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'id' => [ 'type' => 'integer', 'required' => true ],
                ],
            ],
        ] );

        register_rest_route( 'sirapix-pos/v1', '/customers', [
            [
                'methods'  => WP_REST_Server::CREATABLE,
                'callback' => [ __CLASS__, 'post_customer' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'name'  => [ 'type' => 'string', 'required' => true ],
                    'phone' => [ 'type' => 'string', 'required' => true ],
                    'email' => [ 'type' => 'string', 'required' => false ],
                ],
            ],
        ] );

        // Lookup existing customer by phone
        register_rest_route( 'sirapix-pos/v1', '/customers/lookup', [
            [
                'methods'  => WP_REST_Server::READABLE,
                'callback' => [ __CLASS__, 'get_customer_lookup' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'phone' => [ 'type' => 'string', 'required' => true ],
                ],
            ],
        ] );

        register_rest_route( 'sirapix-pos/v1', '/orders', [
            [
                'methods'  => WP_REST_Server::CREATABLE,
                'callback' => [ __CLASS__, 'post_order' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'customer' => [ 'type' => 'object', 'required' => true ],
                    'items'    => [ 'type' => 'array', 'required' => true ],
                    'payment'  => [ 'type' => 'string', 'required' => false, 'default' => 'pos_offline' ],
                    'notes'    => [ 'type' => 'string', 'required' => false ],
                    'origin'   => [ 'type' => 'string', 'required' => false, 'enum' => [ 'in_store', 'phone' ] ],
                ],
            ],
            [
                'methods'  => WP_REST_Server::READABLE,
                'callback' => [ __CLASS__, 'get_orders' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'search'   => [ 'type' => 'string',  'required' => false ],
                    'origin'   => [ 'type' => 'string',  'required' => false, 'enum' => [ 'in_store', 'phone', 'online' ] ],
                    'page'     => [ 'type' => 'integer', 'required' => false, 'default' => 1 ],
                    'per_page' => [ 'type' => 'integer', 'required' => false, 'default' => 30 ],
                ],
            ],
        ] );

        register_rest_route( 'sirapix-pos/v1', '/orders/(?P<id>\d+)', [
            [
                'methods'  => WP_REST_Server::READABLE,
                'callback' => [ __CLASS__, 'get_order_detail' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'id' => [ 'type' => 'integer', 'required' => true ],
                ],
            ],
        ] );

        register_rest_route( 'sirapix-pos/v1', '/orders/(?P<id>\d+)/refund', [
            [
                'methods'  => WP_REST_Server::CREATABLE,
                'callback' => [ __CLASS__, 'post_order_refund' ],
                'permission_callback' => [ __CLASS__, 'permission' ],
                'args' => [
                    'id'            => [ 'type' => 'integer', 'required' => true ],
                    'items'         => [ 'type' => 'array',   'required' => false ],
                    'refund_amount' => [ 'type' => 'number',  'required' => false ],
                    'reason'        => [ 'type' => 'string',  'required' => false ],
                    'restock'       => [ 'type' => 'boolean', 'required' => false, 'default' => false ],
                ],
            ],
        ] );

        // Static JSON generators (admin-only)
        register_rest_route( 'sirapix-pos/v1', '/static/products', [
            [
                'methods'  => WP_REST_Server::CREATABLE,
                'callback' => [ __CLASS__, 'generate_static_products' ],
                'permission_callback' => function(){ return current_user_can('manage_woocommerce'); },
            ],
        ] );
    }

    public static function permission( $request ) {
        if ( ! is_user_logged_in() ) { return false; }
        if ( current_user_can( 'manage_woocommerce' ) ) { return true; }
        return sirapix_wc_pos_user_has_access();
    }

    public static function get_categories( WP_REST_Request $request ) {
        $terms = get_terms( [
            'taxonomy'   => 'product_cat',
            'hide_empty' => true,
            'orderby'    => 'count',
            'order'      => 'DESC',
        ] );
        if ( is_wp_error( $terms ) ) {
            return new WP_REST_Response( [ 'error' => $terms->get_error_message() ], 500 );
        }
        $data = [];
        foreach ( $terms as $t ) {
            $thumb_id = get_term_meta( $t->term_id, 'thumbnail_id', true );
            $img = $thumb_id ? wp_get_attachment_image_url( $thumb_id, 'woocommerce_thumbnail' ) : '';
            $data[] = [
                'id'    => (int) $t->term_id,
                'name'  => $t->name,
                'count' => (int) $t->count,
                'image' => $img,
            ];
        }
        return new WP_REST_Response( [ 'categories' => $data ], 200 );
    }

    public static function get_products( WP_REST_Request $request ) {
        $category_id = absint( $request->get_param( 'category' ) );
        $search      = sanitize_text_field( (string) $request->get_param( 'search' ) );
        $page        = max( 1, absint( $request->get_param( 'page' ) ) );
        $per_page    = min( 60, max( 1, absint( $request->get_param( 'per_page' ) ) ) );

        $args = [
            'status'   => 'publish',
            'paginate' => true,
            'limit'    => $per_page,
            'page'     => $page,
            'return'   => 'objects',
            'type'     => [ 'simple', 'variable' ],
        ];

        if ( $search !== '' ) {
            $args['search'] = $search;
        }

        if ( $category_id ) {
            $term = get_term( $category_id, 'product_cat' );
            if ( $term && ! is_wp_error( $term ) ) {
                $args['category'] = [ $term->slug ];
            }
        }

        $results = wc_get_products( $args );
        $products = [];
        foreach ( $results->products as $p ) {
            if ( ! $p instanceof WC_Product ) { continue; }
            $img_id = $p->get_image_id();
            $img = $img_id ? wp_get_attachment_image_url( $img_id, 'woocommerce_thumbnail' ) : wc_placeholder_img_src();
            $price = (float) wc_get_price_to_display( $p );
            if ( $search !== '' ) {
                $name = $p->get_name();
                if ( stripos( $name, $search ) === false ) { continue; }
            }
            $row = [
                'id'          => $p->get_id(),
                'name'        => $p->get_name(),
                'type'        => $p->get_type(),
                'price'       => $price,
                'price_html'  => wc_price( $price ),
                'image'       => $img,
                'stock'       => $p->get_stock_quantity(),
                'stock_status'=> $p->get_stock_status(),
            ];
            if ( $p->is_type( 'variable' ) ) {
                /** @var WC_Product_Variable $p */
                $min = (float) wc_get_price_to_display( $p, [ 'price' => $p->get_variation_price( 'min', true ) ] );
                $max = (float) wc_get_price_to_display( $p, [ 'price' => $p->get_variation_price( 'max', true ) ] );
                $row['price_from'] = $min;
                $row['price_to']   = $max;
                $row['has_price_range'] = ( $min !== $max );
                // Attribute summary: name/slug/options
                $attrs = [];
                foreach ( $p->get_variation_attributes() as $slug => $opts ) {
                    $tax = wc_attribute_taxonomy_id_by_name( $slug );
                    $name = wc_attribute_label( $slug );
                    $values = array_values( array_filter( array_map( 'wc_clean', (array) $opts ) ) );
                    $attrs[] = [ 'name' => $name, 'slug' => $slug, 'options' => $values ];
                }
                $row['attributes_summary'] = $attrs;
            }
            $products[] = $row;
        }

        return new WP_REST_Response( [
            'page'      => (int) $page,
            'per_page'  => (int) $per_page,
            'total'     => (int) $results->total,
            'max_pages' => (int) $results->max_num_pages,
            'products'  => $products,
        ], 200 );
    }

    public static function get_orders( WP_REST_Request $request ) {
        $search   = sanitize_text_field( (string) $request->get_param( 'search' ) );
        $origin   = sanitize_text_field( (string) $request->get_param( 'origin' ) );
        $page     = max( 1, absint( $request->get_param( 'page' ) ) );
        $per_page = min( 100, max( 1, absint( $request->get_param( 'per_page' ) ) ) );

        // If search looks like a numeric order ID, try direct lookup first.
        if ( $search !== '' && ctype_digit( $search ) ) {
            $order_id = absint( $search );
            $order    = wc_get_order( $order_id );
            if ( $order instanceof WC_Order ) {
                $data = [];
                $data[] = self::format_order_for_pos_list( $order );

                return new WP_REST_Response( [
                    'page'       => 1,
                    'per_page'   => 1,
                    'total'      => 1,
                    'max_pages'  => 1,
                    'orders'     => $data,
                ], 200 );
            }
        }

        $query_args = [
            'type'     => 'shop_order',
            'status'   => array_keys( wc_get_order_statuses() ),
            'paginate' => true,
            'limit'    => $per_page,
            'page'     => $page,
            'orderby'  => 'date',
            'order'    => 'DESC',
        ];

        $meta_query = [];

        // Origin filter based on stored POS origin meta.
        if ( $origin === 'in_store' ) {
            $meta_query[] = [
                'key'   => '_sirapix_pos_origin',
                'value' => __( 'In-Store Purchase', 'sirapix-pos-for-woocommerce' ),
            ];
        } elseif ( $origin === 'phone' ) {
            $meta_query[] = [
                'key'   => '_sirapix_pos_origin',
                'value' => __( 'Phone', 'sirapix-pos-for-woocommerce' ),
            ];
        } elseif ( $origin === 'online' ) {
            // Treat orders without a POS origin meta as "online".
            $meta_query[] = [
                'key'     => '_sirapix_pos_origin',
                'compare' => 'NOT EXISTS',
            ];
        }

        // Search by customer phone/email when search is non-empty and not an order ID.
        if ( $search !== '' ) {
            $customer_meta_query = [ 'relation' => 'OR' ];

            // Phone: use the raw search string for a broad LIKE match. This keeps behavior intuitive
            // for cashiers typing the phone as they see it in WooCommerce (with spaces or symbols).
            $customer_meta_query[] = [
                'key'     => '_billing_phone',
                'value'   => $search,
                'compare' => 'LIKE',
            ];

            // Email: case-insensitive LIKE match.
            $customer_meta_query[] = [
                'key'     => '_billing_email',
                'value'   => $search,
                'compare' => 'LIKE',
            ];

            if ( ! empty( $meta_query ) ) {
                $meta_query = array_merge( [ 'relation' => 'AND' ], $meta_query, [ $customer_meta_query ] );
            } else {
                $meta_query = $customer_meta_query;
            }
        }

        if ( ! empty( $meta_query ) ) {
            $query_args['meta_query'] = $meta_query;
        }

        $results = wc_get_orders( $query_args );

        $orders = [];
        foreach ( $results->orders as $order ) {
            if ( ! $order instanceof WC_Order ) { continue; }
            $orders[] = self::format_order_for_pos_list( $order );
        }

        return new WP_REST_Response( [
            'page'       => (int) $page,
            'per_page'   => (int) $per_page,
            'total'      => (int) $results->total,
            'max_pages'  => (int) $results->max_num_pages,
            'orders'     => $orders,
        ], 200 );
    }

    protected static function format_order_for_pos_list( WC_Order $order ) {
        $order_id   = $order->get_id();
        $created    = $order->get_date_created();
        $date_iso   = $created ? $created->date( 'c' ) : '';

        $order_total         = (float) $order->get_total();
        $order_total_refund  = (float) $order->get_total_refunded();
        $decimals            = wc_get_price_decimals();
        $rounded_total       = wc_round_tax_total( $order_total, $decimals );
        $rounded_total_ref   = wc_round_tax_total( $order_total_refund, $decimals );
        $is_fully_refunded   = ( $rounded_total > 0 && $rounded_total_ref >= $rounded_total );

        $first_name = $order->get_billing_first_name();
        $last_name  = $order->get_billing_last_name();
        $name       = trim( $first_name . ' ' . $last_name );
        if ( $name === '' ) {
            $name = $order->get_formatted_billing_full_name();
        }

        $phone      = $order->get_billing_phone();
        $email      = $order->get_billing_email();
        $origin_raw = $order->get_meta( '_sirapix_pos_origin', true );

        if ( ! $origin_raw ) {
            $origin_raw = __( 'Online', 'sirapix-pos-for-woocommerce' );
        }

        return [
            'id'               => (int) $order_id,
            'order_no'         => $order->get_order_number(),
            'status'           => $order->get_status(),
            'date_created'     => $date_iso,
            'total'            => $order_total,
            'total_refunded'   => $order_total_refund,
            'is_fully_refunded'=> $is_fully_refunded,
            'origin'           => (string) $origin_raw,
            'customer_name'    => (string) $name,
            'customer_phone'   => (string) $phone,
            'customer_email'   => (string) $email,
        ];
    }

    public static function get_order_detail( WP_REST_Request $request ) {
        $id = absint( $request->get_param( 'id' ) );
        if ( ! $id ) {
            return new WP_REST_Response( [ 'error' => 'Invalid order ID' ], 400 );
        }

        $order = wc_get_order( $id );
        if ( ! $order instanceof WC_Order ) {
            return new WP_REST_Response( [ 'error' => 'Order not found' ], 404 );
        }

        $created  = $order->get_date_created();
        $date_iso = $created ? $created->date( 'c' ) : '';

        $order_total        = (float) $order->get_total();
        $order_total_refund  = (float) $order->get_total_refunded();
        $decimals            = wc_get_price_decimals();
        $rounded_total       = wc_round_tax_total( $order_total, $decimals );
        $rounded_total_ref   = wc_round_tax_total( $order_total_refund, $decimals );
        $is_fully_refunded   = ( $rounded_total > 0 && $rounded_total_ref >= $rounded_total );

        $items = [];
        foreach ( $order->get_items() as $item_id => $item ) {
            if ( ! $item instanceof WC_Order_Item_Product ) { continue; }

            $product      = $item->get_product();
            $product_id   = $product ? $product->get_id() : 0;
            $variation_id = $product && $product->is_type( 'variation' ) ? $product->get_id() : 0;

            $qty          = (float) $item->get_quantity();
            // WooCommerce stores refunded quantities as negative numbers; normalise to positive
            $qty_ref_raw  = (float) $order->get_qty_refunded_for_item( $item_id );
            $qty_refunded = abs( $qty_ref_raw );

            $line_total       = (float) $item->get_total();
            $line_refunded_raw = (float) $order->get_total_refunded_for_item( $item_id );
            $line_refunded    = abs( $line_refunded_raw );

            $items[] = [
                'order_item_id'       => (int) $item_id,
                'product_id'          => (int) $product_id,
                'variation_id'        => (int) $variation_id,
                'name'                => (string) $item->get_name(),
                'quantity'            => $qty,
                'quantity_refunded'   => $qty_refunded,
                'quantity_refundable' => max( 0, $qty - $qty_refunded ),
                'line_total'          => $line_total,
                'line_total_refunded' => $line_refunded,
            ];
        }

        $first_name = $order->get_billing_first_name();
        $last_name  = $order->get_billing_last_name();
        $name       = trim( $first_name . ' ' . $last_name );
        if ( $name === '' ) {
            $name = $order->get_formatted_billing_full_name();
        }

        $phone      = $order->get_billing_phone();
        $email      = $order->get_billing_email();
        $origin_raw = $order->get_meta( '_sirapix_pos_origin', true );
        if ( ! $origin_raw ) {
            $origin_raw = __( 'Online', 'sirapix-pos-for-woocommerce' );
        }

        return new WP_REST_Response( [
            'id'               => (int) $order->get_id(),
            'order_no'         => $order->get_order_number(),
            'status'           => $order->get_status(),
            'date_created'     => $date_iso,
            'total'            => $order_total,
            'total_refunded'   => $order_total_refund,
            'is_fully_refunded'=> $is_fully_refunded,
            'origin'           => (string) $origin_raw,
            'customer_name'    => (string) $name,
            'customer_phone'   => (string) $phone,
            'customer_email'   => (string) $email,
            'items'            => $items,
        ], 200 );
    }

    public static function post_order_refund( WP_REST_Request $request ) {
        $id            = absint( $request->get_param( 'id' ) );
        $items_input   = (array) $request->get_param( 'items' );
        $refund_amount = $request->get_param( 'refund_amount' );
        $reason        = sanitize_text_field( (string) $request->get_param( 'reason' ) );
        $restock       = (bool) $request->get_param( 'restock' );

        if ( ! $id ) {
            return new WP_REST_Response( [ 'error' => 'Invalid order ID' ], 400 );
        }

        $order = wc_get_order( $id );
        if ( ! $order instanceof WC_Order ) {
            return new WP_REST_Response( [ 'error' => 'Order not found' ], 404 );
        }

        $line_items = [];

        foreach ( $items_input as $row ) {
            $item_id = absint( $row['order_item_id'] ?? 0 );
            $qty     = floatval( $row['quantity'] ?? 0 );
            if ( ! $item_id || $qty <= 0 ) { continue; }

            $item = $order->get_item( $item_id );
            if ( ! $item instanceof WC_Order_Item_Product ) { continue; }

            $original_qty  = (float) $item->get_quantity();
            $already_ref   = (float) $order->get_qty_refunded_for_item( $item_id );
            $max_refundqty = max( 0, $original_qty - $already_ref );
            $qty           = min( $qty, $max_refundqty );
            if ( $qty <= 0 ) { continue; }

            $line_items[ $item_id ] = [
                'qty' => $qty,
            ];
        }

        if ( empty( $line_items ) && ! is_numeric( $refund_amount ) ) {
            return new WP_REST_Response( [ 'error' => __( 'No valid items to refund.', 'sirapix-pos-for-woocommerce' ) ], 400 );
        }

        $amount = null;
        if ( is_numeric( $refund_amount ) ) {
            $amount = max( 0, floatval( $refund_amount ) );
        }

        $refund_data = [
            'order_id'       => $order->get_id(),
            'amount'         => $amount,
            'line_items'     => $line_items,
            'reason'         => $reason,
            'refund_payment' => false,
            'restock_items'  => $restock,
        ];

        try {
            $refund = wc_create_refund( $refund_data );
            if ( is_wp_error( $refund ) ) {
                return new WP_REST_Response( [ 'error' => $refund->get_error_message() ], 400 );
            }

            if ( $refund instanceof WC_Order_Refund ) {
                $refund->update_meta_data( '_sirapix_pos_refund', 'yes' );

                if ( is_user_logged_in() ) {
                    $cashier_id = get_current_user_id();
                    $cashier    = wp_get_current_user();
                    $refund->update_meta_data( '_sirapix_pos_cashier_id', (int) $cashier_id );
                    if ( $cashier && $cashier instanceof WP_User ) {
                        $refund->update_meta_data( '_sirapix_pos_cashier_name', (string) $cashier->display_name );
                    }
                }

                $refund->save();

                $order_note = __( 'Refund created via POS', 'sirapix-pos-for-woocommerce' );
                if ( $reason !== '' ) {
                    $order_note .= ' - ' . $reason;
                }
                $order->add_order_note( $order_note );
            }

            $order_total        = (float) $order->get_total();
            $order_total_refund = (float) $order->get_total_refunded();
            if ( $order_total > 0 && $order_total_refund >= $order_total ) {
                $order->update_status( 'refunded' );
            }

            return new WP_REST_Response( [
                'order_id'        => (int) $order->get_id(),
                'refund_id'       => $refund instanceof WC_Order_Refund ? (int) $refund->get_id() : 0,
                'order_status'    => $order->get_status(),
                'total'           => (float) $order->get_total(),
                'total_refunded'  => (float) $order->get_total_refunded(),
            ], 200 );
        } catch ( Exception $e ) {
            return new WP_REST_Response( [ 'error' => $e->getMessage() ], 500 );
        }
    }

    public static function get_product_variations( WP_REST_Request $request ) {
        $id = absint( $request->get_param( 'id' ) );
        if ( ! $id ) { return new WP_REST_Response( [ 'error' => 'Invalid product ID' ], 400 ); }
        $product = wc_get_product( $id );
        if ( ! $product || ! $product->is_type( 'variable' ) ) {
            return new WP_REST_Response( [ 'error' => __( 'Product is not variable', 'sirapix-pos-for-woocommerce' ) ], 400 );
        }
        /** @var WC_Product_Variable $product */
        $children = $product->get_children();
        $list = [];
        foreach ( $children as $vid ) {
            $v = wc_get_product( $vid );
            if ( ! $v || ! $v->exists() ) { continue; }
            /** @var WC_Product_Variation $v */
            $atts = [];
            foreach ( $v->get_attributes() as $k => $val ) {
                $atts[ $k ] = is_array( $val ) ? implode( ',', $val ) : (string) $val;
            }
            $img_id = $v->get_image_id() ?: $product->get_image_id();
            $img = $img_id ? wp_get_attachment_image_url( $img_id, 'woocommerce_thumbnail' ) : wc_placeholder_img_src();
            $price = (float) wc_get_price_to_display( $v );
            $list[] = [
                'variation_id' => (int) $v->get_id(),
                'attributes'   => $atts,
                'price'        => $price,
                'price_html'   => wc_price( $price ),
                'stock_status' => $v->get_stock_status(),
                'stock_qty'    => $v->get_stock_quantity(),
                'image'        => $img,
            ];
        }
        return new WP_REST_Response( [ 'variations' => $list ], 200 );
    }

    public static function post_customer( WP_REST_Request $request ) {
        $name  = sanitize_text_field( (string) $request->get_param( 'name' ) );
        $phone = sanitize_text_field( (string) $request->get_param( 'phone' ) );
        $email = sanitize_email( (string) $request->get_param( 'email' ) );
        $uid = sirapix_wc_pos_find_or_create_customer( $name, $phone, $email );
        if ( is_wp_error( $uid ) ) {
            return new WP_REST_Response( [ 'error' => $uid->get_error_message() ], 400 );
        }
        $user = get_user_by( 'id', $uid );
        return new WP_REST_Response( [ 'customer_id' => (int) $uid, 'email' => $user ? $user->user_email : '' ], 200 );
    }

    public static function post_order( WP_REST_Request $request ) {
        $customer = (array) $request->get_param( 'customer' );
        $items    = (array) $request->get_param( 'items' );
        $payment  = sanitize_text_field( (string) $request->get_param( 'payment' ) );
        $notes    = sanitize_text_field( (string) $request->get_param( 'notes' ) );
        $origin_in = sanitize_text_field( (string) $request->get_param( 'origin' ) );
        $origin_in = in_array( $origin_in, [ 'in_store', 'phone' ], true ) ? $origin_in : 'in_store';
        $origin_label = $origin_in === 'phone' ? __( 'Phone', 'sirapix-pos-for-woocommerce' ) : __( 'In-Store Purchase', 'sirapix-pos-for-woocommerce' );
        $override_total = $request->get_param( 'override_total' );
        $override_total = is_null( $override_total ) ? null : floatval( $override_total );

        $name  = isset( $customer['name'] ) ? sanitize_text_field( (string) $customer['name'] ) : '';
        $phone = isset( $customer['phone'] ) ? sanitize_text_field( (string) $customer['phone'] ) : '';
        $email = isset( $customer['email'] ) ? sanitize_email( (string) $customer['email'] ) : '';

        $uid = sirapix_wc_pos_find_or_create_customer( $name, $phone, $email );
        if ( is_wp_error( $uid ) ) {
            return new WP_REST_Response( [ 'error' => $uid->get_error_message() ], 400 );
        }

        if ( empty( $items ) ) {
            return new WP_REST_Response( [ 'error' => __( 'No items in order.', 'sirapix-pos-for-woocommerce' ) ], 400 );
        }

        try {
            $order = wc_create_order();
            // Mark origin for admin visibility
            if ( method_exists( $order, 'set_created_via' ) ) {
                $order->set_created_via( 'sirapix_pos' );
            }
            $order->update_meta_data( '_sirapix_pos_origin', $origin_label );
            // WooCommerce Order Attribution (controls the "Origin" column)
            // Set a deterministic origin so WC doesn't show "Unknown".
            $order->update_meta_data( '_wc_order_attribution_source_type', 'utm' );
            $order->update_meta_data( '_wc_order_attribution_utm_source', $origin_label );
            $order->update_meta_data( '_wc_order_attribution_utm_medium', 'pos' );
            $order->save();
            $order->set_customer_id( $uid );

            if ( is_user_logged_in() ) {
                $cashier_id = get_current_user_id();
                $cashier    = wp_get_current_user();
                $order->update_meta_data( '_sirapix_pos_cashier_id', (int) $cashier_id );
                if ( $cashier && $cashier instanceof WP_User ) {
                    $order->update_meta_data( '_sirapix_pos_cashier_name', (string) $cashier->display_name );
                }
            }

            foreach ( $items as $row ) {
                $pid = absint( $row['product_id'] ?? 0 );
                $qty = max( 1, absint( $row['quantity'] ?? 0 ) );
                $variation_id = absint( $row['variation_id'] ?? 0 );
                $attributes = isset( $row['attributes'] ) && is_array( $row['attributes'] ) ? $row['attributes'] : [];
                if ( ! $qty ) { continue; }

                if ( $variation_id ) {
                    $variation = wc_get_product( $variation_id );
                    if ( ! $variation || ! $variation->is_type( 'variation' ) ) { continue; }
                    $parent_id = $variation->get_parent_id();
                    if ( $pid && $pid !== $parent_id ) { continue; }
                    $order->add_product( $variation, $qty );
                } else {
                    if ( ! $pid ) { continue; }
                    $product = wc_get_product( $pid );
                    if ( ! $product ) { continue; }
                    $order->add_product( $product, $qty );
                }
            }

            if ( $notes ) { $order->add_order_note( $notes ); }

            // Set billing basics
            $user = get_user_by( 'id', $uid );
            $order->set_billing_email( $user ? $user->user_email : '' );
            $order->set_billing_phone( sirapix_wc_pos_normalize_phone( $phone ) );

            // Shipping line: Offline Store (no cost)
            $item = new WC_Order_Item_Shipping();
            $item->set_method_title( __( 'Offline Store', 'sirapix-pos-for-woocommerce' ) );
            $item->set_method_id( 'sirapix_pos_offline_store' );
            $item->set_total( 0 );
            $order->add_item( $item );

            // Payment method label
            $order->set_payment_method( $payment ?: 'pos_offline' );
            $order->set_payment_method_title( __( 'Offline Payment', 'sirapix-pos-for-woocommerce' ) );

            // Initial totals
            $order->calculate_totals( true );

            // Apply override as a negative fee (discount) if provided and valid
            if ( is_float( $override_total ) || is_int( $override_total ) ) {
                $current_total = floatval( $order->get_total() );
                $target_total = max( 0, floatval( $override_total ) );
                if ( $target_total < $current_total ) {
                    $discount = $current_total - $target_total;
                    $fee = new WC_Order_Item_Fee();
                    $fee->set_name( __( 'POS Discount Override', 'sirapix-pos-for-woocommerce' ) );
                    $fee->set_total( 0 - $discount );
                    $order->add_item( $fee );
                    $order->calculate_totals( true );
                    /* translators: 1: original order total, 2: overridden order total. */
                    $order->add_order_note( sprintf( __( 'Total overridden from %1$s to %2$s', 'sirapix-pos-for-woocommerce' ), wc_price( $current_total ), wc_price( $order->get_total() ) ) );
                }
            }

            // Mark paid and completed
            $order->payment_complete();
            $order->update_status( 'completed', __( 'Completed via POS', 'sirapix-pos-for-woocommerce' ), true );

            return new WP_REST_Response( [
                'order_id'   => (int) $order->get_id(),
                'order_no'   => $order->get_order_number(),
                'status'     => $order->get_status(),
                'total_html' => $order->get_formatted_order_total(),
                'order_url'  => wc_get_endpoint_url( 'view-order', $order->get_id(), wc_get_page_permalink( 'myaccount' ) ),
            ], 200 );
        } catch ( Exception $e ) {
            return new WP_REST_Response( [ 'error' => $e->getMessage() ], 500 );
        }
    }
}

Sirapix_WC_POS_REST::init();
