(function(){
  const $ = window.jQuery;
  function ready(fn){ if(document.readyState!=='loading'){ fn(); } else { document.addEventListener('DOMContentLoaded', fn); } }

  ready(function(){
    const posRoot = document.querySelector('.sirapix-pos-wrapper');
    if(!posRoot){ return; }

    const I18N = (window.SIRAPIX_POS && SIRAPIX_POS.i18n) || {};

    // State
    let state = {
      categoryId: null,
      search: '',
      page: 1,
      loading: false,
      maxPages: 1,
      cart: {},
      static: {
        categories: null,
        products: null,
      }
    };

    // Static JSON loaders (products/categories) with mode toggle
    async function tryLoadStaticProducts(){
      // If products mode is forced to online, skip static JSON while online, but
      // still allow static JSON as an offline fallback when the network is down.
      const isOffline = (typeof navigator !== 'undefined' && navigator.onLine === false);
      if(!isOffline && window.SIRAPIX_POS && SIRAPIX_POS.products_mode === 'online'){ return false; }
      if(state.static.products && state.static.categories){ return true; }
      const base = (window.SIRAPIX_POS && SIRAPIX_POS.static_url) ? String(SIRAPIX_POS.static_url) : '';
      if(!base) return false;
      const fname = (window.SIRAPIX_POS && SIRAPIX_POS.static_products_file) ? String(SIRAPIX_POS.static_products_file) : 'products.json';
      try{
        const url = base.replace(/\/$/, '') + '/' + fname;
        const res = await fetch(url, { cache: 'no-cache' });
        if(!res.ok) throw new Error('HTTP '+res.status);
        const data = await res.json();
        if(!data || !Array.isArray(data.products)) throw new Error('invalid');
        state.static.products = data.products || [];
        state.static.categories = data.categories || [];
        return true;
      }catch(e){ return false; }
    }

    // No customers.json: we use REST-only for customer lookup by phone.

    // Cache for variations to reduce lag
    const variationsCache = new Map(); // productId -> variations array
    const variationsInflight = new Map(); // productId -> Promise
    function fetchVariations(productId){
      const id = Number(productId)||0;
      if(!id) return Promise.resolve([]);
      if(variationsCache.has(id)) return Promise.resolve(variationsCache.get(id));
      if(variationsInflight.has(id)) return variationsInflight.get(id);
      const p = api.get(`/sirapix-pos/v1/products/${id}/variations`).then(data=>{
        const vars = (data&&data.variations)||[];
        variationsCache.set(id, vars);
        variationsInflight.delete(id);
        return vars;
      }).catch(err=>{ variationsInflight.delete(id); throw err; });
      variationsInflight.set(id, p);
      return p;
    }

    const STORAGE_KEY = 'sirapix_pos_cart_v1';
    const QUEUE_KEY = 'sirapix_pos_order_queue_v1';
    const SETTINGS_CACHE_KEY = 'sirapix_pos_settings_snapshot_v1';

    async function loadSettingsSnapshot(){
      const url = (window.SIRAPIX_POS && SIRAPIX_POS.settings_snapshot_url) ? String(SIRAPIX_POS.settings_snapshot_url) : '';
      if(!url){ return null; }

      function applySettingsPayload(payload){
        if(!payload || typeof payload !== 'object') return;
        window.SIRAPIX_POS_SETTINGS = payload;
        const settings = payload && payload.settings ? payload.settings : {};
        try {
          const root = document.documentElement;
          if(settings.hasOwnProperty('sirapix_wc_pos_items_per_row')){
            let n = parseInt(settings.sirapix_wc_pos_items_per_row, 10);
            if(!n || isNaN(n)) n = 8;
            if(n < 2) n = 2;
            if(n > 10) n = 10;
            if(root){ root.style.setProperty('--spx-items-per-row', String(n)); }
            if(window.SIRAPIX_POS){ SIRAPIX_POS.items_per_row = n; }
          }
          if(settings.sirapix_wc_pos_products_mode){
            const mode = (settings.sirapix_wc_pos_products_mode === 'online') ? 'online' : 'offline';
            if(window.SIRAPIX_POS){ SIRAPIX_POS.products_mode = mode; }
          }
          if(settings.sirapix_wc_pos_primary_color){
            if(root){ root.style.setProperty('--spx-primary', String(settings.sirapix_wc_pos_primary_color)); }
          }
          if(settings.sirapix_wc_pos_secondary_color){
            if(root){ root.style.setProperty('--spx-secondary', String(settings.sirapix_wc_pos_secondary_color)); }
          }
        } catch(_){ }
      }

      try {
        const res = await fetch(url, { cache: 'no-store' });
        if(!res.ok) throw new Error('HTTP '+res.status);
        const data = await res.json();
        try { localStorage.setItem(SETTINGS_CACHE_KEY, JSON.stringify({ url, data })); } catch(e){}
        applySettingsPayload(data);
        return data;
      } catch(e){
        try {
          const raw = localStorage.getItem(SETTINGS_CACHE_KEY);
          if(!raw) return null;
          const saved = JSON.parse(raw||'null');
          if(saved && saved.data){
            applySettingsPayload(saved.data);
            return saved.data;
          }
        } catch(_){ }
        return null;
      }
    }

    // Kick off settings snapshot load (network-first with offline fallback)
    loadSettingsSnapshot();

    // Toast utility
    let toastContainer = document.querySelector('.spx-toast-container');
    if(!toastContainer){
      toastContainer = document.createElement('div');
      toastContainer.className = 'spx-toast-container';
      document.body.appendChild(toastContainer);
    }
    function showToast(message, { type='success', timeout=2500 }={}){
      try {
        const t = document.createElement('div');
        t.className = 'spx-toast ' + (type||'');
        const icon = (type==='success')
          ? "<span class=\"icon\" aria-hidden=\"true\">"+
            "<svg xmlns='http://www.w3.org/2000/svg' width='18' height='18' viewBox='0 0 24 24' fill='none' stroke='currentColor' stroke-width='2' stroke-linecap='round' stroke-linejoin='round' style='color:var(--spx-primary,#0ea5e9)'>"+
            "<path d='M22 11.08V12a10 10 0 1 1-5.93-9.14'/><polyline points='22 4 12 14.01 9 11.01'/></svg>"+
            "</span>"
          : (type==='warning')
          ? "<span class=\"icon\" aria-hidden=\"true\">"+
            "<svg xmlns='http://www.w3.org/2000/svg' width='18' height='18' viewBox='0 0 24 24' fill='none' stroke='currentColor' stroke-width='2' stroke-linecap='round' stroke-linejoin='round' style='color:#f59e0b'>"+
            "<circle cx='12' cy='12' r='10'/><line x1='12' y1='8' x2='12' y2='12'/><line x1='12' y1='16' x2='12.01' y2='16'/>"+
            "</svg>"+
            "</span>"
          : "<span class=\"icon\" aria-hidden=\"true\">"+
            "<svg xmlns='http://www.w3.org/2000/svg' width='18' height='18' viewBox='0 0 24 24' fill='none' stroke='currentColor' stroke-width='2' stroke-linecap='round' stroke-linejoin='round' style='color:#ef4444'>"+
            "<circle cx='12' cy='12' r='10'/><line x1='12' y1='8' x2='12' y2='12'/><line x1='12' y1='16' x2='12.01' y2='16'/>"+
            "</svg>"+
            "</span>";
        t.innerHTML = icon + '<span>'+ String(message||'') +'</span>';
        toastContainer.appendChild(t);
        setTimeout(()=>{ t.style.opacity = '0'; t.style.transform = 'translateY(6px)'; }, Math.max(500, timeout-200));
        setTimeout(()=>{ t.remove(); }, Math.max(800, timeout));
      } catch(e){}
    }

    // Lightweight modal utility for order completion
    let modalRoot = document.querySelector('.spx-modal-root');
    function ensureModalRoot(){
      if(!modalRoot){
        modalRoot = document.createElement('div');
        modalRoot.className = 'spx-modal-root';
        document.body.appendChild(modalRoot);
      }
    }
    function closeModal(){ if(modalRoot){ modalRoot.innerHTML = ''; modalRoot.style.display='none'; document.body.classList.remove('spx-modal-open'); } }
    function showOrderModal({ order_no, total_html }, receipt){
      ensureModalRoot();
      const items = (receipt?.items||[]).map(it=>`<li class="row"><span class="n">${it.name}</span><span class="q">×${it.qty}</span><span class="p">${Number(it.price||0).toFixed(2)}</span></li>`).join('');
      modalRoot.innerHTML = `
        <div class="spx-modal-overlay"></div>
        <div class="spx-modal">
          <div class="spx-modal-hd">
            <strong>${I18N.order_prefix || 'Order #'}${order_no}</strong>
            <button type="button" class="spx-modal-close" aria-label="${I18N.close || 'Close'}">×</button>
          </div>
          <div class="spx-modal-bd">
            <div class="muted">${I18N.completed_successfully || 'Completed successfully'}</div>
            <ul class="spx-list">${items}</ul>
            <div class="spx-total">${total_html||''}</div>
          </div>
          <div class="spx-modal-ft">
            <button type="button" class="sirapix-btn secondary spx-close">${I18N.close || 'Close'}</button>
            <button type="button" class="sirapix-btn primary spx-print">${I18N.print_receipt || 'Print Receipt'}</button>
          </div>
        </div>`;
      modalRoot.style.display='block';
      document.body.classList.add('spx-modal-open');
      modalRoot.querySelector('.spx-modal-close').addEventListener('click', closeModal);
      modalRoot.querySelector('.spx-close').addEventListener('click', closeModal);
      const printBtn = modalRoot.querySelector('.spx-print');
      if(printBtn){ printBtn.addEventListener('click', function(){ if(lastReceipt){ closeModal(); openPrintWindow(lastReceipt); } }); }
      modalRoot.querySelector('.spx-modal-overlay').addEventListener('click', closeModal);
    }

    async function runImagePrefetchIfRequested(){
      try {
        if(!window.SIRAPIX_POS || !window.localStorage) return;
        if(!SIRAPIX_POS.aggressive_images){
          try{ localStorage.removeItem('sirapix_pos_prefetch_images'); }catch(_e){}
          return;
        }
        const flag = localStorage.getItem('sirapix_pos_prefetch_images');
        if(flag !== '1') return;
        const isOffline = (typeof navigator !== 'undefined' && navigator.onLine === false);
        if(isOffline){
          showToast(I18N.image_prefetch_requires_online || 'Image prefetch requires online connection.', { type:'warning' });
          try{ localStorage.removeItem('sirapix_pos_prefetch_images'); }catch(_e){}
          return;
        }

        ensureModalRoot();
        modalRoot.style.display='block';
        document.body.classList.add('spx-modal-open');
        const totalLabel = I18N.image_prefetch_prep || 'Preparing image prefetch…';
        modalRoot.innerHTML = `
          <div class="spx-modal-overlay"></div>
          <div class="spx-modal spx-prefetch-modal">
            <div class="spx-modal-hd">
              <strong>${I18N.offline_images_prefetch || 'Offline Images Prefetch'}</strong>
            </div>
            <div class="spx-modal-bd">
              <div class="muted" id="spx-prefetch-status">${totalLabel}</div>
              <div class="spx-progress-wrap">
                <div class="spx-progress-bar"><div class="spx-progress-inner" id="spx-prefetch-bar"></div></div>
                <div class="spx-progress-text" id="spx-prefetch-text">0%</div>
              </div>
            </div>
            <div class="spx-modal-ft">
              <button type="button" class="sirapix-btn secondary spx-close-prefetch">${I18N.close || 'Close'}</button>
            </div>
          </div>`;

        const closeBtn = modalRoot.querySelector('.spx-close-prefetch');
        if(closeBtn){ closeBtn.addEventListener('click', function(){ closeModal(); }); }
        const overlay = modalRoot.querySelector('.spx-modal-overlay');
        if(overlay){ overlay.addEventListener('click', closeModal); }
        const bar = document.getElementById('spx-prefetch-bar');
        const text = document.getElementById('spx-prefetch-text');
        const statusEl = document.getElementById('spx-prefetch-status');

        function setProgress(done, total){
          const pct = total > 0 ? Math.round((done/total)*100) : 0;
          if(bar){ bar.style.width = pct + '%'; }
          if(text){ text.textContent = pct + '%'; }
          if(statusEl){ statusEl.textContent = `${I18N.prefetching_images || 'Prefetching images…'} ${done}/${total}`; }
        }

        if(statusEl){ statusEl.textContent = I18N.image_prefetch_loading_products || 'Loading product list…'; }
        const hasStatic = await tryLoadStaticProducts();
        if(!hasStatic || !Array.isArray(state.static.products)){
          if(statusEl){ statusEl.textContent = I18N.image_prefetch_products_missing || 'Could not load static products JSON. Make sure offline products are generated.'; }
          showToast(I18N.image_prefetch_failed || 'Image prefetch failed.', { type:'error' });
          try{ localStorage.removeItem('sirapix_pos_prefetch_images'); }catch(_e){}
          return;
        }

        const urls = new Set();
        (state.static.products||[]).forEach(p=>{
          if(p && typeof p === 'object'){
            if(p.image){ urls.add(String(p.image)); }
            if(Array.isArray(p.variations)){
              p.variations.forEach(v=>{ if(v && v.image){ urls.add(String(v.image)); } });
            }
          }
        });

        const list = Array.from(urls.values());
        const total = list.length;
        if(!total){
          if(statusEl){ statusEl.textContent = I18N.image_prefetch_no_images || 'No product images found to prefetch.'; }
          showToast(I18N.image_prefetch_no_images || 'No product images found to prefetch.', { type:'warning' });
          try{ localStorage.removeItem('sirapix_pos_prefetch_images'); }catch(_e){}
          return;
        }

        if(statusEl){ statusEl.textContent = `Prefetching images… 0/${total}`; }
        setProgress(0, total);

        let done = 0;
        const concurrency = 4;
        async function worker(startIndex){
          for(let i=startIndex; i<total; i+=concurrency){
            const url = list[i];
            try{
              await fetch(url, { cache:'no-cache', mode:'no-cors' });
            }catch(_e){}
            done++;
            setProgress(done, total);
          }
        }

        const workers = [];
        for(let i=0;i<concurrency;i++){
          workers.push(worker(i));
        }
        await Promise.all(workers);

        try{ localStorage.removeItem('sirapix_pos_prefetch_images'); }catch(_e){}
        if(statusEl){ statusEl.textContent = I18N.image_prefetch_completed || 'Image prefetch completed.'; }
        setProgress(total, total);
        showToast(I18N.image_prefetch_completed_toast || 'Image prefetch completed. Product images should now be available offline.', { type:'success' });
      } catch(e){
        try{ localStorage.removeItem('sirapix_pos_prefetch_images'); }catch(_e){}
        showToast(I18N.image_prefetch_failed || 'Image prefetch failed.', { type:'error' });
      }
    }

    // UI elements
    const el = {
      categories: document.getElementById('spx-categories'),
      categoriesSelectWrap: null,
      categoriesSelect: null,
      products: document.getElementById('spx-products'),
      sentinel: document.getElementById('spx-products-sentinel'),
      search: document.getElementById('spx-search'),
      searchSuggest: null,
      cartItems: document.getElementById('spx-cart-items'),
      cartBox: document.querySelector('.sirapix-cart'),
      subtotal: document.getElementById('spx-subtotal'),
      tax: document.getElementById('spx-tax'),
      total: document.getElementById('spx-total'),
      titleTotal: document.getElementById('spx-title-total'),
      clear: document.getElementById('spx-clear'),
      clearTop: document.getElementById('spx-clear-top'),
      complete: document.getElementById('spx-complete'),
      completeTop: document.getElementById('spx-complete-top'),
      print: document.getElementById('spx-print'),
      override: document.getElementById('spx-override-total'),
      note: document.getElementById('spx-order-note'),
      origin: document.querySelectorAll('input[name="spx-origin"]'),
    };

    runImagePrefetchIfRequested();

    // Online/Offline indicator dot + text
    (function(){
      const netDot = document.getElementById('spx-net-dot');
      const netText = document.getElementById('spx-net-text');
      function updateNet(){
        if(!netDot) return;
        const on = (typeof navigator !== 'undefined') ? !!navigator.onLine : true;
        netDot.title = on ? (I18N.status_online || 'Status: Online') : (I18N.status_offline || 'Status: Offline');
        if(on){ netDot.classList.remove('spx-net-offline'); }
        else { netDot.classList.add('spx-net-offline'); }
        if(netText){ netText.textContent = on ? (I18N.status_online || 'Status: Online') : (I18N.status_offline || 'Status: Offline'); }
      }
      updateNet();
      window.addEventListener('online', function(){ updateNet(); });
      window.addEventListener('offline', function(){ updateNet(); });
    })();

    // Ensure buttons are not implicit submitters
    if(el.complete){ el.complete.setAttribute('type','button'); }
    if(el.completeTop){ el.completeTop.setAttribute('type','button'); }

    const refundBtn = document.getElementById('spx-refund-btn');
    if(refundBtn){
      refundBtn.setAttribute('type','button');
      refundBtn.addEventListener('click', function(){ openRefundsDrawer(); });
    }

    // Extra hardening: block rapid double activation on mousedown/dblclick
    function guardClick(e){ if(submittingOrder){ e.preventDefault(); e.stopPropagation(); return false; } }
    if(el.complete){
      el.complete.addEventListener('mousedown', guardClick, { capture:true });
      el.complete.addEventListener('dblclick', function(e){ e.preventDefault(); e.stopPropagation(); }, { capture:true });
    }
    if(el.completeTop){
      el.completeTop.addEventListener('mousedown', guardClick, { capture:true });
      el.completeTop.addEventListener('dblclick', function(e){ e.preventDefault(); e.stopPropagation(); }, { capture:true });
    }

    el.searchSuggest = document.createElement('div');
    el.searchSuggest.className = 'sirapix-search-suggest';
    const searchPanel = el.search.closest('.sirapix-panel') || el.search.parentNode;
    searchPanel.appendChild(el.searchSuggest);
    // Clear (X) button for search — attach to the title container for absolute positioning
    const searchTitle = el.search.closest('.sirapix-panel-title.with-search') || searchPanel;
    const searchClear = document.createElement('button');
    searchClear.type = 'button';
    searchClear.className = 'sirapix-search-clear';
    searchClear.textContent = '×';
    searchTitle.appendChild(searchClear);

    let cartBadge = document.getElementById('spx-cart-count');

    // Helpers
    // Remove error class when user fixes the field
    (function(){
      const nameEl = document.getElementById('spx-customer-name');
      const phoneEl = document.getElementById('spx-customer-phone');
      const emailEl = document.getElementById('spx-customer-email');
      function rm(el){ if(el){ el.classList.remove('spx-field-error'); } }
      // Track whether fields were autofilled; if user types, we won't overwrite on next lookup
      let autofilledName = false;
      let autofilledEmail = false;
      if(nameEl){ nameEl.addEventListener('input', function(){ if((nameEl.value||'').trim().length){ rm(nameEl); autofilledName = false; } }); }
      if(phoneEl){ phoneEl.addEventListener('input', function(){ const v=(phoneEl.value||'').replace(/[^0-9]/g,''); if(v.length){ rm(phoneEl); } }); }
      if(emailEl){ emailEl.addEventListener('input', function(){ if((emailEl.value||'').trim().length){ rm(emailEl); autofilledEmail = false; } }); }

      // Auto-fill by phone lookup
      let lookupTimer = null; let lastLookupVal = '';
      let matchedPhone = '';
      async function tryLookup(){
        if(!phoneEl) return;
        const raw = (phoneEl.value||'').toString();
        const digits = raw.replace(/[^0-9]/g,'');
        if(digits.length < 6){ return; }
        if(digits === lastLookupVal){
          // If phone unchanged, nothing to do
        } else {
          lastLookupVal = digits; // new phone triggers lookup
        }
        try {
          const res = await api.get('/sirapix-pos/v1/customers/lookup', { phone: digits });
          if(res && res.found){
            if(nameEl && ((autofilledName || !nameEl.value || nameEl.value.trim()===''))){ nameEl.value = res.name||''; autofilledName = true; }
            if(emailEl && ((autofilledEmail || !emailEl.value || emailEl.value.trim()===''))){ emailEl.value = res.email||''; autofilledEmail = true; }
            matchedPhone = digits;
          } else {
            matchedPhone = '';
          }
        } catch(e){}
      }
      if(phoneEl){
        phoneEl.addEventListener('input', function(){
          // If user changes phone away from the last matched phone, immediately reset customer fields
          const digits = (phoneEl.value||'').replace(/[^0-9]/g,'');
          if(matchedPhone && digits !== matchedPhone){
            if(nameEl){ nameEl.value = ''; }
            if(emailEl){ emailEl.value = ''; }
            autofilledName = false; autofilledEmail = false;
            matchedPhone = '';
          }
          if(lookupTimer) clearTimeout(lookupTimer);
          lookupTimer = setTimeout(tryLookup, 250);
        });
        phoneEl.addEventListener('blur', function(){ tryLookup(); });
      }
    })();

    function saveCart(){
      try { localStorage.setItem(STORAGE_KEY, JSON.stringify(state.cart||{})); } catch(e){}
    }
    function loadCart(){
      try {
        const raw = localStorage.getItem(STORAGE_KEY);
        if(!raw) return;
        const data = JSON.parse(raw||'{}')||{};
        const restored = {};
        Object.keys(data).forEach(k=>{
          const r = data[k];
          if(r && typeof r === 'object' && r.prod){ restored[k] = { prod: r.prod, qty: Number(r.qty||0) }; }
        });
        state.cart = restored;
      } catch(e){}
    }
    function formatMoney(val){
      const cfg = (window.SIRAPIX_POS && SIRAPIX_POS.currency) || {};
      const decs = (typeof cfg.decimals === 'number') ? cfg.decimals : 2;
      const d = cfg.decimal || '.';
      const t = cfg.thousand || ',';
      const sym = cfg.symbol || '';
      const pos = cfg.position || 'left';
      const n = Number(val||0);
      const parts = n.toFixed(decs).split('.');
      parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, t);
      const num = parts.join(d);
      if(pos === 'left'){ return sym + num; }
      if(pos === 'left_space'){ return sym + ' ' + num; }
      if(pos === 'right'){ return num + sym; }
      if(pos === 'right_space'){ return num + ' ' + sym; }
      return sym + num;
    }
    const api = {
      async get(url, params, fetchOptions){
        const q = new URLSearchParams(params||{}).toString();
        const full = (SIRAPIX_POS.rest_url.replace(/\/$/, '') + url + (q? ('?' + q):''));
        let res;
        try {
          res = await fetch(full, Object.assign({ headers: { 'X-WP-Nonce': SIRAPIX_POS.nonce } }, fetchOptions||{}));
        } catch(err){
          const e = new Error('Network fetch failed'); e._net = true; throw e;
        }
        if(!res.ok){
          // Return server error text if available
          let data = {};
          try { data = await res.json(); } catch(_){}
          const e = new Error((data&&data.error)||('HTTP '+res.status)); e._http = res.status; throw e;
        }
        try { return await res.json(); } catch(_) { return {}; }
      },
      async post(url, body){
        const full = (SIRAPIX_POS.rest_url.replace(/\/$/, '') + url);
        let res;
        try {
          res = await fetch(full, {
            method: 'POST',
            headers: { 'X-WP-Nonce': SIRAPIX_POS.nonce, 'Content-Type': 'application/json' },
            body: JSON.stringify(body||{})
          });
        } catch(err){ const e = new Error('Network fetch failed'); e._net = true; throw e; }
        let data = {};
        try { data = await res.json(); } catch(_){}
        if(!res.ok){ const e = new Error((data&&data.error)||('HTTP '+res.status)); e._http = res.status; throw e; }
        return data;
      }
    };

    // Apply items-per-row setting to CSS variable for the grid
    (function(){
      try{
        const root = document.documentElement;
        if(root && window.SIRAPIX_POS && typeof SIRAPIX_POS.items_per_row !== 'undefined'){
          let n = parseInt(SIRAPIX_POS.items_per_row, 10);
          if(!n || isNaN(n)) n = 8;
          if(n < 2) n = 2;
          if(n > 10) n = 10;
          root.style.setProperty('--spx-items-per-row', String(n));
        }
      }catch(e){}
    })();

    // Offline queue utils
    function readQueue(){
      try { return JSON.parse(localStorage.getItem(QUEUE_KEY)||'[]')||[]; } catch(_){ return []; }
    }
    function writeQueue(list){
      try { localStorage.setItem(QUEUE_KEY, JSON.stringify(list||[])); } catch(_){ }
    }
    function enqueueOrder(payload){
      const list = readQueue();
      const entry = { id: Date.now()+'_'+Math.random().toString(36).slice(2), created_at: Date.now(), payload };
      list.push(entry);
      writeQueue(list);
      updateQueueCount();
      return entry.id;
    }
    async function processQueue(){
      if(typeof navigator !== 'undefined' && navigator.onLine === false) return;
      let list = readQueue();
      if(!list.length) return;
      // Process FIFO
      const copy = [...list];
      for(let i=0;i<copy.length;i++){
        const it = copy[i];
        try {
          const resp = await api.post('/sirapix-pos/v1/orders', it.payload);
          // remove from queue
          list = readQueue().filter(x=>x.id!==it.id);
          writeQueue(list);
          updateQueueCount();
          const num = (resp.order_no||resp.order_id||'');
          const text = I18N.toast_queued_order_synced
            ? I18N.toast_queued_order_synced.replace('%s', num)
            : ('Queued order synced #'+ num);
          showToast(text, { type:'success' });
        } catch(err){
        // If still network error, stop; if server error, drop and notify
        if(err && err._net){ break; }
        const msg = (err && err.message) ? err.message : 'Error';
        const text = I18N.toast_queued_order_failed
          ? (I18N.toast_queued_order_failed.replace('%s', msg))
          : ('Queued order failed: '+ msg);
        showToast(text, { type:'error' });
          // drop this entry to avoid blocking
          list = readQueue().filter(x=>x.id!==it.id);
          writeQueue(list);
          updateQueueCount();
        }
      }
    }
    window.addEventListener('online', ()=>{
      Promise.resolve().then(()=> processQueue()).finally(()=>{ updateQueueCount(); });
    });
    window.addEventListener('offline', ()=>{ updateQueueCount(); });

    // Queue UI helpers
    function updateQueueCount(){
      try {
        const elc = document.getElementById('spx-queue-count');
        const btn = document.getElementById('spx-sync-now');
        const list = readQueue()||[];
        const len = list.length;
        if(elc){ elc.textContent = String(len); }
        if(btn){
          const online = (typeof navigator !== 'undefined') ? !!navigator.onLine : true;
          const shouldEnable = online && len > 0;
          btn.disabled = !shouldEnable;
        }
      } catch(_){}
    }
    function bindQueueUI(){
      const btn = document.getElementById('spx-sync-now');
      if(btn){
        btn.addEventListener('click', function(){
          if(btn.classList.contains('is-syncing')){ return; }
          btn.classList.add('is-syncing');
          btn.disabled = true;
          Promise.resolve().then(()=> processQueue()).catch(()=>{})
            .finally(()=>{
              updateQueueCount();
              btn.classList.remove('is-syncing');
            });
        });
      }
      updateQueueCount();
    }

    // Placeholder renderers
    function renderCategories(items){
      el.categories.innerHTML = (items||[]).map(c => `<div class="cat" data-id="${c.id}">${c.name}</div>`).join('');
      // Build/select mobile dropdown mirror (created once)
      if(!el.categoriesSelectWrap){
        el.categoriesSelectWrap = document.createElement('div');
        el.categoriesSelectWrap.className = 'sirapix-categories-select-wrap';
        el.categoriesSelect = document.createElement('select');
        el.categoriesSelect.id = 'spx-categories-select';
        el.categoriesSelectWrap.appendChild(el.categoriesSelect);
        // Insert before the grid
        el.categories.parentNode.insertBefore(el.categoriesSelectWrap, el.categories);
        // On change, simulate category switch using existing logic
        el.categoriesSelect.addEventListener('change', function(){
          const id = Number(el.categoriesSelect.value)||null;
          const target = el.categories.querySelector(`.cat[data-id="${id}"]`);
          if(target){
            [...el.categories.querySelectorAll('.cat')].forEach(x=>x.classList.remove('active'));
            target.classList.add('active');
          }
          state.categoryId = id;
          state.page = 1; el.products.innerHTML='';
          loadProducts({ reset:true });
        });
      }
      // Populate select options and sync selected
      if(el.categoriesSelect){
        const opts = (items||[]).map(c => `<option value="${c.id}">${c.name}</option>`).join('');
        el.categoriesSelect.innerHTML = opts;
      }
    }
    function renderProducts(items){
      const html = (items||[]).map(p => {
        const fullName = p.name||'';
        const nameAttr = String(fullName).replace(/"/g, '&quot;');
        const shortName = fullName.length > 20 ? (fullName.slice(0,20) + '...') : fullName;
        const type = (p.type ? p.type : ((p.has_price_range || (p.attributes_summary&&p.attributes_summary.length)) ? 'variable' : 'simple'));
        const displayPrice = (type==='variable' && p.has_price_range) ? (formatMoney(p.price_from) + ' - ' + formatMoney(p.price_to)) : (p.price_html||p.price||'');
        return `
        <div class="prod" data-id="${p.id}" data-type="${type}" data-price="${Number(p.price||0)}" data-name="${nameAttr}">
          <img src="${p.image||''}" alt="">
          <div class="meta">
            <span title="${fullName}">${shortName}</span>
            <strong>${displayPrice}</strong>
          </div>
        </div>
      `;}).join('');
      el.products.insertAdjacentHTML('beforeend', html);
    }

    function addToCart(prod){
      if(!state.cart[prod.id]){ state.cart[prod.id] = { prod, qty: 0 }; }
      state.cart[prod.id].qty += 1;
      renderCart();
    }
    function addToCartVariant(parentId, baseName, variation){
      const attrs = variation.attributes||{};
      const attrParts = Object.keys(attrs).map(k=>{
        const label = k.replace(/^attribute_/,'').replace(/^pa_/,'').replace(/_/g,' ');
        return label.charAt(0).toUpperCase()+label.slice(1)+': '+attrs[k];
      });
      const name = baseName + (attrParts.length? (' — ' + attrParts.join(', ')) : '');
      const prod = {
        id: Number(variation.variation_id),
        product_id: Number(parentId),
        variation_id: Number(variation.variation_id),
        attributes: attrs,
        name,
        price: Number(variation.price||0),
        image: variation.image||''
      };
      addToCart(prod);
    }

    function totalQty(){
      return Object.values(state.cart).reduce((s,r)=> s + Number(r.qty||0), 0);
    }

    function bumpCart(){
      // Disabled bounce animation
      return;
    }

    function updateCartBadge(){
      if(!cartBadge) return;
      const q = totalQty();
      if(q > 0){ cartBadge.textContent = q; cartBadge.style.display = 'inline-flex'; }
      else { cartBadge.style.display = 'none'; }
    }

    function renderCart(){
      const rows = Object.values(state.cart);
      let subtotal = 0;
      const html = rows.map(r => {
        const price = Number(r.prod.price||0);
        subtotal += price * r.qty;
        return `
          <div class="sirapix-cart-item" data-id="${r.prod.id}">
            <div class="left">
              ${r.prod.image ? `<img class=\"thumb\" src=\"${r.prod.image}\" alt=\"\">` : ''}
              <div class="name">${r.prod.name}</div>
            </div>
            <div class="sirapix-qty">
              <button class="qty-dec">-</button>
              <span>${r.qty}</span>
              <button class="qty-inc">+</button>
            </div>
          </div>
        `;
      }).join('');
      el.cartItems.innerHTML = html || `<div>${I18N.no_items || 'No items'}</div>`;
      el.subtotal.textContent = formatMoney(subtotal);
      el.tax.textContent = formatMoney(0);
      el.total.textContent = formatMoney(subtotal);
      if(el.titleTotal){ el.titleTotal.textContent = formatMoney(subtotal); }
      updateCartBadge();
      saveCart();
    }

    function showDrawer(contentHtml){
      ensureModalRoot();
      modalRoot.innerHTML = `
        <div class="spx-modal-overlay"></div>
        <div class="spx-drawer" style="position:fixed;right:0;top:0;height:100%;width:65vw;min-width:480px;max-width:1100px;background:#fff;box-shadow:-8px 0 24px rgba(17,24,39,.14);z-index:2147483646;display:flex;flex-direction:column;">
          ${contentHtml}
        </div>`;
      modalRoot.style.display='block';
      document.body.classList.add('spx-modal-open');
      modalRoot.querySelector('.spx-modal-overlay').addEventListener('click', closeModal);
    }

    function ensureRefundStyles(){
      try {
        if(document.getElementById('spx-refund-inline-styles')) return;
        const style = document.createElement('style');
        style.id = 'spx-refund-inline-styles';
        style.textContent = `
          .spx-ref-row{transition:background-color .12s ease,border-color .12s ease;}
          .spx-ref-row:hover{background-color:rgba(14,165,233,0.04);}
          .spx-ref-row.is-active{border-color:var(--spx-primary,#0ea5e9);background-color:rgba(14,165,233,0.08);}
        `;
        document.head.appendChild(style);
      } catch(_e){}
    }

    const refundState = {
      page: 1,
      maxPages: 1,
      search: '',
      origin: '',
      currentOrderId: null,
      currentOrder: null,
    };

    async function loadRefundOrdersInto(elRoot){
      if(!elRoot) return;
      const listEl = elRoot.querySelector('#spx-ref-orders');
      const pageInfo = elRoot.querySelector('#spx-ref-page-info');
      if(listEl){ listEl.innerHTML = '<div class="spx-loading" style="padding:12px;color:#6B7280;">Loading orders…</div>'; }
      try {
        const params = {
          page: refundState.page,
          per_page: 30,
        };
        if(refundState.search){ params.search = refundState.search; }
        if(refundState.origin){ params.origin = refundState.origin; }

        const data = await api.get('/sirapix-pos/v1/orders', params);
        refundState.maxPages = data.max_pages || data.maxPages || 1;
        const orders = data.orders || [];
        if(listEl){
          if(!orders.length){
            listEl.innerHTML = '<div style="padding:12px;color:#6B7280;">No orders found.</div>';
          } else {
            const rows = orders.map(o=>{
              const created = o.date_created ? new Date(o.date_created).toLocaleString() : '';
              const origin = o.origin || '';
              const totalNum = Number(o.total||0);
              const refundedNum = Number(o.total_refunded||0);
              const hasRefund = refundedNum > 0;
              const isFullyRefunded = !!(
                o.is_fully_refunded ||
                (String(o.status||'').toLowerCase()==='refunded') ||
                (totalNum > 0 && refundedNum >= totalNum)
              );

              let statusLabel = '';
              if(hasRefund){
                statusLabel = isFullyRefunded ? 'Fully refunded' : 'Partially refunded';
              }

              const refundedBadge = isFullyRefunded
                ? '<span style="display:inline-flex;align-items:center;padding:2px 6px;border-radius:999px;background:#ECFDF3;color:#16A34A;font-size:10px;font-weight:500;">Refunded</span>'
                : '';

              const titleText = statusLabel
                ? `#${o.order_no||o.id} - ${statusLabel}`
                : `#${o.order_no||o.id}`;

              const customer = (o.customer_name||'') || (o.customer_email||'');
              const total = formatMoney(o.total||0);
              return `<button type="button" class="spx-ref-row" data-id="${o.id}" style="width:100%;text-align:left;border:0;border-bottom:1px solid #E5E7EB;background:#fff;padding:8px 10px;cursor:pointer;">
                <div style="display:flex;align-items:flex-start;justify-content:space-between;gap:8px;">
                  <div style="flex:1 1 auto;min-width:0;">
                    <div style="display:flex;align-items:center;gap:6px;margin-bottom:2px;">
                      <span style="font-weight:600;font-size:12px;">${titleText}</span>
                      ${refundedBadge}
                    </div>
                    <div style="font-size:12px;color:#4B5563;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${customer||'&nbsp;'}</div>
                    <div style="font-size:11px;color:#6B7280;margin-top:2px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${created||''}</div>
                  </div>
                  <div style="flex:0 0 auto;text-align:right;">
                    <div style="font-weight:600;font-size:12px;">${total}</div>
                    <div style="font-size:11px;color:#6B7280;margin-top:2px;">${origin||''}</div>
                  </div>
                </div>
              </button>`;
            }).join('');
            listEl.innerHTML = rows;
          }
        }
        if(pageInfo){ pageInfo.textContent = `Page ${refundState.page} of ${refundState.maxPages}`; }
      } catch(e){
        const msg = (e && e.message) ? e.message : 'Failed to load orders.';
        if(listEl){ listEl.innerHTML = `<div style="padding:12px;color:#B91C1C;">${msg}</div>`; }
        console.error('Failed to load refund orders', e);
      }
    }

    async function loadRefundOrderDetailInto(elRoot, orderId){
      if(!elRoot || !orderId) return;
      const detailEl = elRoot.querySelector('#spx-ref-detail-overlay');
      if(!detailEl) return;
      detailEl.style.display = 'flex';
      detailEl.innerHTML = '<div class="spx-loading" style="padding:12px;color:#6B7280;">Loading order…</div>';
      try {
        const data = await api.get(`/sirapix-pos/v1/orders/${orderId}`);
        refundState.currentOrderId = data.id;
        refundState.currentOrder = data;
        if(detailEl){
          const created = data.date_created ? new Date(data.date_created).toLocaleString() : '';
          const itemsArr = Array.isArray(data.items) ? data.items : [];
          // Normalise refunded/remaining quantities per line defensively in case backend sends negatives
          const allItemsZero = itemsArr.length > 0 && itemsArr.every(function(it){
            const origQty = Number(it.quantity||0);
            const alreadyRef = Math.abs(Number(it.quantity_refunded||0));
            const remaining = Math.max(0, origQty - alreadyRef);
            return remaining <= 0;
          });
          const isFullyRefunded = !!(
            data.is_fully_refunded ||
            (String(data.status||'').toLowerCase()==='refunded') ||
            (Number(data.total_refunded||0) >= Number(data.total||0) && Number(data.total||0) > 0) ||
            allItemsZero
          );
          const header = `
            <div style="display:flex;align-items:center;gap:10px;padding:8px 10px 10px 10px;border-bottom:1px solid #E5E7EB;">
              <button type="button" class="sirapix-btn secondary spx-ref-back" style="flex:0 0 auto;min-width:auto;padding:6px 10px;font-size:12px;display:inline-flex;align-items:center;gap:6px;">
                <span style="display:inline-block;transform:translateY(0.5px);">&#8592;</span>
                <span>Back</span>
              </button>
              <div style="flex:1 1 auto;min-width:0;display:flex;flex-direction:column;gap:2px;">
                <div style="font-weight:600;font-size:13px;">Order #${data.order_no||data.id}</div>
                <div style="font-size:12px;color:#4B5563;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${data.customer_name || data.customer_email || ''}</div>
                <div style="font-size:11px;color:#6B7280;">${created||''}</div>
              </div>
              <div style="flex:0 0 auto;text-align:right;font-size:12px;">
                <div style="font-weight:600;">${formatMoney(data.total||0)}</div>
                <div style="margin-top:2px;color:#6B7280;">Refunded ${formatMoney(data.total_refunded||0)}</div>
              </div>
            </div>`;

          const items = (data.items||[]).map(it=>{
            const origQty = Number(it.quantity||0);
            const alreadyRef = Math.abs(Number(it.quantity_refunded||0));
            const remaining = Math.max(0, origQty - alreadyRef);
            const maxQ = remaining;
            const lineTotal = formatMoney(it.line_total||0);
            const disabledAttr = maxQ <= 0 ? ' disabled' : '';
            const helperText = maxQ <= 0
              ? `Qty: ${origQty} · Refunded: ${alreadyRef} · Remaining: 0`
              : `Qty: ${origQty} · Refunded: ${alreadyRef} · Remaining: ${maxQ}`;
            return `<div class="spx-ref-item" data-item-id="${it.order_item_id}" style="border:1px solid #E5E7EB;border-radius:8px;padding:8px 10px;margin-bottom:6px;">
              <div style="font-weight:500;font-size:12px;margin-bottom:4px;">${it.name}</div>
              <div style="font-size:11px;color:#6B7280;margin-bottom:6px;">${helperText}</div>
              <div style="display:flex;align-items:center;justify-content:space-between;gap:8px;font-size:12px;">
                <label style="display:flex;align-items:center;gap:6px;">Refund qty:
                  <input type="number" min="0" max="${maxQ}" step="1" class="spx-ref-qty" data-item-id="${it.order_item_id}" value="0" style="width:64px;padding:4px 6px;border-radius:6px;border:1px solid #D1D5DB;"${disabledAttr} />
                </label>
                <span style="color:#111827;">Line: ${lineTotal}</span>
              </div>
            </div>`;
          }).join('');

          const controls = `
            <div class="spx-ref-controls" style="margin-top:10px;">
              <label style="display:block;font-size:12px;color:#4B5563;margin-bottom:4px;">Reason</label>
              <textarea id="spx-ref-reason" rows="2" style="width:100%;resize:vertical;border-radius:8px;border:1px solid #D1D5DB;padding:6px 8px;font-size:12px;"></textarea>
              <label style="display:flex;align-items:center;gap:6px;margin:8px 0 10px 0;font-size:12px;color:#4B5563;"><input type="checkbox" id="spx-ref-restock" /> <span>Restock items</span></label>
              <div style="display:flex;gap:8px;justify-content:flex-end;">
                <button type="button" id="spx-ref-full" class="sirapix-btn secondary">Full refund</button>
                <button type="button" id="spx-ref-partial" class="sirapix-btn primary">Apply refund</button>
              </div>
            </div>`;

          detailEl.innerHTML = `
            <div style="display:flex;flex-direction:column;flex:1 1 auto;overflow:auto;">
              ${header}
              <div style="padding:10px;flex:1 1 auto;overflow:auto;">
                <div class="spx-ref-items">${items || '<div style="padding:8px;font-size:12px;color:#6B7280;">No refundable items.</div>'}</div>
                ${controls}
              </div>
            </div>`;

          // Disable controls if the order is fully refunded OR there are no remaining refundable quantities
          (function(){
            const qtyInputs = detailEl.querySelectorAll('.spx-ref-qty');
            let allZero = true;
            qtyInputs.forEach(function(input){
              const maxQ = Number(input.getAttribute('max')||'0');
              if(maxQ > 0){ allZero = false; }
              if(maxQ <= 0){ input.disabled = true; }
            });
            if(isFullyRefunded || allZero){
              const btnFull = detailEl.querySelector('#spx-ref-full');
              const btnPartial = detailEl.querySelector('#spx-ref-partial');
              if(btnFull){ btnFull.disabled = true; }
              if(btnPartial){ btnPartial.disabled = true; }
              qtyInputs.forEach(function(input){ input.disabled = true; });
            }
          })();
        }
      } catch(e){
        if(detailEl){ detailEl.innerHTML = '<div style="padding:12px;color:#B91C1C;">Failed to load order details.</div>'; }
      }
    }

    async function submitRefundFrom(elRoot, mode){
      if(!elRoot || !refundState.currentOrderId){ return; }
      const orderId = refundState.currentOrderId;
      const detailEl = elRoot.querySelector('#spx-ref-detail-overlay');
      const reasonEl = detailEl ? detailEl.querySelector('#spx-ref-reason') : null;
      const restockEl = detailEl ? detailEl.querySelector('#spx-ref-restock') : null;
      const qtyEls = detailEl ? detailEl.querySelectorAll('.spx-ref-qty') : [];
      const items = [];
      qtyEls.forEach(input=>{
        let qty = 0;
        const maxQ = Number(input.getAttribute('max')||'0');
        if(mode==='full'){
          qty = maxQ;
          input.value = String(maxQ);
        } else {
          qty = Number(input.value||'0');
        }
        if(qty > 0){
          items.push({ order_item_id: Number(input.getAttribute('data-item-id')||'0'), quantity: qty });
        }
      });
      const body = {
        id: orderId,
        items: items,
        reason: reasonEl ? reasonEl.value||'' : '',
        restock: restockEl ? !!restockEl.checked : false,
      };
      try {
        const btnFull = detailEl.querySelector('#spx-ref-full');
        const btnPartial = detailEl.querySelector('#spx-ref-partial');
        if(btnFull){ btnFull.disabled = true; }
        if(btnPartial){ btnPartial.disabled = true; }
        const resp = await api.post(`/sirapix-pos/v1/orders/${orderId}/refund`, body);
        showToast(I18N.toast_refund_success || 'Refund created', { type:'success' });
        await loadRefundOrderDetailInto(elRoot, orderId);
        await loadRefundOrdersInto(elRoot);
      } catch(e){
        const msg = (e && e.message) ? e.message : 'Refund failed';
        showToast(msg, { type:'error' });
      }
    }

    function openRefundsDrawer(){
      const header = `
        <div style="display:flex;align-items:center;justify-content:space-between;padding:12px 14px;border-bottom:1px solid #E5E7EB;">
          <strong style="font-family:Inter,-apple-system,Segoe UI,Roboto,Arial,sans-serif;font-size:14px;">Refunds / Orders</strong>
          <button type="button" class="spx-close" aria-label="Close" style="border:none;background:transparent;font-size:18px;line-height:1;cursor:pointer;">×</button>
        </div>`;
      const body = `
        <div style="padding:10px;display:flex;flex-direction:column;gap:8px;flex:1 1 auto;overflow:hidden;position:relative;">
          <div style="display:flex;flex-direction:column;gap:4px;">
            <label style="font-size:12px;">Order ID or customer phone/email</label>
            <input type="text" id="spx-ref-search" class="sirapix-ref-input" placeholder="Order ID or customer phone/email" style="width:100%;" />
          </div>
          <div style="display:flex;flex-direction:column;gap:4px;">
            <label style="font-size:12px;">Origin</label>
            <select id="spx-ref-origin" class="sirapix-ref-input" style="width:100%;">
              <option value="">All</option>
              <option value="in_store">In-store</option>
              <option value="phone">Phone</option>
              <option value="online">Online</option>
            </select>
          </div>
          <div id="spx-ref-orders" class="spx-ref-orders" style="border:1px solid #E5E7EB;border-radius:8px;overflow:auto;min-height:60px;"></div>
          <div style="display:flex;align-items:center;justify-content:space-between;font-size:12px;margin-top:4px;flex:0 0 auto;">
            <button type="button" id="spx-ref-page-prev" class="sirapix-btn secondary" style="min-width:72px;">Prev</button>
            <span id="spx-ref-page-info">Page 1</span>
            <button type="button" id="spx-ref-page-next" class="sirapix-btn secondary" style="min-width:72px;">Next</button>
          </div>
          <div id="spx-ref-detail-overlay" style="position:absolute;inset:0;background:#fff;display:none;flex-direction:column;z-index:1;"></div>
        </div>`;
      ensureRefundStyles();
      showDrawer(header + body);
      const root = modalRoot.querySelector('.spx-drawer');
      const overlay = modalRoot.querySelector('.spx-modal-overlay');
      function escHandler(ev){ if(ev.key === 'Escape'){ ev.preventDefault(); cleanup(); closeModal(); } }
      function cleanup(){ try{ document.removeEventListener('keydown', escHandler, true); }catch(e){} }
      document.addEventListener('keydown', escHandler, true);
      if(overlay){ overlay.onclick = function(){ cleanup(); closeModal(); }; }
      const closeBtn = root.querySelector('.spx-close');
      if(closeBtn){ closeBtn.addEventListener('click', function(){ cleanup(); closeModal(); }); }

      const searchInput = root.querySelector('#spx-ref-search');
      const originSelect = root.querySelector('#spx-ref-origin');
      const prevBtn = root.querySelector('#spx-ref-page-prev');
      const nextBtn = root.querySelector('#spx-ref-page-next');
      const ordersEl = root.querySelector('#spx-ref-orders');
      const detailOverlay = root.querySelector('#spx-ref-detail-overlay');

      refundState.page = 1;
      refundState.maxPages = 1;
      refundState.search = '';
      refundState.origin = '';
      refundState.currentOrderId = null;
      refundState.currentOrder = null;

      let searchTimer = null;
      if(searchInput){
        searchInput.addEventListener('input', function(){
          const v = (searchInput.value||'').trim();
          if(searchTimer) clearTimeout(searchTimer);
          searchTimer = setTimeout(function(){
            refundState.search = v;
            refundState.page = 1;
            loadRefundOrdersInto(root);
          }, 250);
        });
      }
      if(originSelect){
        originSelect.addEventListener('change', function(){
          refundState.origin = originSelect.value||'';
          refundState.page = 1;
          loadRefundOrdersInto(root);
        });
      }
      if(prevBtn){
        prevBtn.addEventListener('click', function(){
          if(refundState.page > 1){ refundState.page--; loadRefundOrdersInto(root); }
        });
      }
      if(nextBtn){
        nextBtn.addEventListener('click', function(){
          if(refundState.page < refundState.maxPages){ refundState.page++; loadRefundOrdersInto(root); }
        });
      }
      if(ordersEl){
        ordersEl.addEventListener('click', function(e){
          const row = e.target.closest('.spx-ref-row');
          if(!row) return;
          // visual active highlight
          try{
            ordersEl.querySelectorAll('.spx-ref-row').forEach(function(btn){ btn.classList.remove('is-active'); });
            row.classList.add('is-active');
          }catch(_e){}
          const id = Number(row.getAttribute('data-id')||'0');
          if(id){
            if(detailOverlay){ detailOverlay.style.display = 'flex'; }
            loadRefundOrderDetailInto(root, id);
          }
        });
      }

      root.addEventListener('click', function(e){
        const t = e.target;
        const backBtn = t && typeof t.closest === 'function' ? t.closest('.spx-ref-back') : null;
        if(backBtn){
          if(detailOverlay){ detailOverlay.style.display = 'none'; }
          refundState.currentOrderId = null;
          refundState.currentOrder = null;
          return;
        }
        if(t.id === 'spx-ref-full'){
          submitRefundFrom(root, 'full');
        } else if(t.id === 'spx-ref-partial'){
          submitRefundFrom(root, 'partial');
        }
      });

      loadRefundOrdersInto(root);
    }

    async function openVariantDrawer(productId, baseName, baseImage){
      try{
        // Open immediately with loading state to avoid perceived lag
        const header = `
          <div style="display:flex;align-items:center;justify-content:space-between;padding:12px 14px;border-bottom:1px solid #E5E7EB;">
            <strong style="font-family:Inter,-apple-system,Segoe UI,Roboto,Arial,sans-serif;font-size:14px;">${baseName}</strong>
            <button type="button" class="spx-close" aria-label="Close" style="border:none;background:transparent;font-size:18px;line-height:1;cursor:pointer;">×</button>
          </div>`;
        const loading = `
            <div style="padding:16px;display:flex;align-items:center;gap:10px;">
              <div class="spx-spinner" style="width:16px;height:16px;border:2px solid #E5E7EB;border-top-color:var(--spx-primary,#0ea5e9);border-radius:50%;animation:spxSpin .8s linear infinite"></div>
              <div style="font-size:13px;color:#6B7280;">Loading variations…</div>
            </div>`;
        showDrawer(header + loading);
        const drawer = modalRoot.querySelector('.spx-drawer');
        const overlay = modalRoot.querySelector('.spx-modal-overlay');
        function escHandler(ev){ if(ev.key === 'Escape'){ ev.preventDefault(); cleanup(); closeModal(); } }
        function cleanup(){ try{ document.removeEventListener('keydown', escHandler, true); }catch(e){} }
        document.addEventListener('keydown', escHandler, true);
        if(overlay){ overlay.onclick = function(){ cleanup(); closeModal(); }; }

        // Use embedded variations from static products if available
        let variations = null;
        if(state.static && state.static.products){
          const p = (state.static.products||[]).find(x=> Number(x.id)===Number(productId));
          if(p && Array.isArray(p.variations)){ variations = p.variations; }
        }
        if(!variations){ variations = await fetchVariations(productId); }
        // Build attribute map
        const attrMap = {};
        variations.forEach(v=>{
          Object.entries(v.attributes||{}).forEach(([k,val])=>{
            const key = String(k);
            if(!attrMap[key]) attrMap[key] = new Set();
            if(val) attrMap[key].add(String(val));
          });
        });
        const selection = {};
        function getMatching(){
          return variations.filter(v=>{
            return Object.entries(selection).every(([k,val])=> !val || (String((v.attributes||{})[k])===String(val)) );
          });
        }
        function renderPills(){
          return Object.entries(attrMap).map(([slug,set])=>{
            const values = Array.from(set.values());
            const label = slug.replace(/^attribute_/,'').replace(/^pa_/,'').replace(/_/g,' ');
            const pretty = label.charAt(0).toUpperCase()+label.slice(1);
            const opts = values.map(val=>{
              const active = (String(selection[slug]||'')===String(val));
              // Disable if no variations would match when choosing this value along with other selections
              const testSel = Object.assign({}, selection, { [slug]: val });
              const possible = variations.some(v=>{
                return Object.entries(testSel).every(([k,sv])=> !sv || (String((v.attributes||{})[k])===String(sv)) );
              });
              const dis = possible ? '' : 'data-disabled="1"';
              return `<button type="button" class="spx-pill ${active?'active':''}" data-attr="${slug}" data-val="${val}" ${dis}>${val}</button>`;
            }).join('');
            return `<div class="spx-attr"><div class="lbl">${pretty}</div><div class="opts">${opts}</div></div>`;
          }).join('');
        }
        function resolveSelected(){
          const match = variations.find(v=>{
            return Object.keys(attrMap).every(k=> String((v.attributes||{})[k]||'') === String(selection[k]||''));
          });
          return match||null;
        }
        let selected = null;
        function buildBody(){
          const pills = renderPills();
          const resolved = resolveSelected();
          selected = resolved;
          const img = (resolved&&resolved.image) ? resolved.image : (baseImage||'');
          const price = resolved ? formatMoney(resolved.price||0) : '';
          return `
            <div style="padding:12px; overflow:auto;flex:1 1 auto;">
              ${img?`<div style="display:flex;justify-content:flex-start;margin-bottom:10px;"><img src="${img}" alt="" style="max-width:200px;width:100%;height:auto;border-radius:8px;"></div>`:''}
              <div class="spx-attrs">${pills}</div>
              <div class="spx-price" style="margin-top:10px;font-weight:700;">${price}</div>
            </div>
            <div style="padding:12px;border-top:1px solid #E5E7EB;flex:0 0 auto;display:flex;gap:10px;justify-content:flex-end;">
              <button type="button" class="sirapix-btn secondary spx-cancel">Cancel</button>
              <button type="button" class="sirapix-btn primary spx-add" ${resolved?'':'disabled'}>Add to Cart</button>
            </div>`;
        }
        drawer.innerHTML = header + buildBody();
        function reRender(){
          const body = buildBody();
          drawer.innerHTML = header + body;
          bindEvents();
        }
        function bindEvents(){
          const x = drawer.querySelector('.spx-close');
          if(x){ x.addEventListener('click', function(){ cleanup(); closeModal(); }); }
          const c = drawer.querySelector('.spx-cancel');
          if(c){ c.addEventListener('click', function(){ cleanup(); closeModal(); }); }
          drawer.querySelectorAll('.spx-pill').forEach(btn=>{
            btn.addEventListener('click', function(){
              if(this.getAttribute('data-disabled')==='1'){ return; }
              const a = this.getAttribute('data-attr');
              const v = this.getAttribute('data-val');
              if(String(selection[a]||'')===String(v)){
                delete selection[a];
              } else {
                selection[a] = v;
              }
              // If all attributes are selected and a concrete variation resolves, add immediately
              const allSelected = Object.keys(attrMap).every(k=> !!selection[k]);
              const resolvedNow = (function(){
                const match = variations.find(v=> Object.keys(attrMap).every(k=> String((v.attributes||{})[k]||'') === String(selection[k]||'')) );
                return match||null;
              })();
              if(allSelected && resolvedNow){
                addToCartVariant(productId, baseName, resolvedNow);
                cleanup(); closeModal(); showToast(I18N.toast_added_to_cart || 'Added to cart', { type:'success' });
                return;
              }
              reRender();
            });
          });
          const addBtn = drawer.querySelector('.spx-add');
          if(addBtn){ addBtn.addEventListener('click', function(){ if(selected){ addToCartVariant(productId, baseName, selected); cleanup(); closeModal(); showToast(I18N.toast_added_to_cart || 'Added to cart', { type:'success' }); } }); }
        }
        bindEvents();
      } catch(e){ console.error(e); showToast(I18N.toast_failed_load_variations || 'Failed to load variations', { type:'error' }); }
    }

    async function loadCategories(){
      try {
        let cats = [];
        const hasStatic = await tryLoadStaticProducts();
        if(hasStatic && Array.isArray(state.static.categories)){
          cats = state.static.categories.slice();
        } else {
          const data = await api.get('/sirapix-pos/v1/categories');
          cats = (data.categories||[]).slice();
        }
        // Sort by product count (desc) so the busiest category appears first
        cats.sort((a,b)=> (Number(b.count||0) - Number(a.count||0)) );
        renderCategories(cats);
        const first = el.categories.querySelector('.cat');
        if(first){
          [...el.categories.querySelectorAll('.cat')].forEach(x=>x.classList.remove('active'));
          first.classList.add('active');
          state.categoryId = Number(first.dataset.id)||null;
          if(el.categoriesSelect){ el.categoriesSelect.value = String(state.categoryId||''); }
        } else {
          state.categoryId = null;
        }
      } catch(e){ console.error(e); }
    }

    async function loadProducts({ reset=false }={}){
      if(state.loading) return;
      if(reset){
        state.page = 1; state.maxPages = 1;
        el.products.innerHTML = '<div class="spx-loading" style="padding:12px;color:#6B7280;">Loading...</div>';
      }
      state.loading = true;
      try {
        const hasStatic = await tryLoadStaticProducts();
        if(hasStatic){
          // Offline/static mode: load ALL products for the current category (no pagination)
          const all = (state.static.products||[]).filter(p=>{
            const byCat = state.categoryId ? (Array.isArray(p.categories) && p.categories.includes(state.categoryId)) : true;
            const bySearch = state.search ? ((p.name||'').toLowerCase().includes(state.search.toLowerCase())) : true;
            return byCat && bySearch;
          });
          state.maxPages = 1;
          const items = all;
          if(reset){ el.products.innerHTML = ''; }
          renderProducts(items);
        } else {
          const data = await api.get('/sirapix-pos/v1/products', {
            category: state.categoryId||'',
            search: state.search||'',
            page: state.page,
            per_page: 24
          });
          state.maxPages = data.max_pages||1;
          if(reset){ el.products.innerHTML = ''; }
          renderProducts(data.products||[]);
        }
      } catch(e){ console.error(e); }
      state.loading = false;
    }

    function positionSuggest(){
      const panel = el.search.closest('.sirapix-panel');
      const inputRect = el.search.getBoundingClientRect();
      const panelRect = panel.getBoundingClientRect();
      const left = inputRect.left - panelRect.left;
      const top = (inputRect.top - panelRect.top) + inputRect.height + 6;
      el.searchSuggest.style.left = left + 'px';
      el.searchSuggest.style.top = top + 'px';
      el.searchSuggest.style.width = inputRect.width + 'px';
    }

    function updateSuggestions(list){
      if(!state.search || !(list&&list.length)) { el.searchSuggest.innerHTML=''; el.searchSuggest.style.display='none'; searchClear.style.display = (el.search.value.trim().length>0?'flex':'none'); return; }
      const items = (list||[]).map(p=>{
        const full = p.name||'';
        const shortName = full.length > 30 ? (full.slice(0,30) + '...') : full;
        const nameAttr = (full||'').replace(/\"/g,'');
        const img = p.image||'';
        const price = typeof p.price === 'number' ? p.price : parseFloat((p.price||'0'))||0;
        const type = (p.type ? p.type : ((p.has_price_range || (p.attributes_summary&&p.attributes_summary.length)) ? 'variable' : 'simple'));
        return `<div class="sug" data-id="${p.id}" data-type="${type}" data-name="${nameAttr}" data-price="${price}" data-image="${img}">`+
               `${img?`<img src="${img}" alt="">`:''}`+
               `<span>${shortName}</span>`+
               `</div>`;
      }).join('');
      el.searchSuggest.innerHTML = items;
      positionSuggest();
      el.searchSuggest.style.display = 'block';
      searchClear.style.display = 'flex';
    }

    // Wire up events
    function flyToCart(fromEl){
      // Disabled fly/bounce animation
      return;
    }

    el.products.addEventListener('click', function(e){
      const card = e.target.closest('.prod');
      if(!card) return;
      // Press animation: scale to 1.02 then back to 1 in 200ms
      card.classList.remove('press-anim'); // reset if rapidly clicked
      void card.offsetWidth; // reflow to restart animation
      card.classList.add('press-anim');
      setTimeout(()=> card.classList.remove('press-anim'), 220);
      const id = Number(card.dataset.id)||Date.now();
      const name = card.getAttribute('data-name') || card.querySelector('.meta span')?.getAttribute('title') || card.querySelector('.meta span')?.textContent || 'Product';
      const priceAttr = card.getAttribute('data-price');
      const image = card.querySelector('img')?.getAttribute('src')||'';
      const price = (priceAttr !== null && priceAttr !== undefined) ? (parseFloat(priceAttr)||0) : 0;
      const type = card.getAttribute('data-type')||'simple';
      if(type==='variable'){
        openVariantDrawer(id, name, image);
      } else {
        flyToCart(card);
        addToCart({ id, name, price, image });
      }
    });

    // Prefetch variations on hover to minimize waiting on click
    el.products.addEventListener('mouseenter', function(e){
      const card = e.target.closest('.prod');
      if(!card) return;
      const type = card.getAttribute('data-type')||'simple';
      if(type==='variable'){
        const id = Number(card.dataset.id)||0;
        if(id && !variationsCache.has(id) && !variationsInflight.has(id)){
          fetchVariations(id).catch(()=>{});
        }
      }
    }, true);

    el.cartItems.addEventListener('click', function(e){
      const row = e.target.closest('.sirapix-cart-item');
      if(!row) return;
      const id = Number(row.dataset.id);
      if(e.target.classList.contains('qty-inc')){
        state.cart[id].qty += 1; renderCart(); bumpCart();
      } else if(e.target.classList.contains('qty-dec')){
        state.cart[id].qty = Math.max(0, state.cart[id].qty - 1);
        if(state.cart[id].qty === 0){ delete state.cart[id]; }
        renderCart();
      }
    });

    function clearCart(){
      state.cart = {};
      renderCart();
      try{ localStorage.removeItem(STORAGE_KEY); }catch(e){}
      showToast(I18N.toast_cart_cleared || 'Cart cleared', { type: 'success' });
    }

    if(el.clear){
      el.clear.addEventListener('click', function(){
        clearCart();
      });
    }
    if(el.clearTop){
      el.clearTop.addEventListener('click', function(){
        clearCart();
      });
    }

    // Debounced search with autocomplete (after 2 chars). Do not change product grid while typing.
    async function searchAllProducts(term, signal){
      const all = [];
      const perPage = 100;
      const first = await api.get('/sirapix-pos/v1/products', { search: term, per_page: perPage, page: 1 }, { signal });
      if(first && Array.isArray(first.products)){ all.push.apply(all, first.products); }
      const maxPages = first && first.max_pages ? first.max_pages : 1;
      for(let p=2; p<=maxPages; p++){
        if(signal && signal.aborted){ break; }
        const data = await api.get('/sirapix-pos/v1/products', { search: term, per_page: perPage, page: p }, { signal });
        if(data && Array.isArray(data.products)){ all.push.apply(all, data.products); }
      }
      return all;
    }

    let t;
    let lastSearchId = 0;
    let currentSearchAbort = null;
    el.search.addEventListener('input', function(){
      const v = el.search.value.trim();
      searchClear.style.display = v.length ? 'flex' : 'none';
      if(t) clearTimeout(t);
      t = setTimeout(function(){
        state.search = v.length >= 2 ? v : '';
        if(state.search){
          const term = state.search.toLowerCase();
          // Prefer static filter if loaded; if it returns no matches, fall back to REST
          if(state.static && state.static.products){
            const list = (state.static.products||[]).filter(p=> (p.name||'').toLowerCase().includes(term));
            if(list && list.length){
              updateSuggestions(list);
              return;
            }
          }
          // Abort previous in-flight request
          if(currentSearchAbort){ try { currentSearchAbort.abort(); } catch(e){} }
          const controller = new AbortController();
          currentSearchAbort = controller;
          const thisId = ++lastSearchId;
          // Fetch suggestions across all pages; do not modify the products grid
          searchAllProducts(state.search, controller.signal)
            .then(list => {
              if(thisId !== lastSearchId) return; // Ignore stale results
              updateSuggestions(list||[]);
            })
            .catch(err => { if(err && err.name !== 'AbortError'){ console.error(err); } });
        } else {
          el.searchSuggest.innerHTML=''; el.searchSuggest.style.display='none';
          // keep existing products grid as-is
        }
      }, 200);
    });

    el.searchSuggest.addEventListener('click', function(e){
      const item = e.target.closest('.sug');
      if(!item) return;
      const id = Number(item.getAttribute('data-id'))||0;
      const name = item.getAttribute('data-name')||'';
      const price = parseFloat(item.getAttribute('data-price')||'0')||0;
      const image = item.getAttribute('data-image')||'';
      const type = item.getAttribute('data-type')||'simple';
      if(type==='variable'){
        openVariantDrawer(id, name, image);
      } else {
        addToCart({ id, name, price, image });
      }
      // Reset search and hide suggestions only; do not touch the products grid
      el.search.value = '';
      state.search = '';
      el.searchSuggest.innerHTML=''; el.searchSuggest.style.display='none';
      searchClear.style.display = 'none';
      // Keep current products grid intact (no reload)
    });

    // Clear button behavior
    searchClear.addEventListener('click', function(){
      el.search.value = '';
      state.search = '';
      el.searchSuggest.innerHTML='';
      el.searchSuggest.style.display='none';
      searchClear.style.display = 'none';
      el.search.focus();
    });

    // Infinite scroll sentinel
    const io = new IntersectionObserver(entries => {
      entries.forEach(en => {
        if(en.isIntersecting && !state.loading){
          if(state.page < state.maxPages){ state.page += 1; loadProducts(); }
        }
      });
    }, { root: null, threshold: 0.1 });
    if(el.sentinel){ io.observe(el.sentinel); }

    // Category switching
    el.categories.addEventListener('click', function(e){
      const c = e.target.closest('.cat');
      if(!c) return;
      [...el.categories.querySelectorAll('.cat')].forEach(x=>x.classList.remove('active'));
      c.classList.add('active');
      state.categoryId = Number(c.dataset.id)||null;
      if(el.categoriesSelect){ el.categoriesSelect.value = String(state.categoryId||''); }
      state.page = 1; el.products.innerHTML='';
      loadProducts({ reset:true });
    });

    // Helpers: normalize phone and validate checkout fields
    function normalizePhone(v){ return (v||'').toString().replace(/[^0-9]/g, ''); }
    function getCheckoutFields(){
      const nameEl = document.getElementById('spx-customer-name');
      const phoneEl = document.getElementById('spx-customer-phone');
      const emailEl = document.getElementById('spx-customer-email');
      const name = (nameEl?.value||'').trim();
      const phoneRaw = (phoneEl?.value||'').trim();
      const emailRaw = (emailEl?.value||'').trim();
      const phone = normalizePhone(phoneRaw);
      let email = emailRaw;
      function markError(el){ if(!el) return; el.classList.add('spx-field-error'); try{ el.focus(); el.select && el.select(); }catch(e){} }
      // Validate phone first, then name
      if(!phone){
        markError(phoneEl);
        const msg = I18N.error_customer_phone_required || 'Customer phone is required';
        throw new Error(msg);
      }
      if(!name){ markError(nameEl); throw new Error('Customer name is required'); }
      if(!email){ email = phone + '@' + (SIRAPIX_POS.domain||'example.com'); }
      return { name, phone, email };
    }

    let lastReceipt = null;
    let submittingOrder = false;

    el.complete.addEventListener('click', async function(){
      if(submittingOrder) return;
      // Lock immediately to avoid rapid double clicks
      submittingOrder = true;
      el.complete.disabled = true; el.complete.textContent = 'Completing...';
      if(el.completeTop){ el.completeTop.disabled = true; el.completeTop.textContent = 'Completing...'; }
      try {
        const fields = getCheckoutFields();
        const cartKeys = Object.keys(state.cart);
        if(cartKeys.length === 0){ alert('Cart is empty'); throw new Error('Cart is empty'); }

        const items = cartKeys.map(k=>{
          const r = state.cart[k];
          const prod = r?.prod||{};
          if(prod.variation_id){
            return { product_id: Number(prod.product_id||0), variation_id: Number(prod.variation_id||0), attributes: prod.attributes||{}, quantity: r.qty };
          }
          return { product_id: Number(prod.id||k), quantity: r.qty };
        });
        // Prepare receipt items snapshot before clearing cart
        const receiptItems = cartKeys.map(k=>({ name: state.cart[k]?.prod?.name||('#'+k), qty: state.cart[k]?.qty||0, price: state.cart[k]?.prod?.price||0 }));
        const payment = (document.querySelector('input[name="spx-payment"]:checked')?.value)||'pos_offline';
        const override_total_raw = (el.override?.value||'').trim();
        const override_total = override_total_raw ? parseFloat(override_total_raw) : null;
        const originEl = document.querySelector('input[name="spx-origin"]:checked');
        const origin = originEl ? originEl.value : 'in_store';
        const originLabel = origin === 'phone' ? 'Phone' : 'In-Store';
        const notes = originLabel;
        const orderPayload = {
          customer: fields,
          items,
          payment,
          override_total,
          notes,
          origin
        };
        let resp;
        try {
          resp = await api.post('/sirapix-pos/v1/orders', orderPayload);
        } catch(err) {
          // Queue on offline/network errors
          const isNet = (err && (err._net || /Network|fetch/i.test(err.message||''))) || (navigator.onLine===false);
          if(isNet){
            enqueueOrder(orderPayload);
            showToast(I18N.toast_order_saved_offline || 'Order is saved offline and queued for sync.', { type:'warning', timeout: 3000 });
            updateQueueCount();
            // Build a local receipt snapshot for offline printing (no order number / QR)
            const totalHtml = el.total ? el.total.textContent : '';
            lastReceipt = {
              order_no: '',
              order_id: null,
              total_html: totalHtml,
              date: new Date().toLocaleString(),
              customer: fields,
              items: receiptItems,
              notes,
              order_url: ''
            };
            if(el.print){ el.print.disabled = false; }
            // Clear cart and reset fields as if completed
            state.cart = {}; renderCart();
            try{ localStorage.removeItem(STORAGE_KEY); }catch(e){}
            document.getElementById('spx-customer-name').value = '';
            document.getElementById('spx-customer-phone').value = '';
            document.getElementById('spx-customer-email').value = '';
            if(el.override){ el.override.value = ''; }
            const originDefault = document.querySelector('input[name="spx-origin"][value="in_store"]');
            if(originDefault){ originDefault.checked = true; }
            return; // skip normal success flow
          }
          throw err;
        }
        // Show overlay modal instead of alert
        showOrderModal({ order_no: resp.order_no, total_html: resp.total_html }, lastReceipt);
        const doneText = I18N.toast_order_completed
          ? I18N.toast_order_completed.replace('%s', resp.order_no)
          : ('Order #'+ resp.order_no +' completed');
        showToast(doneText, { type: 'success' });
        // Store last receipt data for printing (before clearing cart)
        lastReceipt = {
          order_no: resp.order_no,
          order_id: resp.order_id,
          total_html: resp.total_html,
          date: new Date().toLocaleString(),
          customer: fields,
          items: receiptItems,
          notes,
          order_url: resp.order_url || ''
        };
        if(el.print){ el.print.disabled = false; }
        // Now reset cart and fields
        state.cart = {}; renderCart();
        try{ localStorage.removeItem(STORAGE_KEY); }catch(e){}
        document.getElementById('spx-customer-name').value = '';
        document.getElementById('spx-customer-phone').value = '';
        document.getElementById('spx-customer-email').value = '';
        if(el.override){ el.override.value = ''; }
        const originDefault = document.querySelector('input[name="spx-origin"][value="in_store"]');
        if(originDefault){ originDefault.checked = true; }
      } catch(err){
        const baseMsg = err && err.message ? err.message : (I18N.error_failed_complete_order || 'Failed to complete order');
        alert(baseMsg);
        showToast(baseMsg, { type: 'error' });
      } finally {
        const completeLabel = I18N.label_complete_order || 'Complete Order';
        el.complete.disabled = false; el.complete.textContent = completeLabel;
        if(el.completeTop){ el.completeTop.disabled = false; el.completeTop.textContent = completeLabel; }
        submittingOrder = false;
      }
    });
    
    // Delegate top button to the same handler for consistency
    if(el.completeTop){
      el.completeTop.addEventListener('click', function(){
        if(submittingOrder) return;
        if(el.complete && !el.complete.disabled){ el.complete.click(); }
      });
    }

    function openPrintWindow(receipt){
      const lines = (receipt.items||[]).map(it=>{
        const lineTotal = (Number(it.price||0)*Number(it.qty||0)).toFixed(2);
        return `<tr><td>${it.name}</td><td style="text-align:center;">${it.qty}</td><td style="text-align:right;">${Number(it.price||0).toFixed(2)}</td><td style="text-align:right;">${lineTotal}</td></tr>`;
      }).join('');
      const site = (SIRAPIX_POS.site_name||'Store');
      const logo = (SIRAPIX_POS.logo||'');
      const hasOrder = !!(receipt.order_no);
      const orderStr = hasOrder ? String(receipt.order_no||'') : '';
      let orderUrl = hasOrder ? String(receipt.order_url||'') : '';
      if(hasOrder && !orderUrl){
        const originGuess = (typeof location !== 'undefined' && location.origin) ? location.origin : ('https://' + (SIRAPIX_POS.domain || ''));
        const orderId = String(receipt.order_id||'');
        orderUrl = (originGuess.replace(/\/$/, '') + '/my-account/view-order/' + orderId + '/');
      }
      const qrUrl = hasOrder ? ('https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(orderUrl)) : '';
      const titleOrderPart = hasOrder ? (' #'+orderStr) : '';
      const heading = hasOrder ? ('Receipt #'+receipt.order_no) : 'Receipt';
      const qrBlock = hasOrder ? `<div class="qr"><img src="${qrUrl}" alt="QR: ${orderStr}"><div class="label">Scan to view order online</div></div>` : '';
      const html = `<!doctype html><html><head><meta charset="utf-8"><title>Receipt — ${site}${titleOrderPart}</title>
      <style>
        body{font:13px/1.45 -apple-system,Segoe UI,Roboto,Arial,sans-serif;padding:18px;color:#111}
        .hdr{display:flex; align-items:center; justify-content:space-between; gap:12px; border-bottom:2px solid #111; padding-bottom:8px;}
        .hdr img{max-height:36px; width:auto; display:block}
        .brand{display:flex; flex-direction:column}
        .brand .name{font-size:16px; font-weight:800; letter-spacing:.2px}
        .brand .meta{font-size:11px; color:#555}
        h1{font-size:18px;margin:10px 0 6px}
        table{width:100%;border-collapse:collapse;margin-top:8px}
        th,td{padding:6px;border-bottom:1px solid #eee}
        th{text-align:left; font-size:12px; color:#444}
        td:last-child, th:last-child{text-align:right}
        .row-end{display:flex; justify-content:flex-end; margin-top:10px}
        .tot{min-width:240px; border:1px solid #eee; border-radius:8px; padding:8px 10px; font-weight:700}
        .kv{display:flex; justify-content:space-between; margin:2px 0}
        .muted{color:#666;font-size:11px;margin-top:12px}
        .qr{display:flex; align-items:center; gap:10px; margin-top:12px}
        .qr img{width:96px; height:96px}
        .qr .label{font-size:12px; color:#444}
      </style></head>
      <body>
        <div class="hdr">
          ${logo?`<img src="${logo}" alt="${site}">`:''}
          <div class="brand"><div class="name">${site}</div><div class="meta">Point of Sale Receipt</div></div>
          ${qrBlock}
        </div>
        <h1>${heading}</h1>
        <div>Date: ${receipt.date}</div>
        <div>Customer: ${receipt.customer.name} | ${receipt.customer.phone}</div>
        <div>Payment: ${SIRAPIX_POS.labels?.payment_offline || 'Offline Payment'}</div>
        <div>Shipping: ${SIRAPIX_POS.labels?.shipping_offline_store || 'Offline Store'}</div>
        <table><thead><tr><th>Item</th><th>Qty</th><th style="text-align:right;">Price</th><th style="text-align:right;">Total</th></tr></thead><tbody>${lines}</tbody></table>
        <div class="row-end"><div class="tot">
          <div class="kv"><span>Subtotal</span><span>${receipt.total_html||''}</span></div>
        </div></div>
        ${hasOrder && orderUrl ? `<div class="muted">Order URL: ${orderUrl}</div>` : ''}
        <script>window.onload=function(){window.print();setTimeout(function(){window.close();},200);};</script>
      </body></html>`;
      const printWindow = window.open('', 'print', 'height=600,width=800');
      printWindow.document.write(html);
      printWindow.document.close();
    }

    if(el.print){
      el.print.addEventListener('click', function(){
        if(!lastReceipt){ alert('No receipt to print'); return; }
        openPrintWindow(lastReceipt);
      });
    }

    // Initial load: restore cart before any rendering to avoid wiping storage
    loadCart();
    renderCart();
    loadCategories().then(()=> loadProducts({ reset:true }));
    // Try to sync any queued orders on init
    processQueue();
    bindQueueUI();
  });
})();
