/**
 * Simply SEO - Gutenberg Sidebar Panel
 *
 * @package Simply_SEO
 * @since 1.0.0
 */

import { __ } from '@wordpress/i18n';
import { registerPlugin } from '@wordpress/plugins';
import { PluginSidebar, PluginSidebarMoreMenuItem } from '@wordpress/edit-post';
import {
    PanelBody,
    TextControl,
    TextareaControl,
    ToggleControl,
    Button,
} from '@wordpress/components';
import { useSelect } from '@wordpress/data';
import { useEntityProp } from '@wordpress/core-data';
import { MediaUpload, MediaUploadCheck } from '@wordpress/block-editor';

/**
 * SEO Icon component
 */
const SeoIcon = () => (
    <svg
        width="24"
        height="24"
        viewBox="0 0 24 24"
        fill="none"
        xmlns="http://www.w3.org/2000/svg"
    >
        <path
            d="M15.5 14h-.79l-.28-.27A6.471 6.471 0 0016 9.5 6.5 6.5 0 109.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z"
            fill="currentColor"
        />
    </svg>
);

/**
 * Simply SEO Sidebar Component
 */
const SimplySeoSidebar = () => {
    // Get current post type
    const postType = useSelect( ( select ) =>
        select( 'core/editor' ).getCurrentPostType()
    );

    // Get post ID
    const postId = useSelect( ( select ) =>
        select( 'core/editor' ).getCurrentPostId()
    );

    // Get meta values
    const [ meta, setMeta ] = useEntityProp( 'postType', postType, 'meta' );

    // Get post title and excerpt for previews
    const postTitle = useSelect( ( select ) =>
        select( 'core/editor' ).getEditedPostAttribute( 'title' )
    );

    const postExcerpt = useSelect( ( select ) =>
        select( 'core/editor' ).getEditedPostAttribute( 'excerpt' )
    );

    const permalink = useSelect( ( select ) =>
        select( 'core/editor' ).getPermalink()
    );

    // Extract meta values with defaults
    const seoTitle = meta?._simply_seo_title || '';
    const seoDescription = meta?._simply_seo_description || '';
    const seoKeywords = meta?._simply_seo_keywords || '';
    const noindex = meta?._simply_seo_noindex || false;
    const nofollow = meta?._simply_seo_nofollow || false;
    const canonical = meta?._simply_seo_canonical || '';
    const socialImage = meta?._simply_seo_social_image || 0;

    // Get settings from localized data
    const { titleFormat, siteName } = window.simplySeoData || {
        titleFormat: '%title% | %sitename%',
        siteName: '',
    };

    // Calculate preview values
    const previewTitle = seoTitle || postTitle || __( 'Post Title', 'simply-seo' );
    const formattedTitle = titleFormat
        .replace( '%title%', previewTitle )
        .replace( '%sitename%', siteName )
        .replace( '%tagline%', '' );

    const previewDescription = seoDescription || postExcerpt || 
        __( 'No description set. The first part of your content will be used.', 'simply-seo' );

    /**
     * Update meta value
     */
    const updateMeta = ( key, value ) => {
        setMeta( {
            ...meta,
            [ key ]: value,
        } );
    };

    // Get social image data
    const socialImageData = useSelect(
        ( select ) => {
            if ( socialImage ) {
                return select( 'core' ).getMedia( socialImage );
            }
            return null;
        },
        [ socialImage ]
    );

    return (
        <>
            <PluginSidebarMoreMenuItem target="simply-seo-sidebar">
                { __( 'Simply SEO', 'simply-seo' ) }
            </PluginSidebarMoreMenuItem>

            <PluginSidebar
                name="simply-seo-sidebar"
                title={ __( 'Simply SEO', 'simply-seo' ) }
                icon={ <SeoIcon /> }
            >
                { /* SEO Settings Panel */ }
                <PanelBody
                    title={ __( 'SEO Settings', 'simply-seo' ) }
                    initialOpen={ true }
                >
                    <TextControl
                        label={ __( 'SEO Title', 'simply-seo' ) }
                        value={ seoTitle }
                        onChange={ ( value ) => updateMeta( '_simply_seo_title', value ) }
                        placeholder={ postTitle || __( 'Post title', 'simply-seo' ) }
                        help={ __( 'Leave empty to use the post title.', 'simply-seo' ) }
                    />

                    <TextareaControl
                        label={ __( 'Meta Description', 'simply-seo' ) }
                        value={ seoDescription }
                        onChange={ ( value ) => updateMeta( '_simply_seo_description', value ) }
                        placeholder={ __( 'Enter a description...', 'simply-seo' ) }
                        help={
                            seoDescription
                                ? `${ seoDescription.length }/160 ${ __( 'characters', 'simply-seo' ) }`
                                : __( 'Leave empty to use an excerpt.', 'simply-seo' )
                        }
                        rows={ 3 }
                    />

                    <TextControl
                        label={ __( 'Keywords', 'simply-seo' ) }
                        value={ seoKeywords }
                        onChange={ ( value ) => updateMeta( '_simply_seo_keywords', value ) }
                        placeholder={ __( 'keyword1, keyword2, keyword3', 'simply-seo' ) }
                    />

                    <TextControl
                        label={ __( 'Canonical URL', 'simply-seo' ) }
                        value={ canonical }
                        onChange={ ( value ) => updateMeta( '_simply_seo_canonical', value ) }
                        placeholder={ permalink || '' }
                        help={ __( 'Leave empty to use the default URL.', 'simply-seo' ) }
                        type="url"
                    />
                </PanelBody>

                { /* Robots Panel */ }
                <PanelBody
                    title={ __( 'Robots', 'simply-seo' ) }
                    initialOpen={ true }
                >
                    <ToggleControl
                        label={ __( 'noindex', 'simply-seo' ) }
                        checked={ noindex }
                        onChange={ ( value ) => updateMeta( '_simply_seo_noindex', value ) }
                        help={ __( 'Prevent search engines from indexing this page.', 'simply-seo' ) }
                    />

                    <ToggleControl
                        label={ __( 'nofollow', 'simply-seo' ) }
                        checked={ nofollow }
                        onChange={ ( value ) => updateMeta( '_simply_seo_nofollow', value ) }
                        help={ __( 'Prevent search engines from following links on this page.', 'simply-seo' ) }
                    />
                </PanelBody>

                { /* Social Image Panel */ }
                <PanelBody
                    title={ __( 'Social Image', 'simply-seo' ) }
                    initialOpen={ false }
                >
                    <MediaUploadCheck>
                        <MediaUpload
                            onSelect={ ( media ) => updateMeta( '_simply_seo_social_image', media.id ) }
                            allowedTypes={ [ 'image' ] }
                            value={ socialImage }
                            render={ ( { open } ) => (
                                <div>
                                    { socialImageData && (
                                        <div style={ { marginBottom: '12px' } }>
                                            <img
                                                src={ socialImageData.source_url }
                                                alt=""
                                                style={ {
                                                    maxWidth: '100%',
                                                    height: 'auto',
                                                    borderRadius: '2px',
                                                } }
                                            />
                                        </div>
                                    ) }

                                    <Button
                                        onClick={ open }
                                        variant="secondary"
                                        style={ { marginRight: '8px' } }
                                    >
                                        { socialImage
                                            ? __( 'Replace Image', 'simply-seo' )
                                            : __( 'Select Image', 'simply-seo' ) }
                                    </Button>

                                    { socialImage > 0 && (
                                        <Button
                                            onClick={ () => updateMeta( '_simply_seo_social_image', 0 ) }
                                            variant="tertiary"
                                            isDestructive
                                        >
                                            { __( 'Remove', 'simply-seo' ) }
                                        </Button>
                                    ) }
                                </div>
                            ) }
                        />
                    </MediaUploadCheck>

                    <p
                        className="components-base-control__help"
                        style={ { marginTop: '8px' } }
                    >
                        { __( 'Optional. Uses featured image or default image if not set.', 'simply-seo' ) }
                    </p>
                </PanelBody>

                { /* Search Preview Panel */ }
                <PanelBody
                    title={ __( 'Search Preview', 'simply-seo' ) }
                    initialOpen={ false }
                >
                    <div
                        style={ {
                            padding: '12px',
                            backgroundColor: '#fff',
                            borderRadius: '8px',
                            border: '1px solid #ddd',
                        } }
                    >
                        <div
                            style={ {
                                color: '#1a0dab',
                                fontSize: '18px',
                                lineHeight: '1.3',
                                marginBottom: '4px',
                                wordBreak: 'break-word',
                            } }
                        >
                            { formattedTitle.substring( 0, 60 ) }
                            { formattedTitle.length > 60 && '...' }
                        </div>

                        <div
                            style={ {
                                color: '#006621',
                                fontSize: '14px',
                                marginBottom: '4px',
                            } }
                        >
                            { canonical || permalink || window.location.origin + '/...' }
                        </div>

                        <div
                            style={ {
                                color: '#545454',
                                fontSize: '13px',
                                lineHeight: '1.4',
                                wordBreak: 'break-word',
                            } }
                        >
                            { previewDescription.substring( 0, 160 ) }
                            { previewDescription.length > 160 && '...' }
                        </div>
                    </div>
                </PanelBody>
            </PluginSidebar>
        </>
    );
};

// Register the plugin
registerPlugin( 'simply-seo', {
    render: SimplySeoSidebar,
    icon: <SeoIcon />,
} );
