<?php
/**
 * Plugin Name: Simply SEO
 * Plugin URI: https://wijzijnmeo.nl/contact
 * Description: A lightweight SEO plugin with just the essentials: meta titles, descriptions, keywords, and indexing control.
 * Version: 1.1.3
 * Author: MEO
 * Author URI: https://wijzijnmeo.nl
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: simply-seo
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Plugin constants
define( 'SIMPLY_SEO_VERSION', '1.1.3' );
define( 'SIMPLY_SEO_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SIMPLY_SEO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SIMPLY_SEO_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main Simply SEO class
 */
final class Simply_SEO {

    /**
     * Single instance
     */
    private static $instance = null;

    /**
     * Get instance
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Load required files
     */
    private function load_dependencies() {
        require_once SIMPLY_SEO_PLUGIN_DIR . 'includes/class-simply-seo-settings.php';
        require_once SIMPLY_SEO_PLUGIN_DIR . 'includes/class-simply-seo-meta-box.php';
        require_once SIMPLY_SEO_PLUGIN_DIR . 'includes/class-simply-seo-taxonomy.php';
        require_once SIMPLY_SEO_PLUGIN_DIR . 'includes/class-simply-seo-schema.php';
        require_once SIMPLY_SEO_PLUGIN_DIR . 'includes/class-simply-seo-breadcrumbs.php';
        require_once SIMPLY_SEO_PLUGIN_DIR . 'includes/class-simply-seo-redirects.php';
        require_once SIMPLY_SEO_PLUGIN_DIR . 'includes/class-simply-seo-frontend.php';
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Initialize classes
        add_action( 'init', array( $this, 'init_classes' ) );
        
        // Register activation/deactivation hooks
        register_activation_hook( __FILE__, array( $this, 'activate' ) );
        register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
    }

    /**
     * Initialize plugin classes
     */
    public function init_classes() {
        // Settings page (admin only)
        if ( is_admin() ) {
            new Simply_SEO_Settings();
            new Simply_SEO_Meta_Box();
            new Simply_SEO_Taxonomy();
            new Simply_SEO_Redirects();
        }
        
        // Frontend output
        if ( ! is_admin() ) {
            new Simply_SEO_Frontend();
            new Simply_SEO_Schema();
            new Simply_SEO_Breadcrumbs();
            new Simply_SEO_Redirects(); // Also needed on frontend for handling redirects
        }
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options if not exist
        if ( false === get_option( 'simply_seo_settings' ) ) {
            $defaults = array(
                'enabled_post_types'      => array( 'post', 'page' ),
                'noindex_search'          => true,
                'noindex_404'             => true,
                'nofollow_search'         => true,
                'nofollow_404'            => true,
                'default_image'           => 0,
                'title_format'            => '%title% | %sitename%',
                'home_title'              => '',
                'home_description'        => '',
                'breadcrumbs_separator'   => '›',
                'breadcrumbs_home_text'   => '',
                'breadcrumbs_home_icon'   => false,
                'breadcrumbs_show_current'=> true,
                'keep_data_on_uninstall'  => false,
            );
            add_option( 'simply_seo_settings', $defaults, '', 'yes' );
        }
        
        // Create redirects table
        $this->create_redirects_table();
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Create redirects database table
     */
    private function create_redirects_table() {
        global $wpdb;
        
        $table_name      = $wpdb->prefix . 'simply_seo_redirects';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            source_url varchar(500) NOT NULL,
            target_url varchar(500) NOT NULL,
            redirect_type smallint(3) NOT NULL DEFAULT 301,
            hits bigint(20) unsigned NOT NULL DEFAULT 0,
            enabled tinyint(1) NOT NULL DEFAULT 1,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY source_url (source_url(191)),
            KEY enabled (enabled)
        ) $charset_collate;";
        
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );
        
        // Store db version
        update_option( 'simply_seo_db_version', '1.0' );
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        flush_rewrite_rules();
    }

    /**
     * Get plugin settings
     */
    public static function get_settings() {
        $defaults = array(
            'enabled_post_types'      => array( 'post', 'page' ),
            'noindex_search'          => true,
            'noindex_404'             => true,
            'nofollow_search'         => true,
            'nofollow_404'            => true,
            'default_image'           => 0,
            'title_format'            => '%title% | %sitename%',
            'home_title'              => '',
            'home_description'        => '',
            'breadcrumbs_separator'   => '›',
            'breadcrumbs_home_text'   => '',
            'breadcrumbs_home_icon'   => false,
            'breadcrumbs_show_current'=> true,
        );
        
        $settings = get_option( 'simply_seo_settings', $defaults );
        return wp_parse_args( $settings, $defaults );
    }
}

/**
 * Initialize plugin
 */
function simply_seo() {
    return Simply_SEO::get_instance();
}

// Start the plugin
simply_seo();
