<?php
/**
 * Simply SEO Taxonomy
 *
 * Handles SEO fields for taxonomies (categories, tags, custom taxonomies)
 *
 * @package Simply_SEO
 * @since 1.0.2
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Taxonomy class
 */
class Simply_SEO_Taxonomy {

    /**
     * Meta keys for taxonomy
     */
    private $meta_keys = array(
        'title'       => '_simply_seo_title',
        'description' => '_simply_seo_description',
        'noindex'     => '_simply_seo_noindex',
        'nofollow'    => '_simply_seo_nofollow',
        'image'       => '_simply_seo_image',
    );

    /**
     * Constructor
     */
    public function __construct() {
        // Get all public taxonomies
        add_action( 'admin_init', array( $this, 'register_taxonomy_fields' ) );
    }

    /**
     * Register fields for all public taxonomies
     */
    public function register_taxonomy_fields() {
        $taxonomies = get_taxonomies( array( 'public' => true ), 'names' );

        foreach ( $taxonomies as $taxonomy ) {
            // Add fields to add form
            add_action( $taxonomy . '_add_form_fields', array( $this, 'add_taxonomy_fields' ) );
            
            // Add fields to edit form
            add_action( $taxonomy . '_edit_form_fields', array( $this, 'edit_taxonomy_fields' ), 10, 2 );
            
            // Save fields
            add_action( 'created_' . $taxonomy, array( $this, 'save_taxonomy_fields' ) );
            add_action( 'edited_' . $taxonomy, array( $this, 'save_taxonomy_fields' ) );
            
            // Add column to taxonomy list
            add_filter( 'manage_edit-' . $taxonomy . '_columns', array( $this, 'add_taxonomy_column' ) );
            add_filter( 'manage_' . $taxonomy . '_custom_column', array( $this, 'render_taxonomy_column' ), 10, 3 );
        }

        // Enqueue media library on taxonomy pages
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts( $hook ) {
        if ( 'edit-tags.php' === $hook || 'term.php' === $hook ) {
            wp_enqueue_media();
            wp_enqueue_style( 'simply-seo-admin', SIMPLY_SEO_PLUGIN_URL . 'assets/css/admin.css', array(), SIMPLY_SEO_VERSION );
            wp_enqueue_script( 'simply-seo-admin', SIMPLY_SEO_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery' ), SIMPLY_SEO_VERSION, true );
            
            wp_localize_script( 'simply-seo-admin', 'simplySeoAdmin', array(
                'selectImage' => __( 'Select Image', 'simply-seo' ),
                'useImage'    => __( 'Use this image', 'simply-seo' ),
            ) );
        }
    }

    /**
     * Add fields to taxonomy add form
     */
    public function add_taxonomy_fields( $taxonomy ) {
        wp_nonce_field( 'simply_seo_taxonomy', 'simply_seo_taxonomy_nonce' );
        ?>
        <div class="simply-seo-taxonomy-fields">
            <h3><?php esc_html_e( 'Simply SEO', 'simply-seo' ); ?></h3>
            
            <div class="form-field">
                <label for="simply-seo-tax-title"><?php esc_html_e( 'SEO Title', 'simply-seo' ); ?></label>
                <input type="text" name="simply_seo_tax_title" id="simply-seo-tax-title" value="">
                <p class="description"><?php esc_html_e( 'Custom title for this archive page. Leave empty to use the term name.', 'simply-seo' ); ?></p>
            </div>

            <div class="form-field">
                <label for="simply-seo-tax-description"><?php esc_html_e( 'SEO Description', 'simply-seo' ); ?></label>
                <textarea name="simply_seo_tax_description" id="simply-seo-tax-description" rows="3"></textarea>
                <p class="description"><?php esc_html_e( 'Custom meta description for this archive page.', 'simply-seo' ); ?></p>
            </div>

            <div class="form-field">
                <label><?php esc_html_e( 'Robots', 'simply-seo' ); ?></label>
                <div class="robots-options">
                    <label>
                        <input type="checkbox" name="simply_seo_tax_noindex" value="1">
                        <?php esc_html_e( 'noindex', 'simply-seo' ); ?>
                    </label>
                    <label>
                        <input type="checkbox" name="simply_seo_tax_nofollow" value="1">
                        <?php esc_html_e( 'nofollow', 'simply-seo' ); ?>
                    </label>
                </div>
            </div>

            <div class="form-field">
                <label><?php esc_html_e( 'Social Image', 'simply-seo' ); ?></label>
                <input type="hidden" name="simply_seo_tax_image" id="simply-seo-tax-image" value="">
                <div id="simply-seo-tax-image-preview" class="simply-seo-image-preview"></div>
                <button type="button" class="button simply-seo-tax-select-image"><?php esc_html_e( 'Select Image', 'simply-seo' ); ?></button>
                <button type="button" class="button simply-seo-tax-remove-image" style="display:none;"><?php esc_html_e( 'Remove Image', 'simply-seo' ); ?></button>
            </div>
        </div>
        <?php
    }

    /**
     * Add fields to taxonomy edit form
     */
    public function edit_taxonomy_fields( $term, $taxonomy ) {
        $title       = get_term_meta( $term->term_id, $this->meta_keys['title'], true );
        $description = get_term_meta( $term->term_id, $this->meta_keys['description'], true );
        $noindex     = get_term_meta( $term->term_id, $this->meta_keys['noindex'], true );
        $nofollow    = get_term_meta( $term->term_id, $this->meta_keys['nofollow'], true );
        $image       = get_term_meta( $term->term_id, $this->meta_keys['image'], true );
        $image_url   = $image ? wp_get_attachment_image_url( $image, 'medium' ) : '';
        
        wp_nonce_field( 'simply_seo_taxonomy', 'simply_seo_taxonomy_nonce' );
        ?>
        <tr class="form-field">
            <th scope="row">
                <label for="simply-seo-tax-title"><?php esc_html_e( 'SEO Title', 'simply-seo' ); ?></label>
            </th>
            <td>
                <input type="text" name="simply_seo_tax_title" id="simply-seo-tax-title" value="<?php echo esc_attr( $title ); ?>" class="large-text">
                <p class="description"><?php esc_html_e( 'Custom title for this archive page. Leave empty to use the term name.', 'simply-seo' ); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row">
                <label for="simply-seo-tax-description"><?php esc_html_e( 'SEO Description', 'simply-seo' ); ?></label>
            </th>
            <td>
                <textarea name="simply_seo_tax_description" id="simply-seo-tax-description" rows="3" class="large-text"><?php echo esc_textarea( $description ); ?></textarea>
                <p class="description"><?php esc_html_e( 'Custom meta description for this archive page.', 'simply-seo' ); ?></p>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><?php esc_html_e( 'Robots', 'simply-seo' ); ?></th>
            <td>
                <label>
                    <input type="checkbox" name="simply_seo_tax_noindex" value="1" <?php checked( $noindex ); ?>>
                    <?php esc_html_e( 'noindex', 'simply-seo' ); ?>
                </label>
                <br>
                <label>
                    <input type="checkbox" name="simply_seo_tax_nofollow" value="1" <?php checked( $nofollow ); ?>>
                    <?php esc_html_e( 'nofollow', 'simply-seo' ); ?>
                </label>
            </td>
        </tr>

        <tr class="form-field">
            <th scope="row"><?php esc_html_e( 'Social Image', 'simply-seo' ); ?></th>
            <td>
                <input type="hidden" name="simply_seo_tax_image" id="simply-seo-tax-image" value="<?php echo esc_attr( $image ); ?>">
                <div id="simply-seo-tax-image-preview" class="simply-seo-image-preview">
                    <?php if ( $image_url ) : ?>
                        <img src="<?php echo esc_url( $image_url ); ?>" style="max-width:200px;height:auto;">
                    <?php endif; ?>
                </div>
                <p>
                    <button type="button" class="button simply-seo-tax-select-image"><?php esc_html_e( 'Select Image', 'simply-seo' ); ?></button>
                    <button type="button" class="button simply-seo-tax-remove-image" <?php echo ! $image ? 'style="display:none;"' : ''; ?>><?php esc_html_e( 'Remove Image', 'simply-seo' ); ?></button>
                </p>
            </td>
        </tr>
        <?php
    }

    /**
     * Save taxonomy fields
     */
    public function save_taxonomy_fields( $term_id ) {
        // Verify nonce
        if ( ! isset( $_POST['simply_seo_taxonomy_nonce'] ) ||
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['simply_seo_taxonomy_nonce'] ) ), 'simply_seo_taxonomy' ) ) {
            return;
        }

        // Check permissions
        if ( ! current_user_can( 'edit_term', $term_id ) ) {
            return;
        }

        // Save title
        if ( isset( $_POST['simply_seo_tax_title'] ) ) {
            update_term_meta( $term_id, $this->meta_keys['title'], 
                sanitize_text_field( wp_unslash( $_POST['simply_seo_tax_title'] ) ) );
        }

        // Save description
        if ( isset( $_POST['simply_seo_tax_description'] ) ) {
            update_term_meta( $term_id, $this->meta_keys['description'], 
                sanitize_textarea_field( wp_unslash( $_POST['simply_seo_tax_description'] ) ) );
        }

        // Save noindex
        $noindex = isset( $_POST['simply_seo_tax_noindex'] ) && sanitize_text_field( wp_unslash( $_POST['simply_seo_tax_noindex'] ) ) === '1';
        update_term_meta( $term_id, $this->meta_keys['noindex'], $noindex );

        // Save nofollow
        $nofollow = isset( $_POST['simply_seo_tax_nofollow'] ) && sanitize_text_field( wp_unslash( $_POST['simply_seo_tax_nofollow'] ) ) === '1';
        update_term_meta( $term_id, $this->meta_keys['nofollow'], $nofollow );

        // Save image
        if ( isset( $_POST['simply_seo_tax_image'] ) ) {
            update_term_meta( $term_id, $this->meta_keys['image'], 
                absint( $_POST['simply_seo_tax_image'] ) );
        }
    }

    /**
     * Add SEO column to taxonomy list
     */
    public function add_taxonomy_column( $columns ) {
        $columns['simply_seo'] = __( 'SEO', 'simply-seo' );
        return $columns;
    }

    /**
     * Render SEO column content
     */
    public function render_taxonomy_column( $content, $column_name, $term_id ) {
        if ( 'simply_seo' !== $column_name ) {
            return $content;
        }

        $title       = get_term_meta( $term_id, $this->meta_keys['title'], true );
        $description = get_term_meta( $term_id, $this->meta_keys['description'], true );
        $noindex     = get_term_meta( $term_id, $this->meta_keys['noindex'], true );

        $status = array();
        
        if ( $title || $description ) {
            $status[] = '<span style="color:#00a32a;">●</span>';
        } else {
            $status[] = '<span style="color:#dba617;">○</span>';
        }

        if ( $noindex ) {
            $status[] = '<span style="color:#d63638;" title="noindex">N</span>';
        }

        return implode( ' ', $status );
    }
}
