<?php
/**
 * Simply SEO Settings
 *
 * @package Simply_SEO
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Settings class
 */
class Simply_SEO_Settings {

    /**
     * Option name
     */
    private $option_name = 'simply_seo_settings';

    /**
     * Constructor
     */
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_filter( 'plugin_action_links_' . SIMPLY_SEO_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );
    }

    /**
     * Add settings page to admin menu
     */
    public function add_settings_page() {
        add_options_page(
            __( 'Simply SEO', 'simply-seo' ),
            __( 'Simply SEO', 'simply-seo' ),
            'manage_options',
            'simply-seo',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Add settings link to plugins page
     */
    public function add_settings_link( $links ) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            esc_url( admin_url( 'options-general.php?page=simply-seo' ) ),
            __( 'Settings', 'simply-seo' )
        );
        array_unshift( $links, $settings_link );
        return $links;
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets( $hook ) {
        if ( 'settings_page_simply-seo' !== $hook ) {
            return;
        }

        wp_enqueue_media();
        wp_enqueue_style( 'simply-seo-admin', SIMPLY_SEO_PLUGIN_URL . 'assets/css/admin.css', array(), SIMPLY_SEO_VERSION );
        wp_enqueue_script( 'simply-seo-admin', SIMPLY_SEO_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery' ), SIMPLY_SEO_VERSION, true );
        
        wp_localize_script( 'simply-seo-admin', 'simplySeoAdmin', array(
            'selectImage' => __( 'Select Image', 'simply-seo' ),
            'useImage'    => __( 'Use this image', 'simply-seo' ),
        ) );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting(
            'simply_seo_settings_group',
            $this->option_name,
            array(
                'type'              => 'array',
                'sanitize_callback' => array( $this, 'sanitize_settings' ),
                'default'           => array(),
            )
        );
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings( $input ) {
        $sanitized = array();

        // Post types
        $sanitized['enabled_post_types'] = array();
        if ( isset( $input['enabled_post_types'] ) && is_array( $input['enabled_post_types'] ) ) {
            foreach ( $input['enabled_post_types'] as $post_type ) {
                $sanitized['enabled_post_types'][] = sanitize_key( $post_type );
            }
        }

        // Special pages robots
        $sanitized['noindex_search']  = ! empty( $input['noindex_search'] );
        $sanitized['nofollow_search'] = ! empty( $input['nofollow_search'] );
        $sanitized['noindex_404']     = ! empty( $input['noindex_404'] );
        $sanitized['nofollow_404']    = ! empty( $input['nofollow_404'] );

        // Title format
        $sanitized['title_format'] = isset( $input['title_format'] )
            ? sanitize_text_field( $input['title_format'] )
            : '%title% | %sitename%';

        // Homepage
        $sanitized['home_title']       = isset( $input['home_title'] )
            ? sanitize_text_field( $input['home_title'] )
            : '';
        $sanitized['home_description'] = isset( $input['home_description'] )
            ? sanitize_textarea_field( $input['home_description'] )
            : '';

        // Default image
        $sanitized['default_image'] = isset( $input['default_image'] )
            ? absint( $input['default_image'] )
            : 0;

        // Breadcrumbs settings
        $sanitized['breadcrumbs_separator'] = isset( $input['breadcrumbs_separator'] )
            ? sanitize_text_field( $input['breadcrumbs_separator'] )
            : '›';
        
        $sanitized['breadcrumbs_home_text'] = isset( $input['breadcrumbs_home_text'] )
            ? sanitize_text_field( $input['breadcrumbs_home_text'] )
            : '';
        
        $sanitized['breadcrumbs_home_icon'] = ! empty( $input['breadcrumbs_home_icon'] );
        
        $sanitized['breadcrumbs_show_current'] = isset( $input['breadcrumbs_show_current'] )
            ? (bool) $input['breadcrumbs_show_current']
            : true;

        // Options
        $sanitized['keep_data_on_uninstall'] = ! empty( $input['keep_data_on_uninstall'] );

        return $sanitized;
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $settings   = Simply_SEO::get_settings();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab selection doesn't require nonce
        $active_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'general';
        ?>
        <div class="wrap simply-seo-wrap">
            <h1><?php esc_html_e( 'Simply SEO', 'simply-seo' ); ?></h1>

            <nav class="nav-tab-wrapper simply-seo-tabs">
                <a href="<?php echo esc_url( admin_url( 'options-general.php?page=simply-seo&tab=general' ) ); ?>"
                   class="nav-tab <?php echo 'general' === $active_tab ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'General', 'simply-seo' ); ?>
                </a>
                <a href="<?php echo esc_url( admin_url( 'options-general.php?page=simply-seo&tab=homepage' ) ); ?>"
                   class="nav-tab <?php echo 'homepage' === $active_tab ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Homepage', 'simply-seo' ); ?>
                </a>
                <a href="<?php echo esc_url( admin_url( 'options-general.php?page=simply-seo&tab=social' ) ); ?>"
                   class="nav-tab <?php echo 'social' === $active_tab ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Social', 'simply-seo' ); ?>
                </a>
                <a href="<?php echo esc_url( admin_url( 'options-general.php?page=simply-seo&tab=breadcrumbs' ) ); ?>"
                   class="nav-tab <?php echo 'breadcrumbs' === $active_tab ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Breadcrumbs', 'simply-seo' ); ?>
                </a>
                <a href="<?php echo esc_url( admin_url( 'options-general.php?page=simply-seo&tab=redirects' ) ); ?>"
                   class="nav-tab <?php echo 'redirects' === $active_tab ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Redirects', 'simply-seo' ); ?>
                </a>
                <a href="<?php echo esc_url( admin_url( 'options-general.php?page=simply-seo&tab=options' ) ); ?>"
                   class="nav-tab <?php echo 'options' === $active_tab ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Options', 'simply-seo' ); ?>
                </a>
            </nav>

            <?php if ( 'redirects' === $active_tab ) : ?>
                <?php $this->render_redirects_tab( $settings ); ?>
            <?php elseif ( 'options' === $active_tab ) : ?>
                <form action="options.php" method="post">
                    <?php settings_fields( 'simply_seo_settings_group' ); ?>
                    <input type="hidden" name="simply_seo_active_tab" value="<?php echo esc_attr( $active_tab ); ?>">
                    <?php $this->render_hidden_general_fields( $settings ); ?>
                    <?php $this->render_hidden_homepage_fields( $settings ); ?>
                    <?php $this->render_hidden_social_fields( $settings ); ?>
                    <?php $this->render_hidden_breadcrumbs_fields( $settings ); ?>
                    <?php $this->render_options_tab( $settings ); ?>
                    <?php submit_button(); ?>
                </form>
            <?php else : ?>
            <form action="options.php" method="post">
                <?php settings_fields( 'simply_seo_settings_group' ); ?>
                
                <!-- Hidden field to track active tab -->
                <input type="hidden" name="simply_seo_active_tab" value="<?php echo esc_attr( $active_tab ); ?>">

                <?php if ( 'general' === $active_tab ) : ?>
                    <?php $this->render_general_tab( $settings ); ?>
                    <?php $this->render_hidden_homepage_fields( $settings ); ?>
                    <?php $this->render_hidden_social_fields( $settings ); ?>
                    <?php $this->render_hidden_breadcrumbs_fields( $settings ); ?>
                    <?php $this->render_hidden_options_fields( $settings ); ?>
                <?php elseif ( 'homepage' === $active_tab ) : ?>
                    <?php $this->render_hidden_general_fields( $settings ); ?>
                    <?php $this->render_homepage_tab( $settings ); ?>
                    <?php $this->render_hidden_social_fields( $settings ); ?>
                    <?php $this->render_hidden_breadcrumbs_fields( $settings ); ?>
                    <?php $this->render_hidden_options_fields( $settings ); ?>
                <?php elseif ( 'social' === $active_tab ) : ?>
                    <?php $this->render_hidden_general_fields( $settings ); ?>
                    <?php $this->render_hidden_homepage_fields( $settings ); ?>
                    <?php $this->render_social_tab( $settings ); ?>
                    <?php $this->render_hidden_breadcrumbs_fields( $settings ); ?>
                    <?php $this->render_hidden_options_fields( $settings ); ?>
                <?php elseif ( 'breadcrumbs' === $active_tab ) : ?>
                    <?php $this->render_hidden_general_fields( $settings ); ?>
                    <?php $this->render_hidden_homepage_fields( $settings ); ?>
                    <?php $this->render_hidden_social_fields( $settings ); ?>
                    <?php $this->render_breadcrumbs_tab( $settings ); ?>
                    <?php $this->render_hidden_options_fields( $settings ); ?>
                <?php endif; ?>

                <?php submit_button( __( 'Save Settings', 'simply-seo' ) ); ?>
            </form>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render hidden General fields
     */
    private function render_hidden_general_fields( $settings ) {
        $post_types = get_post_types( array( 'public' => true ), 'objects' );
        ?>
        <!-- Hidden General Fields -->
        <?php foreach ( $post_types as $post_type ) :
            if ( 'attachment' === $post_type->name ) continue;
            $checked = in_array( $post_type->name, $settings['enabled_post_types'], true );
            if ( $checked ) :
        ?>
        <input type="hidden" name="simply_seo_settings[enabled_post_types][]" value="<?php echo esc_attr( $post_type->name ); ?>">
        <?php 
            endif;
        endforeach; ?>
        <input type="hidden" name="simply_seo_settings[title_format]" value="<?php echo esc_attr( $settings['title_format'] ); ?>">
        <?php if ( $settings['noindex_search'] ) : ?>
        <input type="hidden" name="simply_seo_settings[noindex_search]" value="1">
        <?php endif; ?>
        <?php if ( $settings['nofollow_search'] ) : ?>
        <input type="hidden" name="simply_seo_settings[nofollow_search]" value="1">
        <?php endif; ?>
        <?php if ( $settings['noindex_404'] ) : ?>
        <input type="hidden" name="simply_seo_settings[noindex_404]" value="1">
        <?php endif; ?>
        <?php if ( $settings['nofollow_404'] ) : ?>
        <input type="hidden" name="simply_seo_settings[nofollow_404]" value="1">
        <?php endif; ?>
        <?php
    }

    /**
     * Render hidden Homepage fields
     */
    private function render_hidden_homepage_fields( $settings ) {
        ?>
        <!-- Hidden Homepage Fields -->
        <input type="hidden" name="simply_seo_settings[home_title]" value="<?php echo esc_attr( $settings['home_title'] ); ?>">
        <input type="hidden" name="simply_seo_settings[home_description]" value="<?php echo esc_attr( $settings['home_description'] ); ?>">
        <?php
    }

    /**
     * Render hidden Social fields
     */
    private function render_hidden_social_fields( $settings ) {
        ?>
        <!-- Hidden Social Fields -->
        <input type="hidden" name="simply_seo_settings[default_image]" value="<?php echo esc_attr( $settings['default_image'] ); ?>">
        <?php
    }

    /**
     * Render General tab
     */
    private function render_general_tab( $settings ) {
        $post_types = get_post_types( array( 'public' => true ), 'objects' );
        ?>
        <div class="simply-seo-settings-content">
            <!-- Post Types -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Post Types', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'Enable SEO fields for these post types.', 'simply-seo' ); ?></p>
                    <table class="form-table simply-seo-toggles">
                        <tbody>
                            <?php foreach ( $post_types as $post_type ) :
                                if ( 'attachment' === $post_type->name ) continue;
                                $checked = in_array( $post_type->name, $settings['enabled_post_types'], true );
                            ?>
                            <tr>
                                <th scope="row"><?php echo esc_html( $post_type->labels->name ); ?></th>
                                <td>
                                    <label class="simply-seo-toggle">
                                        <input type="checkbox"
                                               name="simply_seo_settings[enabled_post_types][]"
                                               value="<?php echo esc_attr( $post_type->name ); ?>"
                                               <?php checked( $checked ); ?>>
                                        <span class="simply-seo-toggle-slider"></span>
                                    </label>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Title Format -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Title Format', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'Configure how page titles are formatted.', 'simply-seo' ); ?></p>
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row">
                                    <label for="simply-seo-title-format"><?php esc_html_e( 'Format', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <input type="text"
                                           id="simply-seo-title-format"
                                           name="simply_seo_settings[title_format]"
                                           value="<?php echo esc_attr( $settings['title_format'] ); ?>"
                                           class="regular-text"
                                           placeholder="%title% | %sitename%">
                                    <p class="description">
                                        <?php esc_html_e( 'Available placeholders:', 'simply-seo' ); ?>
                                        <code>%title%</code>, <code>%sitename%</code>, <code>%tagline%</code>
                                    </p>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Special Pages -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Special Pages', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'Configure robots meta for special pages.', 'simply-seo' ); ?></p>
                    <table class="form-table simply-seo-toggles">
                        <tbody>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Search Results', 'simply-seo' ); ?></th>
                                <td>
                                    <div class="simply-seo-toggle-group">
                                        <label class="simply-seo-toggle">
                                            <input type="checkbox"
                                                   name="simply_seo_settings[noindex_search]"
                                                   value="1"
                                                   <?php checked( $settings['noindex_search'] ); ?>>
                                            <span class="simply-seo-toggle-slider"></span>
                                            <span class="simply-seo-toggle-text">noindex</span>
                                        </label>
                                        <label class="simply-seo-toggle">
                                            <input type="checkbox"
                                                   name="simply_seo_settings[nofollow_search]"
                                                   value="1"
                                                   <?php checked( $settings['nofollow_search'] ); ?>>
                                            <span class="simply-seo-toggle-slider"></span>
                                            <span class="simply-seo-toggle-text">nofollow</span>
                                        </label>
                                    </div>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php esc_html_e( '404 Page', 'simply-seo' ); ?></th>
                                <td>
                                    <div class="simply-seo-toggle-group">
                                        <label class="simply-seo-toggle">
                                            <input type="checkbox"
                                                   name="simply_seo_settings[noindex_404]"
                                                   value="1"
                                                   <?php checked( $settings['noindex_404'] ); ?>>
                                            <span class="simply-seo-toggle-slider"></span>
                                            <span class="simply-seo-toggle-text">noindex</span>
                                        </label>
                                        <label class="simply-seo-toggle">
                                            <input type="checkbox"
                                                   name="simply_seo_settings[nofollow_404]"
                                                   value="1"
                                                   <?php checked( $settings['nofollow_404'] ); ?>>
                                            <span class="simply-seo-toggle-slider"></span>
                                            <span class="simply-seo-toggle-text">nofollow</span>
                                        </label>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render Homepage tab
     */
    private function render_homepage_tab( $settings ) {
        ?>
        <div class="simply-seo-settings-content">
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Homepage SEO', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'Set SEO data for your homepage.', 'simply-seo' ); ?></p>
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row">
                                    <label for="simply-seo-home-title"><?php esc_html_e( 'Homepage Title', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <input type="text"
                                           id="simply-seo-home-title"
                                           name="simply_seo_settings[home_title]"
                                           value="<?php echo esc_attr( $settings['home_title'] ); ?>"
                                           class="large-text"
                                           placeholder="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>">
                                    <p class="description">
                                        <?php esc_html_e( 'Leave empty to use the site title.', 'simply-seo' ); ?>
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="simply-seo-home-description"><?php esc_html_e( 'Homepage Description', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <textarea id="simply-seo-home-description"
                                              name="simply_seo_settings[home_description]"
                                              rows="3"
                                              class="large-text"
                                              placeholder="<?php echo esc_attr( get_bloginfo( 'description' ) ); ?>"><?php echo esc_textarea( $settings['home_description'] ); ?></textarea>
                                    <p class="description">
                                        <?php esc_html_e( 'Leave empty to use the site tagline.', 'simply-seo' ); ?>
                                    </p>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render Social tab
     */
    private function render_social_tab( $settings ) {
        $image_id  = $settings['default_image'];
        $image_url = $image_id ? wp_get_attachment_image_url( $image_id, 'medium' ) : '';
        ?>
        <div class="simply-seo-settings-content">
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Default Social Image', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'This image will be used for social sharing when no featured image or custom social image is set.', 'simply-seo' ); ?></p>
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row">
                                    <label><?php esc_html_e( 'Image', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <div class="simply-seo-image-upload">
                                        <input type="hidden"
                                               id="simply-seo-default-image"
                                               name="simply_seo_settings[default_image]"
                                               value="<?php echo esc_attr( $image_id ); ?>">

                                        <div id="simply-seo-image-preview" class="simply-seo-image-preview">
                                            <?php if ( $image_url ) : ?>
                                                <img src="<?php echo esc_url( $image_url ); ?>" alt="">
                                            <?php endif; ?>
                                        </div>

                                        <p class="simply-seo-image-buttons">
                                            <button type="button" id="simply-seo-select-image" class="button">
                                                <?php esc_html_e( 'Select Image', 'simply-seo' ); ?>
                                            </button>
                                            <button type="button"
                                                    id="simply-seo-remove-image"
                                                    class="button"
                                                    <?php echo ! $image_id ? 'style="display:none;"' : ''; ?>>
                                                <?php esc_html_e( 'Remove Image', 'simply-seo' ); ?>
                                            </button>
                                        </p>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render hidden Breadcrumbs fields
     */
    private function render_hidden_breadcrumbs_fields( $settings ) {
        ?>
        <!-- Hidden Breadcrumbs Fields -->
        <input type="hidden" name="simply_seo_settings[breadcrumbs_separator]" value="<?php echo esc_attr( $settings['breadcrumbs_separator'] ); ?>">
        <input type="hidden" name="simply_seo_settings[breadcrumbs_home_text]" value="<?php echo esc_attr( $settings['breadcrumbs_home_text'] ); ?>">
        <?php if ( $settings['breadcrumbs_home_icon'] ) : ?>
        <input type="hidden" name="simply_seo_settings[breadcrumbs_home_icon]" value="1">
        <?php endif; ?>
        <?php if ( $settings['breadcrumbs_show_current'] ) : ?>
        <input type="hidden" name="simply_seo_settings[breadcrumbs_show_current]" value="1">
        <?php endif; ?>
        <?php
    }

    /**
     * Render Breadcrumbs tab
     */
    private function render_breadcrumbs_tab( $settings ) {
        ?>
        <div class="simply-seo-settings-content">
            <!-- Usage -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'How to Use Breadcrumbs', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'Add breadcrumbs to your theme using one of these methods:', 'simply-seo' ); ?></p>
                    
                    <h4><?php esc_html_e( 'Shortcode', 'simply-seo' ); ?></h4>
                    <p><code>[simply_seo_breadcrumbs]</code></p>
                    <p class="description"><?php esc_html_e( 'Use this in pages, posts, or widgets.', 'simply-seo' ); ?></p>
                    
                    <h4><?php esc_html_e( 'PHP Template Function', 'simply-seo' ); ?></h4>
                    <p><code>&lt;?php simply_seo_breadcrumbs(); ?&gt;</code></p>
                    <p class="description"><?php esc_html_e( 'Add this to your theme template files (e.g., header.php, single.php).', 'simply-seo' ); ?></p>
                    
                    <h4><?php esc_html_e( 'Example Output', 'simply-seo' ); ?></h4>
                    <div style="background:#f0f0f1;padding:15px;border-radius:4px;margin-top:10px;display:flex;align-items:center;gap:8px;flex-wrap:wrap;">
                        <?php
                        $home_text = $settings['breadcrumbs_home_text'] ? $settings['breadcrumbs_home_text'] : 'Home';
                        $sep = $settings['breadcrumbs_separator'];
                        
                        if ( $settings['breadcrumbs_home_icon'] ) {
                            ?>
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"></path><polyline points="9 22 9 12 15 12 15 22"></polyline></svg>
                            <?php
                        } else {
                            echo esc_html( $home_text );
                        }
                        ?>
                        <span style="color:#666;"><?php echo esc_html( $sep ); ?></span>
                        <span>Category</span>
                        <span style="color:#666;"><?php echo esc_html( $sep ); ?></span>
                        <span style="color:#666;">Current Page</span>
                    </div>
                </div>
            </div>

            <!-- Settings -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Breadcrumbs Settings', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row">
                                    <label for="simply-seo-breadcrumbs-separator"><?php esc_html_e( 'Separator', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <select id="simply-seo-breadcrumbs-separator" name="simply_seo_settings[breadcrumbs_separator]">
                                        <option value="›" <?php selected( $settings['breadcrumbs_separator'], '›' ); ?>>› (default)</option>
                                        <option value="/" <?php selected( $settings['breadcrumbs_separator'], '/' ); ?>>/</option>
                                        <option value="»" <?php selected( $settings['breadcrumbs_separator'], '»' ); ?>>»</option>
                                        <option value="→" <?php selected( $settings['breadcrumbs_separator'], '→' ); ?>>→</option>
                                        <option value="|" <?php selected( $settings['breadcrumbs_separator'], '|' ); ?>>|</option>
                                        <option value="-" <?php selected( $settings['breadcrumbs_separator'], '-' ); ?>>-</option>
                                        <option value="•" <?php selected( $settings['breadcrumbs_separator'], '•' ); ?>>•</option>
                                    </select>
                                    <p class="description"><?php esc_html_e( 'Character between breadcrumb items.', 'simply-seo' ); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="simply-seo-breadcrumbs-home-icon"><?php esc_html_e( 'Home Link', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <fieldset>
                                        <label class="simply-seo-toggle">
                                            <input type="checkbox"
                                                   id="simply-seo-breadcrumbs-home-icon"
                                                   name="simply_seo_settings[breadcrumbs_home_icon]"
                                                   value="1"
                                                   <?php checked( $settings['breadcrumbs_home_icon'] ); ?>>
                                            <span class="simply-seo-toggle-slider"></span>
                                            <span class="simply-seo-toggle-text"><?php esc_html_e( 'Use home icon instead of text', 'simply-seo' ); ?></span>
                                        </label>
                                    </fieldset>
                                    <div id="simply-seo-home-icon-preview" style="margin-top:10px;display:<?php echo $settings['breadcrumbs_home_icon'] ? 'flex' : 'none'; ?>;align-items:center;gap:8px;">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"></path><polyline points="9 22 9 12 15 12 15 22"></polyline></svg>
                                        <span class="description" style="margin:0;"><?php esc_html_e( 'This icon will be used', 'simply-seo' ); ?></span>
                                    </div>
                                </td>
                            </tr>
                            <tr id="simply-seo-home-text-row" <?php echo $settings['breadcrumbs_home_icon'] ? 'style="display:none;"' : ''; ?>>
                                <th scope="row">
                                    <label for="simply-seo-breadcrumbs-home-text"><?php esc_html_e( 'Home Text', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <input type="text"
                                           id="simply-seo-breadcrumbs-home-text"
                                           name="simply_seo_settings[breadcrumbs_home_text]"
                                           value="<?php echo esc_attr( $settings['breadcrumbs_home_text'] ); ?>"
                                           class="regular-text"
                                           placeholder="Home">
                                    <p class="description"><?php esc_html_e( 'Text for the home link. Leave empty for "Home".', 'simply-seo' ); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="simply-seo-breadcrumbs-show-current"><?php esc_html_e( 'Current Page', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <fieldset>
                                        <label class="simply-seo-toggle">
                                            <input type="checkbox"
                                                   id="simply-seo-breadcrumbs-show-current"
                                                   name="simply_seo_settings[breadcrumbs_show_current]"
                                                   value="1"
                                                   <?php checked( $settings['breadcrumbs_show_current'] ); ?>>
                                            <span class="simply-seo-toggle-slider"></span>
                                            <span class="simply-seo-toggle-text"><?php esc_html_e( 'Show current page in breadcrumbs', 'simply-seo' ); ?></span>
                                        </label>
                                    </fieldset>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Advanced Usage -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Advanced Usage', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'Override settings using shortcode attributes or function parameters:', 'simply-seo' ); ?></p>
                    
                    <h4><?php esc_html_e( 'Shortcode with custom options', 'simply-seo' ); ?></h4>
                    <p><code>[simply_seo_breadcrumbs separator="/" home_text="Start" show_current="false"]</code></p>
                    
                    <h4><?php esc_html_e( 'PHP with custom options', 'simply-seo' ); ?></h4>
<pre style="background:#f0f0f1;padding:15px;border-radius:4px;overflow-x:auto;"><code>&lt;?php 
simply_seo_breadcrumbs( array(
    'separator'    => '/',
    'home_text'    => 'Start',
    'show_home'    => true,
    'show_current' => false,
) ); 
?&gt;</code></pre>
                    
                    <h4><?php esc_html_e( 'Available Parameters', 'simply-seo' ); ?></h4>
                    <table class="widefat" style="margin-top:10px;">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Parameter', 'simply-seo' ); ?></th>
                                <th><?php esc_html_e( 'Description', 'simply-seo' ); ?></th>
                                <th><?php esc_html_e( 'Default', 'simply-seo' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td><code>separator</code></td>
                                <td><?php esc_html_e( 'Character between items', 'simply-seo' ); ?></td>
                                <td><code>›</code></td>
                            </tr>
                            <tr>
                                <td><code>home_text</code></td>
                                <td><?php esc_html_e( 'Text for home link', 'simply-seo' ); ?></td>
                                <td><code>Home</code></td>
                            </tr>
                            <tr>
                                <td><code>show_home</code></td>
                                <td><?php esc_html_e( 'Show home link', 'simply-seo' ); ?></td>
                                <td><code>true</code></td>
                            </tr>
                            <tr>
                                <td><code>show_current</code></td>
                                <td><?php esc_html_e( 'Show current page', 'simply-seo' ); ?></td>
                                <td><code>true</code></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render Redirects tab
     */
    private function render_redirects_tab( $settings ) {
        // Enqueue redirects script
        wp_enqueue_script( 'simply-seo-redirects', SIMPLY_SEO_PLUGIN_URL . 'assets/js/redirects.js', array( 'jquery' ), SIMPLY_SEO_VERSION, true );
        
        wp_localize_script( 'simply-seo-redirects', 'simplySeoRedirects', array(
            'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
            'nonce'         => wp_create_nonce( 'simply_seo_redirects' ),
            'confirmDelete' => __( 'Are you sure you want to delete this redirect?', 'simply-seo' ),
            'saving'        => __( 'Saving...', 'simply-seo' ),
            'saved'         => __( 'Saved!', 'simply-seo' ),
            'error'         => __( 'Error saving redirect', 'simply-seo' ),
        ) );

        $redirects = new Simply_SEO_Redirects();
        $redirects->render_tab( $settings );
    }

    /**
     * Render Options tab
     */
    private function render_options_tab( $settings ) {
        ?>
        <div class="simply-seo-settings-content">
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Data Management', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Uninstall Behavior', 'simply-seo' ); ?></th>
                                <td>
                                    <fieldset>
                                        <label class="simply-seo-toggle">
                                            <input type="checkbox"
                                                   name="simply_seo_settings[keep_data_on_uninstall]"
                                                   value="1"
                                                   <?php checked( ! empty( $settings['keep_data_on_uninstall'] ) ); ?>>
                                            <span class="simply-seo-toggle-slider"></span>
                                            <span class="simply-seo-toggle-text"><?php esc_html_e( 'Keep data when plugin is deleted', 'simply-seo' ); ?></span>
                                        </label>
                                        <p class="description" style="margin-top:10px;">
                                            <?php esc_html_e( 'When enabled, all SEO data (settings, post meta, redirects, etc.) will be preserved if you delete the plugin. Useful if you plan to reinstall later.', 'simply-seo' ); ?>
                                        </p>
                                    </fieldset>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Data Stored by Simply SEO', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <p class="description"><?php esc_html_e( 'The following data is stored by Simply SEO and will be removed on uninstall (unless "Keep data" is enabled):', 'simply-seo' ); ?></p>
                    <ul style="list-style:disc;margin-left:20px;margin-top:10px;">
                        <li><strong><?php esc_html_e( 'Plugin Settings', 'simply-seo' ); ?></strong> - <?php esc_html_e( 'All configuration options', 'simply-seo' ); ?></li>
                        <li><strong><?php esc_html_e( 'Post/Page SEO Data', 'simply-seo' ); ?></strong> - <?php esc_html_e( 'Titles, descriptions, keywords, robots settings', 'simply-seo' ); ?></li>
                        <li><strong><?php esc_html_e( 'Taxonomy SEO Data', 'simply-seo' ); ?></strong> - <?php esc_html_e( 'Category/tag SEO settings', 'simply-seo' ); ?></li>
                        <li><strong><?php esc_html_e( 'Redirects', 'simply-seo' ); ?></strong> - <?php esc_html_e( 'All 301/302 redirects and hit counts', 'simply-seo' ); ?></li>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render hidden Options fields
     */
    private function render_hidden_options_fields( $settings ) {
        ?>
        <!-- Hidden Options Fields -->
        <?php if ( ! empty( $settings['keep_data_on_uninstall'] ) ) : ?>
        <input type="hidden" name="simply_seo_settings[keep_data_on_uninstall]" value="1">
        <?php endif; ?>
        <?php
    }
}
