<?php
/**
 * Simply SEO Schema
 *
 * Handles Schema.org JSON-LD structured data output
 *
 * @package Simply_SEO
 * @since 1.0.2
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Schema class
 */
class Simply_SEO_Schema {

    /**
     * Settings
     */
    private $settings;

    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = Simply_SEO::get_settings();
        
        // Output schema in head
        add_action( 'wp_head', array( $this, 'output_schema' ), 5 );
    }

    /**
     * Output schema JSON-LD
     */
    public function output_schema() {
        $schemas = array();

        // Website schema (always)
        $schemas[] = $this->get_website_schema();

        // Organization/Person schema (homepage)
        if ( is_front_page() ) {
            $schemas[] = $this->get_organization_schema();
        }

        // Article schema (singular posts)
        if ( is_singular( 'post' ) ) {
            $schemas[] = $this->get_article_schema();
        }

        // WebPage schema (pages)
        if ( is_singular( 'page' ) ) {
            $schemas[] = $this->get_webpage_schema();
        }

        // BreadcrumbList schema
        if ( ! is_front_page() ) {
            $breadcrumb_schema = $this->get_breadcrumb_schema();
            if ( $breadcrumb_schema ) {
                $schemas[] = $breadcrumb_schema;
            }
        }

        // CollectionPage for archives
        if ( is_archive() ) {
            $schemas[] = $this->get_collection_schema();
        }

        // Filter empty schemas
        $schemas = array_filter( $schemas );

        if ( empty( $schemas ) ) {
            return;
        }

        // Output each schema
        foreach ( $schemas as $schema ) {
            $this->output_json_ld( $schema );
        }
    }

    /**
     * Output JSON-LD script tag
     */
    private function output_json_ld( $schema ) {
        if ( empty( $schema ) ) {
            return;
        }

        $json = wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT );
        
        // Output JSON-LD - the JSON is safe as wp_json_encode handles encoding
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD must not be HTML escaped
        echo '<script type="application/ld+json">' . "\n" . $json . "\n</script>\n";
    }

    /**
     * Get Website schema
     */
    private function get_website_schema() {
        $schema = array(
            '@context' => 'https://schema.org',
            '@type'    => 'WebSite',
            '@id'      => home_url( '/#website' ),
            'url'      => home_url( '/' ),
            'name'     => get_bloginfo( 'name' ),
        );

        $tagline = get_bloginfo( 'description' );
        if ( $tagline ) {
            $schema['description'] = $tagline;
        }

        // Add search action
        $schema['potentialAction'] = array(
            '@type'       => 'SearchAction',
            'target'      => array(
                '@type'        => 'EntryPoint',
                'urlTemplate'  => home_url( '/?s={search_term_string}' ),
            ),
            'query-input' => 'required name=search_term_string',
        );

        return $schema;
    }

    /**
     * Get Organization schema
     */
    private function get_organization_schema() {
        $schema = array(
            '@context' => 'https://schema.org',
            '@type'    => 'Organization',
            '@id'      => home_url( '/#organization' ),
            'name'     => get_bloginfo( 'name' ),
            'url'      => home_url( '/' ),
        );

        // Add logo if site icon exists
        $site_icon_id = get_option( 'site_icon' );
        if ( $site_icon_id ) {
            $icon_url = wp_get_attachment_image_url( $site_icon_id, 'full' );
            if ( $icon_url ) {
                $schema['logo'] = array(
                    '@type'      => 'ImageObject',
                    'url'        => $icon_url,
                    '@id'        => home_url( '/#logo' ),
                    'caption'    => get_bloginfo( 'name' ),
                );
                $schema['image'] = array( '@id' => home_url( '/#logo' ) );
            }
        }

        return $schema;
    }

    /**
     * Get Article schema for posts
     */
    private function get_article_schema() {
        $post = get_queried_object();
        
        if ( ! $post ) {
            return null;
        }

        $schema = array(
            '@context'         => 'https://schema.org',
            '@type'            => 'Article',
            '@id'              => get_permalink( $post->ID ) . '#article',
            'headline'         => get_the_title( $post->ID ),
            'url'              => get_permalink( $post->ID ),
            'datePublished'    => get_the_date( 'c', $post->ID ),
            'dateModified'     => get_the_modified_date( 'c', $post->ID ),
            'mainEntityOfPage' => array(
                '@type' => 'WebPage',
                '@id'   => get_permalink( $post->ID ),
            ),
            'isPartOf'         => array(
                '@id' => home_url( '/#website' ),
            ),
        );

        // Add author
        $author_id = $post->post_author;
        $author_name = get_the_author_meta( 'display_name', $author_id );
        if ( $author_name ) {
            $schema['author'] = array(
                '@type' => 'Person',
                'name'  => $author_name,
                'url'   => get_author_posts_url( $author_id ),
            );
        }

        // Add publisher
        $schema['publisher'] = array(
            '@id' => home_url( '/#organization' ),
        );

        // Add featured image
        if ( has_post_thumbnail( $post->ID ) ) {
            $image_id   = get_post_thumbnail_id( $post->ID );
            $image_data = wp_get_attachment_image_src( $image_id, 'full' );
            if ( $image_data ) {
                $schema['image'] = array(
                    '@type'  => 'ImageObject',
                    'url'    => $image_data[0],
                    'width'  => $image_data[1],
                    'height' => $image_data[2],
                );
            }
        }

        // Add description
        $description = get_post_meta( $post->ID, '_simply_seo_description', true );
        if ( ! $description && $post->post_excerpt ) {
            $description = $post->post_excerpt;
        }
        if ( ! $description ) {
            $description = wp_trim_words( $post->post_content, 30, '...' );
        }
        if ( $description ) {
            $schema['description'] = wp_strip_all_tags( $description );
        }

        // Add word count
        $word_count = str_word_count( wp_strip_all_tags( $post->post_content ) );
        if ( $word_count > 0 ) {
            $schema['wordCount'] = $word_count;
        }

        // Add categories as keywords
        $categories = get_the_category( $post->ID );
        if ( ! empty( $categories ) ) {
            $keywords = wp_list_pluck( $categories, 'name' );
            $schema['keywords'] = implode( ', ', $keywords );
        }

        return $schema;
    }

    /**
     * Get WebPage schema for pages
     */
    private function get_webpage_schema() {
        $post = get_queried_object();
        
        if ( ! $post ) {
            return null;
        }

        $schema = array(
            '@context'      => 'https://schema.org',
            '@type'         => 'WebPage',
            '@id'           => get_permalink( $post->ID ) . '#webpage',
            'url'           => get_permalink( $post->ID ),
            'name'          => get_the_title( $post->ID ),
            'datePublished' => get_the_date( 'c', $post->ID ),
            'dateModified'  => get_the_modified_date( 'c', $post->ID ),
            'isPartOf'      => array(
                '@id' => home_url( '/#website' ),
            ),
        );

        // Add description
        $description = get_post_meta( $post->ID, '_simply_seo_description', true );
        if ( ! $description && $post->post_excerpt ) {
            $description = $post->post_excerpt;
        }
        if ( $description ) {
            $schema['description'] = wp_strip_all_tags( $description );
        }

        // Add featured image
        if ( has_post_thumbnail( $post->ID ) ) {
            $image_id   = get_post_thumbnail_id( $post->ID );
            $image_data = wp_get_attachment_image_src( $image_id, 'full' );
            if ( $image_data ) {
                $schema['primaryImageOfPage'] = array(
                    '@type'  => 'ImageObject',
                    'url'    => $image_data[0],
                    'width'  => $image_data[1],
                    'height' => $image_data[2],
                );
            }
        }

        return $schema;
    }

    /**
     * Get BreadcrumbList schema
     */
    private function get_breadcrumb_schema() {
        $breadcrumbs = $this->get_breadcrumb_items();
        
        if ( empty( $breadcrumbs ) || count( $breadcrumbs ) < 2 ) {
            return null;
        }

        $items = array();
        $position = 1;

        foreach ( $breadcrumbs as $crumb ) {
            $items[] = array(
                '@type'    => 'ListItem',
                'position' => $position,
                'name'     => $crumb['name'],
                'item'     => $crumb['url'],
            );
            $position++;
        }

        return array(
            '@context'        => 'https://schema.org',
            '@type'           => 'BreadcrumbList',
            'itemListElement' => $items,
        );
    }

    /**
     * Get breadcrumb items for schema
     */
    private function get_breadcrumb_items() {
        $items = array();

        // Home
        $items[] = array(
            'name' => __( 'Home', 'simply-seo' ),
            'url'  => home_url( '/' ),
        );

        if ( is_singular() ) {
            $post = get_queried_object();
            
            // Add category for posts
            if ( 'post' === $post->post_type ) {
                $categories = get_the_category( $post->ID );
                if ( ! empty( $categories ) ) {
                    $category = $categories[0];
                    
                    // Get parent categories
                    $parents = get_ancestors( $category->term_id, 'category' );
                    $parents = array_reverse( $parents );
                    
                    foreach ( $parents as $parent_id ) {
                        $parent = get_term( $parent_id, 'category' );
                        $items[] = array(
                            'name' => $parent->name,
                            'url'  => get_term_link( $parent ),
                        );
                    }
                    
                    $items[] = array(
                        'name' => $category->name,
                        'url'  => get_term_link( $category ),
                    );
                }
            }
            
            // Add parent pages
            if ( 'page' === $post->post_type && $post->post_parent ) {
                $parents = get_ancestors( $post->ID, 'page' );
                $parents = array_reverse( $parents );
                
                foreach ( $parents as $parent_id ) {
                    $items[] = array(
                        'name' => get_the_title( $parent_id ),
                        'url'  => get_permalink( $parent_id ),
                    );
                }
            }

            // Current page
            $items[] = array(
                'name' => get_the_title( $post->ID ),
                'url'  => get_permalink( $post->ID ),
            );

        } elseif ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            
            // Get parent terms
            if ( $term->parent ) {
                $parents = get_ancestors( $term->term_id, $term->taxonomy );
                $parents = array_reverse( $parents );
                
                foreach ( $parents as $parent_id ) {
                    $parent = get_term( $parent_id, $term->taxonomy );
                    $items[] = array(
                        'name' => $parent->name,
                        'url'  => get_term_link( $parent ),
                    );
                }
            }
            
            $items[] = array(
                'name' => $term->name,
                'url'  => get_term_link( $term ),
            );

        } elseif ( is_post_type_archive() ) {
            $post_type = get_queried_object();
            $items[] = array(
                'name' => $post_type->labels->name,
                'url'  => get_post_type_archive_link( $post_type->name ),
            );

        } elseif ( is_author() ) {
            $author = get_queried_object();
            $items[] = array(
                'name' => $author->display_name,
                'url'  => get_author_posts_url( $author->ID ),
            );

        } elseif ( is_search() ) {
            $items[] = array(
                'name' => sprintf(
                    /* translators: %s: search query */
                    __( 'Search: %s', 'simply-seo' ),
                    get_search_query()
                ),
                'url'  => get_search_link(),
            );

        } elseif ( is_404() ) {
            $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '/';
            $items[] = array(
                'name' => __( 'Page not found', 'simply-seo' ),
                'url'  => home_url( $request_uri ),
            );
        }

        return $items;
    }

    /**
     * Get CollectionPage schema for archives
     */
    private function get_collection_schema() {
        $schema = array(
            '@context' => 'https://schema.org',
            '@type'    => 'CollectionPage',
            'isPartOf' => array(
                '@id' => home_url( '/#website' ),
            ),
        );

        if ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            
            $schema['@id']  = get_term_link( $term ) . '#webpage';
            $schema['url']  = get_term_link( $term );
            $schema['name'] = $term->name;
            
            // Check for custom SEO title
            $seo_title = get_term_meta( $term->term_id, '_simply_seo_title', true );
            if ( $seo_title ) {
                $schema['name'] = $seo_title;
            }
            
            // Add description
            $seo_description = get_term_meta( $term->term_id, '_simply_seo_description', true );
            if ( $seo_description ) {
                $schema['description'] = $seo_description;
            } elseif ( $term->description ) {
                $schema['description'] = $term->description;
            }

        } elseif ( is_post_type_archive() ) {
            $post_type = get_queried_object();
            
            $schema['@id']  = get_post_type_archive_link( $post_type->name ) . '#webpage';
            $schema['url']  = get_post_type_archive_link( $post_type->name );
            $schema['name'] = $post_type->labels->name;

        } elseif ( is_author() ) {
            $author = get_queried_object();
            
            $schema['@type'] = 'ProfilePage';
            $schema['@id']   = get_author_posts_url( $author->ID ) . '#webpage';
            $schema['url']   = get_author_posts_url( $author->ID );
            $schema['name']  = $author->display_name;
        }

        return $schema;
    }
}
