<?php
/**
 * Simply SEO Redirects
 *
 * Handles 301/302 URL redirections using custom database table
 *
 * @package Simply_SEO
 * @since 1.0.6
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Redirects class
 */
class Simply_SEO_Redirects {

    /**
     * Table name
     */
    private $table_name;

    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'simply_seo_redirects';
        
        // Handle redirects on frontend (early priority)
        add_action( 'template_redirect', array( $this, 'handle_redirects' ), 1 );
        
        // AJAX handlers
        add_action( 'wp_ajax_simply_seo_save_redirect', array( $this, 'ajax_save_redirect' ) );
        add_action( 'wp_ajax_simply_seo_delete_redirect', array( $this, 'ajax_delete_redirect' ) );
        add_action( 'wp_ajax_simply_seo_toggle_redirect', array( $this, 'ajax_toggle_redirect' ) );
    }

    /**
     * Handle redirects on frontend
     */
    public function handle_redirects() {
        if ( is_admin() ) {
            return;
        }

        global $wpdb;
        
        // Get current request path
        $request_uri  = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        $current_path = wp_parse_url( $request_uri, PHP_URL_PATH );
        $current_path = rtrim( $current_path, '/' );
        
        if ( empty( $current_path ) ) {
            $current_path = '/';
        }

        // Try to get redirect from cache first
        $cache_key = 'simply_seo_redirect_' . md5( $current_path );
        $redirect  = wp_cache_get( $cache_key, 'simply_seo' );

        if ( false === $redirect ) {
            $table = $this->table_name;

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $redirect = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM `{$table}` WHERE enabled = 1 AND (source_url = %s OR source_url = %s) LIMIT 1", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $current_path,
                    $current_path . '/'
                )
            );

            // Cache for 1 hour (even if null to prevent repeated queries)
            wp_cache_set( $cache_key, $redirect ? $redirect : 'none', 'simply_seo', HOUR_IN_SECONDS );
        }

        // Check if we got a valid redirect (not 'none' placeholder)
        if ( $redirect && 'none' !== $redirect ) {
            $target = esc_url_raw( $redirect->target_url );
            
            // Make sure target is absolute URL
            if ( strpos( $target, 'http' ) !== 0 ) {
                $target = home_url( $target );
            }

            // Prevent redirect loops
            $target_path = wp_parse_url( $target, PHP_URL_PATH );
            if ( rtrim( $target_path, '/' ) !== $current_path ) {
                // Increment hit count (don't wait for this)
                $table = $this->table_name;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE `{$table}` SET hits = hits + 1 WHERE id = %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        $redirect->id
                    )
                );
                
                // Add target host to allowed redirect hosts
                add_filter( 'allowed_redirect_hosts', array( $this, 'allow_redirect_host' ) );
                
                // phpcs:ignore WordPress.Security.SafeRedirect.wp_redirect_wp_redirect -- External redirects intentionally allowed
                wp_redirect( $target, (int) $redirect->redirect_type );
                exit;
            }
        }
    }

    /**
     * Allow redirect host for external redirects
     *
     * @param array $hosts Allowed hosts.
     * @return array
     */
    public function allow_redirect_host( $hosts ) {
        // Get all unique target hosts from redirects
        global $wpdb;
        $table = $this->table_name;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $targets = $wpdb->get_col(
            "SELECT DISTINCT target_url FROM `{$table}` WHERE enabled = 1" // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        );
        
        if ( $targets ) {
            foreach ( $targets as $target ) {
                $host = wp_parse_url( $target, PHP_URL_HOST );
                if ( $host && ! in_array( $host, $hosts, true ) ) {
                    $hosts[] = $host;
                }
            }
        }
        
        return $hosts;
    }

    /**
     * Get all redirects
     */
    public function get_redirects() {
        global $wpdb;
        $table = $this->table_name;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $results = $wpdb->get_results(
            "SELECT * FROM `{$table}` ORDER BY created_at DESC", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            ARRAY_A
        );
        
        return $results ? $results : array();
    }

    /**
     * Render redirects tab content
     */
    public function render_tab( $settings ) {
        $redirects = $this->get_redirects();
        ?>
        <div class="simply-seo-settings-content">
            <!-- Add New Redirect -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Add New Redirect', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <form id="simply-seo-add-redirect" class="simply-seo-redirect-form">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="redirect-from"><?php esc_html_e( 'From URL', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <code><?php echo esc_url( home_url() ); ?></code>
                                    <input type="text" id="redirect-from" name="from" class="regular-text" placeholder="/old-page/" required>
                                    <p class="description"><?php esc_html_e( 'The relative path to redirect from (e.g., /old-page/)', 'simply-seo' ); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="redirect-to"><?php esc_html_e( 'To URL', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <input type="text" id="redirect-to" name="to" class="regular-text code" placeholder="https://example.com/new-page/ or /new-page/" required>
                                    <p class="description"><?php esc_html_e( 'The destination URL (absolute or relative)', 'simply-seo' ); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="redirect-type"><?php esc_html_e( 'Redirect Type', 'simply-seo' ); ?></label>
                                </th>
                                <td>
                                    <select id="redirect-type" name="type">
                                        <option value="301"><?php esc_html_e( '301 - Permanent', 'simply-seo' ); ?></option>
                                        <option value="302"><?php esc_html_e( '302 - Temporary', 'simply-seo' ); ?></option>
                                    </select>
                                    <p class="description"><?php esc_html_e( '301 for permanent moves (SEO-friendly), 302 for temporary redirects.', 'simply-seo' ); ?></p>
                                </td>
                            </tr>
                        </table>
                        <p class="submit">
                            <button type="submit" class="button button-primary"><?php esc_html_e( 'Add Redirect', 'simply-seo' ); ?></button>
                            <span class="simply-seo-status"></span>
                        </p>
                    </form>
                </div>
            </div>

            <!-- Existing Redirects -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'Existing Redirects', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <?php if ( empty( $redirects ) ) : ?>
                        <p class="description"><?php esc_html_e( 'No redirects configured yet.', 'simply-seo' ); ?></p>
                    <?php else : ?>
                        <table class="wp-list-table widefat fixed striped" id="simply-seo-redirects-table">
                            <thead>
                                <tr>
                                    <th scope="col" class="column-enabled"><?php esc_html_e( 'Active', 'simply-seo' ); ?></th>
                                    <th scope="col" class="column-from"><?php esc_html_e( 'From', 'simply-seo' ); ?></th>
                                    <th scope="col" class="column-to"><?php esc_html_e( 'To', 'simply-seo' ); ?></th>
                                    <th scope="col" class="column-type"><?php esc_html_e( 'Type', 'simply-seo' ); ?></th>
                                    <th scope="col" class="column-hits"><?php esc_html_e( 'Hits', 'simply-seo' ); ?></th>
                                    <th scope="col" class="column-actions"><?php esc_html_e( 'Actions', 'simply-seo' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $redirects as $redirect ) : ?>
                                    <tr data-id="<?php echo esc_attr( $redirect['id'] ); ?>">
                                        <td class="column-enabled">
                                            <label class="simply-seo-toggle simply-seo-toggle-small">
                                                <input type="checkbox" class="simply-seo-redirect-toggle" <?php checked( $redirect['enabled'], '1' ); ?>>
                                                <span class="simply-seo-toggle-slider"></span>
                                            </label>
                                        </td>
                                        <td class="column-from">
                                            <code><?php echo esc_html( $redirect['source_url'] ); ?></code>
                                        </td>
                                        <td class="column-to">
                                            <a href="<?php echo esc_url( $redirect['target_url'] ); ?>" target="_blank" rel="noopener">
                                                <?php echo esc_html( $redirect['target_url'] ); ?>
                                            </a>
                                        </td>
                                        <td class="column-type">
                                            <span class="simply-seo-badge simply-seo-badge-<?php echo esc_attr( $redirect['redirect_type'] ); ?>">
                                                <?php echo esc_html( $redirect['redirect_type'] ); ?>
                                            </span>
                                        </td>
                                        <td class="column-hits">
                                            <?php echo esc_html( number_format_i18n( $redirect['hits'] ) ); ?>
                                        </td>
                                        <td class="column-actions">
                                            <button type="button" class="button-link simply-seo-delete-redirect" title="<?php esc_attr_e( 'Delete', 'simply-seo' ); ?>">
                                                <span class="dashicons dashicons-trash"></span>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Help -->
            <div class="postbox">
                <div class="postbox-header">
                    <h2><?php esc_html_e( 'When to Use Redirects', 'simply-seo' ); ?></h2>
                </div>
                <div class="inside">
                    <div class="simply-seo-help-grid">
                        <div>
                            <h4><?php esc_html_e( '301 - Permanent Redirect', 'simply-seo' ); ?></h4>
                            <ul>
                                <li><?php esc_html_e( 'Page has permanently moved to a new URL', 'simply-seo' ); ?></li>
                                <li><?php esc_html_e( 'Changed permalink structure', 'simply-seo' ); ?></li>
                                <li><?php esc_html_e( 'Merged multiple pages into one', 'simply-seo' ); ?></li>
                                <li><?php esc_html_e( 'Transfers SEO value to the new URL', 'simply-seo' ); ?></li>
                            </ul>
                        </div>
                        <div>
                            <h4><?php esc_html_e( '302 - Temporary Redirect', 'simply-seo' ); ?></h4>
                            <ul>
                                <li><?php esc_html_e( 'Page is temporarily unavailable', 'simply-seo' ); ?></li>
                                <li><?php esc_html_e( 'A/B testing different pages', 'simply-seo' ); ?></li>
                                <li><?php esc_html_e( 'Maintenance mode for specific pages', 'simply-seo' ); ?></li>
                                <li><?php esc_html_e( 'Does not transfer SEO value', 'simply-seo' ); ?></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * AJAX: Save redirect
     */
    public function ajax_save_redirect() {
        check_ajax_referer( 'simply_seo_redirects', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied', 'simply-seo' ) ) );
        }

        $from = isset( $_POST['from'] ) ? sanitize_text_field( wp_unslash( $_POST['from'] ) ) : '';
        $to   = isset( $_POST['to'] ) ? esc_url_raw( wp_unslash( $_POST['to'] ) ) : '';
        $type = isset( $_POST['type'] ) && $_POST['type'] === '302' ? 302 : 301;

        if ( empty( $from ) || empty( $to ) ) {
            wp_send_json_error( array( 'message' => __( 'Both From and To URLs are required', 'simply-seo' ) ) );
        }

        // Ensure from starts with /
        if ( strpos( $from, '/' ) !== 0 ) {
            $from = '/' . $from;
        }

        global $wpdb;
        $table = $this->table_name;

        // Check for duplicates
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $exists = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM `{$table}` WHERE source_url = %s OR source_url = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                rtrim( $from, '/' ),
                rtrim( $from, '/' ) . '/'
            )
        );

        if ( $exists ) {
            wp_send_json_error( array( 'message' => __( 'A redirect from this URL already exists', 'simply-seo' ) ) );
        }

        // Insert new redirect
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->insert(
            $this->table_name,
            array(
                'source_url'    => $from,
                'target_url'    => $to,
                'redirect_type' => $type,
                'enabled'       => 1,
                'hits'          => 0,
            ),
            array( '%s', '%s', '%d', '%d', '%d' )
        );

        if ( false === $result ) {
            wp_send_json_error( array( 'message' => __( 'Failed to save redirect', 'simply-seo' ) ) );
        }

        // Clear redirect cache
        wp_cache_flush_group( 'simply_seo' );

        wp_send_json_success( array(
            'message' => __( 'Redirect added successfully', 'simply-seo' ),
            'id'      => $wpdb->insert_id,
        ) );
    }

    /**
     * AJAX: Delete redirect
     */
    public function ajax_delete_redirect() {
        check_ajax_referer( 'simply_seo_redirects', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied', 'simply-seo' ) ) );
        }

        $id = isset( $_POST['id'] ) ? absint( $_POST['id'] ) : 0;

        if ( empty( $id ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid redirect ID', 'simply-seo' ) ) );
        }

        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $wpdb->delete(
            $this->table_name,
            array( 'id' => $id ),
            array( '%d' )
        );

        if ( false === $result ) {
            wp_send_json_error( array( 'message' => __( 'Failed to delete redirect', 'simply-seo' ) ) );
        }

        // Clear redirect cache
        wp_cache_flush_group( 'simply_seo' );

        wp_send_json_success( array( 'message' => __( 'Redirect deleted', 'simply-seo' ) ) );
    }

    /**
     * AJAX: Toggle redirect enabled/disabled
     */
    public function ajax_toggle_redirect() {
        check_ajax_referer( 'simply_seo_redirects', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied', 'simply-seo' ) ) );
        }

        $id      = isset( $_POST['id'] ) ? absint( $_POST['id'] ) : 0;
        $enabled = isset( $_POST['enabled'] ) && $_POST['enabled'] === 'true' ? 1 : 0;

        if ( empty( $id ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid redirect ID', 'simply-seo' ) ) );
        }

        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $wpdb->update(
            $this->table_name,
            array( 'enabled' => $enabled ),
            array( 'id' => $id ),
            array( '%d' ),
            array( '%d' )
        );

        if ( false === $result ) {
            wp_send_json_error( array( 'message' => __( 'Failed to update redirect', 'simply-seo' ) ) );
        }

        // Clear redirect cache
        wp_cache_flush_group( 'simply_seo' );

        wp_send_json_success( array( 'message' => __( 'Redirect updated', 'simply-seo' ) ) );
    }

    /**
     * Drop redirects table (for uninstall)
     */
    public static function drop_table() {
        global $wpdb;
        $table_name = esc_sql( $wpdb->prefix . 'simply_seo_redirects' );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query( "DROP TABLE IF EXISTS `{$table_name}`" );
    }
}
