<?php
/**
 * Simply SEO Meta Box
 *
 * @package Simply_SEO
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Meta Box class
 */
class Simply_SEO_Meta_Box {

    /**
     * Meta keys
     */
    private $meta_keys = array(
        'title'        => '_simply_seo_title',
        'description'  => '_simply_seo_description',
        'keywords'     => '_simply_seo_keywords',
        'noindex'      => '_simply_seo_noindex',
        'nofollow'     => '_simply_seo_nofollow',
        'canonical'    => '_simply_seo_canonical',
        'social_image' => '_simply_seo_social_image',
    );

    /**
     * Constructor
     */
    public function __construct() {
        add_action( 'init', array( $this, 'register_meta' ) );
        add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_gutenberg_assets' ) );
        
        // Classic editor fallback
        add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
        add_action( 'save_post', array( $this, 'save_meta_box' ), 10, 2 );
        
        // Enqueue admin scripts for classic editor
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_classic_editor_assets' ) );
    }

    /**
     * Enqueue scripts for classic editor
     */
    public function enqueue_classic_editor_assets( $hook ) {
        if ( 'post.php' !== $hook && 'post-new.php' !== $hook ) {
            return;
        }

        // Only load for enabled post types
        $settings   = Simply_SEO::get_settings();
        $post_types = $settings['enabled_post_types'];
        $screen     = get_current_screen();
        
        if ( ! $screen || ! in_array( $screen->post_type, $post_types, true ) ) {
            return;
        }

        // Check if block editor is being used for this specific post
        $is_block_editor = false;
        
        if ( function_exists( 'use_block_editor_for_post_type' ) ) {
            $is_block_editor = use_block_editor_for_post_type( $screen->post_type );
        }
        
        // Also check if Classic Editor plugin is forcing classic mode
        if ( class_exists( 'Classic_Editor' ) ) {
            $is_block_editor = false;
        }
        
        // Check for specific post override (classic editor plugin or user preference)
        if ( 'post.php' === $hook ) {
            global $post;
            if ( $post && function_exists( 'use_block_editor_for_post' ) ) {
                $is_block_editor = use_block_editor_for_post( $post );
            }
        }

        // Only load for classic editor
        if ( $is_block_editor ) {
            return;
        }

        wp_enqueue_media();
        wp_enqueue_script( 'simply-seo-admin', SIMPLY_SEO_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery' ), SIMPLY_SEO_VERSION, true );
        
        wp_localize_script( 'simply-seo-admin', 'simplySeoAdmin', array(
            'selectImage'       => __( 'Select Image', 'simply-seo' ),
            'selectSocialImage' => __( 'Select Social Image', 'simply-seo' ),
            'useImage'          => __( 'Use this image', 'simply-seo' ),
        ) );
    }

    /**
     * Register meta fields for REST API
     */
    public function register_meta() {
        $settings    = Simply_SEO::get_settings();
        $post_types  = $settings['enabled_post_types'];

        foreach ( $post_types as $post_type ) {
            register_post_meta( $post_type, $this->meta_keys['title'], array(
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'auth_callback'     => function() {
                    return current_user_can( 'edit_posts' );
                },
            ) );

            register_post_meta( $post_type, $this->meta_keys['description'], array(
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_textarea_field',
                'auth_callback'     => function() {
                    return current_user_can( 'edit_posts' );
                },
            ) );

            register_post_meta( $post_type, $this->meta_keys['keywords'], array(
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'auth_callback'     => function() {
                    return current_user_can( 'edit_posts' );
                },
            ) );

            register_post_meta( $post_type, $this->meta_keys['noindex'], array(
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'boolean',
                'sanitize_callback' => 'rest_sanitize_boolean',
                'auth_callback'     => function() {
                    return current_user_can( 'edit_posts' );
                },
            ) );

            register_post_meta( $post_type, $this->meta_keys['nofollow'], array(
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'boolean',
                'sanitize_callback' => 'rest_sanitize_boolean',
                'auth_callback'     => function() {
                    return current_user_can( 'edit_posts' );
                },
            ) );

            register_post_meta( $post_type, $this->meta_keys['canonical'], array(
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'string',
                'sanitize_callback' => 'esc_url_raw',
                'auth_callback'     => function() {
                    return current_user_can( 'edit_posts' );
                },
            ) );

            register_post_meta( $post_type, $this->meta_keys['social_image'], array(
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'integer',
                'sanitize_callback' => 'absint',
                'auth_callback'     => function() {
                    return current_user_can( 'edit_posts' );
                },
            ) );
        }
    }

    /**
     * Enqueue Gutenberg assets
     */
    public function enqueue_gutenberg_assets() {
        $settings   = Simply_SEO::get_settings();
        $post_types = $settings['enabled_post_types'];
        $screen     = get_current_screen();

        // Only load on enabled post types
        if ( ! in_array( $screen->post_type, $post_types, true ) ) {
            return;
        }

        $asset_file = SIMPLY_SEO_PLUGIN_DIR . 'build/index.asset.php';
        
        if ( file_exists( $asset_file ) ) {
            $asset = include $asset_file;
        } else {
            $asset = array(
                'dependencies' => array(
                    'wp-plugins',
                    'wp-edit-post',
                    'wp-element',
                    'wp-components',
                    'wp-data',
                    'wp-i18n',
                    'wp-compose',
                ),
                'version' => SIMPLY_SEO_VERSION,
            );
        }

        wp_enqueue_script(
            'simply-seo-gutenberg',
            SIMPLY_SEO_PLUGIN_URL . 'build/index.js',
            $asset['dependencies'],
            $asset['version'],
            true
        );

        wp_localize_script( 'simply-seo-gutenberg', 'simplySeoData', array(
            'metaKeys'     => $this->meta_keys,
            'titleFormat'  => $settings['title_format'],
            'siteName'     => get_bloginfo( 'name' ),
            'siteTagline'  => get_bloginfo( 'description' ),
        ) );

        wp_set_script_translations( 'simply-seo-gutenberg', 'simply-seo' );
    }

    /**
     * Add classic editor meta box
     */
    public function add_meta_box() {
        $settings   = Simply_SEO::get_settings();
        $post_types = $settings['enabled_post_types'];

        foreach ( $post_types as $post_type ) {
            add_meta_box(
                'simply-seo-meta-box',
                __( 'Simply SEO', 'simply-seo' ),
                array( $this, 'render_meta_box' ),
                $post_type,
                'normal',
                'high',
                array(
                    '__block_editor_compatible_meta_box' => true,
                    '__back_compat_meta_box'             => false,
                )
            );
        }
    }

    /**
     * Render classic editor meta box
     */
    public function render_meta_box( $post ) {
        wp_nonce_field( 'simply_seo_meta_box', 'simply_seo_meta_box_nonce' );

        $title        = get_post_meta( $post->ID, $this->meta_keys['title'], true );
        $description  = get_post_meta( $post->ID, $this->meta_keys['description'], true );
        $keywords     = get_post_meta( $post->ID, $this->meta_keys['keywords'], true );
        $noindex      = get_post_meta( $post->ID, $this->meta_keys['noindex'], true );
        $nofollow     = get_post_meta( $post->ID, $this->meta_keys['nofollow'], true );
        $canonical    = get_post_meta( $post->ID, $this->meta_keys['canonical'], true );
        $social_image = get_post_meta( $post->ID, $this->meta_keys['social_image'], true );
        
        $image_url = $social_image ? wp_get_attachment_image_url( $social_image, 'medium' ) : '';
        ?>
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row">
                    <label for="simply-seo-title"><?php esc_html_e( 'SEO Title', 'simply-seo' ); ?></label>
                </th>
                <td>
                    <input type="text" 
                           id="simply-seo-title" 
                           name="simply_seo_title" 
                           value="<?php echo esc_attr( $title ); ?>" 
                           class="large-text"
                           placeholder="<?php echo esc_attr( $post->post_title ); ?>">
                    <p class="description">
                        <?php esc_html_e( 'Leave empty to use the post title.', 'simply-seo' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="simply-seo-description"><?php esc_html_e( 'Meta Description', 'simply-seo' ); ?></label>
                </th>
                <td>
                    <textarea id="simply-seo-description" 
                              name="simply_seo_description" 
                              rows="3" 
                              class="large-text"><?php echo esc_textarea( $description ); ?></textarea>
                    <p class="description">
                        <?php esc_html_e( 'Leave empty to use an excerpt from the content.', 'simply-seo' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="simply-seo-keywords"><?php esc_html_e( 'Keywords', 'simply-seo' ); ?></label>
                </th>
                <td>
                    <input type="text" 
                           id="simply-seo-keywords" 
                           name="simply_seo_keywords" 
                           value="<?php echo esc_attr( $keywords ); ?>" 
                           class="large-text"
                           placeholder="<?php esc_attr_e( 'keyword1, keyword2, keyword3', 'simply-seo' ); ?>">
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="simply-seo-canonical"><?php esc_html_e( 'Canonical URL', 'simply-seo' ); ?></label>
                </th>
                <td>
                    <input type="url" 
                           id="simply-seo-canonical" 
                           name="simply_seo_canonical" 
                           value="<?php echo esc_url( $canonical ); ?>" 
                           class="large-text"
                           placeholder="<?php echo esc_url( get_permalink( $post->ID ) ); ?>">
                    <p class="description">
                        <?php esc_html_e( 'Leave empty to use the default URL. Use this to point to the original content if this is a duplicate.', 'simply-seo' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <?php esc_html_e( 'Robots', 'simply-seo' ); ?>
                </th>
                <td>
                    <fieldset>
                        <label style="display:block;margin-bottom:4px;">
                            <input type="checkbox" 
                                   name="simply_seo_noindex" 
                                   value="1" 
                                   <?php checked( $noindex ); ?>>
                            <?php esc_html_e( 'noindex', 'simply-seo' ); ?>
                        </label>
                        <label style="display:block;">
                            <input type="checkbox" 
                                   name="simply_seo_nofollow" 
                                   value="1" 
                                   <?php checked( $nofollow ); ?>>
                            <?php esc_html_e( 'nofollow', 'simply-seo' ); ?>
                        </label>
                    </fieldset>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label><?php esc_html_e( 'Social Image', 'simply-seo' ); ?></label>
                </th>
                <td>
                    <input type="hidden" 
                           id="simply-seo-social-image" 
                           name="simply_seo_social_image" 
                           value="<?php echo esc_attr( $social_image ); ?>">
                    
                    <button type="button" id="simply-seo-select-social-image" class="button">
                        <?php esc_html_e( 'Select Image', 'simply-seo' ); ?>
                    </button>
                    
                    <button type="button" 
                            id="simply-seo-remove-social-image" 
                            class="button" 
                            style="<?php echo $social_image ? '' : 'display:none;'; ?>">
                        <?php esc_html_e( 'Remove Image', 'simply-seo' ); ?>
                    </button>
                    
                    <div id="simply-seo-social-image-preview">
                        <?php if ( $image_url ) : ?>
                            <img src="<?php echo esc_url( $image_url ); ?>" style="max-width:300px;height:auto;margin-top:10px;">
                        <?php endif; ?>
                    </div>
                    
                    <p class="description">
                        <?php esc_html_e( 'Optional. Uses featured image or default image if not set.', 'simply-seo' ); ?>
                    </p>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Save classic editor meta box
     */
    public function save_meta_box( $post_id, $post ) {
        // Verify nonce
        if ( ! isset( $_POST['simply_seo_meta_box_nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['simply_seo_meta_box_nonce'] ) ), 'simply_seo_meta_box' ) ) {
            return;
        }

        // Check autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        // Check permissions
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Check if post type is enabled
        $settings = Simply_SEO::get_settings();
        if ( ! in_array( $post->post_type, $settings['enabled_post_types'], true ) ) {
            return;
        }

        // Save title
        if ( isset( $_POST['simply_seo_title'] ) ) {
            update_post_meta( $post_id, $this->meta_keys['title'], 
                sanitize_text_field( wp_unslash( $_POST['simply_seo_title'] ) ) );
        }

        // Save description
        if ( isset( $_POST['simply_seo_description'] ) ) {
            update_post_meta( $post_id, $this->meta_keys['description'], 
                sanitize_textarea_field( wp_unslash( $_POST['simply_seo_description'] ) ) );
        }

        // Save keywords
        if ( isset( $_POST['simply_seo_keywords'] ) ) {
            update_post_meta( $post_id, $this->meta_keys['keywords'], 
                sanitize_text_field( wp_unslash( $_POST['simply_seo_keywords'] ) ) );
        }

        // Save noindex
        $noindex = isset( $_POST['simply_seo_noindex'] ) && $_POST['simply_seo_noindex'] === '1';
        update_post_meta( $post_id, $this->meta_keys['noindex'], $noindex );

        // Save nofollow
        $nofollow = isset( $_POST['simply_seo_nofollow'] ) && $_POST['simply_seo_nofollow'] === '1';
        update_post_meta( $post_id, $this->meta_keys['nofollow'], $nofollow );

        // Save canonical
        if ( isset( $_POST['simply_seo_canonical'] ) ) {
            update_post_meta( $post_id, $this->meta_keys['canonical'], 
                esc_url_raw( wp_unslash( $_POST['simply_seo_canonical'] ) ) );
        }

        // Save social image
        if ( isset( $_POST['simply_seo_social_image'] ) ) {
            update_post_meta( $post_id, $this->meta_keys['social_image'], 
                absint( $_POST['simply_seo_social_image'] ) );
        }
    }
}
