<?php
/**
 * Simply SEO Frontend
 *
 * @package Simply_SEO
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Frontend class
 */
class Simply_SEO_Frontend {

    /**
     * Settings
     */
    private $settings;

    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = Simply_SEO::get_settings();
        
        // Output meta tags
        add_action( 'wp_head', array( $this, 'output_meta_tags' ), 1 );
        
        // Filter document title
        add_filter( 'pre_get_document_title', array( $this, 'filter_title' ), 15 );
        add_filter( 'document_title_parts', array( $this, 'filter_title_parts' ), 15 );
        
        // Add robots meta
        add_filter( 'wp_robots', array( $this, 'filter_robots' ), 15 );

        // Filter canonical URL
        add_filter( 'get_canonical_url', array( $this, 'filter_canonical_url' ), 15, 2 );
    }

    /**
     * Output meta tags in head
     */
    public function output_meta_tags() {
        $description  = $this->get_description();
        $keywords     = $this->get_keywords();
        $image        = $this->get_social_image();
        $title        = $this->get_title();
        $canonical    = $this->get_canonical_url();
        $url          = $canonical ? $canonical : $this->get_current_url();

        echo "\n<!-- Simply SEO -->\n";

        // Meta description
        if ( $description ) {
            printf(
                '<meta name="description" content="%s">' . "\n",
                esc_attr( $description )
            );
        }

        // Meta keywords
        if ( $keywords ) {
            printf(
                '<meta name="keywords" content="%s">' . "\n",
                esc_attr( $keywords )
            );
        }

        // Open Graph tags
        if ( $title ) {
            printf(
                '<meta property="og:title" content="%s">' . "\n",
                esc_attr( $title )
            );
        }

        if ( $description ) {
            printf(
                '<meta property="og:description" content="%s">' . "\n",
                esc_attr( $description )
            );
        }

        printf(
            '<meta property="og:url" content="%s">' . "\n",
            esc_url( $url )
        );

        printf(
            '<meta property="og:site_name" content="%s">' . "\n",
            esc_attr( get_bloginfo( 'name' ) )
        );

        if ( is_singular() ) {
            echo '<meta property="og:type" content="article">' . "\n";
        } else {
            echo '<meta property="og:type" content="website">' . "\n";
        }

        if ( $image ) {
            printf(
                '<meta property="og:image" content="%s">' . "\n",
                esc_url( $image )
            );
        }

        // Twitter Card tags
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";

        if ( $title ) {
            printf(
                '<meta name="twitter:title" content="%s">' . "\n",
                esc_attr( $title )
            );
        }

        if ( $description ) {
            printf(
                '<meta name="twitter:description" content="%s">' . "\n",
                esc_attr( $description )
            );
        }

        if ( $image ) {
            printf(
                '<meta name="twitter:image" content="%s">' . "\n",
                esc_url( $image )
            );
        }

        echo "<!-- /Simply SEO -->\n\n";
    }

    /**
     * Filter document title
     */
    public function filter_title( $title ) {
        // Homepage
        if ( is_front_page() || is_home() ) {
            $home_title = $this->settings['home_title'];
            if ( $home_title ) {
                return $home_title;
            }
        }

        // Singular posts/pages
        if ( is_singular() ) {
            $post_id  = get_queried_object_id();
            $seo_title = get_post_meta( $post_id, '_simply_seo_title', true );
            
            if ( $seo_title ) {
                return $this->format_title( $seo_title );
            }
        }

        return $title;
    }

    /**
     * Filter title parts
     */
    public function filter_title_parts( $parts ) {
        // Apply title format for singular content
        if ( is_singular() && isset( $parts['title'] ) ) {
            $post_id   = get_queried_object_id();
            $seo_title = get_post_meta( $post_id, '_simply_seo_title', true );
            
            if ( $seo_title ) {
                $parts['title'] = $seo_title;
            }
        }

        return $parts;
    }

    /**
     * Format title with placeholders
     */
    private function format_title( $title ) {
        $format = $this->settings['title_format'];
        
        $replacements = array(
            '%title%'    => $title,
            '%sitename%' => get_bloginfo( 'name' ),
            '%tagline%'  => get_bloginfo( 'description' ),
        );

        return str_replace( 
            array_keys( $replacements ), 
            array_values( $replacements ), 
            $format 
        );
    }

    /**
     * Get title for current page
     */
    private function get_title() {
        // Homepage
        if ( is_front_page() || is_home() ) {
            $title = $this->settings['home_title'];
            if ( $title ) {
                return $title;
            }
            return get_bloginfo( 'name' );
        }

        // Singular
        if ( is_singular() ) {
            $post_id   = get_queried_object_id();
            $seo_title = get_post_meta( $post_id, '_simply_seo_title', true );
            
            if ( $seo_title ) {
                return $this->format_title( $seo_title );
            }
            
            return $this->format_title( get_the_title( $post_id ) );
        }

        // Archives
        if ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            if ( $term ) {
                $seo_title = get_term_meta( $term->term_id, '_simply_seo_title', true );
                if ( $seo_title ) {
                    return $this->format_title( $seo_title );
                }
            }
            return $this->format_title( single_term_title( '', false ) );
        }

        if ( is_author() ) {
            return $this->format_title( get_the_author() );
        }

        if ( is_post_type_archive() ) {
            return $this->format_title( post_type_archive_title( '', false ) );
        }

        // Search
        if ( is_search() ) {
            return $this->format_title(
                /* translators: %s: search query */
                sprintf( __( 'Search Results for: %s', 'simply-seo' ), get_search_query() )
            );
        }

        // 404
        if ( is_404() ) {
            return $this->format_title( __( 'Page Not Found', 'simply-seo' ) );
        }

        return get_bloginfo( 'name' );
    }

    /**
     * Get description for current page
     */
    private function get_description() {
        // Homepage
        if ( is_front_page() || is_home() ) {
            $description = $this->settings['home_description'];
            if ( $description ) {
                return $this->truncate_description( $description );
            }
            return $this->truncate_description( get_bloginfo( 'description' ) );
        }

        // Singular
        if ( is_singular() ) {
            $post_id     = get_queried_object_id();
            $description = get_post_meta( $post_id, '_simply_seo_description', true );
            
            if ( $description ) {
                return $this->truncate_description( $description );
            }

            // Fallback to excerpt or content
            $post = get_post( $post_id );
            if ( $post ) {
                if ( $post->post_excerpt ) {
                    return $this->truncate_description( $post->post_excerpt );
                }
                
                $content = wp_strip_all_tags( $post->post_content );
                $content = strip_shortcodes( $content );
                return $this->truncate_description( $content );
            }
        }

        // Category/Tag/Taxonomy
        if ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            if ( $term ) {
                // Check for custom SEO description
                $seo_description = get_term_meta( $term->term_id, '_simply_seo_description', true );
                if ( $seo_description ) {
                    return $this->truncate_description( $seo_description );
                }
                // Fallback to term description
                if ( $term->description ) {
                    return $this->truncate_description( $term->description );
                }
            }
        }

        return '';
    }

    /**
     * Truncate description to recommended length
     */
    private function truncate_description( $text, $length = 160 ) {
        $text = wp_strip_all_tags( $text );
        $text = preg_replace( '/\s+/', ' ', $text );
        $text = trim( $text );

        if ( strlen( $text ) <= $length ) {
            return $text;
        }

        $text = substr( $text, 0, $length );
        $text = substr( $text, 0, strrpos( $text, ' ' ) );
        
        return $text . '...';
    }

    /**
     * Get keywords for current page
     */
    private function get_keywords() {
        if ( ! is_singular() ) {
            return '';
        }

        $post_id  = get_queried_object_id();
        $keywords = get_post_meta( $post_id, '_simply_seo_keywords', true );

        return $keywords ? sanitize_text_field( $keywords ) : '';
    }

    /**
     * Get social image for current page
     */
    private function get_social_image() {
        // Singular pages
        if ( is_singular() ) {
            $post_id = get_queried_object_id();

            // 1. Custom social image
            $social_image_id = get_post_meta( $post_id, '_simply_seo_social_image', true );
            if ( $social_image_id ) {
                $image_url = wp_get_attachment_image_url( $social_image_id, 'large' );
                if ( $image_url ) {
                    return $image_url;
                }
            }

            // 2. Featured image
            if ( has_post_thumbnail( $post_id ) ) {
                $image_url = get_the_post_thumbnail_url( $post_id, 'large' );
                if ( $image_url ) {
                    return $image_url;
                }
            }
        }

        // Taxonomy archives
        if ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            if ( $term ) {
                $term_image_id = get_term_meta( $term->term_id, '_simply_seo_image', true );
                if ( $term_image_id ) {
                    $image_url = wp_get_attachment_image_url( $term_image_id, 'large' );
                    if ( $image_url ) {
                        return $image_url;
                    }
                }
            }
        }

        // Default image
        $default_image_id = $this->settings['default_image'];
        if ( $default_image_id ) {
            $image_url = wp_get_attachment_image_url( $default_image_id, 'large' );
            if ( $image_url ) {
                return $image_url;
            }
        }

        return '';
    }

    /**
     * Get current URL
     */
    private function get_current_url() {
        global $wp;
        return home_url( add_query_arg( array(), $wp->request ) );
    }

    /**
     * Filter robots meta
     */
    public function filter_robots( $robots ) {
        // Check singular posts
        if ( is_singular() ) {
            $post_id  = get_queried_object_id();
            $noindex  = get_post_meta( $post_id, '_simply_seo_noindex', true );
            $nofollow = get_post_meta( $post_id, '_simply_seo_nofollow', true );
            
            if ( $noindex ) {
                $robots['noindex'] = true;
                unset( $robots['max-image-preview'] );
            }
            
            if ( $nofollow ) {
                $robots['nofollow'] = true;
            }
        }

        // Check taxonomy archives
        if ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            if ( $term ) {
                $noindex  = get_term_meta( $term->term_id, '_simply_seo_noindex', true );
                $nofollow = get_term_meta( $term->term_id, '_simply_seo_nofollow', true );
                
                if ( $noindex ) {
                    $robots['noindex'] = true;
                    unset( $robots['max-image-preview'] );
                }
                
                if ( $nofollow ) {
                    $robots['nofollow'] = true;
                }
            }
        }

        // Check search page
        if ( is_search() ) {
            if ( $this->settings['noindex_search'] ) {
                $robots['noindex'] = true;
            }
            if ( $this->settings['nofollow_search'] ) {
                $robots['nofollow'] = true;
            }
        }

        // Check 404 page
        if ( is_404() ) {
            if ( $this->settings['noindex_404'] ) {
                $robots['noindex'] = true;
            }
            if ( $this->settings['nofollow_404'] ) {
                $robots['nofollow'] = true;
            }
        }

        return $robots;
    }

    /**
     * Get canonical URL for current page
     */
    private function get_canonical_url() {
        if ( ! is_singular() ) {
            return '';
        }

        $post_id   = get_queried_object_id();
        $canonical = get_post_meta( $post_id, '_simply_seo_canonical', true );

        return $canonical ? esc_url( $canonical ) : '';
    }

    /**
     * Filter WordPress canonical URL
     */
    public function filter_canonical_url( $canonical_url, $post ) {
        if ( ! $post ) {
            return $canonical_url;
        }

        $custom_canonical = get_post_meta( $post->ID, '_simply_seo_canonical', true );

        if ( $custom_canonical ) {
            return esc_url( $custom_canonical );
        }

        return $canonical_url;
    }
}
